/***************************************************************************
 *   Copyright (C) 2001 by Rick L. Vinyard, Jr.                            *
 *   rvinyard@cs.nmsu.edu                                                  *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU Lesser General Public License as        *
 *   published by the Free Software Foundation version 2.1.                *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU Lesser General Public      *
 *   License along with this library; if not, write to the                 *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA              *
 ***************************************************************************/
#ifndef CONEXUSREAD_ERROR_H
#define CONEXUSREAD_ERROR_H

#include <conexus/error.h>

namespace conexus
  {

  /**
   * Base class for all receiving errors.
   * @author Rick L Vinyard Jr
   * @ingroup errors
   */
  class read_error: public conexus_error
    {
    public:
    read_error(int e=0): conexus_error("Unknown error, socket receive failed.",e,"read")
      { }
    read_error(const std::string s, int e=0): conexus_error(s,e,"read")
      { }
    }
  ;

  namespace error
    {

    namespace read
      {

      /**
       * EBADF: Thrown when equivalent BSD socket API error is encountered. May be caught alone
       * or via parent class.
       * @author Rick L Vinyard Jr
       * @ingroup errors
       */
      class bad_fd: public read_error
        {
        public:
          bad_fd(): read_error("sockfd is not a valid descriptor.", EBADF)
          { }
        }
      ;

      /**
       * ECONNREFUSED
       * A remote host refused to allow the network connection (typically
       * because it is not running the requested service).
       * @author Rick L Vinyard Jr
       * @ingroup errors
       */
      class connection_refused: public read_error
        {
        public:
          connection_refused(): read_error("A remote host refused to allow the network connection.", ECONNREFUSED)
          { }
        }
      ;

      /**
       * ENOTCONN
       * The socket is associated with a connection-oriented protocol and
       * has not been connected (see connect(2) and accept(2)).
       * @author Rick L Vinyard Jr
       * @ingroup errors
       */
      class not_connected: public read_error
        {
        public:
          not_connected(): read_error("The socket is associated with a connection-oriented protocol and has not been connected", ENOTCONN)
          { }
        }
      ;

      /**
       * ENOTSOCK
       * The argument s is not a socket.
       * @author Rick L Vinyard Jr
       * @ingroup errors
       */
      class not_socket: public read_error
        {
        public:
          not_socket(): read_error("The argument s is not a socket.", ENOTSOCK)
          { }
        }
      ;

      /**
       * EAGAIN
       * The  socket  is  marked  non-blocking  and the receive operation
       * would block, or a receive timeout had been set and  the  timeout
       * expired before data was received.
       * @author Rick L Vinyard Jr
       * @ingroup errors
       */
      class would_block: public read_error
        {
        public:
          would_block(): read_error("The socket is marked non-blocking and  the  requested  operation would block.", EAGAIN)
          { }
        }
      ;

      /**
       * EINTR
       * The  receive  was interrupted by delivery of a signal before any
       * data were available.
       * @author Rick L Vinyard Jr
       * @ingroup errors
       */
      class interrupted: public read_error
        {
        public:
          interrupted(): read_error("The  receive  was interrupted by delivery of a signal before any data were available.", EINTR)
          { }
        }
      ;

      /**
       * EFAULT
       * An invalid user space address was specified for a parameter.
       * @author Rick L Vinyard Jr
       * @ingroup errors
       */
      class invalid_user_space_address: public read_error
        {
        public:
          invalid_user_space_address(): read_error("An invalid user space address was specified for a parameter.", EFAULT)
          { }
        }
      ;

      /**
       * EINVAL
       * Invalid argument passed.
       * @author Rick L Vinyard Jr
       * @ingroup errors
       */
      class invalid_argument: public read_error
        {
        public:
          invalid_argument(): read_error("Invalid argument passed.", EINVAL)
          { }
        }
      ;

      /**
       * ENOMEM
       * No memory available.
       * @author Rick L Vinyard Jr
       * @ingroup errors
       */
      class no_memory: public read_error
        {
        public:
          no_memory(): read_error("No memory available.", ENOMEM)
          { }
        }
      ;

      /**
       * Not opened.
       * @author Rick L Vinyard Jr
       * @ingroup errors
       */
      class not_opened: public read_error
      {
      public:
      not_opened(): read_error("Trying to read from a device that can't be opened.", -1)
        { }
      }
      ;

      /**
       * Not bound.
       * @author Rick L Vinyard Jr
       * @ingroup errors
       */
      class not_bound: public read_error
      {
      public:
      not_bound(): read_error("Trying to read from a socket that can't be bound.", -1)
        { }
      }
      ;


    }

  }

}

#endif // CONEXUSSEND_ERROR_H
