/***************************************************************************
 *   Copyright (C) 2001 by Rick L. Vinyard, Jr.                            *
 *   rvinyard@cs.nmsu.edu                                                  *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU Lesser General Public License as        *
 *   published by the Free Software Foundation version 2.1.                *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU Lesser General Public      *
 *   License along with this library; if not, write to the                 *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA              *
 ***************************************************************************/
#ifndef CONEXUSBIND_ERROR_H
#define CONEXUSBIND_ERROR_H

#include <conexus/error.h>

namespace conexus
  {

    /**
     * Base class for all binding errors.
     * @author Rick L Vinyard Jr
     * @ingroup errors
     */
    class bind_error: public conexus_error
      {
      public:
      bind_error(int e=0): conexus_error("Unknown error, socket cannot be bound.", e,"bind")
        { }
      bind_error(const std::string s, int e=0): conexus_error(s, e,"bind")
        { }
      }
    ;

  namespace error
  {

  namespace bind
      {

      /**
       * EBADF: Thrown when equivalent BSD socket API error is encountered. May be caught alone
       * or via parent class.
       * @author Rick L Vinyard Jr
       * @ingroup errors
       */
      class bad_fd: public bind_error
        {
        public:
          bad_fd(): bind_error("sockfd is not a valid descriptor.", EBADF)
          { }
        }
      ;

      /**
       * EINVAL: Thrown when equivalent BSD socket API error is encountered. May be caught alone
       * or via parent class.
       * @author Rick L Vinyard Jr
       * @ingroup errors
       */
      class bound_socket: public bind_error
        {
        public:
          bound_socket(): bind_error("The socket is already bound to an address.", EINVAL)
          { }
        }
      ;

      /**
       * EACCES: Thrown when equivalent BSD socket API error is encountered. May be caught alone
       * or via parent class.
       * @author Rick L Vinyard Jr
       * @ingroup errors
       */
      class access: public bind_error
        {
        public:
          access(): bind_error("The address is protected, and the user is not the super-user.", EACCES)
          { }
        }
      ;

      /**
       * ENOTSOCK: Thrown when equivalent BSD socket API error is encountered. May be caught alone
       * or via parent class.
       * @author Rick L Vinyard Jr
       * @ingroup errors
       */
      class not_socket: public bind_error
        {
        public:
          not_socket(): bind_error("Argument is a descriptor for a file, not a socket.", ENOTSOCK)
          { }
        }
      ;

      /**
       * EINVAL: Thrown when equivalent UNIX domain (AF_UNIX) API error is encountered. May be caught alone
       * or via parent class.
       * @author Rick L Vinyard Jr
       * @ingroup errors
       */
      class invalid_address: public bind_error
        {
        public:
          invalid_address(): bind_error("The addrlen is wrong, or the socket was not in the AF_UNIX family.", EINVAL)
          { }
        }
      ;

      /**
       * EROFS: Thrown when equivalent UNIX domain (AF_UNIX) API error is encountered. May be caught alone
       * or via parent class.
       * @author Rick L Vinyard Jr
       * @ingroup errors
       */
      class read_only_filesystem: public bind_error
        {
        public:
          read_only_filesystem(): bind_error("The socket inode would reside on a read-only file system.", EROFS)
          { }
        }
      ;

      /**
       * EFAULT: Thrown when equivalent UNIX domain (AF_UNIX) API error is encountered. May be caught alone
       * or via parent class.
       * @author Rick L Vinyard Jr
       * @ingroup errors
       */
      class segfault: public bind_error
        {
        public:
          segfault(): bind_error("addr points outside the user’s accessible address space.", EFAULT)
          { }
        }
      ;

      /**
       * ENAMETOOLONG: Thrown when equivalent UNIX domain (AF_UNIX) API error is encountered. May be caught alone
       * or via parent class.
       * @author Rick L Vinyard Jr
       * @ingroup errors
       */
      class name_too_long: public bind_error
        {
        public:
          name_too_long(): bind_error("addr is too long.", ENAMETOOLONG)
          { }
        }
      ;

      /**
       * ENOENT: Thrown when equivalent UNIX domain (AF_UNIX) API error is encountered. May be caught alone
       * or via parent class.
       * @author Rick L Vinyard Jr
       * @ingroup errors
       */
      class not_exist: public bind_error
        {
        public:
          not_exist(): bind_error("The file does not exist.", ENOENT)
          { }
        }
      ;

      /**
       * ENOMEM: Thrown when equivalent UNIX domain (AF_UNIX) API error is encountered. May be caught alone
       * or via parent class.
       * @author Rick L Vinyard Jr
       * @ingroup errors
       */
      class no_kernel_memory: public bind_error
        {
        public:
          no_kernel_memory(): bind_error("Insufficient kernel memory was available.", ENOMEM)
          { }
        }
      ;

      /**
       * ENOTDIR: Thrown when equivalent UNIX domain (AF_UNIX) API error is encountered. May be caught alone
       * or via parent class.
       * @author Rick L Vinyard Jr
       * @ingroup errors
       */
      class not_directory: public bind_error
        {
        public:
          not_directory(): bind_error("A component of the path prefix is not a directory.", ENOTDIR)
          { }
        }
      ;

      /**
       * ELOOP: Thrown when equivalent UNIX domain (AF_UNIX) API error is encountered. May be caught alone
       * or via parent class.
       * @author Rick L Vinyard Jr
       * @ingroup errors
       */
      class loop: public bind_error
        {
        public:
          loop(): bind_error("Too many symbolic links were encountered in resolving addr.", ELOOP)
          { }
        }
      ;

      /**
       * Thrown when bind is requested and udp socket doesn't have an address
       * @author Rick L Vinyard Jr
       * @ingroup errors
       */
      class no_address: public bind_error
        {
        public:
          no_address(): bind_error("cannot bind socket without an address.", -1)
          { }
        };

      /**
       * Thrown when bind is requested of the Socket class
       * @author Rick L Vinyard Jr
       * @ingroup errors
       */
      class no_socket_bind: public bind_error
      {
      public:
      no_socket_bind(): bind_error("The Socket class bind method cannot be called without an address.", -1)
        { }
      };

      /**
       * Thrown when bind is requested and a socket cannot be opened
       * @author Rick L Vinyard Jr
       * @ingroup errors
       */
      class not_open: public bind_error
      {
      public:
      not_open(): bind_error("Socket could not be opened for bind method.", -1)
        { }
      };

      }

  }

}

#endif // CONEXUSBIND_ERROR_H
