/***************************************************************************
 *   Copyright (C) 2001 by Rick L. Vinyard, Jr.                            *
 *   rvinyard@cs.nmsu.edu                                                  *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU Lesser General Public License as        *
 *   published by the Free Software Foundation version 2.1.                *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU Lesser General Public      *
 *   License along with this library; if not, write to the                 *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA              *
 ***************************************************************************/
#ifndef CONEXUSERROR_H
#define CONEXUSERROR_H

/**
 * @defgroup errors
 */

#include <stdexcept>
#include <errno.h>
#include <sstream>

namespace conexus
  {

  /**
   * Base class for all conexus library errors.
   * @author Rick L Vinyard Jr
   * @ingroup errors
   */
  class conexus_error: public std::runtime_error
    {
    public:
    conexus_error(const std::string s="Unknown error.", int e=0, const std::string c=""):
          std::runtime_error(s),
          m_error_number(e),
        m_error_string(s),
        m_class(c)
      {
        if (m_error_number == 0)
          m_error_number = errno;
        std::ostringstream ostr;
      ostr << "conexus:" << m_class << ":";
        if (m_error_number != 0)
          ostr << "errno[" << m_error_number << "]: ";
        else
          ostr << " ";
        ostr << m_error_string;
        m_return_string = ostr.str();
      }

      ~conexus_error() throw () { }

      virtual const char * what () const throw () {
        return m_return_string.c_str();
      }

    protected:
      int m_error_number;
      std::string m_return_string;
      std::string m_error_string;
      std::string m_class;
    }
  ;

  }

#include <conexus/error_bind.h>
#include <conexus/error_close.h>
#include <conexus/error_connect.h>
#include <conexus/error_listen.h>
#include <conexus/error_open.h>
#include <conexus/error_read.h>
#include <conexus/error_state.h>
#include <conexus/error_write.h>

namespace conexus {

typedef enum ErrorType {
ERROR_OPEN,
    ERROR_BIND,
    ERROR_CLOSE,
    ERROR_CONNECT,
    ERROR_LISTEN,
    ERROR_READ,
    ERROR_STATE,
    ERROR_WRITE,
  } ErrorType;

void throw_error(int error_num, ErrorType type) throw (conexus_error);

void throw_bind_error(int error_num) throw (bind_error);
void throw_close_error(int error_num) throw (close_error);
void throw_connect_error(int error_num) throw (connect_error);
void throw_listen_error(int error_num) throw (listen_error);
void throw_open_error(int error_num) throw (open_error);
void throw_read_error(int error_num) throw (read_error);
void throw_write_error(int error_num) throw (write_error);

}

#endif

