/***************************************************************************
 *   Copyright (C) 2001 by Rick L. Vinyard, Jr.                            *
 *   rvinyard@cs.nmsu.edu                                                  *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU Lesser General Public License as        *
 *   published by the Free Software Foundation version 2.1.                *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU Lesser General Public      *
 *   License along with this library; if not, write to the                 *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA              *
 ***************************************************************************/
#include "error.h"

namespace conexus {

void throw_error(int error_num, ErrorType type) throw (conexus_error) {
  switch (type) {
  case ERROR_BIND:
    throw_bind_error(error_num);
    break;
  case ERROR_CLOSE:
    throw_close_error(error_num);
    break;
  case ERROR_CONNECT:
    throw_connect_error(error_num);
    break;
  case ERROR_LISTEN:
    throw_listen_error(error_num);
    break;
  case ERROR_OPEN:
    throw_open_error(error_num);
    break;
  case ERROR_READ:
    throw_read_error(error_num);
    break;
  case ERROR_WRITE:
    throw_write_error(error_num);
    break;
  }

  throw conexus_error();
}

void throw_open_error(int error_num) throw (open_error) {
  switch (error_num) {
  case EPROTONOSUPPORT:
    throw error::open::protocol_not_supported();
  case EAFNOSUPPORT:
    throw error::open::af_not_supported();
  case ENFILE:
    throw error::open::kernel_memory();
  case EMFILE:
    throw error::open::process_file();
  case EACCES:
    throw error::open::access();
  case ENOBUFS:
  case ENOMEM:
    throw error::open::no_memory();
  case EINVAL:
    throw error::open::invalid_protocol();
  default:
  throw open_error();
  }
}

void throw_close_error(int error_num) throw (close_error) {
  switch (error_num) {
    case EBADF:
    throw error::close::bad_fd();
    break;
  case EINTR:
    throw error::close::interrupted();
    break;
  default:
  throw close_error();
  }
}

void throw_read_error(int error_num) throw (read_error) {
  switch (error_num) {
  case EBADF:
    throw error::read::bad_fd();
  case ECONNREFUSED:
    throw error::read::connection_refused();
  case ENOTCONN:
    throw error::read::not_connected();
  case ENOTSOCK:
    throw error::read::not_socket();
  case EAGAIN:
    throw error::read::would_block();
  case EINTR:
    throw error::read::interrupted();
  case EFAULT:
    throw error::read::invalid_user_space_address();
  case EINVAL:
    throw error::read::invalid_argument();
  case ENOMEM:
    throw error::read::no_memory();
  default:
     throw read_error();
  }
}

void throw_write_error(int error_num) throw (write_error) {
  switch (error_num) {
    case EWOULDBLOCK:
    throw error::write::would_block();
    case EBADF:
    throw error::write::bad_fd();
    case ECONNRESET:
    throw error::write::connection_reset();
    case EDESTADDRREQ:
    throw error::write::destination_address_required();
    case EFAULT:
    throw error::write::invalid_user_space_address();
    case EINTR:
    throw error::write::interrupted();
    case EINVAL:
    throw error::write::invalid_argument();
    case EISCONN:
    throw error::write::is_connected();
    case EMSGSIZE:
    throw error::write::message_size();
    case ENOBUFS:
    throw error::write::no_buffers();
    case ENOMEM:
    throw error::write::no_memory();
    case ENOTCONN:
    throw error::write::not_connected();
    case ENOTSOCK:
    throw error::write::not_socket();
    case EOPNOTSUPP:
    throw error::write::operation_not_supported();
    case EPIPE:
    throw error::write::pipe();
    case ECONNREFUSED:
    throw error::write::connection_refused();
    case EACCES:
      throw error::write::permission_denied();
  default:
    throw write_error();
  }
}

void throw_bind_error(int error_num) throw (bind_error) {
  switch (error_num) {
  case EBADF:
    throw error::bind::bad_fd();
  case EINVAL:
    throw error::bind::bound_socket();
  case EACCES:
    throw error::bind::access();
  case ENOTSOCK:
    throw error::bind::not_socket();
  case EROFS:
    throw error::bind::read_only_filesystem();
  case EFAULT:
    throw error::bind::segfault();
  case ENAMETOOLONG:
    throw error::bind::name_too_long();
  case ENOENT:
    throw error::bind::not_exist();
  case ENOMEM:
    throw error::bind::no_kernel_memory();
  case ENOTDIR:
    throw error::bind::not_directory();
  case ELOOP:
    throw error::bind::loop();
  default:
    throw bind_error();
  }
}

void throw_connect_error(int error_num) throw (connect_error) {
  switch (error_num) {
  case EBADF:
    throw error::connect::bad_fd();
  case EFAULT:
    throw error::connect::segfault();
  case ENOTSOCK:
    throw error::connect::not_socket();
  case EISCONN:
    throw error::connect::connected_socket();
  case ECONNREFUSED:
    throw error::connect::connection_refused();
  case ETIMEDOUT:
    throw error::connect::timeout();
  case ENETUNREACH:
    throw error::connect::network_unreachable();
  case EADDRINUSE:
    throw error::connect::address_in_use();
  case EINPROGRESS:
    throw error::connect::not_immediate();
  case EALREADY:
    throw error::connect::in_progress();
  case EAGAIN:
    throw error::connect::try_again();
  case EAFNOSUPPORT:
    throw error::connect::af_not_supported();
  case EACCES:
  case EPERM:
    throw error::connect::access();
  default:
    throw connect_error();
  }
}

void throw_listen_error(int error_num) throw (listen_error) {
  switch (error_num) {
  case EADDRINUSE:
    throw error::listen::address_in_use();
  case EBADF:
    throw error::listen::bad_fd();
  case ENOTSOCK:
    throw error::listen::not_socket();
  case EOPNOTSUPP:
    throw error::listen::operation_not_supported();
  default:
    throw listen_error();
  }
}

}
