#!/bin/bash

# This file is part of Clusterix, Copyright (C) 2004 Alessandro Manzini, 
# email: a.manzini@infogroup.it

# Clusterix is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# Clusterix is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with Clusterix; if not, write to the Free Software
# Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

# Clusterixwsd version 5.1
# 0. The cluster write information in local quorum file and a timestamp
# in the quorum file of other node
# 1. Put date and node in the log informations
# 2. Rewritten initializing sequence
# 3. added start service without cluster
# 4. added status information to private network
# 5. added killing hungup ssh writer processes
# 6. added comment for syntax using freebsd
# 7. substitute argument auxw with auxww to command ps for more long command
# 8. added backup interface for cluster hearbeat information
# 9. added use of public interface also for information in case
#    of fail of other private interface
# 10. added control on public interface of the node. It takes action
#     according the status of the public interface of both of the nodes
# 11. made a change to control routine. before that a node is called down 
#     it require now 2 fails one after 5 sec the other
# 12. Added support to cluster to function only with public interface
#     and rewritten use of chooseinterface
# 13. Change start command to take advantage of freebsd rtprio command
#     to set low priority to cluster process
# 14. Added remote start stop startservice stopservice on the other node
# 15. Added reading vip address from external file
# 16. Added control that the node are not both active at the same time
# 17. Added support to use rsh instead ssh to improve performance of
#     exchanging information of the cluster. Usefull with slow network.
# 18. Corrected an error in the lunch of remote stop in teh stopservicestop
# 19. Changed killremotestop. It wait a timeout before to kill the remote stop
#     process. It is necessary to give time to disk to dismount
# 20. Patched all the variables containing pid of process
# 21. Introduced number of retries before to say service down. (variable $countfailedservicenum and $failedtrysec)
# 22. Added stop of cluster without stopping the service
# 23. Added general control routine that make several tries of launching control program
# 24. Changed control_process script
# 25. Changed writestopinf function. Now it write STOPPED on remote node.
# 26. Changed the script to start,stop service. It restart the service if service dont start
# 27. Changed wirte on quorum device. Now it dont display the initailization date but the
#     last start service date of the node
# 28. All command are put in variables
# 29. It is used now a configuration file
# 30. You can specify operating system type Linux or FreeBSD without
#     any other setting changes in the routines of his file.
# 31. The writes in the quorum file are put in only one ssh command.
# 32. Added support for Solaris.
# 33. Changed control_process and control_control and script to pass ps and ps option
# 34. Changed killing of hanged processes remote writing
# 35. Changed start script to send mail alert if it fail. Also
# changed start script to pass all paremeters from conf file
# 36. Changed to pass others parameter for control start and stop
# 37. Added time in sec before to begin to check process after the start
#     of service.
# 38. Changed stop service script to introduce a second stop script
# 39. Stopservice now before stop also the controlservice routine
# 40. Changed the routine that decide if the other node is down
# 41. Added writedatenow and active as command argument of cluster
# to force wirite date and active status on quorum device
# 42. Insert possibility to mount disk (because before it was
# only possible with clusterix wuth shared disk)
# 43. Patched killremotestop
# 44. Added force crash of machine if the umount dont function
# 45. Added possibility to make fsck before to mount device
# 46. Changed the name of interface variable in configuration file
# 47. Introduce locks for start and stop sequence and change script.sh
# to mantain lock during multiple start stop sequence
# 48. Changed send arp procedure to permit virtual ip on more interfaces
# 49. Changed procedure to obtain macaddress.now macaddress is obtained
# dinamically by the interface setting.

# Configuration file
conffile="/opt/clusterix/clusterixwsd.conf"
. $conffile

# Functions

# OS depending functions


#Linux

# Read virtual ip addreess from external file
readexternalvipfilelinux () {
if [ -f $externalvipfile ]
then
count="1"
for ip in `$grep -v ^# $externalvipfile`
do
        eval vip${count}="$ip"
        vipinterfacenum=`$echo $ip | cut -d. -f4`
        eval netmask${count}="$netmaskpublicnetwork"
        eval broadcast${count}="$broadcastpublicnetwork"
        eval interfacenumber${count}="$vipinterfacenum"
        eval interface${count}="$interfacepublicnetwork"
        count=$(($count + 1))
done
else
$echo "$date $iam: $externalvipfile no such file. Cannot start cluster."
exit 1
fi
}

# set virtual ip address
setvipcommandformatlinux () {
count="$1"
interface="$(eval $echo \$interface${count})"
interfacenumber="$(eval $echo \$interfacenumber${count})"
vip="$(eval $echo \$vip${count})"
netmask="$($ifconfig $interface | $grep inet| $awk -F: '{print $4}')"
broadcast="$($ifconfig $interface | $grep inet| $awk  '{print $3}'| $awk -F: '{print $2}')"
iammacaddress="$($ifconfig $interface | $grep $interface | $awk '{print $5}')"
$echo "$date $iam: Configuring virtual ip address $vip ..."
$ifconfig $interface:$interfacenumber $vip netmask $netmask broadcast $broadcast up
$echo "$date $iam: Publishing macaddress $iammacaddress for interface $interface for $iam ..."
$send_arp $vip $iammacaddress $broadcast ffffffffffff $interface
}

# unset virtual ip address
unsetvipcommandformatlinux () {
count="$1"
interface="$(eval $echo \$interface${count})"
interfacenumber="$(eval $echo \$interfacenumber${count})"
$echo "$date $iam: Deconfiguring interface $interface:$interfacenumber ..."
$ifconfig $interface:$interfacenumber down
}

# Control if virtual ip address is up or down
controlactiveiplinux () {
count="$1"
interface="$(eval $echo \$interface${count})"
interfacenumber="$(eval $echo \$interfacenumber${count})"
controlipup=$($ifconfig | $grep "$interface:$interfacenumber")
}

# Kill all process on device. Take the devicetomount as argument
killprocesslinux () {
for pidopenfile in `$lsof $1 | $awk '{print $2}' |$uniq | $grep -v PID`; do if [ ! -z "$pidopenfile" ]; then $kill -9 $pidopenfile; fi; done
}


# define the ping command to establish when network is down
pingcommandlinux="$ping -b -c 3 -f -w 2"

# FreeBSD. The same as before but for FreeBSD

readexternalvipfilefreebsd () {
if [ -f $externalvipfile ]
then
count="1"
for ip in `$grep -v ^# $externalvipfile`
do
        eval vip${count}="$ip"
        eval interface${count}="$interfacepublicnetwork"
        count=$(($count + 1))
done
else
$echo "$date $iam: $externalvipfile no such file. Cannot start cluster."
exit 1
fi
}

setvipcommandformatfreebsd () {
count="$1"
interface="$(eval $echo \$interface${count})"
vip="$(eval $echo \$vip${count})"
$echo "$date $iam: Configuring virtual ip address $vip ..."
$echo "$date $iam: Publishing macaddress $iammacaddress for $iam ..."
$ifconfig $interface $vip netmask 255.255.255.255 alias
}

unsetvipcommandformatfreebsd () {
vip="$(eval $echo \$vip${count})"
count="$1"
interface="$(eval $echo \$interface${count})"
$echo "$date $iam: Deconfiguring interface $interface with vip $vip ..."
$ifconfig $interface $vip netmask 255.255.255.255 -alias
}

controlactiveipfreebsd () {
count="$1"
vip="$(eval $echo \$vip${count})"
controlipup=$($ifconfig | $grep "$vip")
}

killprocessfreebsd () {
for pidopenfile in `$fstat -f $1 | awk '{print $6}' | $grep -v INUM`; do if [ ! -z "$pidopenfile" ]; then kill -9 $pidopenfile; fi; done
}

pingcommandfreebsd="$ping -c 3 -f -t 2"

# Solaris

# Read virtual ip addreess from external file
readexternalvipfilesolaris () {
if [ -f $externalvipfile ]
then
count="1"
for ip in `$grep -v ^# $externalvipfile`
do
        eval vip${count}="$ip"
        vipinterfacenum=`$echo $ip | cut -d. -f4`
        eval netmask${count}="$netmaskpublicnetwork"
        eval broadcast${count}="$broadcastpublicnetwork"
        eval interfacenumber${count}="$vipinterfacenum"
        eval interface${count}="$interfacepublicnetwork"
        count=$(($count + 1))
done
else
$echo "$date $iam: $externalvipfile no such file. Cannot start cluster."
exit 1
fi
}

# set virtual ip address
setvipcommandformatsolaris () {
count="$1"
interface="$(eval $echo \$interface${count})"
interfacenumber="$(eval $echo \$interfacenumber${count})"
vip="$(eval $echo \$vip${count})"
netmask="$(eval $echo \$netmask${count})"
broadcast="$(eval $echo \$broadcast${count})"
$echo "$date $iam: Configuring virtual ip address $vip ..."
$ifconfig $interface:$interfacenumber plumb
$ifconfig $interface:$interfacenumber $vip up
$echo "$date $iam: Publishing macaddress $iammacaddress for $iam ..."
}

# unset virtual ip address
unsetvipcommandformatsolaris () {
count="$1"
interface="$(eval $echo \$interface${count})"
interfacenumber="$(eval $echo \$interfacenumber${count})"
$echo "$date $iam: Deconfiguring interface $interface:$interfacenumber ..."
$ifconfig $interface:$interfacenumber unplumb
}

# Control if virtual ip address is up or down
controlactiveipsolaris () {
count="$1"
interface="$(eval $echo \$interface${count})"
interfacenumber="$(eval $echo \$interfacenumber${count})"
controlipup="$($ifconfig -a | $grep "$interface:$interfacenumber")"
}

# Kill all process on device. Take the devicetomount as argument
killprocesssolaris () {
$fstat -k $1
}


# define the ping command to establish when network is down
pingcommandsolaris="$ping -c 1"


# Choose operating system
chooseos () {
case "$operatingsystem" in

Linux)
                controlactiveip=controlactiveiplinux
                unsetvipcommandformat=unsetvipcommandformatlinux
                setvipcommandformat=setvipcommandformatlinux
                readexternalvipfile=readexternalvipfilelinux
                killprocess=killprocesslinux
                pingcommand="$pingcommandlinux"
                vipdef=vipdeflinux
;;
FreeBSD)
                controlactiveip=controlactiveipfreebsd
                unsetvipcommandformat=unsetvipcommandformatfreebsd
                setvipcommandformat=setvipcommandformatfreebsd
                readexternalvipfile=readexternalvipfilefreebsd
                killprocess=killprocessfreebsd
                pingcommand="$pingcommandfreebsd"
                vipdef=vipdeffreebsd
;;
Solaris)
                controlactiveip=controlactiveipsolaris
                unsetvipcommandformat=unsetvipcommandformatsolaris
                setvipcommandformat=setvipcommandformatsolaris
                readexternalvipfile=readexternalvipfilesolaris
                killprocess=killprocesssolaris
                pingcommand="$pingcommandsolaris"
                vipdef=vipdefsolaris
;;
[Ww][Ii][Nn][Dd][Oo][Ww][Ss]*)
$echo "$date $iam: Windows OS? are you joking!?"
exit 1
;;
*)
$echo "$date $iam: wrong operating system!"
exit 1
;;
esac
}

# End of operating system depend function and variables 
###################################################################

# Set variable for the interfaces
setinterface () {
eval ${node1}ipprivatenetwork1="$node1ipprivatenetwork1"
eval ${node2}ipprivatenetwork1="$node2ipprivatenetwork1"
eval ${node1}ipprivatenetwork2="$node1ipprivatenetwork2"
eval ${node2}ipprivatenetwork2="$node2ipprivatenetwork2"
eval ${node1}ippublicnetwork="$node1ippublicnetwork"
eval ${node2}ippublicnetwork="$node2ippublicnetwork"
}

# Variable definition to start virtual ip address
# with freebsd send_arp and mac are not necessary.The gratuitous arp is send
# by ifconfig program
setvip () {
if [ $useexternalvipfile = "on" ]
then
$echo "$date $iam: Reading virtual ip addrees from $externalvipfile ..."
$readexternalvipfile
else
$vipdef
fi
}

# Kill of the remote command stop if it remain hanged
# pay attention to the pattern of grep
killremotestop () {
count="0"
sleep 4
while [ $count -lt $timeoutkillremotestop ]
do
sleep 1
pidremotestop=$($ps $psoption | $grep "$networktouse"| $grep "$pathcluster stopservice" | $grep -v grep| $awk '{print $2}')
if [ ! -z "$pidremotestop" ]
then
count=$(($count + 1))
$echo "$date $iam: Waiting for process $pidremotestop to stop other node. Number of try $count..."
else
$echo "$date $iam: Ok. The remote stopping process is terminated."
count="$timeoutkillremotestop"
fi
done
pidremotestop=$($ps $psoption | $grep "$networktouse"| $grep "$pathcluster stopservice" | $grep -v grep| awk '{print $2}')
if [ ! -z "$pidremotestop" ]
then
$echo "$date $iam: Timeout reached for process $pidremotestop to stop other node. I kill it."
$kill -9 $pidremotestop
fi
}
 
# Kill of the remote command write if it remain hanged
# pay attention to the pattern of grep.It must be as in writeinf function.
killremotewrite () {
pidremotewrite=$($ps $psoption | $grep "$ssh $networktouse $echo" | grep "$quorumfile" | $grep -v grep | $awk '{print $2}')
if [ ! -z "$pidremotewrite" ]
then
sleep 2
pidremotewrite=$($ps $psoption | $grep "$ssh $networktouse $echo" | grep "$quorumfile" | $grep -v grep | $awk '{print $2}')
if [ ! -z "$pidremotewrite" ]
then
$echo "$date $iam: Found hanged process with pid: $pidremotewrite ."
$echo "$date $iam: Killing hanged process with pid: $pidremotewrite ..."
$kill -9 $pidremotewrite
fi
fi
}

# function that check if umount command is failed or not.
# if it fails the cluster force a crash of the machine to
# prevent the mount of the disk on both of the node and to
# preserve data integrety of the disks
checkmountcommand () {
count="$1"
devicetomount="$(eval $echo \$devicetomount${count})"
$mount | grep "$devicetomount"
if [ $? -eq 0 ]
then
checkresult=1
$echo "$($datepath): Failed to umount device $devicetomount." >> $log
fi
}

# function that force a crash in the node
crashnode () {
$crashcommand
}

# Define the syntax of the command that mount the device
mountcommandformat () {
count="$1"
devicetomount="$(eval $echo \$devicetomount${count})"
mountpoint="$(eval $echo \$mountpoint${count})"
if [ "$fsckbeforemount" = "ON" ]; then $echo "$date $iam: Performing fsck on $devicetomount ..."; $fsck $fsckoption $devicetomount; $echo "$date $iam: Done." ; fi
$echo "$date $iam: Mounting $devicetomount on  $mountpoint"
$mount $mountoption $devicetomount $mountpoint
}

# Define the syntax of the command that umount the device
umountcommandformat () {
count="$1"
devicetomount="$(eval $echo \$devicetomount${count})"
$echo "$date $iam: Umounting $devicetomount..."
$umount $umountoption $devicetomount
}

# Define the syntax of the command that kill the processes over a filesystem
killprocessformat () {
count="$1"
devicetomount="$(eval $echo \$devicetomount${count})"
$echo "$date $iam: Killing processes on $devicetomount.."
$killprocess $devicetomount
}


# Find the names of the nodes over whom the software is running.

whoami () {
iam="$hostname"
if [ $iam = "$node1" ]
then
youare="$node2"
blocknum iam youare
else
youare="$node1"
blocknum youare iam
fi
} 

notbothactive () {
if [  "$iamactive" = "yes" ]  &&  [ "$youareactive" = "yes" ]
then
$echo "$date $iam: Found service duplication! All the nodes own the service! Forcing restart of all the nodes..."
startserviceforce
fi
}

checkroutine () {
trycountnum="$1"
tryprogram="$2"
tryfailedsec="$3"
trymessage="$4"
trycount=1
onoff="$(($trycountnum + 1))"

$tryprogram
if [ $? -ne 0 ]
then
	$echo "$($datepath): Found $trymessage failed. Number of tries: ${trycount}.  Retrying..." >> $log
trycount=$(($trycount + 1))
	while [ $trycount -le $trycountnum ]
	do 
	sleep $tryfailedsec
	$tryprogram
		if [ $? -ne 0 ]
		then
		$echo "$($datepath): Found $trymessage failed. Number of tries: ${trycount}.  Retrying..." >> $log
		else
		$echo "$($datepath): $trymessage is now good after $trycount number of tries.Ok..." >> $log
		trycount=$(($trycountnum + 5))
		fi
	trycount=$(($trycount + 1))
	done

	if [ "$trycount" -eq "$onoff" ]
	then
	$echo "$($datepath): $trymessage failed for $countnum times. I assume it is down..."  >> $log
	return 1
	else
	return 0
	fi
fi
}

# Establish which interface to use on the bases of configuration variables
interfaceactive () {
case "$PRNI" in
on)
	case "$BNI" in
	on)
	$echo "$date $iam: Cluster is using 2 private interface..."
	chooseinterface=chooseinterface3
	;;
	off)
	$echo "$date $iam: Cluster is using 1 private interface..."
	chooseinterface=chooseinterface2
	;;
 	*)
	$echo "$date $iam: Error in variables BNI. Possible values are on or off."	
	exit 1
	;;
	esac
;;
off)
	case "$BNI" in
        on)
	$echo "$date $iam: Error: backup interface cannot be on if private interface is off."
	exit 1
	;;
	off)
	$echo "$date $iam: Warning: Cluster have no private interface..."
	chooseinterface=chooseinterface1
	;;
 	*)
        $echo "$date $iam: Error in variables BNI. Possible values are on or off."
	exit 1
        ;;
	esac
;;
*)
$echo "$date $iam: Error in variables PRNI. Possible values are on or off."
exit 1
;;
esac
}

# Execute the ping on the other host
remoteping () {                    
killremotewrite
$ssh $networktouse "$pingcommand $broadcastpublicnetwork" 1> /dev/null 2> /dev/null
}

# Command to stop service and vip in the other node trought private network
remotestop () {
$chooseinterface
if [ $networktouse != "none" ]
then
killremotewrite
$ssh $networktouse "$pathcluster stopservice"
else
$echo "$date $iam: All available interfaces are down:  cannot stop service on node $youare."
fi
}

# Command to execute cluster command on other node
remote () {                                                            
command="$1"
$chooseinterface
if [ $networktouse != "none" ]
then
killremotewrite
$ssh $networktouse "$pathcluster $command > /dev/null 2>&1"
else
$echo "$date $iam: All available interfaces are down:  cannot start service on node $youare."
fi
}


# Check if private network 1 is up
checkprivatenetwork1 () {
$pingcommand $(eval $echo \$${youare}ipprivatenetwork1) 1> /dev/null 2> /dev/null
}

# Check if private backup network is up
checkprivatenetwork2 () {
$pingcommand $(eval $echo \$${youare}ipprivatenetwork2) 1> /dev/null 2> /dev/null
}

# Check if public interface is up
checkpublicnetwork () {
$pingcommand $broadcastpublicnetwork > /dev/null 2> /dev/null
}

# Check if public interface of other node is up
checkpublicothernode () {
$pingcommand $(eval $echo \$${youare}ippublicnetwork) 1> /dev/null 2> /dev/null
if [ $? -ne 0 ]                         
then
sleep 1
$pingcommand $(eval $echo \$${youare}ippublicnetwork) > /dev/null 2> /dev/null
fi
}

# Choose the network to use by status
chooseinterface3 () {
# Chech private network 1
checkprivatenetwork1
if [ $? -eq 0 ]
then
networktouse="$(eval $echo \$${youare}ipprivatenetwork1)"
else
$echo "$date $iam: Private network interface $(eval $echo \$${youare}ipprivatenetwork1) is down. Try to use an other network interface..."
# Try private network 2
	checkprivatenetwork2
		if [ $? -eq 0 ]
		then
		$echo "$date $iam: ...OK Private network interface $(eval $echo \$${youare}ipprivatenetwork2) is up. I use it ..."
		networktouse="$(eval $echo \$${youare}ipprivatenetwork2)"
		else
		$echo "$date $iam: Private network interface $(eval $echo \$${youare}ipprivatenetwork2) is down. Try to use public network interface..."
# Try to use public interface
	checkpublicothernode
			if [ $? -eq 0 ]
        		then
			$echo "$date $iam: ...OK Private network interface $(eval $echo \$${youare}ippublicnetwork) is up. I use it ..."
        		networktouse="$(eval $echo \$${youare}ippublicnetwork)"
			else
			$echo "$date $iam: Public network interface $(eval $echo \$${youare}ippublicnetwork) of node $youare is down. I cant communicate with $youare."
			networktouse="none"
			fi
		fi
	fi
}

chooseinterface2 () {
# Chech private network 1
checkprivatenetwork1
if [ $? -eq 0 ]
then
networktouse="$(eval $echo \$${youare}ipprivatenetwork1)"
else
$echo "$date $iam: Private network interface $(eval $echo \$${youare}ipprivatenetwork1) is down. Try to use an other network interface..."
checkpublicothernode
                if [ $? -eq 0 ]
                then
                $echo "$date $iam: ...OK Public network interface $(eval $echo \$${youare}ippublicnetwork) is up. I use it ..."
                networktouse="$(eval $echo \$${youare}ippublicnetwork)"
                else
                $echo "$date $iam: Public network interface $(eval $echo \$${youare}ippublicnetwork) of node $youare is down. I cant communicate with $youare."
                networktouse="none"                              
                fi
fi
}

chooseinterface1 () {
# Try to use public interface
checkpublicothernode
if [ $? -eq 0 ]
then
$echo "$date $iam: ...OK Private network interface $(eval $echo \$${youare}ippublicnetwork) is up. I use it ..."
networktouse="$(eval $echo \$${youare}ippublicnetwork)"
else
$echo "$date $iam: Public network interface $(eval $echo \$${youare}ippublicnetwork) of node $youare is down. I cant communicate with $youare."
networktouse="none"                              
fi
}

chooseinterfacestatus () {
checkpublicnetwork
if [ $? -eq 0 ]     
then
echo $echooption "${statusok}OK${statusclose}:		PUBLICN 		-------> $broadcastpublicnetwork is up" 1> $tty 2> /dev/null
else
echo $echooption "${statusko}ERROR${statusclose}:		PUBLICN 		-------> $broadcastpublicnetwork is down" 1> $tty 2> /dev/null
fi
checkpublicothernode
if [ $? -eq 0 ]
then
echo $echooption "${statusok}OK${statusclose}:		PUBLICP2P $(eval $echo \$${iam}ippublicnetwork) -------> $(eval $echo \$${youare}ippublicnetwork) is up" 1> $tty 2> /dev/null
else
echo $echooption "${statusko}ERROR${statusclose}:		PUBLICP2P $(eval $echo \$${iam}ippublicnetwork) ---------> $(eval $echo \$${youare}ippublicnetwork) is down" 1> $tty 2> /dev/null
fi
if [ $PRNI != "off" ]
then
checkprivatenetwork1
        if [ $? -eq 0 ]
        then
        echo $echooption "${statusok}OK${statusclose}:		PRIVATEN1 $(eval $echo \$${iam}ipprivatenetwork1) -------> $(eval $echo \$${youare}ipprivatenetwork1) is up" 1> $tty 2> /dev/null      
        else
        echo $echooption "${statusko}ERROR${statusclose}:	PRIVATEN1 $(eval $echo \$${iam}ipprivatenetwork1) -------> $(eval $echo \$${youare}ipprivatenetwork1) is down" 1> $tty 2> /dev/null
        fi
else
echo $echooption "${statuswn}WARNING${statusclose}:	PRIVATEN1 is disabled in configuration file.Not using it..." 1> $tty 2> /dev/null
fi
if [ $BNI != "off" ]
then
checkprivatenetwork2
        if [ $? -eq 0 ]
        then
        echo $echooption "${statusok}OK${statusclose}:		PRIVATEN2 $(eval $echo \$${iam}ipprivatenetwork2) -------> $(eval $echo \$${youare}ipprivatenetwork2) is up" 1> $tty 2> /dev/null
        else
        echo $echooption "${statusko}ERROR${statusclose}:	PRIVATEN2 $(eval $echo \$${iam}ipprivatenetwork2) -------> $(eval $echo \$${youare}ipprivatenetwork2) is down" 1> $tty 2> /dev/null
        fi
else
echo $echooption "${statuswn}WARNING${statusclose}:	PRIVATEN2 is disabled in configuration file.Not using it..." 1> $tty 2> /dev/null
fi
#networktouse="$(eval $echo \$${youare}ippublicnetwork)"


}

# Configure the virtual ip address and interface
startvip () {
count=1
while [ $count -le $numvip ]
do
vipcount="$(eval $echo \$vip${count})"
$controlactiveip "$count"
if [ ! -z "$vipcount" ] && [ -z "$controlipup" ]
then
$setvipcommandformat "$count"
fi
count=$(($count + 1))
done
}

# deconfigure virtual ip address and interface
stopvip () {
count=1
while [ $count -le $numvip ]
do
vipcount="$(eval $echo \$vip${count})"
$controlactiveip "$count"
if [ ! -z "$vipcount" ] && [ ! -z "$controlipup" ]
then
$unsetvipcommandformat "$count"
fi
count=$(($count + 1))
done
}

# Mount all the defined devices of shared disk
mountcommand () {
devicetomount
mountpoint
count=1
while [ $count -le $numdevice ]
do
devicetomount="$(eval $echo \$devicetomount${count})"
if [ ! -z "$devicetomount" ]
then
mountcommandformat "$count"
fi
count=$(($count + 1))
done
}

# Umount all the defined devices of shared disk
umountcommand () {
devicetomount
mountpoint
checkresult=0
count=1
while [ $count -le $numdevice ]
do
devicetomount="$(eval $echo \$devicetomount${count})"
if [ ! -z "$devicetomount" ]
then
umountcommandformat "$count"
checkmountcommand "$count"
fi
count=$(($count + 1))
done
if [ $checkresult -ne 0 ]
then
return 1
else
return 0
fi
}

# Kill all the process that use the mounted devices of shared disk
killcommand () {
devicetomount
mountpoint
count=1
while [ $count -le $numdevice ]
do
devicetomount="$(eval $echo \$devicetomount${count})"
if [ ! -z "$devicetomount" ]
then
killprocessformat "$count"
fi
count=$(($count + 1))
done
}


# Display information from quorum device
printinf () {
difference=$(($datenum - $datenumnode))
if [ $difference -ge $timeoutdisk ]
then printvar="${statusko}ERROR ERROR ERROR${statusclose}: $node is DOWN! Information is out of date!  ${statusko}ERROR ERROR ERROR${statusclose}"
else
printvar="${statusok}OK OK OK${statusclose}: $node is up and running... ${statusok}OK OK OK${statusclose}"
fi
$echo ""
echo $echooption "$printvar"
$echo "Cluster Information for node ### $node ### :"
$echo ""
$echo "Date of Last  Check of quorum device: 			\"$datenode\""
$echo "Secs from 1970 of Last Check of quorum device: 		$datenumnode"
$echo "Status of this node: 					$statusnode"
$echo "Active (if this node own the service):	 		$activenode"
$echo "Status of Other Node: 					$statusothernode"
$echo "Date of last service start on this node ($node):	\"$startserviceforcedate\""
$echo "Initialization Date of this node ($node): 		\"$initializationdatenode\""
$echo ""
}

printlocks () {
$echo "Cluster Locks Information for node ### $node ### :"
$echo ""
$echo "Start Locks: $lockstartservicebefore , $lockstartserviceafter"
$echo "Stop Locks:  $lockstopservice"
$echo ""
}

readlocks () {
if [ "$1" = "iam" ]
then
lockstartservicebefore=$($dd bs=$blocksize count=1 skip=$(eval $echo \$${1}blockstartservicebefore) conv=sync,notrunc if=$quorumfile)
lockstartserviceafter=$($dd bs=$blocksize count=1 skip=$(eval $echo \$${1}blockstartserviceafter) conv=sync,notrunc if=$quorumfile)
lockstopservice=$($dd bs=$blocksize count=1 skip=$(eval $echo \$${1}blockstopservice) conv=sync,notrunc if=$quorumfile)

else
lockstartservicebefore=$($ssh $networktouse $dd bs=$blocksize count=1 skip=$(eval $echo \$${1}blockstartservicebefore) conv=sync,notrunc if=$quorumfile)
lockstartserviceafter=$($ssh $networktouse $dd bs=$blocksize count=1 skip=$(eval $echo \$${1}blockstartserviceafter) conv=sync,notrunc if=$quorumfile)
lockstopservice=$($ssh $networktouse $dd bs=$blocksize count=1 skip=$(eval $echo \$${1}blockstopservice) conv=sync,notrunc if=$quorumfile)

fi
}

# Chech if a node is initialized.
checkinitialization () {
readinitialization iam
difference=$(($datenum - $iaminitializationdatenum))
if [ $difference -gt 0 ]
then
$echo "Node Cluster $iam is initialized at $aiminitializationdate ."
else
$echo "Node Cluster  $iam not initialized."
fi
}

checklockstartlocal () {
iamlockstartservicebefore=$($dd bs=$blocksize count=1 skip=$iamblockstartservicebefore conv=sync,notrunc if=$quorumfile)
iamlockstartserviceafter=$($dd bs=$blocksize count=1 skip=$iamblockstartserviceafter conv=sync,notrunc if=$quorumfile)
iamlockstopservice=$($dd bs=$blocksize count=1 skip=$iamblockstopservice conv=sync,notrunc if=$quorumfile)
if [ "$iamlockstartservicebefore" = "FREE" ] && [ "$iamlockstartserviceafter" = "FREE" ] && [ "$iamlockstopservice" = "FREE" ]
then
$echo "$date $iam: no start locks found on $iam. go on..."
else
$echo "$date $iam: Start locks found on $iam. Start sequence aborted."
exit 1
fi
}

checklockstartremote () {
$chooseinterface
if [ $networktouse != "none" ]
then
youarelockstartservicebefore=$($ssh $networktouse $dd bs=$blocksize count=1 skip=$youareblockstartservicebefore conv=sync,notrunc if=$quorumfile)
youarelockstartserviceafter=$($ssh $networktouse $dd bs=$blocksize count=1 skip=$youareblockstartserviceafter conv=sync,notrunc if=$quorumfile)
youarelockstopservice=$($ssh $networktouse $dd bs=$blocksize count=1 skip=$youareblockstopservice conv=sync,notrunc if=$quorumfile)
if [ "$youarelockstartservicebefore" = "FREE" ] && [ "$youarelockstartserviceafter" = "FREE" ] && [ "$youarelockstopservice" = "FREE" ]
then
$echo "$date $iam: no start locks found on $youare. go on..."
else
$echo "$date $iam: Start locks found on $youare. Start sequence aborted."
exit 1
fi
else
$echo "$date $iam: All avalaible network are down: Cannot check for locks on $youare.go on..."
fi
}

checklockstop () {
iamlockstartserviceafter=$($dd bs=$blocksize count=1 skip=$iamblockstartserviceafter conv=sync,notrunc if=$quorumfile)
iamlockstopservice=$($dd bs=$blocksize count=1 skip=$iamblockstopservice conv=sync,notrunc if=$quorumfile)
if [ "$iamlockstartserviceafter" = "FREE" ] && [ "$iamlockstopservice" = "FREE" ]
then
$echo "$date $iam: no stop locks found on $iam. go on..."
else
$echo "$date $iam: Stop locks found on $iam. Start sequence aborted."
exit 1
fi
}


# Start of the service
startservice () {
if [ $iamactive = "no" ]
then
startserviceforce
fi
}


# Force start service without checking who is active
startserviceforce () {
checklockstartlocal
checklockstartremote
$echo "$date $iam: Setting startbeforestop lock..."
writestartlocalbefore "LOCKED" 2> /dev/null
$echo "$date $iam: Resetting all services..."
$pathcluster stopservice
$echo "$date $iam: Setting startafterstop lock..."
writestartlocalafter "LOCKED" 2> /dev/null
$echo "$date $iam: Launching stop services on other node ($youare) ..."
remotestop &
killremotestop
mountcommand
startvip
$echo "$date $iam: Starting service on node $iam..."
mailsend
$script "$servicename" "$log" "$start_service_comand" "$stop_service_comand" "$includestringstart" "$excludestringstart" "$includestringstop" "$excludestringstop" "$timeoutstop" "$timeoutstart" "$ps" "$psoption" "$mailto" "$iam" "$countstartlimit" "$countstart" "$begincheck" "$stop_service_comand_2" "$timeoutstop_2" start
iamactive="yes"
$echo "$date" | $dd bs=$blocksize count=1 seek=$(eval $echo $iamblockstartserviceforcedate) conv=sync,notrunc of=$quorumfile
$echo "$date $iam: Unsetting startafterstop lock..."
writestartlocalafter "FREE" 2> /dev/null
$echo "$date $iam: Unsetting startbeforestop lock..."
writestartlocalbefore "FREE" 2> /dev/null
}

# Force stop service
stopserviceforce () {
checklockstop 2> /dev/null
$pathcluster stopcontrol
$echo "$date $iam: Setting stop lock..."
writestoplocal "LOCKED" 2> /dev/null
$echo "$date $iam: Stopping service on node $iam..."
$script "$servicename" "$log" "$start_service_comand" "$stop_service_comand" "$includestringstart" "$excludestringstart" "$includestringstop" "$excludestringstop" "$timeoutstop" "$timeoutstart" "$ps" "$psoption" "$mailto" "$iam" "$countstartlimit" "$countstart" "$begincheck" "$stop_service_comand_2" "$timeoutstop_2" stop
killcommand
checkroutine "$umountcountnum" umountcommand "$umountfailedsec" "$umountfaildmessage"
if [ $? -ne 0 ]
then
$echo "$date $iam: Failed to umount device. I force crash of node $iam to preserve data integrity..."
crashnode
fi
stopvip
iamactive="no"
$echo "$date $iam: Unsetting stop lock..."
writestoplocal "FREE" 2> /dev/null
}

# Stop service
stopservice () {
if [  $iamactive = "yes" ]
then
stopserviceforce
fi
}

# Stop of the cluster program that check quorum device
stopcluster () {
pidcluster=$($ps $psoption | $grep "$startclusterforeground" | $grep -v grep | $awk '{print $2}')
if [ ! -z "$pidcluster" ]
then
$kill -9 $pidcluster
$echo "$date $iam: Killed process \"$startclusterforeground\" with pid $pidcluster."
fi
$echo "$date $iam: Cluster stopped on node $iam."
}


# Write information on quorum device
writeinf () {
$echo "$date" | $dd bs=$blocksize count=1 seek=$(eval $echo \$${1}blockdate) conv=sync,notrunc of=$quorumfile
$echo "$datenum" | $dd bs=$blocksize count=1 seek=$(eval $echo \$${1}blockdatenum) conv=sync,notrunc of=$quorumfile
$echo "OK" | $dd bs=$blocksize count=1 seek=$(eval $echo \$${1}blockstatusnode) conv=sync,notrunc of=$quorumfile
$echo "$iamactive" | $dd bs=$blocksize count=1 seek=$(eval $echo \$${1}blockactive) conv=sync,notrunc of=$quorumfile
$echo "$youarestatus" | $dd bs=$blocksize count=1 seek=$(eval $echo \$${1}blockstatusothernode) conv=sync,notrunc of=$quorumfile
# For the checking of the other host, i write timestamp on the other node
$chooseinterface
if [ $networktouse != "none" ]
then
killremotewrite
$ssh $networktouse "$echo "$datenum" | $dd bs=$blocksize count=1 seek=$(eval $echo \$${1}blockdatenumothernode) conv=sync,notrunc of=$quorumfile; $echo "$iamactive" | $dd bs=$blocksize count=1 seek=$(eval $echo \$${1}blockactiveothernode) conv=sync,notrunc of=$quorumfile" &
else
$echo "$date $iam: All available interfaces are down:  cannot write information on node $youare."
fi
}

# Write information on quorum device before to stop cluster program
writestopinf () {
$echo "$iamactive" | $dd bs=$blocksize count=1 seek=$(eval $echo \$${1}blockactive) conv=sync,notrunc of=$quorumfile
$echo "STOPPED" | $dd bs=$blocksize count=1 seek=$(eval $echo \$${1}blockstatusnode) conv=sync,notrunc of=$quorumfile
$chooseinterface
if [ $networktouse != "none" ]
then
killremotewrite
$ssh $networktouse "$echo "no" | $dd bs=$blocksize count=1 seek=$(eval $echo \$${1}blockactiveothernode) conv=sync,notrunc of=$quorumfile" &
else
$echo "$date $iam: All available interfaces are down:  cannot write information on node $youare."
fi
}


# Send email alert
mailsend() {
mail -s "Start services on node $iam" $mailto <<EOF

${iam}: Launching start of services on node $iam...

EOF
}

checkstart () {
iamdatenum=$($dd bs=$blocksize count=1 skip=$iamblockdatenum conv=sync,notrunc if=$quorumfile)
difference=$(($datenum - $iamdatenum))
difference1=$(($checkfreq + $checkfreq))
if [ $difference -le $difference1 ]
then
$echo "$date $iam: Cluster already started..."
exit 1
fi
}

# Functions specific for cluster without shared disk. 

# Block information definition for the raw device
blocknum () {
# Information written by node1
eval ${1}blocknode=0
eval ${1}blockdate=1
eval ${1}blockdatenum=2
eval ${1}blockstatusnode=3
eval ${1}blockactive=4
eval ${1}blockstatusothernode=5
eval ${1}blockinitializationdatenum=6
eval ${1}blockinitializationdate=7
eval ${1}blockdatenumothernode=8
eval ${1}blockactiveothernode=9
eval ${1}blockstartserviceforcedate=10
eval ${1}blockstartservicebefore=11
eval ${1}blockstartserviceafter=12
eval ${1}blockstopservice=13

# Information taken by node2
eval ${2}blocknode=20
eval ${2}blockdate=21
eval ${2}blockdatenum=22
eval ${2}blockstatusnode=23
eval ${2}blockactive=24
eval ${2}blockstatusothernode=25
eval ${2}blockinitializationdatenum=26
eval ${2}blockinitializationdate=27
eval ${2}blockdatenumothernode=28
eval ${2}blockactiveothernode=29
eval ${2}blockstartserviceforcedate=30
eval ${2}blockstartservicebefore=31
eval ${2}blockstartserviceafter=32
eval ${2}blockstopservice=33
}

#  Read information from quorum file
readinf () {
if [ "$1" = "iam" ] 
then
node=$($dd bs=$blocksize count=1 skip=$(eval $echo \$${1}blocknode) conv=sync,notrunc if=$quorumfile)
datenumnode=$($dd bs=$blocksize count=1 skip=$(eval $echo \$${1}blockdatenum) conv=sync,notrunc if=$quorumfile)
datenode=$($dd bs=$blocksize count=1 skip=$(eval $echo \$${1}blockdate) conv=sync,notrunc if=$quorumfile)
statusnode=$($dd bs=$blocksize count=1 skip=$(eval $echo \$${1}blockstatusnode) conv=sync,notrunc if=$quorumfile)
activenode=$($dd bs=$blocksize count=1 skip=$(eval $echo \$${1}blockactive) conv=sync,notrunc if=$quorumfile)
statusothernode=$($dd bs=$blocksize count=1 skip=$(eval $echo \$${1}blockstatusothernode) conv=sync,notrunc if=$quorumfile)
initializationdatenumnode=$($dd bs=$blocksize count=1 skip=$(eval $echo \$${1}blockinitializationdatenum) conv=sync,notrunc if=$quorumfile)
initializationdatenode=$($dd bs=$blocksize count=1 skip=$(eval $echo \$${1}blockinitializationdate) conv=sync,notrunc if=$quorumfile)
startserviceforcedate=$($dd bs=$blocksize count=1 skip=$(eval $echo \$${1}blockstartserviceforcedate) conv=sync,notrunc if=$quorumfile)

else

node=$($ssh $networktouse $dd bs=$blocksize count=1 skip=$(eval $echo \$${1}blocknode) conv=sync,notrunc if=$quorumfile)
datenumnode=$($ssh $networktouse $dd bs=$blocksize count=1 skip=$(eval $echo \$${1}blockdatenum) conv=sync,notrunc if=$quorumfile)
datenode=$($ssh $networktouse $dd bs=$blocksize count=1 skip=$(eval $echo \$${1}blockdate) conv=sync,notrunc if=$quorumfile)
statusnode=$($ssh $networktouse $dd bs=$blocksize count=1 skip=$(eval $echo \$${1}blockstatusnode) conv=sync,notrunc if=$quorumfile)
activenode=$($ssh $networktouse $dd bs=$blocksize count=1 skip=$(eval $echo \$${1}blockactive) conv=sync,notrunc if=$quorumfile)
statusothernode=$($ssh $networktouse $dd bs=$blocksize count=1 skip=$(eval $echo \$${1}blockstatusothernode) conv=sync,notrunc if=$quorumfile)
initializationdatenumnode=$($ssh $networktouse $dd bs=$blocksize count=1 skip=$(eval $echo \$${1}blockinitializationdatenum) conv=sync,notrunc if=$quorumfile)
initializationdatenode=$($ssh $networktouse $dd bs=$blocksize count=1 skip=$(eval $echo \$${1}blockinitializationdate) conv=sync,notrunc if=$quorumfile)
startserviceforcedate=$($ssh $networktouse $dd bs=$blocksize count=1 skip=$(eval $echo \$${1}blockstartserviceforcedate) conv=sync,notrunc if=$quorumfile)
fi
}

# Read initialization information for quorum file
readinitialization () {
if [ "$1" = "iam" ]
then
set ${1}initializationdatenum=$($dd bs=$blocksize count=1 skip={1}blockinitializationdatenum conv=sync,notrunc if=$quorumfile)
set ${1}initializationdate=$($dd bs=$blocksize count=1 skip={1}blockinitializationdate conv=sync,notrunc if=$quorumfile)
else
set ${1}initializationdatenum=$($ssh $(eval $echo \$${youare}ipprivatenetwork1) $dd bs=$blocksize count=1 skip={1}blockinitializationdatenum conv=sync,notrunc if=$quorumfile)
set ${1}initializationdate=$($ssh $(eval $echo \$${youare}ipprivatenetwork1) $dd bs=$blocksize count=1 skip={1}blockinitializationdate conv=sync,notrunc if=$quorumfile)
fi
}


# Initialize a node
initialization () {
if [ ! -z "$quorumfile" ]
then
$chooseinterface
if [ $networktouse = "none" ]
then
$echo "$date $iam: All available interfaces are down:  cannot write information on node $youare."
else
touch $quorumfile
$ssh $networktouse "touch $quorumfile"
if [ "$1" = "iam" ]
then
$echo "$date $iam: Begin initialization quorum device for node ${!1}..."
$echo "${!1}" | $dd bs=$blocksize count=1 seek=$(eval $echo \$${1}blocknode) conv=sync,notrunc of=$quorumfile
$echo "Block $(eval $echo \$${1}blocknode) initialized on node ${!1} in $quorumfile."
$echo "$date" | $dd bs=$blocksize count=1 seek=$(eval $echo \$${1}blockdate) conv=sync,notrunc of=$quorumfile
$echo "Block $(eval $echo \$${1}blockdate) initialized on node ${!1} in $quorumfile."
$echo "$datenum" | $dd bs=$blocksize count=1 seek=$(eval $echo \$${1}blockdatenum) conv=sync,notrunc of=$quorumfile
$echo "Block $(eval $echo \$${1}blockdatenum) initialized on node ${!1} in $quorumfile."
$echo "Initializing.." | $dd bs=$blocksize count=1 seek=$(eval $echo \$${1}blockstatusnode) conv=sync,notrunc of=$quorumfile
$echo "Block $(eval $echo \$${1}blockstatusnode) initialized on node ${!1} in $quorumfile."
$echo "Initializing.." | $dd bs=$blocksize count=1 seek=$(eval $echo \$${1}blockactive) conv=sync,notrunc of=$quorumfile
$echo "Block $(eval $echo \$${1}blockactive) initialized on node ${!1} in $quorumfile."
$echo "Initializing.." | $dd bs=$blocksize count=1 seek=$(eval $echo \$${1}blockstatusothernode) conv=sync,notrunc of=$quorumfile
$echo "Block $(eval $echo \$${1}blockstatusothernode) initialized on node ${!1} in $quorumfile."
$echo "$datenum" | $dd bs=$blocksize count=1 seek=$(eval $echo \$${1}blockinitializationdatenum) conv=sync,notrunc of=$quorumfile
$echo "Block $(eval $echo \$${1}blockstatusothernode) initialized on node ${!1} in $quorumfile."
$echo "$date" | $dd bs=$blocksize count=1 seek=$(eval $echo \$${1}blockinitializationdate) conv=sync,notrunc of=$quorumfile
$echo "Block $(eval $echo \$${1}blockinitializationdate) initialized on node ${!1} in $quorumfile."
$ssh $networktouse "$echo "$datenum" | $dd bs=$blocksize count=1 seek=$(eval $echo \$${1}blockdatenumothernode) conv=sync,notrunc of=$quorumfile"
$echo "Block $(eval $echo \$${1}blockdatenumothernode) initialized on node $youare in $quorumfile."
$ssh $networktouse "$echo "Initializing.." | $dd bs=$blocksize count=1 seek=$(eval $echo \$${1}blockactiveothernode) conv=sync,notrunc of=$quorumfile"
$echo "Block $(eval $echo \$${1}blockactiveothernode) initialized on node $youare in $quorumfile."
$echo "$date" | $dd bs=$blocksize count=1 seek=$(eval $echo \$${1}blockstartserviceforcedate) conv=sync,notrunc of=$quorumfile
$echo "Block $(eval $echo \$${1}blockstartserviceforcedate) initialized on node ${!1} in $quorumfile."
$echo "FREE" | $dd bs=$blocksize count=1 seek=$(eval $echo \$${1}blockstartservicebefore) conv=sync,notrunc of=$quorumfile
$echo "Block $(eval $echo \$${1}blockstartservicebefore) initialized on node ${!1} in $quorumfile."
$echo "FREE" | $dd bs=$blocksize count=1 seek=$(eval $echo \$${1}blockstartserviceafter) conv=sync,notrunc of=$quorumfile
$echo "Block $(eval $echo \$${1}blockstartserviceafter) initialized on node ${!1} in $quorumfile."
$echo "FREE" | $dd bs=$blocksize count=1 seek=$(eval $echo \$${1}blockstopservice) conv=sync,notrunc of=$quorumfile
$echo "Block $(eval $echo \$${1}blockstopservice) initialized on node ${!1} in $quorumfile."
$echo "$date $iam: Node Cluster ${!1} is initialized at $date ."
else
$ssh $networktouse "$echo "${!1}" | $dd bs=$blocksize count=1 seek=$(eval $echo \$${1}blocknode) conv=sync,notrunc of=$quorumfile"
$echo "Block $(eval $echo \$${1}blocknode) initialized on node ${!1} in $quorumfile."
$ssh $networktouse "$echo "$date" | $dd bs=$blocksize count=1 seek=$(eval $echo \$${1}blockdate) conv=sync,notrunc of=$quorumfile"
$echo "Block $(eval $echo \$${1}blockdate) initialized on node ${!1} in $quorumfile."
$ssh $networktouse "$echo "$datenum" | $dd bs=$blocksize count=1 seek=$(eval $echo \$${1}blockdatenum) conv=sync,notrunc of=$quorumfile"
$echo "Block $(eval $echo \$${1}blockdatenum) initialized on node ${!1} in $quorumfile."
$ssh $networktouse "$echo "Initializing.." | $dd bs=$blocksize count=1 seek=$(eval $echo \$${1}blockstatusnode) conv=sync,notrunc of=$quorumfile"
$echo "Block $(eval $echo \$${1}blockstatusnode) initialized on node ${!1} in $quorumfile."
$ssh $networktouse "$echo "Initializing.." | $dd bs=$blocksize count=1 seek=$(eval $echo \$${1}blockactive) conv=sync,notrunc of=$quorumfile"
$echo "Block $(eval $echo \$${1}blockactive) initialized on node ${!1} in $quorumfile."
$ssh $networktouse "$echo "Initializing.." | $dd bs=$blocksize count=1 seek=$(eval $echo \$${1}blockstatusothernode) conv=sync,notrunc of=$quorumfile"
$echo "Block $(eval $echo \$${1}blockstatusothernode) initialized on node ${!1} in $quorumfile."
$ssh $networktouse "$echo "$datenum" | $dd bs=$blocksize count=1 seek=$(eval $echo \$${1}blockinitializationdatenum) conv=sync,notrunc of=$quorumfile"
$echo "Block $(eval $echo \$${1}blockinitializationdatenum) initialized on node ${!1} in $quorumfile."
$ssh $networktouse "$echo "$date" | $dd bs=$blocksize count=1 seek=$(eval $echo \$${1}blockinitializationdate) conv=sync,notrunc of=$quorumfile"
$echo "Block $(eval $echo \$${1}blockinitializationdate) initialized on node ${!1} in $quorumfile."
$echo "$datenum" | $dd bs=$blocksize count=1 seek=$(eval $echo \$${1}blockdatenumothernode) conv=sync,notrunc of=$quorumfile
$echo "Block $(eval $echo \$${1}blockdatenumothernode) initialized on node $iam in $quorumfile."
$echo "Initializing.." | $dd bs=$blocksize count=1 seek=$(eval $echo \$${1}blockactiveothernode) conv=sync,notrunc of=$quorumfile
$echo "Block $(eval $echo \$${1}blockactiveothernode) initialized on node $iam in $quorumfile."
$ssh $networktouse "$echo "$date" | $dd bs=$blocksize count=1 seek=$(eval $echo \$${1}blockstartserviceforcedate) conv=sync,notrunc of=$quorumfile"
$echo "Block $(eval $echo \$${1}blockstartserviceforcedate) initialized on node ${!1} in $quorumfile."
$ssh $networktouse "$echo "FREE" | $dd bs=$blocksize count=1 seek=$(eval $echo \$${1}blockstartservicebefore) conv=sync,notrunc of=$quorumfile"
$echo "Block $(eval $echo \$${1}blockstartservicebefore) initialized on node ${!1} in $quorumfile."
$ssh $networktouse "$echo "FREE" | $dd bs=$blocksize count=1 seek=$(eval $echo \$${1}blockstartserviceafter) conv=sync,notrunc of=$quorumfile"
$echo "Block $(eval $echo \$${1}blockstartserviceafter) initialized on node ${!1} in $quorumfile."
$ssh $networktouse "$echo "FREE" | $dd bs=$blocksize count=1 seek=$(eval $echo \$${1}blockstopservice) conv=sync,notrunc of=$quorumfile"
$echo "Block $(eval $echo \$${1}blockstopservice) initialized on node ${!1} in $quorumfile."
$echo "$date $iam: Node Cluster ${!1} is initialized at $date ."
fi
fi
else
$echo "$date $node you have to define the quorum file.." 1> $tty
$echo "$date $node Stop initialization.." 1> $tty
fi
}

# Write the actual date in the quorum device
writedate () {
$echo "$date" | $dd bs=$blocksize count=1 seek=$(eval $echo \$iamblockdate) conv=sync,notrunc of=$quorumfile
$echo "Block $(eval $echo \$iamblockdate) initialized on node ${iam} in $quorumfile to value \"$date\". " 1> $tty
$echo "$datenum" | $dd bs=$blocksize count=1 seek=$(eval $echo \$iamblockdatenum) conv=sync,notrunc of=$quorumfile
$echo "Block $(eval $echo \$iamblockdatenum) reinitialized on node ${iam} in $quorumfile to value $datenum." 1> $tty
}

writeactive () {
if [ "$1" != "yes" ] && [ "$1" != "no" ]
then
$echo "Usage: $pathcluster writeactive yes|no." 1> $tty
else
$echo "$1" | $dd bs=$blocksize count=1 seek=$(eval $echo \$iamblockactive) conv=sync,notrunc of=$quorumfile
$echo "Block $(eval $echo \$iamblockactive) initialized on node ${iam} in $quorumfile to value $1." 1> $tty
fi
}

writestartlocalbefore () {
$echo "$1" | $dd bs=$blocksize count=1 seek=$(eval $echo \$iamblockstartservicebefore) conv=sync,notrunc of=$quorumfile
}

writestartlocalafter () {
$echo "$1" | $dd bs=$blocksize count=1 seek=$(eval $echo \$iamblockstartserviceafter) conv=sync,notrunc of=$quorumfile
}

writestoplocal () {
$echo "$1" | $dd bs=$blocksize count=1 seek=$(eval $echo \$iamblockstopservice) conv=sync,notrunc of=$quorumfile
}


# Routine that control if the other node is up and writing on quorum device
controlquorum () {
datemom=$youaredatenum
youaredatenum=$($dd bs=$blocksize count=1 skip=$youareblockdatenumothernode conv=sync if=$quorumfile)
if [ -z "$datemom" ]; then datemom=1 ; fi
difference=$(($datenum - $youaredatenum))
if [ $difference -ge $timeoutdisk ]
then
	if [ $datemom -eq $youaredatenum ]
	then
	$echo "$date $iam: $youare not polling disk..."
	return 1
	else
	$echo "$date $iam: Probable time mismatch between the nodes."
        return 0
	fi
else
	if [ $difference -lt -10 ]
	then
		if [ $datemom -eq $youaredatenum ]
        	then
        	$echo "$date $iam: time mismatch but $youare seem not polling disk..."
        	return 1
        	else
       		$echo "$date $iam: Probable time mismatch between the nodes."
        	return 0
        	fi
	else
	return 0
	fi
fi
}

# Chech if the other node is accessing the quorum device in the last (timeout
# variable) secs. If no, the node start the service but only if the node is inactive.
control () {
checkroutine "$trycountnumpubnet" checkpublicnetwork "$trysecintpubnet" "$trymessagepubnet"
if [ $? -ne 0 ]
then
$echo "$date $iam: my public interface is down. Checking public interface of $youare..."
$chooseinterface                
        if [ $networktouse != "none" ]
        then
        remoteping
                if [ $? -eq 0 ]
                then          
                $echo "$date $iam: ...Public interface of $youare is up. So i stop service in $iam and pass them to $youare"  
                stopserviceforce
                else      
                $echo "$date $iam: ...Public interface of $youare is down too.All nodes are unreacheble."                     
                $echo "$date $iam: So i dont take any action."
                fi              
        else     
        $echo "$date $iam: All my interface are down."
        $echo "$date $iam: So i stop service on this node $iam."
        stopserviceforce        
        fi

else

	checkroutine "$trycountnumquorum" controlquorum "$trysecintquorum" "$trymessagequorum"
	if [ $? -ne 0 ]
	then
	youarestatus=KO
	startservice
	else
	youarestatus=OK
		if [ $youareactive = "no" ]
		then
		startservice
		fi
	notbothactive
	fi
fi
}

#  Check for the active node
activenode () {
count=1
iamactive="yes"
while [ $count -le $numvip ]
do
vipcount="$(eval $echo \$vip${count})"
$controlactiveip "$count"
if [  ! -z "$vipcount" ]  &&  [ -z "$controlipup" ]
then
iamactive="no"
fi
count=$(($count + 1))
done
youareactive=$($dd bs=$blocksize count=1 skip=$youareblockactiveothernode conv=sync,notrunc if=$quorumfile)
if [ "$youareactive" != "yes" ]
then
youareactive="no"
else
youareactive="yes"
fi
}

# Main
exec 1>> $log
whoami
chooseos
setinterface
setvip
$echo "$date $iam: execute command \"$pathcluster $1 $2\"..."
interfaceactive
if [ $? -ne 0 ]
then
exit 1
else

case "$1" in

startforeground)
checkstart  2> /dev/null 
$echo ""
$echo "$date $iam: Starting cluster..."
$echo ""
while (true)
do
date=$($datepath)
datenum=$($datenumpath)
activenode  2> /dev/null
control 2> /dev/null
writeinf iam 2> /dev/null
sleep $checkfreq
done
;;

stop)
activenode 2> /dev/null
stopservice 
writestopinf iam 2> /dev/null
stopcluster 
;;

status)
readinf iam 1> $tty 2> /dev/null
printinf 1> $tty 2> /dev/null
readlocks iam 1> $tty 2> /dev/null
printlocks 1> $tty 2> /dev/null
$chooseinterface
if [ $networktouse != "none" ]
then
readinf youare 1> $tty 2> /dev/null
printinf 1> $tty 2> /dev/null
readlocks youare 1> $tty 2> /dev/null
printlocks 1> $tty 2> /dev/null
else
echo $echooption "${statusko}ERROR${statusclose}: All avalaible network are down:  cannot take information about $youare status." 1> $tty 2> /dev/null
fi
chooseinterfacestatus
;;

clearlocks)
$echo "$date $iam: Unsetting startafterstop lock..." 1> $tty 2> /dev/null
writestartlocalafter "FREE"
$echo "$date $iam: Unsetting startbeforestop lock..." 1> $tty 2> /dev/null
writestartlocalbefore "FREE"
$echo "$date $iam: Unsetting stop lock..." 1> $tty 2> /dev/null
writestoplocal "FREE"

;;

setlocks)
$echo "$date $iam: Setting startafterstop lock..." 1> $tty 2> /dev/null
writestartlocalafter "LOCKED"
$echo "$date $iam: Setting startbeforestop lock..." 1> $tty 2> /dev/null
writestartlocalbefore "LOCKED"
$echo "$date $iam: Setting stop lock..." 1> $tty 2> /dev/null
writestoplocal "LOCKED"

;;

initialize)
initialization iam 1> $tty 2> /dev/null
initialization youare 1> $tty 2> /dev/null
;;

stopservice)
activenode 2> /dev/null
stopserviceforce
writestopinf iam 2> /dev/null
;;

startservice)
activenode 2> /dev/null
startserviceforce
;;

startserviceifnotactive)
activenode 2> /dev/null
startservice
;;

stopcluster)
activenode 2> /dev/null
stopcluster
remote stopclusterhere
;;

stopclusterhere)
stopcluster
;;

startcontrol)
$echo "$date $iam: Starting control process..." 1> $tty
$control_process "$control_control" "$pathcluster stopall" "$controlscript" "$log" "$checkprocfreq" "$countfailedservicenum" "$failedtrysec" "$trymessageservice" "$ps" "$psoption" > /dev/null 2>&1 &
;;
stopcontrol)
stringcontrol="$control_process $control_control"
pid=$($ps $psoption | $grep "$stringcontrol" | $grep -v grep | $awk '{print $2}')
stringcontrol1="$control_control $control_process"
pid1=$($ps $psoption | $grep "$stringcontrol1" | $grep -v grep | $awk '{print $2}')
if [ ! -z "$pid" ]
then
$kill -9 $pid $pid1
$echo "$date $iam: Killed processes $pid $pid1." 1> $tty
else
$echo "$date $iam: $control_process is not running." 1> $tty
fi
;;
statuscontrol)
stringcontrol="$control_process $control_control"
stringcontrol1="$control_control $control_process"
pid=$($ps $psoption | $grep "$stringcontrol" | $grep -v grep | $awk '{print $2}')
pid1=$($ps $psoption | $grep "$stringcontrol1" | $grep -v grep | $awk '{print $2}')
if [ ! -z "$pid" ]
then
$echo "$date $iam: $control_process is running with pid $pid." 1> $tty
else
$echo "$date $iam: $control_process is not running." 1> $tty
fi
if [ ! -z "$pid1" ]
then
$echo "$date $iam: $control_control is running with pid $pid1." 1> $tty
else
$echo "$date $iam: $control_control is not running." 1> $tty
fi
;;
startall)
$startclusterforeground &
sleep 1
$pathcluster startcontrol
;;
stopall)
$pathcluster stopcontrol
$pathcluster stop
;;
start)
$startclusterforeground &
;;
writedatenow)
writedate
;;
writeactive)
writeactive $2
;;
version) $echo "$version"  1> $tty 
;;
remote)
if [ $2 != "start" ] && [ $2 != "stop" ] && [ $2 != "startservice" ] && [ $2 != "stopservice" ] && [ $2 != "clearlocks" ] && [ $2 != "setlocks" ]
then
$echo "$version..." 1> $tty
$echo "Usage: $pathcluster remote {start|stop|startservice|stopservice|clearlocks|setlocks}" 1> $tty
$echo "" 1> $tty
else
remote $2
fi
;;

*)
	$echo "$version..." 1> $tty
        $echo "Usage: $pathcluster {start|stop|startforeground|startall|stopall|status|initialize|startservice|stopservice|stopcluster|stopclusterhere|startcontrol|stopcontrol|statuscontrol|writedatenow|writeactive||clearlocks|version}" 1> $tty
	$echo "" 1> $tty
	$echo "start:	 		start cluster in background: service and check quorum device." 1> $tty
	$echo "stop: 			stop cluster: service and check quorum device." 1> $tty
	$echo "status: 		Retrieve status information of the 2 nodes." 1> $tty
	$echo "initialize: 		Initialize quorum device." 1> $tty
	$echo "startforeground: 	start cluster in foreground: service and check quorum device." 1> $tty
	$echo "startcontrol: 		start processes that control the availibility of service." 1> $tty
	$echo "stopcluster: 		stop the cluster system without stopping the service on both nodes." 1> $tty
	$echo "stopclusterhere: 	stop the cluster system without stopping the service on this node." 1> $tty
	$echo "stopcontrol: 		stop processes that control the availibility of service." 1> $tty
	$echo "statuscontrol: 		status of processes that control the availibility of service." 1> $tty
	$echo "startservice: 		start only the service not the cluster system." 1> $tty
	$echo "startserviceifnotactive:	start only the service not the cluster system only if the node is not active." 1> $tty
	$echo "stopservice: 		stop only the service not the check of quorum device." 1> $tty
	$echo "stopall: 		stop service,stop check quorum device and stop processes that control the availibility of service." 1> $tty
	$echo "startall: 		start service,start check quorum device and start processes that control the availibility of service. Use this only if the cluster is down also in the other node.Otherwise use $pathcluster start." 1> $tty
	$echo "remote {start|stop|startservice|stopservice|clearlocks}: start,stop,startservice,stopservice,clearlocks on the other node." 1> $tty
	$echo "writeactive {yes|no}: Write yes,no for the status on quorum device." 1> $tty
	$echo "writedatenow: Write actual date on quorum device." 1> $tty
	$echo "clearlocks: Set FREE status on all the locks." 1> $tty
	$echo "version: 		Program version." 1> $tty
	        exit 1

esac

fi
