/*
 * This file is part of ClanBomber;
 * you can get it at "http://www.nongnu.org/clanbomber".
 *
 * Copyright (C) 1999-2004, 2007 Andreas Hundt, Denis Oliver Kropp
 * Copyright (C) 2009 Rene Lopez <rsl@members.fsf.org>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 */

#include <cmath>
#include <string>
#include "boost/format.hpp"

///#include <directfb.h>
#include "SDL.h"

#include "ClanBomber.h"
#include "GameStatus.h"
#include "ServerSetup.h"
#include "Client.h"
#include "GameObject.h"
#include "GameConfig.h"
#include "Timer.h"
#include "Bomber.h"
#include "Map.h"
#include "Server.h"
#include "Chat.h"
#include "Utils.h"

static GameStatus* game_status = NULL;
static SimpleTimer demo_mode_timer;
static float pokal_scroll_in = 800;

GameStatus::GameStatus(ClanBomberApplication* _app)
{
	app = _app;
	game_status = this;
}

void GameStatus::show()
{
	pokal_scroll_in = 800;
	demo_mode_timer.reset();
	if (end_of_game) {
          Resources::Gamestatus_klatsch()->play();
	}
	bool server_acts = ClanBomberApplication::is_server(); 
	bool client_acts = ClanBomberApplication::is_client();
	bool space_pressed = false;
	bool escape_pressed = false;

	bool space = true;//Is this really needed

	while (((server_acts || !client_acts) && !space_pressed) ||
           (!server_acts && client_acts && !ClanBomberApplication::get_client()->server_started_new_map())) {
		if (server_acts) {
			ClanBomberApplication::get_server()->disconnect_dead_clients();
			ClanBomberApplication::get_server()->send_SERVER_KEEP_ALIVE();
		}
		else if (client_acts) {
			if (escape_pressed && !space_pressed) {
				ClientSetup::end_session();
				ClanBomberApplication::get_client()->send_CLIENT_DISCONNECT();
				break;
			}
			else {
				ClanBomberApplication::get_client()->disconnect_from_server();
				ClanBomberApplication::get_client()->send_CLIENT_KEEP_ALIVE();
			}
			space_pressed = false;
			escape_pressed = false;
		}
		if (!end_of_game && server_acts && ClanBomberApplication::get_server()->is_in_demo_mode()) {
			if (demo_mode_timer.elapsed()>NET_SERVER_PAUSE_MILLISECONDS_BETWEEN_MAPS) {
				break;
			}
		}
		if (!server_acts && client_acts && ClanBomberApplication::get_client()->end_game()) {
			break;
		}
		//DFBInputEvent evt;
		SDL_Event event;
		//while (keybuffer->GetEvent( keybuffer, DFB_EVENT(&evt) ) == DFB_OK) {
		while (SDL_PollEvent(&event)) {
		  if (event.type == SDL_KEYDOWN) {
		    switch (event.key.keysym.sym) {
		    case SDLK_BACKSPACE:
		      if (server_acts) {
			Chat::show();
			ServerSetup::enter_chat_message(true);
			ClanBomberApplication::get_client()->reset_new_chat_message_arrived();
			Chat::hide();
		      }
		      else if (client_acts) {
			Chat::show();
			ClientSetup::enter_chat_message(true);
			ClanBomberApplication::get_client()->reset_new_chat_message_arrived();
			Chat::hide();
		      }
		      break;
		    case SDLK_SPACE:
		      space_pressed = true;
		    case SDLK_ESCAPE:
		      escape_pressed = true;
		    case SDLK_LEFT://Is a need for this
		      //space = DIKS_DOWN;
		      break;
		    default: ;
		    }
		  }
		}
		draw();
	}
	CB_FillScreen(0, 0, 0);
	CB_Flip();
	CB_FillScreen(0, 0, 0);
	CB_Flip();
}

GameStatus::~GameStatus()
{
}

void GameStatus::analyze_game()
{
	end_of_game = false;
	winner = NULL;
	//CL_Iterator<Bomber> bomber_object_counter(app->bomber_objects);
	for(std::list<Bomber*>::iterator bomber_object_iter = app->bomber_objects.begin();
	    bomber_object_iter != app->bomber_objects.end();
	    bomber_object_iter++) {
	  if (!(*bomber_object_iter)->is_dead()) {
	    winner = (*bomber_object_iter);
	  }
	  if ((*bomber_object_iter)->get_points() == Config::get_points_to_win() ) {
	    end_of_game = true;
	  }
	}
}

bool GameStatus::get_end_of_game()
{
	return end_of_game;
}

void GameStatus::draw()
{
  //using namespace std;
  bool server_acts = ClanBomberApplication::is_server();
  bool client_acts = ClanBomberApplication::is_client();
  Resources::Gamestatus_background()->blit(0, 0);
  //primary->SetColor( primary, 0xFF, 0xFF, 0xFF, 0xFF );
  if (server_acts || !client_acts) {
    if (server_acts && Chat::enabled()) {
      if (ClanBomberApplication::get_server()->is_in_demo_mode() && !game_status->end_of_game) {
	float seconds = (NET_SERVER_PAUSE_MILLISECONDS_BETWEEN_MAPS - demo_mode_timer.elapsed()) / 1000;
	std::string nstr = str(boost::format(_("PRESS SPACE TO CONTINUE  (autostart %.02f s)")) % seconds);
	//primary->DrawString( primary, CL_String(), -1, 360, 570, DSTF_TOPCENTER );
	Resources::Font_small()->render(nstr, 360, 570,
                                        cbe::FontAlignment_0topcenter);
      }
      else {
	//primary->DrawString( primary, "PRESS SPACE TO CONTINUE", -1, 400, 570, DSTF_TOPCENTER );
	Resources::Font_small()->render(_("PRESS SPACE TO CONTINUE"), 400, 570,
                                        cbe::FontAlignment_0topcenter);
      }
    }
    else {
      //primary->DrawString( primary, "PRESS SPACE TO CONTINUE", -1, 400, 570, DSTF_TOPCENTER );
      Resources::Font_small()->render(_("PRESS SPACE TO CONTINUE"), 400, 570,
                                      cbe::FontAlignment_0topcenter);
    }
  }
  if (!game_status->winner) {
    //primary->DrawString( primary, "Draw Game", -1, 500, 40, DSTF_TOPCENTER );
    Resources::Font_big()->render(_("Draw Game"), 500, 40,
                                  cbe::FontAlignment_0topcenter);
  }
  else {
    std::string nstr = str(boost::format(_("%s won")) % game_status->winner->get_name());
    //primary->DrawString( primary, game_status->winner->get_name() + " won", -1, 500, 40, DSTF_TOPCENTER );
    Resources::Font_big()->render(nstr, 500, 40,
                                  cbe::FontAlignment_0topcenter);
  }
  int nr = 0;
  //CL_Iterator<Bomber> bomber_object_counter(game_status->app->bomber_objects);
  for(std::list<Bomber*>::iterator bomber_object_iter = game_status->app->bomber_objects.begin();
      bomber_object_iter != game_status->app->bomber_objects.end() && nr < 8;
      bomber_object_iter++) {
    nr++;
    if (!client_acts) {
      (*bomber_object_iter)->act();
    }
    //primary->DrawString( primary, bomber_object_counter()->get_name(), -1, 70, 157+nr*43, DSTF_TOPLEFT );
    Resources::Font_big()->render((*bomber_object_iter)->get_name(), 70,
                                  157+nr*43, cbe::FontAlignment_0topleft);
    if ((*bomber_object_iter) == game_status->winner) {
      if (!game_status->end_of_game) {
	GameObject* obj = static_cast<GameObject*>(*bomber_object_iter);
	if (obj != NULL) {
	  obj->show( 5+(nr%2)*20, 150+nr*43);
	}
	for (int i=0; i<(*bomber_object_iter)->get_points()-1; i++) {
	  Resources::Gamestatus_tools()->put_screen( 300+i*43, 150+nr*43, 0 );
	}
	Resources::Gamestatus_tools()->put_screen( std::max(257+(*bomber_object_iter)->get_points()*43, (int)pokal_scroll_in), 150+nr*43, 0 );
      }
      else {   
	float scalefactor;
	scalefactor = (float)(sin(Timer::get_time()/100.0f)+2.0f)/2;
	GameObject* obj = static_cast<GameObject*>(*bomber_object_iter);
	if (obj != NULL) {
	  obj->show( (int)(5+(800-pokal_scroll_in)/3), (int)(150+nr*43),scalefactor);
	}
	for (int i=0; i<(*bomber_object_iter)->get_points()-1; i++) {
	  Resources::Gamestatus_tools()->put_screen((int)(320+i*43-20*scalefactor), (int)(170+nr*43-20*scalefactor),scalefactor,scalefactor, 0 );
	}
	Resources::Gamestatus_tools()->put_screen( std::max((int)(277+(*bomber_object_iter)->get_points()*43-20*scalefactor), (int)pokal_scroll_in), (int)(170+nr*43-20*scalefactor), scalefactor,scalefactor, 0 );
      }
    }
    else {   
      GameObject* obj = static_cast<GameObject*>(*bomber_object_iter);
      if (obj != NULL) {
	obj->show( 5+(nr%2)*20, 150+nr*43);
      }
      for (int i=0; i<(*bomber_object_iter)->get_points(); i++) {
	Resources::Gamestatus_tools()->put_screen( 300+i*43, 150+nr*43, 0 );
      }
    }
  }
  if (game_status->end_of_game) {
    //primary->DrawString( primary, "the Match", -1, 500, 80, DSTF_TOPCENTER );
    Resources::Font_big()->render("the Match", 500, 80,
                                  cbe::FontAlignment_0topcenter);
  }
  else {
    if (server_acts) {
      game_status->app->get_server()->send_update_messages_to_clients(ClanBomberApplication::get_server_frame_counter());
    }
    //primary->DrawString( primary, "NEXT LEVEL", -1, 785, 10, DSTF_TOPRIGHT );
    Resources::Font_small()->render(_("NEXT LEVEL"), 785, 10,
                                    cbe::FontAlignment_0topright);
    if (!server_acts && client_acts) {
      std::string temp_string = game_status->app->map->current_server_map_name;
      //TODO should be uppercasing eliminated      
      //temp_string.to_upper();
      //primary->DrawString( primary, temp_string, -1, 785, 125, DSTF_TOPRIGHT );
      Resources::Font_small()->render(temp_string, 785, 125,
                                      cbe::FontAlignment_0topright);
    }
    else {
      std::string temp_string = game_status->app->map->get_name();
      //temp_string.to_upper();
      //primary->DrawString( primary, temp_string, -1, 785, 125, DSTF_TOPRIGHT );
      Resources::Font_small()->render(temp_string, 785, 125,
                                      cbe::FontAlignment_0topright);
    }
    game_status->app->map->show_preview(790-119,30,0.18f);
  }
  Chat::draw();
  if ((server_acts || client_acts) && !Chat::enabled()) {
    //primary->SetColor(primary, 255, 255, 255, 255);
    //primary->DrawString(primary, "BACKSPACE:", -1, 640, 580, DSTF_TOPLEFT);
    Resources::Font_small()->render(_("BACKSPACE:"), 640, 580,
                                    cbe::FontAlignment_0topleft);
    //primary->DrawString(primary, "CHAT", -1, 745, 580, DSTF_TOPLEFT);
    Resources::Font_small()->render(_("CHAT"), 745, 580,
                                    cbe::FontAlignment_0topleft);
  }
  if (server_acts) {
    if (!Chat::enabled() && pokal_scroll_in <= 100) {
      ServerSetup::show_chat_request();
    }
  }
  else if (client_acts) {
    if (!Chat::enabled() && pokal_scroll_in <= 100) {
      ClientSetup::show_chat_request();
    }
  }
  CB_Flip();
  if (pokal_scroll_in > 100) {
    if (!game_status->end_of_game) {
      pokal_scroll_in -= Timer::time_elapsed(true)*1000.0f;
    }
    else {   
      pokal_scroll_in -= Timer::time_elapsed(true)*333.0f;
    }
  }
}
