/* Copyright (c) 2017-2018 Marvin Eagle <trailblazer77_2000@yahoo.com>
 * 
 *		This file is part of Chord Sequencer.
 * 
 * 		Chord Sequencer is a program to create and interactively play
 * 		series of chords in the live performance, controlling chord
 * 		change by pedal (or other similar device). It uses midi and
 * 		must be connected to an external synthesizer. 
 *
 *    Chord Sequencer is free software: you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation, either version 3 of the License, or
 *    (at your option) any later version.
 *
 *    Chord Sequencer is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *   along with Chord Sequencer.  If not, see <https://www.gnu.org/licenses/>.
 * 
 * 		Author: Marvin Eagle <trailblazer77_2000@yahoo.com>
 * 
*/

#include "note.h"
#include <stdio.h>
#include <iostream>
//  g++  -c note.cpp -o note.o `pkg-config gtkmm-3.0 --cflags --libs`

Note::Note(int midi_num)
{	
	normal_x = 0;
	midi_number = midi_num;
	midi_instrument = 0;
	if(midi_number != -1000) 
	{
		if (is_black(midi_num) )
		{
			black = true;
			display = 3; // #
		}
		else
		{
			black = false;
			display = 2; // regular
		}
		position_x = 0; 
		position_y = calculate_y(midi_num); 
		
		display_options = find_display_options(midi_num);
	//	printf("display options are: %d\n", display_options);
	}
	else // dummy note
	{
		position_x = -1000;
		position_y = -1000;
		display = 2;
	}
	selected = false;
	velocity = 50;
	midi_channel = 0;
}

// this function initializes display_options number where each digit
// corresponds to an appropriate option, starting from bottom of the
// radio buttons list (highest number)
int Note::find_display_options(int midi_num)
{
	int simplified_midi_num =  midi_num % 12;
	
	switch (simplified_midi_num)
	{
		case 0: // C
			return 320;
		case 1: // C sharp or D flat
			return 431;
		case 2: // D
			return 420;
		case 3: // D sharp or E flat
			return 310;
		case 4: // E
			return 421;
		case 5: // F
			return 320;
		case 6: // F sharp of G flat
			return 431;
		case 7: // G
			return 420;
		case 8: // G sharp or A flat
			return 31;
		case 9: // A
			return 420;
		case 10: // A sharp or B flat
			return 310;
		case 11: // B
			return 421;
	}
}

// This function uses values of display and midi_number to change position_y
// after display value changed by pressing radio buttons
void Note::update_display()
{
	if(display == 0) // double flat
		position_y = calculate_y(midi_number + 2);
	else if(display == 1) // flat
		position_y = calculate_y(midi_number + 1);
	else if(display == 2) // normal
		position_y = calculate_y(midi_number);
	else if(display == 3) // sharp
		position_y = calculate_y(midi_number - 1);
	else if(display == 4) // double sharp
		position_y = calculate_y(midi_number - 2);
}

int Note::calculate_y(int midi_num)
{	
	// black keys are used as sharp notes
	if (midi_num >= 36 && midi_num <= 42) // lowest C is 36
	{
		// interval is 15 here
		switch (midi_num)
		{
			case 42: case 41:
				return 436;
			case 40: 
				return 451;
			case 39: case 38:
				return 466;
			case 37: case 36:
				return 481;
		}
	}
	else if(midi_num >= 43 && midi_num <= 78)
	{
		// 78 is F# (highest horizontal line) with y = 61
		// 43 is G (lowest horizontal line) with y = 421
		// interval is 18 here
		switch (midi_num)
		{
			case 78: case 77:
				return 61;
			case 76:
				return 79;
			case 75: case 74:
				return 97;
			case 73: case 72:
				return 115;
			case 71:
				return 133;
			case 70: case 69:
				return 151;
			case 68: case 67:
				return 169;
			case 66: case 65:
				return 187;
			case 64: 
				return 205;
			case 63: case 62:
				return 223;
			case 61: case 60:
				return 241;
			case 59:
				return 259;
			case 58: case 57:
				return 277;
			case 56: case 55:
				return 295;
			case 54: case 53:
				return 313;
			case 52:
				return 331;
			case 51: case 50:
				return 349;
			case 49: case 48:
				return 367;
			case 47:
				return 385;
			case 46: case 45:
				return 403;
			case 44: case 43:
				return 421;
		}
	}
	else if(midi_num >= 79 && midi_num <= 84)
	{
		// 84 is top C with y = 1
		// 79 is G  with y = 46
		//  interval is 15 here
		switch(midi_num)
		{
			case 84:
				return 1;
			case 83: 
				return 16;
			case 82: case 81:
				return 31;
			case 80: case 79:
				return 46;
		}
	}
		
}

bool Note::is_black(int midi_num)
{
	int simplified_midi_num =  midi_num % 12;
	switch (simplified_midi_num)
	{
		case 1: case 3: case 6: case 8: case 10:
			return true;
		case 0: case 2: case 4: case 5: case 7:
		case 9: case 11:
			return false;
	}
}

// This function uses values of display and normal_x to calculate new
// postion_x
void Note::update_x()
{
	if(display == 0) // double flat
		position_x = normal_x + 7;
	else if(display == 1) //flat
		position_x = normal_x + 5;
	else if(display == 2) //normal
		position_x = normal_x;
	else if(display == 3) //sharp
		position_x = normal_x + 5;
	else if(display == 4) //double sharp
		position_x = normal_x + 7;
}

Note::~Note()
{ 
	
}

