/* Copyright (c) 2017-2018 Marvin Eagle <trailblazer77_2000@yahoo.com>
 * 
 *		This file is part of Chord Sequencer.
 * 
 * 		Chord Sequencer is a program to create and interactively play
 * 		series of chords in the live performance, controlling chord
 * 		change by pedal (or other similar device). It uses midi and
 * 		must be connected to an external synthesizer. 
 *
 *    Chord Sequencer is free software: you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation, either version 3 of the License, or
 *    (at your option) any later version.
 *
 *    Chord Sequencer is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *   along with Chord Sequencer.  If not, see <https://www.gnu.org/licenses/>.
 * 
 * 		Author: Marvin Eagle <trailblazer77_2000@yahoo.com>
 * 
*/

#include "mybutton.h"
#include <gdkmm/general.h>  // for cairo helper functions
#include <iostream>
#include <cstring>
#include <string>
#include <pangomm/fontdescription.h>
#include <glibmm/ustring.h>
#include <config.h>
#include "gettext.h"

#define _(string) gettext (string)

// The whole project is 7000 lines in 19 files
// Author : Marvin Eagle (except for RtMidi library, which has it's own authors)
// Version Sequencer 1.0 completed Dec. 19, 2017

//   g++  -c mybutton.cpp -o mybutton.o `pkg-config gtkmm-3.0 --cflags --libs`
//   g++  note.o mybutton.o mywidget.o myarea.o main.o examplewindow.o -o main `pkg-config gtkmm-3.0 --cflags --libs`

MyButton::MyButton()
  //The GType name will actually be gtkmm__CustomObject_mywidget
{
	set_has_window(false);
	over = false;
	occupied = false;
	pressed = false;
	button_type = 0; // 1 for chord_radio_buttons, 2 for modal buttons, 
					 // 3 for mult_select button,  4 for section buttons
	button_name = 0; // 1 for "Edit chord" modal button
					//  2 for "Play chord" modal button
					//  3 for "Edit note" modal button
					//  4 for "Sequence" modal button
					//  5 for mult_select button
					//  6 for "Sections" modal button
	
	sect_start = 0;
	sect_end = 0;
	sect_chord_start = 0;
	sect_chord_end = 0;
	sect_chord_length = 0;
	sect_pix_length = 0;
	sect_name = NULL;
}

MyButton::~MyButton()
{}

Gtk::SizeRequestMode MyButton::get_request_mode_vfunc() const
{
  //Accept the default value supplied by the base class.
  return Gtk::Widget::get_request_mode_vfunc();
}

//Discover the total amount of minimum space and natural space needed by
//this widget.

void MyButton::get_preferred_width_vfunc(int& minimum_width, int& natural_width) const
{
  minimum_width = 0;
  natural_width = 53;
}

void MyButton::get_preferred_height_for_width_vfunc(int /* width */,
   int& minimum_height, int& natural_height) const
{
  minimum_height = 9;
  natural_height = 12;
}

void MyButton::get_preferred_height_vfunc(int& minimum_height, int& natural_height) const
{
  minimum_height = 9;
  natural_height = 12;
}

void MyButton::get_preferred_width_for_height_vfunc(int /* height */,
   int& minimum_width, int& natural_width) const
{
  minimum_width = 0;
  natural_width = 53;
}

void MyButton::on_size_allocate(Gtk::Allocation& allocation)
{
  //Do something with the space that we have actually been given:
  //(We will not be given heights or widths less than we have requested, though
  //we might get more)
 
  //Use the offered allocation for this container:
  set_allocation(allocation);
 
}

void MyButton::on_map()
{
  //Call base class:
  Gtk::Widget::on_map();
}

void MyButton::on_unmap()
{
  //Call base class:
  Gtk::Widget::on_unmap();
}

void MyButton::on_realize()
{
  //Do not call base class Gtk::Widget::on_realize().
  //It's intended only for widgets that set_has_window(false).

  //Call base class:
  Gtk::Widget::on_realize();
}

void MyButton::on_unrealize()
{
   //Call base class:
  Gtk::Widget::on_unrealize();
}

void MyButton::draw_text(const Cairo::RefPtr<Cairo::Context>& cr, const Glib::ustring& text,
                       int x, int y, int size)
{
 /*  Set the text font and font settings */

   Pango::FontDescription font;
   font.set_family("Gyneric BRK");
   font.set_absolute_size(size * PANGO_SCALE);
   font.set_weight(Pango::WEIGHT_NORMAL);
   
    /*  Create text layout and get text size params */
    Glib::RefPtr<Pango::Layout> layout = create_pango_layout(text);
    Pango::Rectangle rect = layout->get_pixel_logical_extents();
    int length = rect.get_width();
	double scale = 1;
    if(button_type == 4 ) // section button
    {
		// center text inside the button
		x = ( (float)sect_pix_length - (float) length) / 2;
	}
	else if(button_type == 2 ) // modal button 
	{
		//printf("Text length is %d\n", length);		
		if(length > 50)
		{
			scale =	 50 / (double) length;
			// printf("Scale is %f\n", scale);
		}
	}
	else if(button_type == 3) // multiple select button
	{
		//printf("Text length is %d\n", length);		
		if(length > 70)
		{
			scale =	 70 / (double) length;
			// printf("Scale is %f\n", scale);
		}
	}
    layout->set_font_description(font);
 
    cr->save();
    cr->move_to(x, y);
	cr->scale(scale, 1);
    layout->show_in_cairo_context(cr);
    cr->restore();
}

bool MyButton::on_draw(const Cairo::RefPtr<Cairo::Context>& cr)
{
	int xs, ys, kl, kh;
	double red, green, blue, red_over, green_over, blue_over;
	double red_pressed, green_pressed, blue_pressed;
	red_pressed = 0.035; green_pressed = 0.74; blue_pressed = 0.69;
	
	if(button_type == 1) // chord_radio_button
	{
		red = 0.11; green = 0.5; blue = 0.41;
		if(occupied)
		{
			red = 0.42; green = 0.02; blue = 0.45;
		}
		red_over = 0.375; green_over = 0.62; blue_over = 0.60; // mouse over colors
		
		xs = 0; 
		ys = 0;
		kl = 53;
		kh = 12;
	}
	else if(button_type == 2 ) // modal radio button 
	{
		red_over = 0.035; green_over = 0.62; blue_over = 0.60; // mouse over colors
		red = 0.11; green = 0.5; blue = 0.41;
		red_pressed = 0.2; green_pressed = 0.9; blue_pressed = 0.85;
		xs = 0; 
		ys = 0;
		kl = 80;
		kh = 24;
	}
	else if(button_type == 3 ) // mult_select button
	{
		red_over = 0.035; green_over = 0.62; blue_over = 0.60; // mouse over colors
		red = 0.11; green = 0.5; blue = 0.41;
		red_pressed = 0.2; green_pressed = 0.9; blue_pressed = 0.85;
		xs = 0; 
		ys = 0;
		kl = 80;
		kh = 24;
		
	}
	else if(button_type == 4) // section button
	{
		red = 0.44; green = 0.5; blue = 0.47;
		if(occupied)
		{
			red = 0.42; green = 0.02; blue = 0.45;
		}
		red_pressed = 0.09; green_pressed = 0.40; blue_pressed = 0.41;
		// mouse over colors rgb(84, 110, 97)
		//red_over = 0.33; green_over = 0.43; blue_over = 0.38; 
		red_over = 0.09; green_over = 0.40; blue_over = 0.41; // same as pressed
		xs = 0; 
		ys = 0;
		kl = sect_pix_length;
		kh = 30;
		
	}   
		cr->move_to(xs, ys);
		cr->line_to(kl, ys);
		cr->line_to(kl, kh);
		cr->line_to(xs, kh);
		cr->close_path();
		if( (! pressed) && (! over) )
			cr->set_source_rgb(red, green, blue);
		else if( (button_type == 4) && !pressed && over ) // section buttons
			cr->set_source_rgb(0.38, 0.46, 0.42);// rgb(97, 117, 107)
		else if( (button_type == 3) && pressed && over ) // mult_select button
			cr->set_source_rgb(0.45, 0.78, 0.58);
		else if (over)
			cr->set_source_rgb(red_over, green_over, blue_over);
		else if (pressed)
			cr->set_source_rgb(red_pressed, green_pressed, blue_pressed);
		cr->fill ();
		cr->move_to(kl, kh);
		cr->line_to(xs, kh);
		cr->line_to(xs, ys);
		cr->line_to(kl, ys);
		cr->close_path();
		cr->set_source_rgb(0, 0, 0);
		cr->stroke();	
		
		if(button_type == 2 || button_type == 3) // modal button or mult_select
		{
			// Draw button text:
			cr->set_source_rgb(0.1, 0.1, 0.1);
					// 1 for "Edit chord"
					//  2 for "Play chord"
					//  3 for "Edit note"
					//  4 for "Sequence"
			if (button_name == 1)
				draw_text(cr, _("Edit chord"), 9, 3, 15);
			else if(button_name == 2)
				draw_text(cr, _("Play chord"), 9, 3, 15);
			else if(button_name == 3)
				draw_text(cr, _("Edit note"), 9, 3, 15);
			else if(button_name == 4)
				draw_text(cr, _("Sequence"), 9, 3, 15);
			else if(button_name == 5)
				draw_text(cr, "Mult. Select", 4, 3, 13);
			else if(button_name == 6)
				draw_text(cr, "Sections", 9, 3, 15);
		
		}
		
		if(button_type == 4) // section buttons
		{
			if(sect_name != NULL)
			{
				const Glib::ustring text = * sect_name;
				draw_text(cr, text, 9, 6, 15);
			
			}
		}
	
  return true;
}
