/* Copyright (c) 2017-2018 Marvin Eagle <trailblazer77_2000@yahoo.com>
 * 
 *		This file is part of Chord Sequencer.
 * 
 * 		Chord Sequencer is a program to create and interactively play
 * 		series of chords in the live performance, controlling chord
 * 		change by pedal (or other similar device). It uses midi and
 * 		must be connected to an external synthesizer. 
 *
 *    Chord Sequencer is free software: you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation, either version 3 of the License, or
 *    (at your option) any later version.
 *
 *    Chord Sequencer is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *   along with Chord Sequencer.  If not, see <https://www.gnu.org/licenses/>.
 * 
 * 		Author: Marvin Eagle <trailblazer77_2000@yahoo.com>
 * 
*/

#include "customradiobutton.h"
#include "customradiowidget.h"
#include <gdkmm/general.h>  // for cairo helper functions
#include <gtkmm/widget.h>
#include "note.h"
#include <config.h>
#include "gettext.h"

#define _(string) gettext (string)

//  g++  -c customradiobutton.cpp -o customradiobutton.o `pkg-config gtkmm-3.0 --cflags --libs`

// This is a custom widget, that is used not only as a radio button, but 
// also as a label, spacer and vertical separator
// It has as it's children EventBox and inner class CustomRadioWidget 
// inside the EventBox. The inner class has a separate header file.
CustomRadioButton::CustomRadioButton() :
	inner ( new CustomRadioWidget() ), // allocate and initialize the inner field
	e_box ( new Gtk::EventBox() )
	
{
	e_box->add( *inner);
	e_box->set_events(Gdk::BUTTON_PRESS_MASK);
}

CustomRadioButton::~CustomRadioButton() 
{
   delete inner; // don't forget to delete the memory
}

// Constructor for inner class
CustomRadioButton::CustomRadioWidget::CustomRadioWidget()
{
	set_has_window(false);
	active = false;
	type = 0;   // 1 is for blank spacer
				// 2 is for play radio button
				// 3 is for display radio button
				// 4 is for custom labels
				// 5 is for vertical separator
	name = 0;   // 1 is for "Play note"
				// 2 is for "Play chord"
				// 3 is for "Don't play"
				// 4 is for "Note: "
				// 5 is for "Instrument: "
				
				// 6 is for double sharp sign
				// 7 is for sharp sign
				// 8 is for regular
				// 9 is for flat sign
				// 10 is for double flat sign
}

void CustomRadioButton::CustomRadioWidget::draw_text(const Cairo::RefPtr<Cairo::Context>& cr,
			const Glib::ustring& text, int x, int y, int size)
{
 /*  Set the text font and font settings */
	
   Pango::FontDescription font;
   font.set_family("Gyneric BRK");
   font.set_absolute_size(size * PANGO_SCALE);
   font.set_weight(Pango::WEIGHT_NORMAL);
   
    /*  Create text layout and get text size params */
    auto layout = create_pango_layout(text);
    layout->set_font_description(font);
 
    cr->save();
    cr->set_source_rgb(0, 0, 0);
    cr->move_to(x, y);
    layout->show_in_cairo_context(cr);
    cr->restore();
}


void  CustomRadioButton::CustomRadioWidget::draw_double_flat(const Cairo::RefPtr<Cairo::Context>& cr,
					int x, int y, double scale)
{
	// note must be shifted to the right 7px
    // draw double flat
    double red, green, blue;
	cr->save();
	cr->translate(x, y);
	cr->scale(0.75 * scale, scale);
	draw_flat(cr, -1, 0, 1.0);
	draw_flat(cr, 10, 0, 1.0);
	cr->restore();
	
}

void  CustomRadioButton::CustomRadioWidget::draw_flat(const Cairo::RefPtr<Cairo::Context>& cr,
					int x, int y, double scale)
{
	// draw flat // d2b399ff Inkscape line color for black background images
	// Note must be shifted to the right 5px
	double red, green, blue;
	int x_start = 4;
	int y_start = 2.5;
	cr->save();
	cr->translate(x, y);
	cr->scale(scale, scale);
	red = 0; green = 0; blue = 0; 
	
	cr->move_to(x_start, y_start + 2);
	cr->line_to(x_start + 2, y_start + 2);
	cr->line_to(x_start + 2, y_start + 24);
	cr->line_to(x_start, y_start + 26);
	cr->close_path();
	cr->set_source_rgb(red, green, blue);
	cr->set_line_width(0.5);
	cr->fill();
	
	cr->move_to(x_start + 2, y_start + 14);
	cr->curve_to(x_start + 10.9, y_start + 8.9, x_start + 12, y_start + 16,
					x_start + 12, y_start + 16);
	cr->curve_to(x_start + 12, y_start + 20.4, x_start + 3.6, y_start + 23, 
					x_start , y_start + 27 );
	cr->line_to(x_start, y_start + 25.5);
	cr->curve_to(x_start + 5.1, y_start + 21, x_start + 8, y_start + 18.7,
					x_start + 7.2, y_start + 15.2);
	cr->curve_to( x_start + 5.5, y_start + 13.1, x_start + 3.5, y_start + 13.9,
					x_start + 2, y_start + 15.7);
	cr->close_path();
	cr->fill();
	
	cr->restore();
}

void  CustomRadioButton::CustomRadioWidget::draw_double_sharp(const Cairo::RefPtr<Cairo::Context>& cr,
								int x, int y, double scale)
{
	// Note must be shifted to the right 7px
	double red, green, blue;
	cr->save();
	cr->translate(x, y);	
	
	// draw double sharp
	cr->scale(scale, scale);
	red = 0; green = 0; blue = 0;
	
	int x_start = 2;
	int y_start = 12; 
	cr->move_to(x_start, y_start);
	cr->line_to(x_start + 5, y_start);
	cr->curve_to(x_start + 5, y_start + 5, x_start + 11, y_start + 11, 
					x_start + 16, y_start + 11); 
	cr->line_to(x_start + 16, y_start + 16);
	cr->line_to(x_start + 11, y_start + 16);
	cr->curve_to(x_start + 11, y_start + 11, x_start + 5, y_start + 5,
					x_start, y_start + 5);
	cr->close_path();
	cr->set_source_rgb(red, green, blue);
	cr->set_line_width(0.2);
	cr->fill();
	
	cr->move_to(x_start + 16, y_start);
	cr->line_to(x_start + 16, y_start + 5);
	cr->curve_to(x_start + 11, y_start + 5, x_start + 5, y_start + 11,
					x_start + 5, y_start + 16);
	cr->line_to(x_start, y_start + 16);
	cr->line_to(x_start, y_start + 11);
	cr->curve_to(x_start + 5, y_start + 11, x_start + 11, y_start + 5,
					x_start + 11, y_start);
	cr->close_path();
	cr->fill(); 
	cr->restore();
	
}

void CustomRadioButton::CustomRadioWidget::draw_sharp(const Cairo::RefPtr<Cairo::Context>& cr,
					int x, int y, double scale)
{
	// Note must be shifted to the right 5px
	double red, green, blue;
	cr->save();
	cr->translate(x, y);	
	cr->scale(scale, scale);
	// draw sharp
	red = 0; green = 0; blue = 0;
	
	int x_start = 3.5;
	int y_start = 0;
	cr->set_source_rgb(red, green, blue);
	cr->move_to(x_start, y_start + 1);
	cr->line_to(x_start, y_start + 23);
	cr->stroke();
	cr->move_to(x_start + 5, y_start);
	cr->line_to(x_start + 5, y_start + 22);
	cr->stroke();
	cr->move_to(x_start - 3, y_start + 8);
	cr->line_to(x_start + 8, y_start + 6);
	cr->stroke();
	cr->move_to(x_start - 3, y_start + 16);
	cr->line_to(x_start + 8, y_start + 14);
	cr->stroke();
	cr->restore();

}

bool CustomRadioButton::CustomRadioWidget::on_draw(const Cairo::RefPtr<Cairo::Context>& cr)
{
	// draw a circle
	//cr->arc(0.0, 0.0, 1.0, 0.0, 2 * M_PI);
	if(type == 1) // spacer
	{
		cr->move_to(0, 0);
		cr->line_to(100, 0);
		cr->line_to(100, 15);
		cr->line_to(0, 15);
		cr->close_path();
		cr->set_source_rgba(0.8, 0.8, 0.8, 0); // transparent
		cr->fill();
	}
	else if(type == 2) // radio buttons
	{
		int length;
		if(name == 1 || name == 2 || name == 3) // play radio buttons
			length = 100;
		else if( name > 3 && name < 9) // display radio buttons
			length = 75;
		cr->move_to(0, 0);
		cr->line_to(length, 0);
		cr->line_to(length, 20);
		cr->line_to(0, 20);
		cr->close_path();
		cr->set_source_rgba(1, 1, 1, 0); // transparent
		cr->fill();
		cr->save();
		cr->translate(9, 11);
		cr->set_source_rgb(0.3, 0.4, 0.5);
		if(active == true)
		{
			cr->arc(0.0, 0.0, 4.0, 0.0, 2 * M_PI);
			cr->fill();
		}
		cr->arc(0.0, 0.0, 7.0, 0.0, 2 * M_PI);
		cr->stroke();
		cr->restore();
		
		cr->set_source_rgb(0, 0, 0);
		if(name == 1)
			draw_text(cr, _("Play Note"),  25, -2, 14);
		else if(name == 2)
			draw_text(cr, _("Play Chord"), 25, -2, 14);
		else if(name == 3)
			draw_text(cr, _("Don't Play"), 25, -2, 14);
		else if(name == 6)
			draw_double_sharp(cr, 25, -6, 0.85);
		else if(name == 7)
			draw_sharp(cr, 25, 0, 0.85);
		else if(name == 9)
			draw_flat(cr, 25, 0, 0.75);
		else if(name == 10)
			draw_double_flat(cr, 25, -3, 0.85);
	}
	else if (type == 4) // custom labels
	{
		cr->move_to(0, 0);
		cr->line_to(80, 0);
		cr->line_to(80, 25);
		cr->line_to(0, 25);
		cr->close_path();
		cr->set_source_rgba(0.8, 0.8, 0.8, 0); // transparent
		cr->fill();
		
		cr->set_source_rgb(0, 0, 0);
		if(name == 4)
			draw_text(cr, _("Note "),  5, 2, 14);
		else if(name == 5)
			draw_text(cr, _("Instrument"), 5, 2, 14);
	}
	else if (type == 5) // vertical separator
	{
		cr->move_to(0, 0);
		cr->line_to(4, 0);
		cr->line_to(4, 60);
		cr->line_to(0, 60);
		cr->close_path();
		cr->set_source_rgba(0.7, 0.7, 0.7, 1); 
		cr->fill();
	}
}



