/* Copyright (c) 2017-2018 Marvin Eagle <trailblazer77_2000@yahoo.com>
 * 
 *		This file is part of Chord Sequencer.
 * 
 * 		Chord Sequencer is a program to create and interactively play
 * 		series of chords in the live performance, controlling chord
 * 		change by pedal (or other similar device). It uses midi and
 * 		must be connected to an external synthesizer. 
 *
 *    Chord Sequencer is free software: you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation, either version 3 of the License, or
 *    (at your option) any later version.
 *
 *    Chord Sequencer is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *   along with Chord Sequencer.  If not, see <https://www.gnu.org/licenses/>.
 * 
 * 		Author: Marvin Eagle <trailblazer77_2000@yahoo.com>
 * 
 *
 * C++ Program to Implement Doubly Linked List
 * Compile lines: 
 *  g++  -c dllist_main.cpp -o main.o `pkg-config gtkmm-3.0 --cflags --libs`
 *  g++  -c custom_dllist.cpp -o dllist.o `pkg-config gtkmm-3.0 --cflags --libs`
 *  g++ main.o dllist.o -o llist.out `pkg-config gtkmm-3.0 --cflags --libs`
 * 
 * 
 * or using two files only:
 * g++  custom_dllist.cpp -o llist.out `pkg-config gtkmm-3.0 --cflags --libs`
 */
 
#include "custom_dllist.h"


Dlist::Dlist()
{}


// Appends new node to the end of the Dlist, and links it to the data,
// given head of the Dlist and data. Returns (possibly new) head of the list.
Dlnode * Dlist::dl_append(Dlnode * head, gpointer data)
{
	Dlnode *s, *node;
	node = dl_create(); 
	node->data = data;
	node->next = NULL;
	if (head == NULL)
	{
		node->prev = NULL;
		head = node;
	}
	else
	{
		s = head;
		while (s->next != NULL)
			s = s->next;
		s->next = node;
		node->prev = s;
	}
	return head;
}

// Appends new node to the end of the Dlist, and links it to the section_buttons,
// given head of the Dlist and section_buttons. Returns (possibly new) head of the list.
Dlnode * Dlist::dl_append_but(Dlnode * head, gpointer buttons)
{
	Dlnode *s, *node;
	node = dl_create(); 
	node->section_button = buttons;
	node->next = NULL;
	if (head == NULL)
	{
		node->prev = NULL;
		head = node;
	}
	else
	{
		s = head;
		while (s->next != NULL)
			s = s->next;
		s->next = node;
		node->prev = s;
	}
	return head;
}

// Appends new node to the end of the Dlist, and links it to the eventboxes,
// given head of the Dlist and eventboxes. Returns (possibly new) head of the list.
Dlnode * Dlist::dl_append_evbox(Dlnode * head, gpointer evbox)
{
	Dlnode *s, *node;
	node = dl_create(); 
	node->eventbox = evbox;
	node->next = NULL;
	if (head == NULL)
	{
		node->prev = NULL;
		head = node;
	}
	else
	{
		s = head;
		while (s->next != NULL)
			s = s->next;
		s->next = node;
		node->prev = s;
	}
	return head;
}

// Creates Double Linked List (head node), and links it to the data
Dlnode * Dlist::dl_create(gpointer element)
{
	Dlnode * node;
	node = new(Dlnode);
	node->prev = NULL;
	node->next = NULL;
	node->data = element;
	node->name = NULL;
	
	return node;
}

//Creates Double Linked List (head node), without any data in it.
Dlnode * Dlist::dl_create()
{
	Dlnode * node;
	node = new(Dlnode);
	node->prev = NULL;
	node->next = NULL;
	node->name = NULL;
	node->sections = NULL;
	node->section_bases = NULL;
	node->info = 0;
	node->section_button = NULL;
	node->eventbox = NULL;
	node->data = NULL;
	
	return node;
}


// Finds last node in the Dlist, given any node
Dlnode * Dlist::dl_last(Dlnode * anynode)
{
	Dlnode * node;
	node = anynode;
	while(node->next != NULL)
	{
		node = node->next;
	}
	return node;
}

// Finds the node at nth position, counting from 0, given head and position.
// Returns NULL if node is not found.
Dlnode * Dlist::dl_nth_node(Dlnode * head, guint position)
{
	Dlnode * node;
	node = head;
	guint count = 0;
		
	while(count != position && node != NULL)
	{
		node = node->next;
		count++;
	}
	if(node == NULL)
	{
		std::cout<<"There is no Dlist node at the given position"<<std::endl;
	}
	return node;
}


// Adds node to the beginning of the Dlist and links it to the section_buttons,
// given head of the list and data. Returns new head of the Dlist.
Dlnode * Dlist::dl_prepend_but(Dlnode * head, gpointer element)
{
	struct Dlnode *node;
	node = dl_create();
	node->prev = NULL;
	node->section_button = element;
	node->next = head;
	head->prev = node;
	head = node;
	
	return head;
}

// Adds node to the beginning of the Dlist and links it to the eventboxes,
// given head of the list and eventboxes. Returns new head of the Dlist.
Dlnode * Dlist::dl_prepend_evbox(Dlnode * head, gpointer evbox)
{
	struct Dlnode *node;
	node = dl_create();
	node->prev = NULL;
	node->eventbox = evbox;
	node->next = head;
	head->prev = node;
	head = node;
	
	return head;
}


// Adds node to the beginning of the Dlist and links it to the data,
// given head of the list and data. Returns new head of the Dlist.
Dlnode * Dlist::dl_prepend(Dlnode * head, gpointer element)
{
	struct Dlnode *node;
	node = dl_create();
	node->prev = NULL;
	node->data = element;
	node->next = head;
	head->prev = node;
	head = node;
	
	return head;
}

// Deletes node from Dlist, given head and node to delete.
// Does not delete the data linked to the node. Returns new head node, 
// or NULL if there is only one node in the Dlist.
Dlnode * Dlist::dl_delete_node(Dlnode * head, Dlnode * node_to_delete)
{
	struct Dlnode *node, *qs;
	
	 /*first node deletion*/
	if (head == node_to_delete)
	{
		node = head;
		if(head->next != NULL) // check if the head is the only node
		{
			head = head->next;
			head->prev = NULL; 
			delete node; 
			return head;
		}
		else  // head is the only node
		{
			std::cout << "Warning: head is being deleted in Dlist" << std::endl;
			delete node;
		}
		return NULL;
	}
	/*Middle node deletion*/
	qs = head;
	while (qs->next->next != NULL)
	{   
		/*Element deleted in between*/
		if (qs->next == node_to_delete)  
		{
			node = qs->next;
			qs->next = node->next;
			node->next->prev = qs;
			delete node;
			return head;
		}
		qs = qs->next;
	}
	/*Last node deletion*/
	if (qs->next == node_to_delete)    
	{ 	
		node = qs->next;
		delete node;
		qs->next = NULL;
		return head;
	}
	std::cout<<"Node  not found"<<std::endl;
}

// Finds and returns node which is linked to the data, given head node and data
// Returns NULL if the element is not found
Dlnode * Dlist::dl_find(Dlnode * head, gpointer data)
{
	Dlnode * node;
	node = head;
	while(node != NULL)
	{
		if(node->data == data)
			return node;
		else
			node = node->next;
	}
	return node;
}

// Finds and returns node which is linked to the section_buttons, given
// head node and section_buttons.
// Returns NULL if the element is not found.
Dlnode * Dlist::dl_find_but(Dlnode * head, gpointer data)
{
	Dlnode * node;
	node = head;
	while(node != NULL)
	{
		if(node->section_button == data)
			return node;
		else
			node = node->next;
	}
	return node;
}

// Finds and returns node which is linked to the eventboxes, given
// head node and eventboxes.
// Returns NULL if the element is not found.
Dlnode * Dlist::dl_find_evbox(Dlnode * head, gpointer evbox)
{
	Dlnode * node;
	node = head;
	while(node != NULL)
	{
		if(node->eventbox == evbox)
			return node;
		else
			node = node->next;
	}
	return node;
}

// This function counts all nodes, given head node
int Dlist::dl_count(Dlnode * head)
{
	Dlnode * node;
	int count;
	count = 0;
	node = head;
	
	while (true)
	{
		if(node != NULL)
		{
			count++;
			if(node->next != NULL)
				node = node->next;
			else
				return count;
		}
	}
	return count;
}


// Finds the index of the element containing the given data (starting from 0).
// Returns the index of the element, or -1  if data is not found.
int Dlist::dl_index(Dlnode * head, gpointer data)
{
	Dlnode * node;
	int count = 0;
	node = head;
	while(node != NULL)
	{
		if(node->data == data)
			break;
		
		node = node->next;
		count++;
	}
	if(node == NULL)
		count = -1;
	return count;
}

// Finds the index of the node (starting from 0), given head node and node
// to search for. Returns the index of the node, or -1  if node is not found.
int Dlist::dl_node_index(Dlnode * head, Dlnode * node_to_find)
{
	Dlnode * node;
	int count = 0;
	node = head;
	while(node != NULL)
	{
		if(node == node_to_find)
			break;
		
		node = node->next;
		count++;
	}
	if(node == NULL)
		count = -1;
	return count;
}

// Finds node that equals a given node and inserts a new node linked to 
// the given data before this node that was found. If node is not found,
// appends a new node to the end of the list. Returns (possibly new) head 
// of the list.
Dlnode * Dlist::dl_insert_before(Dlnode * head, Dlnode * sibling, gpointer data)
{	
	Dlnode * node, * new_node, * previous;
	node = head;
	previous = NULL;
	
	while(node != sibling && node != NULL)
	{
		previous = node;
		node = node->next;
	}
	// std::cout<<"Found node to insert before: " << node->info << std::endl;
	if(node != sibling) // node was not found
	{
		puts("Error: node to insert before was not found!");
		return head;		
	}
	new_node = dl_create();
	new_node->data = data;
	new_node->next = node;
	if(node != NULL)
		node->prev = new_node;
	new_node->prev = previous;
	if(previous != NULL)
		previous->next = new_node;
	 
	if(sibling == head)
		head = new_node;
	
	return head;
}


// Finds node that equals a given node (sibling) and inserts a new node 
// linked with section_button link to 
// the given data before this node that was found. If sibling is NULL,
// appends a new node to the end of the list. Returns (possibly new) head 
// of the list.
Dlnode * Dlist::dl_insert_before_but(Dlnode * head, Dlnode * sibling, gpointer data)
{	
	Dlnode * node, * new_node, * previous;
	node = head;
	previous = NULL;
	
	while(node != sibling && node != NULL)
	{
		previous = node;
		node = node->next;
	}
	// std::cout<<"Found node to insert before: " << node->info << std::endl;
	if(node != sibling) // node was not found
	{
		puts("Error: node to insert before was not found!");
		return head;		
	}
	new_node = dl_create();
	new_node->section_button = data;
	new_node->next = node;
	if(node != NULL)
		node->prev = new_node;
	new_node->prev = previous;
	if(previous == NULL)
		puts("inside dl_insert_before_but previous == NULL");
	if(previous != NULL)
		previous->next = new_node;
	 
	if(sibling == head)
		head = new_node;
	
	return head;
}

// Finds the node at nth position, counting from 0, given head and position.
// Returns NULL if node is not found. Returns node->data if it is found.
gpointer Dlist::dl_nth_data(Dlnode * head, guint position)
{
	Dlnode * node;
	node = head;
	guint count = 0;
		
	while(count != position && node != NULL)
	{
		node = node->next;
		count++;
	}
	if(node == NULL)
	{
		std::cout<<"There is no Dlist node at the given position"<<std::endl;
		return (gpointer) NULL;
	}
	return (gpointer) (node->data);
}

// Removes a given node from Dlist, without freeing it. Returns (possibly
// new) head of the list, or NULL if head is the only node.  The removed
// element's prev and next links are set to NULL, so that it becomes a 
// self-contained list with one element.
Dlnode * Dlist::dl_remove_node(Dlnode * head, Dlnode * node_to_remove)
{
	struct Dlnode *node, *qs;
	
	 /*first node removal*/
	if (head == node_to_remove)
	{
		node = head;
		if(head->next != NULL) // check if the head is the only node
		{
			head = head->next;
			head->prev = NULL; 
			node->next = NULL;
			node->prev = NULL;
			
			return head;
		}
		else  // head is the only node
		{
			std::cout << "Warning: head is being removed in Dlist" << std::endl;
			free(node);
		}
		return NULL;
	}
		/*Middle node removal*/
	qs = head;
	while (qs->next->next != NULL)
	{   
		/*Element deleted in between*/
		if (qs->next == node_to_remove)  
		{
			node = qs->next;
			qs->next = node->next;
			node->next->prev = qs;
			node->next = NULL;
			node->prev = NULL;
			
			return head;
		}
		qs = qs->next;
	}
		/*Last node removal*/
	if (qs->next == node_to_remove)    
	{ 	
		node = qs->next;
		node->next = NULL;
		node->prev = NULL;
		qs->next = NULL;
		return head;
	}
	std::cout<<"Node  not found"<<std::endl;
}

