#! /bin/sh
### checkurl --- Check if a file was changed.

## Copyright (C)  2002,2003  Marco Parrone.

## Author: Marco Parrone
## Maintainer: Marco Parrone <marc0@autistici.org>
## Keywords: URL changes monitor checker
## Language: Shell

## This program is free software; you can redistribute it and/or modify
## it under the terms of the GNU General Public License as published by
## the Free Software Foundation; either version 2 of the License, or
## (at your option) any later version.

## This program is distributed in the hope that it will be useful,
## but WITHOUT ANY WARRANTY; without even the implied warranty of
## MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
## GNU General Public License for more details.

## You should have received a copy of the GNU General Public License
## along with this program; if not, write to the Free Software
## Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

### Commentary:

## This script checks if a file changed, then it sends a diff to the
## user.

### Code:

m4_include(config.m4)
m4_changequote({{{, }}})

## Set the default values.

case $CU_VERSION in
    "") CU_VERSION=$CONF_VERSION ;;
esac

case $CU_VERBOSE in
    "") CU_VERBOSE="no" ;;
esac

case $CU_PRE_SUBJ in
    "") CU_PRE_SUBJ="[checkurl]" ;;
esac

case $CU_HTML2TEXT in
    "") CU_HTML2TEXT="no" ;;
esac

case $DIFF_FLAGS in
    "") DIFF_FLAGS=$CONF_DIFF_FLAGS ;;
esac

case $WGET_FLAGS in
    "") WGET_FLAGS=$CONF_WGET_FLAGS ;;
esac

case $HTML2TEXT_FLAGS in
    "") HTML2TEXT_FLAGS=$CONF_HTML2TEXT_FLAGS ;;
esac

case $CU_HOME in
    "") CU_HOME="$HOME/.checkurl" ;;
esac

case $CU_CONF_FILE in
    "") CU_CONF_FILE="$CU_HOME/conf.sh" ;;
esac

case $CU_ADDITIONS in
    "") CU_ADDITIONS="no" ;;
esac

case $CU_NOT_ADDITIONS in
    "") CU_NOT_ADDITIONS="no" ;;
esac

case $CU_SUBTRACTIONS in
    "") CU_SUBTRACTIONS="no" ;;
esac

case $CU_NOT_SUBTRACTIONS in
    "") CU_NOT_SUBTRACTIONS="no" ;;
esac

case $CU_MIN_LINES in
    "") CU_MIN_LINES=0;
esac

case $CU_MAX_LINES in
    "") CU_MAX_LINES=0;
esac

alias html2text=$CONF_HTML2TEXT_PRG

## Parse the command line options.

TMP=`getopt -o hVc:p:D:Hd:w:g:vaAsSm:M: --long help,version,conf:,pre-subject:,directory:,html2text,diff-flags:,wget-flags,html2text-flags:,verbose,additions,not-additions,subtractions,not-subtractions,min-lines,max-lines -n checkurl -- ${1+"$@"}`

if test $? != 0 ; then
    exit 1
fi

eval set -- "$TMP"

while true ; do
    case "$1" in
	-h|--help)
	echo "checkurl $CU_VERSION"
	echo
	echo "Purpose:"
	echo "  checkurl is a remote file checker."
	echo
	echo "Usage: checkurl [OPTIONS]... <URL>"
	echo "   -h           --help               Print help and exit."
	echo "   -V           --version            Print version and exit."
	echo "   -c FILENAME  --conf=FILENAME      Set the configuration file."
	echo "   -p PREFIX    --pre-subject=PREFIX The prefix of the subject."
	echo "   -D DIRECTORY --directory=DIRECTORY The checkurl's home directory."
	echo "   -H           --html2text          Convert to plain text."
	echo "   -d FLAGS     --diff-flags=FLAGS   Options for diff."
	echo "   -w FLAGS     --wget-flags=FLAGS   Options for wget."
	echo "   -g FLAGS     --html2text-flags=FLAGS  Options for the converter."
	echo "   -a           --additions          Get only the additions."
	echo "   -A           --not-additions      Do not get additions."
	echo "   -s           --subtractions       Get only the subtractions."
	echo "   -S           --not-subtractions   Do not get subtractions."
	echo "   -m LINES     --min-lines          Don't send the diff if it's shorter"
	echo "                                      than LINES."
	echo "   -M LINES     --max-lines          Don't send the diff if it's longer"
	echo "                                      than LINES."
	echo "   -v           --verbose            Run in verbose mode."
	shift ; exit 0 ;;
	-V|--version)
	echo "checkurl $CU_VERSION"
	shift ; exit 0 ;;
	-c|--conf)
	CU_CONF_FILE=$2
	shift 2 ;;
	-p|--pre-subject)
	ARG_CU_PRE_SUBJ=$2
	shift 2 ;;
	-D|--directory)
	ARG_CU_HOME=$2
	shift 2 ;;
	-H|--html2text)
	ARG_CU_HTML2TEXT="yes"
	shift ;;
	-d|--diff-flags)
	ARG_DIFF_FLAGS=$2
	shift 2 ;;
	-w|--wget-flags)
	ARG_WGET_FLAGS=$2
	shift 2 ;;
	-g|--html2text-flags)
	ARG_HTML2TEXT_FLAGS=$2
	shift 2 ;;
	-v|--verbose)
	ARG_CU_VERBOSE="yes"
	shift ;;
	-a|--additions)
	ARG_CU_ADDITIONS="yes"
	shift ;;
	-A|--not-additions)
	ARG_CU_NOT_ADDITIONS="yes"
	shift ;;
	-s|--subtractions)
	ARG_CU_SUBTRACTIONS="yes"
	shift ;;
	-S|--not-subtractions)
	ARG_CU_NOT_SUBTRACTIONS="yes"
	shift ;;
	-m|--min-lines)
	ARG_CU_MIN_LINES=$2
	shift 2 ;;
	-M|--max-lines)
	ARG_CU_MAX_LINES=$2
	shift 2 ;;
	--) shift ; break ;;
	*) echo "Internal error!" ; exit 1 ;;
    esac
done

## Load in the configuration file.

if test -f $CU_CONF_FILE ; then

    case $CU_VERBOSE in
	"yes") echo "Loading configuration file \"$CU_CONF_FILE\"." ;;
    esac

    . $CU_CONF_FILE

fi

## Set the command line options.

case $CU_VERBOSE in
    "yes") echo "Setting command line options." ;;
esac

case $ARG_CU_PRE_SUBJ in
    "") ;;
    *) CU_PRE_SUBJ=$ARG_CU_PRE_SUBJ ;;
esac

case $ARG_CU_HOME in
    "") ;;
    *) CU_HOME=$ARG_CU_HOME ;;
esac

case $ARG_CU_HTML2TEXT in
    "") ;;
    *) CU_HTML2TEXT=$ARG_CU_HTML2TEXT ;;
esac

case $ARG_DIFF_FLAGS in
    "") ;;
    *) DIFF_FLAGS=$ARG_DIFF_FLAGS ;;
esac

case $ARG_WGET_FLAGS in
    "") ;;
    *) WGET_FLAGS=$ARG_WGET_FLAGS ;;
esac

case $ARG_HTML2TEXT_FLAGS in
    "") ;;
    *) HTML2TEXT_FLAGS=$ARG_HTML2TEXT_FLAGS ;;
esac

case $ARG_CU_VERBOSE in
    "") ;;
    *) CU_VERBOSE=$ARG_CU_VERBOSE ;;
esac

case $ARG_CU_ADDITIONS in
    "") ;;
    *) CU_ADDITIONS=$ARG_CU_ADDITIONS ;;
esac

case $ARG_CU_NOT_ADDITIONS in
    "") ;;
    *) CU_NOT_ADDITIONS=$ARG_CU_NOT_ADDITIONS ;;
esac

case $ARG_CU_SUBTRACTIONS in
    "") ;;
    *) CU_SUBTRACTIONS=$ARG_CU_SUBTRACTIONS ;;
esac

case $ARG_CU_NOT_SUBTRACTIONS in
    "") ;;
    *) CU_NOT_SUBTRACTIONS=$ARG_CU_NOT_SUBTRACTIONS ;;
esac

case $ARG_CU_MIN_LINES in
    "") ;;
    *) CU_MIN_LINES=$ARG_CU_MIN_LINES ;;
esac

case $ARG_CU_MAX_LINES in
    "") ;;
    *) CU_MAX_LINES=$ARG_CU_MAX_LINES ;;
esac


## Make the user checkurl's directory.

case $CU_VERBOSE in
    "yes") echo "Making sure that the checkurl's home exists." ;;
esac

if test -d $CU_HOME; then :; else

    case $CU_VERBOSE in
	"yes") echo "Making checkurl's home directory \"$CU_HOME\"." ;;
    esac

    mkdir $CU_HOME \
	|| (echo "checkurl: Cannot make checkurl's home \"$CU_HOME\"."
	    exit 1 )
fi

## The URL to handle.

case $CU_VERBOSE in
    "yes") echo "Checking if an URL was given." ;;
esac

case $1 in
  "") echo "checkurl: URL required." ; exit 1 ;;
esac

CU_URL=$1

## Make a filename

case $CU_VERBOSE in
    "yes") echo "Making filename." ;;
esac

CU_FILENAME=`echo $CU_URL | tr / _`

## Rename the old files.

case $CU_VERBOSE in
    "yes") echo "Renaming the old files if needed." ;;
esac

if test -f $CU_HOME/$CU_FILENAME ; then

    case $CU_VERBOSE in
	"yes") echo "Renaming $CU_FILENAME." ;;
    esac

    mv $CU_HOME/$CU_FILENAME $CU_HOME/$CU_FILENAME~ \
	|| ( echo "checkurl: cannot rename \"$CU_FILENAME\"." ; exit 1 )

    if test $? != 0 ; then
	exit 1
    fi

fi

## Download the new file.

case $CU_VERBOSE in
    "yes") echo "Downloading \"$CU_URL\"." ;;
esac

wget $WGET_FLAGS -O $CU_HOME/$CU_FILENAME.tmp $CU_URL

if test $? != 0 ; then
    echo "checkurl: Cannot download \"$CU_URL\"."
    rm $CU_HOME/$CU_FILENAME.tmp
    exit 1
fi

mv $CU_HOME/$CU_FILENAME.tmp $CU_HOME/$CU_FILENAME

## Make the diff.

case $CU_VERBOSE in
    "yes") echo "Making diff." ;;
esac

if test -f $CU_HOME/$CU_FILENAME~ ; then 
	
    diff $DIFF_FLAGS $CU_HOME/$CU_FILENAME~ $CU_HOME/$CU_FILENAME \
	> $CU_HOME/$CU_FILENAME.diff

    DIFF_RESULT=$?

    case $CU_VERBOSE in
	"yes") echo "Diffed." ;;
    esac

fi

case $CU_VERBOSE in
    "yes") echo "Diff result: \"$DIFF_RESULT\"" ;;
esac

## Convert the new file or diff.

case $CU_VERBOSE in
    "yes") echo "Convert the new file or diff: \"$CU_HTML2TEXT\"." ;;
esac

SUF=""

case $CU_HTML2TEXT in
    "yes")

    SUF=".txt"
    
    if test -f $CU_HOME/$CU_FILENAME~ ; then 
	
	if test $DIFF_RESULT == 1 ; then
		
	    html2text $HTML2TEXT_FLAGS < $CU_HOME/$CU_FILENAME~ \
		> $CU_HOME/$CU_FILENAME$SUF~
	    
	    html2text $HTML2TEXT_FLAGS < $CU_HOME/$CU_FILENAME \
		> $CU_HOME/$CU_FILENAME$SUF
	    
	    diff $DIFF_FLAGS $CU_HOME/$CU_FILENAME$SUF~ \
		$CU_HOME/$CU_FILENAME$SUF > $CU_HOME/$CU_FILENAME$SUF.diff

	    DIFF_RESULT=$?

	    case $CU_VERBOSE in
		"yes") echo "Diff result: \"$DIFF_RESULT\"." ;;
	    esac

	else

	    html2text $HTML2TEXT_FLAGS < $CU_HOME/$CU_FILENAME \
		> $CU_HOME/$CU_FILENAME$SUF

	fi
	
    else

	html2text $HTML2TEXT_FLAGS < $CU_HOME/$CU_FILENAME \
	    > $CU_HOME/$CU_FILENAME$SUF

    fi

    ;;

esac


## Handle `--additions' (`-a'), `--not-additions' (`-A'),
## `--subtractions' (`-s'), `--not-subtractions' (`-S').

case $CU_VERBOSE in
    "yes") echo "Handling -aAsS."
esac

case $DIFF_RESULT in
    1)
    
    case $CU_ADDITIONS in
	"yes")

	cp $CU_HOME/$CU_FILENAME$SUF.diff $CU_HOME/$CU_FILENAME$SUF.diff~
	grep '^\(+\|>\)' < $CU_HOME/$CU_FILENAME$SUF.diff~ \
	    > $CU_HOME/$CU_FILENAME$SUF.diff

	;;
    esac


    case $CU_SUBTRACTIONS in
	"yes")

	cp $CU_HOME/$CU_FILENAME$SUF.diff $CU_HOME/$CU_FILENAME$SUF.diff~
	grep '^\(-\|<\)' < $CU_HOME/$CU_FILENAME$SUF.diff~ \
	    > $CU_HOME/$CU_FILENAME$SUF.diff

	;;
    esac


    case $CU_NOT_ADDITIONS in
	"yes")

	cp $CU_HOME/$CU_FILENAME$SUF.diff $CU_HOME/$CU_FILENAME$SUF.diff~
	grep -v '^\(+\|>\)' < $CU_HOME/$CU_FILENAME$SUF.diff~ \
	    > $CU_HOME/$CU_FILENAME$SUF.diff

	;;
    esac


    case $CU_NOT_SUBTRACTIONS in
	"yes")

	cp $CU_HOME/$CU_FILENAME$SUF.diff $CU_HOME/$CU_FILENAME$SUF.diff~
	grep -v '^\(-\|<\)' < $CU_HOME/$CU_FILENAME$SUF.diff~ \
	    > $CU_HOME/$CU_FILENAME$SUF.diff

	;;
    esac

    ;;

esac

## Handle `--min-lines' (`-m') and `--max-lines' (`-M')

case $CU_VERBOSE in
    "yes") echo "Handling -mM." ;;
esac

CU_NO_SEND="no"

if test $CU_MIN_LINES != 0 ; then
    LINES=($(wc -l $CU_HOME/$CU_FILENAME$SUF.diff)+0)

    if test $LINES -lt $CU_MIN_LINES ; then
	case $CU_VERBOSE in
	    "yes") echo "Diff too short, will not be sent." ;;
        esac
	CU_NO_SEND="yes"
    fi
fi

if test $CU_MAX_LINES != 0 ; then
    LINES=($(wc -l $CU_HOME/$CU_FILENAME$SUF.diff)+0)

    if test $LINES -gt $CU_MAX_LINES ; then
	case $CU_VERBOSE in
	    "yes") echo "Diff too long, will not be sent." ;;
        esac
	CU_NO_SEND="yes"
    fi
fi

## Send the diff or the new file.

case $CU_VERBOSE in
    "yes") echo "Sending diff or new file in appropriate."
esac

if test $CU_NO_SEND != "yes" ; then
    if test -f $CU_HOME/$CU_FILENAME~ ; then 
	if test $DIFF_RESULT == 1 ; then

	    cat $CU_HOME/$CU_FILENAME$SUF.diff \
		| mail -s "$CU_PRE_SUBJ $CU_URL" $USER

	    case $CU_VERBOSE in
		"yes") echo "Sent diff." ;;
	    esac

	fi
    else

	case $CU_VERBOSE in
	    "yes") echo "Sending new file." ;;
	esac

	cat $CU_HOME/$CU_FILENAME$SUF | mail -s "$CU_PRE_SUBJ $CU_URL" $USER

    fi
fi

## Remove unused files.

FILES_TO_RM="$CU_HOME/$CU_FILENAME.txt.diff \
    $CU_HOME/$CU_FILENAME.txt.diff~ \
    $CU_HOME/$CU_FILENAME.diff \
    $CU_HOME/$CU_FILENAME.txt~ \
    $CU_HOME/$CU_FILENAME~ \
    $CU_HOME/$CU_FILENAME.txt"

for x in $FILES_TO_RM ; do
    if test -f $CU_HOME/$CU_FILENAME$SUF ; then
	rm $CU_HOME/$CU_FILENAME$SUF

	case $CU_VERBOSE in
	    "yes") echo "Removed \"$x\"." ;;
	esac
    fi
    case $CU_VERBOSE in
	"yes") echo "Not removed \"$x\"." ;;
    esac
done

## Exit
    
exit 0

#### checkurl ends here
