/* utilfuncs.h: an assortment of mathematical functions convenient for
 * my research problems but possibly irrelevant to everyone else. Too 
 * high-level or arbitrary in coverage to go in mathdefs. Comment out
 * of the Makefile if you like.
 *
 * Channelflow-0.9
 *
 * Copyright (C) 2001-2005  John F. Gibson
 *
 * John F. Gibson
 * Center for Nonlinear Sciences
 * School of Physics
 * Georgia Institute of Technology
 * Atlanta, GA 30332-0430
 *
 * gibson@cns.physics.gatech.edu
 * jfg@member.fsf.org
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, U
 */

#ifndef CHANNELFLOW_UTILFUNCS_H
#define CHANNELFLOW_UTILFUNCS_H

#include "channelflow/mathdefs.h"
#include "channelflow/array.h"
#include "channelflow/flowfield.h"
#include "channelflow/dns.h"


// Some convenient analytic functions for initializing plane Couette flow
BasisFunc makeroll(Real Lx, Real Lz, Real a, Real b, int Ny); // kx,kz=0,1 roll
BasisFunc makewave(Real Lx, Real Lz, Real a, Real b, int Ny); // kx,kz=1,0 wave
BasisFunc makemeen(Real Lx, Real Lz, Real a, Real b, int Ny); // kx,kz=0,0 mean
BasisFunc makewobl(Real Lx, Real Lz, Real a, Real b, int Ny); // kx,kz=1,1 wave
BasisFunc makeprtb(Real Lx, Real Lz, Real a, Real b, int Ny); // kx,kz=1

// Interpolate (N-1)th order polynomial approximation to f at x, 
// given {fn = f(xn) | n=0,...N-1}

Real polyInterp(const array<Real>& fn, const array<Real>& xn, Real x);


// secantSearch and bisectSearch:
// Find root of f(x) between a and b to tolerance feps from polynomial 
// interpolant of {fn = f(xn)} I.e. rtn x st abs(f(x)) < feps and a <= x <= b. 
// A polynomial-tailored search would be faster, but more complex and liable 
// to fail. Cost is negligible anyway, so stick with simple and foolproof algs.

Real secantSearch(Real a, Real b, array<Real>& fn, const array<Real>& xn, 
		  Real feps=1e-14, int maxsteps=50);

Real bisectSearch(Real a, Real b, array<Real>& fn, const array<Real>& xn, 
		  Real feps=1e-14, int maxsteps=50);


class FieldSeries {
public:
  FieldSeries();
  FieldSeries(int N); // Interpolate with (N-1)th-order polynomial

  void push(const FlowField& f, Real t);
  void interpolate(FlowField& f, Real t) const;
  bool full() const;

private:
  array<Real> t_;
  array<FlowField> f_;
  int emptiness_;  // init val N, decrement in push(), stacks are full at 0
};
  

#endif
