/* utilfuncs.cpp: an assortment of mathematical functions convenient for
 * my research problems but probably irrelevant to everyone else
 *
 * Channelflow-0.9
 *
 * Copyright (C) 2001-2005  John F. Gibson
 *
 * John F. Gibson
 * Center for Nonlinear Sciences
 * School of Physics
 * Georgia Institute of Technology
 * Atlanta, GA 30332-0430
 *
 * gibson@cns.physics.gatech.edu
 * jfg@member.fsf.org
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, U
 */

#include "channelflow/utilfuncs.h"

using namespace std;

// Algorithm from Numerical Recipes in C pg 110;
Real polyInterp(const array<Real>& fa, const array<Real>& xa, Real x) {
  assert(fa.N() == xa.N());
  int N = fa.N();
  array<Real> C(N);
  array<Real> D(N);
  
  Real xdiff = abs(x-xa[0]);
  Real xdiff2 = 0.0;
  Real df;

  int i_closest = 0;
  for (int i=0; i<N; ++i) {
    if ((xdiff2 = abs(x-xa[i])) < xdiff) {
      i_closest = i;
      xdiff = xdiff2;
    }
    C[i] = fa[i];
    D[i] = fa[i];
  }
  
  // Initial approx to f(x)
  Real fx = fa[i_closest--];

  for (int m=1; m<N; ++m) {
    for (int i=0; i<N-m; ++i) {
      Real C_dx = xa[i]-x;
      Real D_dx = xa[i+m] - x;
      Real C_D = C[i+1]-D[i];
      Real denom = C_dx - D_dx;
      if (denom == 0.0)
	cferror("polyinterp(fa,xa,x) : two values of xa has are equal!");
      denom = C_D/denom;
      C[i] = C_dx*denom;
      D[i] = D_dx*denom;
    }
    fx += (df=(2*(i_closest+1) < (N-m)? C[i_closest+1] : D[i_closest--]));
  }
  return fx;
}

Real secantSearch(Real a, Real b, array<Real>& fn, const array<Real>& xn, 
		  Real feps, int maxsteps) {
  Real fa = polyInterp(fn, xn, a);
  Real fb = polyInterp(fn, xn, b);
  if (fa*fb > 0) 
    cferror("secantSearch(a,b,fn,xn) : a and b don't bracket a zero");

  Real c = a - fa * (b-a)/(fb-fa);
  Real fc;

  for (int n=0; n<maxsteps; ++n) {
    c = a - fa * (b-a)/(fb-fa);
    fc = polyInterp(fn,xn,c);
    if (abs(fc) < feps) 
      break;
    if (fc*fa > 0) {
      a = c;
      fa = fc;
    }
    else {
      b = c;
      fb = fc;
    }
  }
  return c;
}


Real bisectSearch(Real a, Real b, array<Real>& fn, const array<Real>& xn, 
		  Real feps, int maxsteps) {
  Real c = 0.5*(a+b)/2; // GCC-3.3.5 complains if not explicitly init'ed
  Real fc;
  Real fa = polyInterp(fn, xn, a);
  Real fb = polyInterp(fn, xn, b);
  if (fa*fb > 0) 
    cferror("bisectSearch(a,b,fn,xn) : a and b don't bracket a zero");

  for (int n=0; n<maxsteps; ++n) {
    c = 0.5*(a+b);
    fc = polyInterp(fn,xn,c);
    if (abs(fc) < feps) 
      break;
    if (fc*fa > 0) {
      a = c;
      fa = fc;
    }
    else {
      b = c;
      fb = fc;
    }
  }
  return c;
}


FieldSeries::FieldSeries() 
  :
  t_(0),
  f_(0),
  emptiness_(0)
{}

FieldSeries::FieldSeries(int N) 
  :
  t_(N),
  f_(N),
  emptiness_(N)
{}

//Still working on first draft of these functions


void FieldSeries::push(const FlowField& f, Real t) {
  for (int n=f_.N()-1; n>0; --n) {
    if (f_[n].congruent(f_[n-1]))
      swap(f_[n], f_[n-1]);
    else
      f_[n] = f_[n-1];
    t_[n] = t_[n-1];
  }
  if (f_.N() > 0) {
    f_[0] = f;
    t_[0] = t;
  }
  --emptiness_;
}

bool FieldSeries::full() const {
  return (emptiness_ == 0) ? true : false;
}

void FieldSeries::interpolate(FlowField& f, Real t) const {
  if (!(this->full())) 
    cferror("FieldSeries::interpolate(Real t, FlowField& f) : FieldSeries is not completely initialized. Take more time steps before interpolating");

  for (int n=0; n<f_.N(); ++n) {
    assert(f_[0].congruent(f_[n]));
    ;
  }

  if (f_[0].xzstate() == Spectral) {
    int Mx = f_[0].Mx();
    int Mz = f_[0].Mz();
    int Ny = f_[0].Ny();
    int Nd = f_[0].Nd();
    int N = f_.N();
    array<Real> fn(N);
    for (int i=0; i<Nd; ++i) 
      for (int ny=0; ny<Ny; ++ny) 
	for (int mx=0; mx<Mx; ++mx) 
	  for (int mz=0; mz<Mz; ++mz) {
	    for (int n=0; n<N; ++n)
	      fn[n] = Re(f_[n].cmplx(mx,ny,mz,i));
	    Real a = polyInterp(fn,t_,t);

	    for (int n=0; n<N; ++n)
	      fn[n] = Im(f_[n].cmplx(mx,ny,mz,i));
	    Real b = polyInterp(fn,t_,t);

	    f.cmplx(mx,ny,mz,i) = Complex(a,b);
	  }
  }
  else {
    int Nx = f_[0].Nx();
    int Ny = f_[0].Ny();
    int Nz = f_[0].Nz();
    int Nd = f_[0].Nd();
    int N = f_.N();
    array<Real> fn(N);
    for (int i=0; i<Nd; ++i) 
      for (int ny=0; ny<Ny; ++ny) 
	for (int nx=0; nx<Nx; ++nx) 
	  for (int nz=0; nz<Nz; ++nz) {
	    for (int n=0; n<N; ++n)
	      fn[n] = f_[n](nx,ny,nz,i);
	    f(nx,ny,nz,i) = polyInterp(fn,t_,t);
	  }
  }
}
      
// This formula recreates (roughly) the kx,kz=1,0 rolls seen in PCF flow.
// Derived in 9/5/05 notes and uinit.m, tinkered with until looked like DNS.
BasisFunc makeroll(Real Lx, Real Lz, Real a, Real b, int Ny) {
  if (Ny < 5) 
    cferror("Ny too small. Set Ny >= 5");
  
  BasisFunc rtn(Ny, 0, 1, Lx, Lz, a, b, Spectral);

  rtn.v().re[0] =  3.0/32.0;
  rtn.v().re[2] = -1.0/8.0;
  rtn.v().re[4] =  1.0/32.0;
  
  ChebyCoeff tmp = diff(rtn.v().re);
  rtn.w().im = tmp;
  rtn.w().im *= Lz/(2*pi);
  
  tmp.setToZero();
  tmp.setState(Physical);
  Vector y = chebypoints(Ny,-1,1);
  for (int n=0; n<Ny; ++n) {
    Real yp = 1 + 2*(y[n]-b)/(b-a);
    tmp[n] = 2*(pow(yp,6) - 0.7*pow(yp,2) - 0.3);
  }
  tmp.makeSpectral();
  rtn.u().re = tmp;
  
  rtn *= -1.0/L2Norm(rtn);
  return rtn;
}

BasisFunc makewave(Real Lx, Real Lz, Real a, Real b, int Ny) {
  BasisFunc rtn(Ny, 1, 0, Lx, Lz, a, b, Physical);
  
  Vector y = chebypoints(Ny,-1,1);
  for (int n=0; n<Ny; ++n) {
    Real yp = 1 + 2*(y[n]-b)/(b-a);
    Real p = 1-yp*yp;
    Real pp = p*p;
    rtn.w().re[n] = 0.7*p*sin(pi*yp);
    rtn.w().im[n] = p*cos(pi*yp);
    rtn.v().re[n] = 0.04*pp*sin(pi*yp);
    rtn.v().im[n] = 0.04*pp*cos(pi*yp);
  }
  rtn.makeSpectral();
  ChebyCoeff tmp = diff(rtn.v().re);
  rtn.u().im = tmp;
  rtn.u().im *= Lx/(2*pi);
  
  tmp = diff(rtn.v().im);
  rtn.u().re = tmp;
  rtn.u().re *= -Lx/(2*pi);
  
  rtn *= 1.0/L2Norm(rtn);
  return rtn;
}

BasisFunc makemeen(Real Lx, Real Lz, Real a, Real b, int Ny) {

  BasisFunc rtn(Ny, 0, 0, Lx, Lz, a, b, Physical);
  Vector y = chebypoints(Ny,-1,1);
  for (int n=0; n<Ny; ++n) 
    rtn.u().re[n] =  -sin(pi*y[n]);
  rtn.makeSpectral();
  rtn *= 1.0/L2Norm(rtn);
  return rtn;
}  

BasisFunc makewobl(Real Lx, Real Lz, Real a, Real b, int Ny) {
  BasisFunc rtn(Ny, 1, 1, Lx, Lz, a, b, Physical);
  
  Vector y = chebypoints(Ny,-1,1);
  for (int n=0; n<Ny; ++n) {
    Real yp = 1 + 2*(y[n]-b)/(b-a);
    Real p = 1-yp*yp;
    Real pp = p*p;
    rtn.u().re[n] = p*cos(pi*yp);
    rtn.u().im[n] = -p*sin(pi*yp);
    rtn.v().re[n] = 0.2*pp*cos(pi*yp);
    rtn.v().im[n] = -0.2*pp*sin(pi*yp);
  }
  rtn.makeSpectral();
  ChebyCoeff tmp;

  // Build up Re(w) = -Lz/Lx  Re(u) - Lz/2pikz Im(v')
  //                = -Lz/Lx [Re(u) + Lx/2pikz Im(v')]
  diff(rtn.v().im, tmp);
  rtn.w().re = tmp;
  rtn.w().re *= Lx/(2*pi);
  rtn.w().re += rtn.u().re;
  rtn.w().re *= -Lz/Lx;
  
  // Build up Im(w) = -Lz/Lx  Im(u) + Lz/2pikz Re(v')
  //                = -Lz/Lx [Im(u) - Lx/2pikz Re(v')]
  diff(rtn.v().re, tmp);
  rtn.w().im = tmp;
  rtn.w().im *= -Lx/(2*pi);
  rtn.w().im += rtn.u().im;
  rtn.w().im *= -Lz/Lx;
  
  rtn *= 1.0/L2Norm(rtn);
  return rtn;
}


BasisFunc makeprtb(Real Lx, Real Lz, Real a, Real b, int Ny) {
  BasisFunc rtn(Ny, 1, 1, Lx, Lz, a, b, Physical);
  
  Vector y = chebypoints(Ny,-1,1);
  for (int n=0; n<Ny; ++n) {
    Real yp = 1 + 2*(y[n]-b)/(b-a);
    Real p = 1-yp*yp;
    Real pp = p*p;
    rtn.u().re[n] = p*(1+yp);
    rtn.v().im[n] = pp*(1+yp);
  }
  rtn.makeSpectral();
  ChebyCoeff tmp;

  // Build up Re(w) = -Lz/Lx  Re(u) - Lz/2pikz Im(v')
  //                = -Lz/Lx [Re(u) + Lx/2pikz Im(v')]
  diff(rtn.v().im, tmp);
  rtn.w().re = tmp;
  rtn.w().re *= Lx/(2*pi);
  rtn.w().re += rtn.u().re;
  rtn.w().re *= -Lz/Lx;
  
  rtn *= 1.0/L2Norm(rtn);
  return rtn;
}
