/* flowfield.h: Class for N-dim Fourier x Chebyshev x Fourier expansions
 * Channelflow-0.9
 *
 * Copyright (C) 2001-2005  John F. Gibson
 *
 * John F. Gibson
 * Center for Nonlinear Sciences
 * School of Physics
 * Georgia Institute of Technology
 * Atlanta, GA 30332-0430
 *
 * gibson@cns.physics.gatech.edu
 * jfg@member.fsf.org
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, U
 */

#include <fstream>
#include <iomanip>

#include "flowfield.h"
#include "chebyshev.h"

const Real EPSILON=1e-4;

FlowField::FlowField()
  :
  Nx_(0),
  Ny_(0),
  Nz_(0),
  Nzpad_(0),
  Nzpad2_(0),
  Nd_(0),
  Lx_(0),
  Lz_(0),
  a_(0),
  b_(0),
  dealiasIO_(false),
  rdata_(0),
  cdata_(0),
  scratch_(0),
  xzstate_(Spectral),
  ystate_(Spectral),
  xz_plan_(0),
  xz_iplan_(0),
  y_plan_(0)
{}

FlowField::FlowField(int Nx, int Ny, int Nz, int Nd, Real Lx, Real Lz, 
		     Real a, Real b, fieldstate xzstate, fieldstate ystate, 
		     int fftw_flags) 
  :
  Nx_(0),
  Ny_(0),
  Nz_(0),
  Nzpad_(0),
  Nzpad2_(0),
  Nd_(0),
  Lx_(0),
  Lz_(0),
  a_(0),
  b_(0),
  dealiasIO_(false),
  rdata_(0),
  cdata_(0),
  scratch_(0),
  xzstate_(xzstate),
  ystate_(ystate),
  xz_plan_(0),
  xz_iplan_(0),
  y_plan_(0)
{
  // This isn't in classic C++ initialization style, but it consolidates
  // all resize and initialization code, and the relative overhead is neglgble.
  resize(Nx, Ny, Nz, Nd, Lx, Lz, a, b, fftw_flags);
}

FlowField::FlowField(int Nx, int Ny, int Nz, int Nd, int tensorOrder, Real Lx, 
		     Real Lz, Real a, Real b, fieldstate xzstate, 
		     fieldstate ystate, int fftw_flags) 
  :
  Nx_(0),
  Ny_(0),
  Nz_(0),
  Nzpad_(0),
  Nzpad2_(0),
  Nd_(pow(Nd,tensorOrder)),
  Lx_(0),
  Lz_(0),
  a_(0),
  b_(0),
  dealiasIO_(false),
  rdata_(0),
  cdata_(0),
  scratch_(0),
  xzstate_(xzstate),
  ystate_(ystate),
  xz_plan_(0),
  xz_iplan_(0),
  y_plan_(0)
{
  // This isn't in classic C++ initialization style, but it consolidates
  // all resize and initialization code, and the relative overhead is neglgble.
  resize(Nx, Ny, Nz, Nd_, Lx, Lz, a, b, fftw_flags);
}

FlowField::FlowField(const FlowField& f) 
  :
  Nx_(0),
  Ny_(0),
  Nz_(0),
  Nzpad_(0),
  Nzpad2_(0),
  Nd_(0),
  Lx_(0),
  Lz_(0),
  a_(0),
  b_(0),
  dealiasIO_(f.dealiasIO_),
  rdata_(0),
  cdata_(0),
  scratch_(0),
  xzstate_(Spectral),
  ystate_(Spectral),
  xz_plan_(0),
  xz_iplan_(0),
  y_plan_(0)
{
  resize(f.Nx_, f.Ny_, f.Nz_, f.Nd_, f.Lx_, f.Lz_, f.a_, f.b_);
  setState(f.xzstate_, f.ystate_);
  dealiasIO_ = f.dealiasIO_;
  int N = Nd_*Ny_*Nx_*Nzpad_;
  for (int i=0; i<N; ++i)
    rdata_[i] = f.rdata_[i];

  fftw_initialize(FFTW_ESTIMATE);
}

FlowField::FlowField(const string& filebase) 
  :
  Nx_(0),
  Ny_(0),
  Nz_(0),
  Nzpad_(0),
  Nzpad2_(0),
  Nd_(0),
  Lx_(0.0),
  Lz_(0.0),
  a_(0.0),
  b_(0.0),
  dealiasIO_(false),
  rdata_(0),
  cdata_(0),
  scratch_(0),
  xzstate_(Spectral),
  ystate_(Spectral),
  xz_plan_(0),
  xz_iplan_(0),
  y_plan_(0)
{
  
  string filename(filebase);
  filename += string(".ff");
  ifstream is(filename.c_str(), ios::in | ios::binary);

  // If filebase.ff doesn't work, try filebase alone.
  if (!is.good()) {
    is.close();
    is.open(filebase.c_str(), ios::in | ios::binary);
    if (is.good()) 
      filename = filebase;
  }
  if (!is.good()) {
    cerr << "FlowField::FlowField(filebase) : can't open file " 
	 << filename << " or " << filebase << endl;
    exit(1);
  }

  int major_version, minor_version, update_version;
  read(is, major_version);
  read(is, minor_version);
  read(is, update_version);
  if (major_version != 0 || minor_version != 9) {
    cerr << "FlowField::FlowField(filebase) : file " << filename
	 << " is not in version > 0.9.16 format.\n"
	 << "Version appears to be " << major_version << '.' 
	 << minor_version << '.' << update_version << endl;
    cerr << "Try using FlowField::FlowField(filebase, chflow_version) constructor.\n";
    exit(1);
  }
    
  read(is, Nx_);
  read(is, Ny_);
  read(is, Nz_);
  read(is, Nd_);
  read(is, xzstate_);
  read(is, ystate_);
  read(is, Lx_);
  read(is, Lz_);
  read(is, a_);
  read(is, b_);
  read(is, dealiasIO_);
  resize(Nx_, Ny_, Nz_, Nd_, Lx_, Lz_, a_, b_);

  // Read data only for non-aliased modes, assume 0 for aliased.
  if (dealiasIO_ == true && xzstate_ == Spectral) {
    int Nxd=2*(Nx_/6);
    int Nzd=2*(Nz_/3)+1;

    // In innermost loop, array index is (nz + Nzpad2_*(nx + Nx_*(ny + Ny_*i))),
    // which is the same as the FlowField::flatten function.
    for (int i=0; i<Nd_; ++i) {
      for (int ny=0; ny<Ny_; ++ny) {

	for (int nx=0; nx<=Nxd; ++nx) {
	  for (int nz=0; nz<=Nzd; ++nz)
	    read(is, rdata_[flatten(nx,ny,nz,i)]);
	  for (int nz=Nzd+1; nz<Nzpad_; ++nz)
	    rdata_[flatten(nx,ny,nz,i)] = 0.0;
	}
	for (int nx=Nxd+1; nx<=Nxd; ++nx) 
	  for (int nz=0; nz<=Nzpad_; ++nz)
	    rdata_[flatten(nx,ny,nz,i)] = 0.0;
	
	for (int nx=Nx_-Nxd; nx<Nx_; ++nx) {
	  for (int nz=0; nz<=Nzd; ++nz)
	    read(is, rdata_[flatten(nx,ny,nz,i)]);
	  for (int nz=Nzd+1; nz<Nzpad_; ++nz)
	    rdata_[flatten(nx,ny,nz,i)] = 0.0;
	}
      }
    }
  }
  else {
    int N = Nd_*Ny_*Nx_*Nzpad_;
    for (int i=0; i<N; ++i)
      read(is, rdata_[i]);
  }
}

FlowField::FlowField(const string& filebase, int major, int minor, int update)
  :
  Nx_(0),
  Ny_(0),
  Nz_(0),
  Nzpad_(0),
  Nzpad2_(0),
  Nd_(0),
  Lx_(0.0),
  Lz_(0.0),
  a_(0.0),
  b_(0.0),
  dealiasIO_(false),
  rdata_(0),
  cdata_(0),
  scratch_(0),
  xzstate_(Spectral),
  ystate_(Spectral),
  xz_plan_(0),
  xz_iplan_(0),
  y_plan_(0)
{
  
  string filename(filebase);
  filename += string(".ff");
  ifstream is(filename.c_str(), ios::in | ios::binary);
  
  // If filebase.ff doesn't work, try filebase alone.
  if (!is.good()) 
    is.open(filebase.c_str(), ios::in | ios::binary);

  if (!is.good()) {
    cerr << "FlowField::FlowField(filebase, major,minor,update) : can't open file " 
	 << filename << " or " << filebase << endl;
    exit(1);
  }

  if (major > 0 || minor > 9) {
    cerr << "FlowField::FlowField(filebase, major,minor,update) :\n"
	 << major << '.' << minor << '.' << update << " is higher than 0.9.x" << endl;
    exit(1);
  }
  
  if (update >= 16) {
    (*this) = FlowField(filebase); // not efficient but saves code redundancy
    return;
  }
  else {
    is.read((char*) &Nx_, sizeof(int));
    is.read((char*) &Ny_, sizeof(int));
    is.read((char*) &Nz_, sizeof(int));
    is.read((char*) &Nd_, sizeof(int));
    is >> xzstate_;
    is >> ystate_;
    is.read((char*) &Lx_, sizeof(Real));
    is.read((char*) &Lz_, sizeof(Real));
    is.read((char*) &a_, sizeof(Real));
    is.read((char*) &b_, sizeof(Real));
    char s;
    is.get(s);
    dealiasIO_ = (s=='1') ? true : false;

    //readLittleEndian(is, Nx_);
    //readLittleEndian(is, Ny_);
    //readLittleEndian(is, Nz_);
    //readLittleEndian(is, Nd_);
    //read(is, xzstate_);
    //read(is, ystate_);
    //readLittleEndian(is, Lx_);
    //readLittleEndian(is, Lz_);
    //readLittleEndian(is, a_);
    //readLittleEndian(is, b_);
    //read(is, dealiasIO_);
    resize(Nx_, Ny_, Nz_, Nd_, Lx_, Lz_, a_, b_);

    // Read data only for non-aliased modes, assume 0 for aliased.
    if (dealiasIO_ == true && xzstate_ == Spectral) {
      int Nxd=2*(Nx_/6);
      int Nzd=2*(Nz_/3)+1;

      // In innermost loop, array index is (nz + Nzpad2_*(nx + Nx_*(ny+Ny_*i)))
      // which is the same as the FlowField::flatten function.
      for (int i=0; i<Nd_; ++i) {
	for (int ny=0; ny<Ny_; ++ny) {

	  for (int nx=0; nx<=Nxd; ++nx) {
	    for (int nz=0; nz<=Nzd; ++nz)
	      is.read((char*) (rdata_ + flatten(nx,ny,nz,i)), sizeof(Real));
	    //readLittleEndian(is, rdata_[flatten(nx,ny,nz,i)]);
	    for (int nz=Nzd+1; nz<Nzpad_; ++nz)
	      rdata_[flatten(nx,ny,nz,i)] = 0.0;
	  }
	  for (int nx=Nxd+1; nx<=Nxd; ++nx) 
	    for (int nz=0; nz<=Nzpad_; ++nz)
	      rdata_[flatten(nx,ny,nz,i)] = 0.0;
	
	  for (int nx=Nx_-Nxd; nx<Nx_; ++nx) {
	    for (int nz=0; nz<=Nzd; ++nz)
	      is.read((char*) (rdata_ + flatten(nx,ny,nz,i)), sizeof(Real));
	    //readLittleEndian(is, rdata_[flatten(nx,ny,nz,i)]);
	    for (int nz=Nzd+1; nz<Nzpad_; ++nz)
	      rdata_[flatten(nx,ny,nz,i)] = 0.0;
	  }
	}
      }
    }
    else {
      int N = Nd_*Ny_*Nx_*Nzpad_;
      for (int i=0; i<N; ++i)
	is.read((char*) (rdata_ + i), sizeof(Real));
      //readLittleEndian(is, rdata_[i]);
    }
  }
}

Vector FlowField::xgridpts() const {
  Vector xpts(Nx_);
  for (int nx=0; nx<Nx_; ++nx)
    xpts[nx] = x(nx);
  return xpts;
}
Vector FlowField::ygridpts() const {
  Vector ypts(Ny_);
  Real c = 0.5*(b_+a_);
  Real r = 0.5*(b_-a_);
  Real piN = pi/(Ny_-1);
  for (int ny=0; ny<Ny_; ++ny)
    ypts[ny] = c + r*cos(piN*ny);
  return ypts;
}
Vector FlowField::zgridpts() const {
  Vector zpts(Nz_);
  for (int nz=0; nz<Nz_; ++nz)
    zpts[nz] = z(nz);
  return zpts;
}

FlowField& FlowField::operator=(const FlowField& f) {
  resize(f.Nx_, f.Ny_, f.Nz_, f.Nd_, f.Lx_, f.Lz_, f.a_, f.b_);
  setState(f.xzstate_, f.ystate_);
  dealiasIO_ = f.dealiasIO_;
  int Ntotal = Nx_ * Ny_ * Nzpad_ * Nd_;
  for (int i=0; i<Ntotal; ++i)
    rdata_[i] = f.rdata_[i];
  return *this;
}

FlowField::~FlowField() {
  if (scratch_) fftw_free(scratch_);
  if (rdata_) fftw_free(rdata_);
  if (y_plan_) fftw_destroy_plan(y_plan_);
  if (xz_plan_) fftw_destroy_plan(xz_plan_);
  if (xz_iplan_) fftw_destroy_plan(xz_iplan_);
}

bool FlowField::isNull() {
  return (Nx_ == 0 && Ny_ == 0 && Nz_ == 0 && Nd_ == 0);
}

void FlowField::resize(int Nx, int Ny, int Nz, int Nd, Real Lx, Real Lz, 
		       Real a, Real b, int fftw_flags) {

  // INEFFICIENT if geometry doesn't change. Should check.
  if (scratch_) fftw_free(scratch_);
  if (rdata_) fftw_free(rdata_);
  if (y_plan_) fftw_destroy_plan(y_plan_);
  if (xz_plan_) fftw_destroy_plan(xz_plan_);
  if (xz_iplan_) fftw_destroy_plan(xz_iplan_);
  
  Nx_ = Nx;
  Ny_ = Ny;
  Nz_ = Nz;
  Nd_ = Nd;
  Lx_ = Lx;
  Lz_ = Lz;
  a_ = a;
  b_ = b;

  assert(Nx_>=0);
  assert(Ny_>=0);
  assert(Nz_>=0);
  assert(Nd_>=0);
  assert(Lx_>=0);
  assert(Lz_>=0);
  assert(b_ >= a_);
  
  Nzpad_ = 2*(Nz_/2+1);
  Nzpad2_ = Nz_/2+1;
  rdata_ = (Real*) fftw_malloc((Nx_*Ny_*Nzpad_*Nd_)*sizeof(Real));
  cdata_ = (Complex*) rdata_;
    
  int N = Nd_*Ny_*Nx_*Nzpad_;
  for (int i=0; i<N; ++i)
    rdata_[i] = 0.0;

  scratch_ = (Real*) fftw_malloc(Ny_*sizeof(Real));

  fftw_initialize(fftw_flags);
}
  
void FlowField::fftw_initialize(int flags) {
  flags = flags | FFTW_DESTROY_INPUT;
  flags = FFTW_MEASURE | FFTW_DESTROY_INPUT;
  
  if (Nx_ !=0 && Nz_ !=0) {
    fftw_complex* fcdata = (fftw_complex*) cdata_;

    const int howmany = Ny_*Nd_;
    const int rank = 2;

    // These params describe the structure of the real-valued array 
    int real_n[rank];
    real_n[0] = Nx_;
    real_n[1] = Nz_;
    int real_embed[rank];
    real_embed[0] = Nx_;
    real_embed[1] = Nzpad_;
    const int real_stride = 1;
    const int real_dist = Nx_*Nzpad_;

    // These params describe the structure of the complex-valued array 
    int cplx_n[rank];
    cplx_n[0] = Nx_;
    cplx_n[1] = Nzpad2_;
    int cplx_embed[rank];
    cplx_embed[0] = Nx_;
    cplx_embed[1] = Nzpad2_;
    const int cplx_stride = 1;
    const int cplx_dist = Nx_*Nzpad2_;

    // Real -> Complex transform parameters
    xz_plan_ = 
      fftw_plan_many_dft_r2c(rank, real_n, howmany, 
			     rdata_, real_embed, real_stride, real_dist, 
			     fcdata, cplx_embed, cplx_stride, cplx_dist, 
			     flags);

    xz_iplan_ = 
      fftw_plan_many_dft_c2r(rank, real_n, howmany, 
			     fcdata, cplx_embed, cplx_stride, cplx_dist, 
			     rdata_, real_embed, real_stride, real_dist, 
			     flags);
  }
  else {
    xz_plan_ = 0;
    xz_iplan_ = 0;
  }
  if (Ny_ != 0) 
    y_plan_  = fftw_plan_r2r_1d(Ny_,scratch_,scratch_,FFTW_REDFT00, flags);
  else 
   y_plan_  = 0;
}

void FlowField::rescale(Real Lx, Real Lz) {
  assertState(Spectral, Spectral);
  Real scalev = Lx_/Lx;
  Real scalew = (Lx*Lz_)/(Lz*Lx_);
  
  for (int ny=0; ny<numYmodes(); ++ ny)
    for (int mx=0; mx<Mx(); ++mx)
      for (int mz=0; mz<Mz(); ++ mz) {
	cmplx(mx,ny,mz,1) *= scalev;
	cmplx(mx,ny,mz,2) *= scalew;
      }
  Lx_ = Lx;
  Lz_ = Lz;
}

void FlowField::interpolate(const FlowField& ff) {
  FlowField& g = *this;
  FlowField& f = (FlowField&) ff;

  fieldstate fxstate = f.xzstate();
  fieldstate fystate = f.ystate();
  fieldstate gxstate = g.xzstate();
  fieldstate gystate = g.ystate();
  
  // Always check these, debugging or not. 
  if (f.Lx() != g.Lx()  || f.Lz() != g.Lz() || f.a() != g.a()  ||
      f.b() != g.b() || f.Nd() != g.Nd()) {
    cerr << "FlowField::interpolate(const FlowField& f) error:\n"
	 << "FlowField doesn't match argument f geometrically.\n";
    exit(1);
  }
  if (f.Nx()>g.Nx()  || f.Ny()>g.Ny()  || f.Nz()>g.Nz()) {
    cerr << "FlowField::interpolate(const FlowField& f) error:\n"
	 << "interpolation requires Nx>=f.Nx, Ny>=f.Ny, Nz>=f.Nz\n";
    exit(1);
  }
  f.makeSpectral();
  g.setState(Spectral, Spectral);
  g.setToZero();
  for (int i=0; i<f.Nd(); ++i) 
    for (int ny=0; ny<f.Ny(); ++ny) {
      for (int kx=f.kxmin(); kx<f.kxmax(); ++kx) {
	int fmx = f.mx(kx);
	int gmx = g.mx(kx);
	for (int kz=f.kzmin(); kz<=f.kzmax(); ++kz) {
	  int fmz = f.mz(kz);
	  int gmz = g.mz(kz);
	  g.cmplx(gmx,ny,gmz,i) = f.cmplx(fmx,ny,fmz,i);
	}
      }
      // Fourier discretizations go from -Mx/2+1 <= kx <= Mx/2
      // So the last mode kx=Mx/2 has an implicit -kx counterpart that is
      // not included in the Fourier representation. For f, this mode is
      // implicitly/automatically included in the Fourier transforms. 
      // But if g has more x modes than f, we need to assign
      // g(-kxmax) = conj(f(kxmax)) explicitly.
      if (g.Nx() > f.Nx()) {
	int kx = f.kxmax();
	int fmx = f.mx(kx);
	int gmx = g.mx(-kx);
	for (int kz=f.kzmin(); kz<=f.kzmax(); ++kz) {
	  int fmz = f.mz(kz);
	  int gmz = g.mz(kz);
	  g.cmplx(gmx,ny,gmz,i) = conj(f.cmplx(fmx,ny,fmz,i));
	}
      }
    }
  f.makeState(fxstate, fystate);
  g.makeState(gxstate, gystate);
}

/****************************************************************************
// Commented out and replced with code that only interpolates onto a finer grid
void FlowField::interpolate(const FlowField& u) {
  assert(u.Lx() == Lx_ && u.Lz() == Lz_);
  assert(a_>=u.a() && b_<=u.b());
  assert(Nd_ == u.Nd_);
  u.assertState(Spectral, Spectral);

  int kxmn = Greater(kxmin(), u.kxmin());
  int kxmx = lesser(kxmax(), u.kxmax());
  int kzmn = Greater(kzmin(), u.kzmin()); // == 0
  int kzmx = lesser(kzmax(), u.kzmax());
  int uNy = u.Ny();

  setState(Spectral, Spectral);
  setToZero();

  // chebyshev discretization is the same, copy coeffs
  if (Ny_ == uNy &&  a_ == u.a_ && b_ == u.b_) {
    for (int i=0; i<Nd_; ++i) {
      for (int kx=kxmn; kx<=kxmx; ++kx) {
	int umx = u.mx(kx);
	int tmx = mx(kx);
	for (int kz=kzmn; kz<=kzmx; ++kz) {
	  int umz = u.mz(kz);
	  int tmz = mz(kz); 
	  for (int ny=0; ny<uNy; ++ny) 
	    cmplx(tmx, ny, tmz, i) = u.cmplx(umx, ny, umz, i);
	}
      }
    }
  }  
  else {
    // chebyshev discretization is the different, interpolate coeffs
    ComplexChebyCoeff uprof(u.Ny(), u.a(), u.b(), Spectral);
    ComplexChebyCoeff tprof(Ny_, a_, b_, Physical);
    ChebyTransform trans(Ny_);
    
    for (int i=0; i<Nd_; ++i) {
      for (int kx=kxmn; kx<=kxmx; ++kx) {
	int umx = u.mx(kx);
	int tmx = mx(kx);
	for (int kz=kzmn; kz<=kzmx; ++kz) {
	  int umz = u.mz(kz);
	  int tmz = mz(kz); 
	  for (int uny=0; uny<uNy; ++uny) 
	    uprof.set(uny, u.cmplx(umx, uny, umz, i));
	  tprof.interpolate(uprof);
	  tprof.makeSpectral(trans);
	  for (int tny=0; tny<Ny_; ++tny) 
	    cmplx(tmx, tny, tmz, i) = tprof[tny];
	}
      }
    }
  }
  //makeSpectral();
}
*********************************************************/

/*****************************************************
void FlowField::reflect(const FlowField& u) {
  assert(u.Lx() == Lx_ && u.Lz() == Lz_);
  assert((a_+b_)/2 == u.a() && b_ <= u.b() && b_ > u.a());
  assert(Nd_ == u.Nd_);
  u.assertState(Spectral, Spectral);

  setState(Spectral, Physical);
  setToZero();
  
  ComplexChebyCoeff uprof(u.Ny(), u.a(), u.b(), Spectral);
  ComplexChebyCoeff tprof(Ny_, a_, b_, Physical);

  int kxmn = Greater(kxmin(), u.kxmin());
  int kxmx = lesser(kxmax(), u.kxmax());
  int kzmn = Greater(kzmin(), u.kzmin()); // == 0
  int kzmx = lesser(kzmax(), u.kzmax());
  int uNy = u.Ny();

  for (int i=0; i<Nd_; ++i) {
    // Copy kx==0 coeffs.
    parity p = (i==1) ? Even : Odd;
    for (int kx=kxmn; kx<=kxmx; ++kx) {
      int unx = u.nx(kx);
      int tnx = nx(kx);
      for (int kz=kzmn; kz<=kzmx; ++kz) {
	int unz = u.nz(kz);
	int tnz = nz(kz);
	for (int uny=0; uny<uNy; ++uny) 
	  uprof.set(uny, u.cmplx(unx, uny, unz, i));
	tprof.reflect(uprof,p);
	for (int tny=0; tny<Ny_; ++tny) 
	  cmplx(tnx, tny, tnz, i) = tprof[tny];
      }
    }
  }
  //makeSpectral();
}
********************************************/

/*****************************************************************
void FlowField::interpolate(const FlowField& U) {
  U.assertState(Spectral, Spectral);
  assert(congruent(U));
  setToZero();
  // Interpolation of (x,z) variation is done by simply copying the common
  // Spectral modes. The Spectral mode coefficients can be copied, too,
  // if we're increasing the number of Spectral modes. But if we're 
  // truncating the expansion, we have to use interpolation in order to 
  // meet the boundary conditions at y=+-1.
  int kxm = (kxmax() < U.kxmax()) ? kxmax()-1 : U.kxmax(); 
  int kzm = lesser(kzmax(), U.kzmax());

  //char* t = " x ";
  //cout << "FlowField::interpolate() {" << endl;
  //cout << "from " << U.Nx_ << t << U.Ny_ << t << U.Nz_ << t << U.Nd_ << endl;
  //cout << "  to " << Nx_   << t <<   Ny_ << t <<   Nz_ << t <<   Nd_ << endl;
  //cout << "kxm,kzm == " << kxm << ',' << kzm << endl;
  // Rescale velocities to keep div==0 and u velocities unchanged.

  setState(Spectral, Spectral);
  for (int i=0; i<Nd_; ++i) {
    // Copy kx==0 coeffs.
    for (int kz=0; kz<=kzm; ++kz) 
      for (int ny=0; ny<Ny_; ++ny) 
	cmplx(0, ny, kz, i) = U.cmplx(0, ny, kz, i);
    // Copy kx>0 coeffs, which have negative counterparts.
    for (int kx=1; kx<=kxm; ++kx) {
      for (int kz=0; kz<=kzm; ++kz) 
	for (int ny=0; ny<Ny_; ++ny) {
	  cmplx(kx, ny, kz, i) = U.cmplx(kx, ny, kz, i);
	  cmplx(Nx_-kx, ny, kz, i) = U.cmplx(U.Nx_-kx, ny, kz,i);
	}
    }
  }
  return;
}
****************************************************************/

void FlowField::optimizeFFTW(int flags) {
  if (y_plan_) fftw_destroy_plan(y_plan_);
  if (xz_plan_) fftw_destroy_plan(xz_plan_);
  if (xz_iplan_) fftw_destroy_plan(xz_iplan_);

  flags = flags | FFTW_DESTROY_INPUT;

  fftw_initialize(flags);
}

ComplexChebyCoeff FlowField::profile(int mx, int mz, int i) const {
  ComplexChebyCoeff rtn(Ny_, a_, b_, ystate_);
  if (xzstate_== Spectral) {
    for (int ny=0; ny<Ny_; ++ny)
      rtn.set(ny, cmplx(mx, ny, mz, i));
  }
  else 
    for (int ny=0; ny<Ny_; ++ny)
      rtn.re[ny] = (*this)(mx, ny, mz, i);
    
  return rtn;
}

void FlowField::addProfile(const ChebyCoeff& profile, int i) {
  assert(xzstate_== Spectral);
  assert(ystate_ == profile.state());
  int kx=0;
  int kz=0;
  int m_x = mx(kx);
  int m_z = mz(kz);
  for (int ny=0; ny<Ny_; ++ny) 
    cmplx(m_x, ny, m_z, i) += Complex(profile[ny], 0.0);
}

void FlowField::addProfile(const ComplexChebyCoeff& profile, 
			   int m_x, int m_z, int i, bool addconj) {
  assert(xzstate_== Spectral);
  assert(ystate_ == profile.state());
  int k_x = kx(m_x);
  int k_z = kz(m_z);
  for (int ny=0; ny<Ny_; ++ny) 
    cmplx(m_x, ny, m_z, i) += profile[ny];
  if (addconj && k_x !=0 && k_z ==0) {
    m_x = mx(-k_x);
    for (int ny=0; ny<Ny_; ++ny) 
      cmplx(m_x,ny,m_z, i) += conj(profile[ny]);
  }
}

BasisFunc FlowField::profile(int mx, int mz) const {
  if (xzstate_== Spectral) {
    int k_x = kx(mx);
    int k_z = kz(mz);
    BasisFunc rtn(Ny_, k_x, k_z, Lx_, Lz_, a_, b_, ystate_);
  
    int Nd = lesser(Nd_, 3);
    for (int i=0; i<Nd; ++i)
      for (int ny=0; ny<Ny_; ++ny)
	rtn[i].set(ny, cmplx(mx, ny, mz, i));
    return rtn;
  }
  else {
    BasisFunc rtn(Ny_, 0, 0, Lx_, Lz_, a_, b_, ystate_);
  
    int Nd = lesser(Nd_, 3);
    for (int i=0; i<Nd; ++i)
      for (int ny=0; ny<Ny_; ++ny)
	rtn[i].re[ny] = (*this)(mx, ny, mz, i);
    return rtn;
  }
}

void FlowField::addProfile(const BasisFunc& profile, bool addconj) {
  assert(xzstate_== Spectral);
  assert(ystate_ == profile.state());
  assert(Nd_ == profile.Nd());
  int m_x = mx(profile.kx());
  int m_z = mz(profile.kz());
  for (int i=0; i<Nd_; ++i)
    for (int ny=0; ny<Ny_; ++ny) 
      cmplx(m_x, ny, m_z, i) += profile[i][ny];

  if (addconj && profile.kx() !=0 && profile.kz() ==0) {
    m_x = mx(-profile.kx());
    for (int i=0; i<Nd_; ++i)
      for (int ny=0; ny<Ny_; ++ny) 
	cmplx(m_x, ny, m_z, i) += conj(profile[i][ny]);
  }
}

void FlowField::translate(Real delta_x, Real delta_z) {
  fieldstate xzs = xzstate_;
  makeState(Spectral, ystate_);
  for (int i=0; i<Nd_; ++i)
    for (int ny=0; ny<Ny_; ++ny) 
      for (int mx=0; mx<Mx(); ++mx) {
	Complex cx = exp(Complex(0.0, 2*pi*kx(mx)*delta_x/Lx_));
	for (int mz=0; mz<Mz(); ++mz) {
	  Complex cz = exp(Complex(0.0, 2*pi*kz(mz)*delta_z/Lz_));
	  cmplx(mx,ny,mz,i) *= cx*cz;
	}
      }
  makeState(xzs, ystate_);
}
	  


bool FlowField::geomCongruent(const FlowField& v) const {
  return (Nx_ == v.Nx_ &&
	  Ny_ == v.Ny_ &&
	  Nz_ == v.Nz_ &&
	  Lx_ == v.Lx_ &&
	  Lz_ == v.Lz_ &&
	  a_ == v.a_ &&
	  b_ == v.b_);
}

bool FlowField::congruent(const FlowField& v) const {
  return (Nx_ == v.Nx_ &&
	  Ny_ == v.Ny_ &&
	  Nz_ == v.Nz_ &&
	  Nd_ == v.Nd_ &&
	  Lx_ == v.Lx_ &&
	  Lz_ == v.Lz_ &&
	  a_ == v.a_ &&
	  b_ == v.b_ &&
	  xzstate_ == v.xzstate_ &&
	  ystate_ == v.ystate_);
}

bool FlowField::congruent(const BasisFunc& phi) const {
  return (Ny_==phi.Ny() && Lx_==phi.Lx() && Lz_==phi.Lz() &&
	  a_==phi.a() && b_==phi.b() && ystate_==phi.state());
}


FlowField& FlowField::operator*=(Real x) {
  int Ntotal = Nx_ * Ny_ * Nzpad_ * Nd_;
  for (int i=0; i<Ntotal; ++i)
    rdata_[i] *= x;
  return *this;
}

FlowField& FlowField::operator*=(Complex z) {
  assert(xzstate_ == Spectral);
  int Ntotal = Nx_ * Ny_ * Nzpad2_ * Nd_;
  for (int i=0; i<Ntotal; ++i)
    cdata_[i] *= z;
  return *this;
}

FlowField& FlowField::operator+=(const ChebyCoeff& U) {
  assert(xzstate_== Spectral);
  assert(ystate_ == U.state());
  for (int ny=0; ny<Ny_; ++ny) 
    cmplx(0, ny, 0, 0) += Complex(U[ny], 0.0);
  return *this;
}

FlowField& FlowField::operator-=(const ChebyCoeff& U) {
  assert(xzstate_== Spectral);
  assert(ystate_ == U.state());
  for (int ny=0; ny<Ny_; ++ny) 
    cmplx(0, ny, 0, 0) -= Complex(U[ny], 0.0);
  return *this;
}

FlowField& FlowField::operator+=(const ComplexChebyCoeff& U) {
  assert(xzstate_== Spectral);
  assert(ystate_ == U.state());
  for (int ny=0; ny<Ny_; ++ny) 
    cmplx(0, ny, 0, 0) += U[ny];
  return *this;
}

FlowField& FlowField::operator-=(const ComplexChebyCoeff& U) {
  assert(xzstate_== Spectral);
  assert(ystate_ == U.state());
  for (int ny=0; ny<Ny_; ++ny) 
    cmplx(0, ny, 0, 0) -= U[ny];
  return *this;
}

FlowField& FlowField::operator+=(const BasisFunc& profile){
  addProfile(profile);
  return *this;
}
FlowField& FlowField::operator-=(const BasisFunc& profile){
  BasisFunc copy(profile);
  copy *= -1;
  addProfile(copy);
  return *this;
}
FlowField& FlowField::operator+=(const FlowField& U) {
  assert(congruent(U));
  int Ntotal = Nx_ * Ny_ * Nzpad_ * Nd_;
  for (int i=0; i<Ntotal; ++i)
    rdata_[i] += U.rdata_[i];
  return *this;
}
FlowField& FlowField::operator -= (const FlowField& U) {
  assert(congruent(U));
  int Ntotal = Nx_ * Ny_ * Nzpad_ * Nd_;
  for (int i=0; i<Ntotal; ++i)
    rdata_[i] -= U.rdata_[i];
  return *this;
}
FlowField& FlowField::operator *= (const FlowField& U) {
  assert(congruent(U));
  if (xzstate_ == Spectral) {
    int Ntotal = Nx_ * Ny_ * Nzpad2_ * Nd_;
    for (int i=0; i<Ntotal; ++i)
      cdata_[i] *= U.cdata_[i];
  }
  else {
    int Ntotal = Nx_ * Ny_ * Nzpad_ * Nd_;
    for (int i=0; i<Ntotal; ++i)
      rdata_[i] *= U.rdata_[i];
  }
  return *this;
}
  
void FlowField::realfft_xz() {
  assert (xzstate_ == Physical);
  fftw_execute(xz_plan_);
  // args are (plan, howmany, in, istride, idist, ostride, odist)
  //rfftwnd_real_to_complex(xz_plan_, Nd_*Ny_, rdata_, 1, Nx_*Nzpad_, 0,0,0);
  int Ntotal = Nd_* Nx_ * Ny_ * Nzpad_;
  Real scale = 1.0/(Nx_*Nz_);
  for (int i=0; i<Ntotal; ++i)
    rdata_[i] *= scale;
  xzstate_ = Spectral;
}

void FlowField::irealfft_xz() {
  assert(xzstate_ == Spectral);
  //cout << "rfftwnd_complex_to_real : " << endl;
  //cout << "howmany = " << Nd_*Ny_ << endl;
  //cout << "idist = " << Nx_*Nzpad_/2 << endl;

  fftw_execute(xz_iplan_);
  //rfftwnd_complex_to_real(xz_iplan_, Nd_*Ny_, (fftw_complex*)rdata_, 1, Nx_*Nzpad_/2, 0,0,0);
  xzstate_ = Physical;
}

void FlowField::chebyfft_y() {
  assert(ystate_ == Physical);
  Real nrm = 1.0/(Ny_-1); // needed because FFTW does unnormalized transforms
  for (int i=0; i<Nd_; ++i) 
    for (int nx=0; nx<Nx_; ++nx) 
      for (int nz=0; nz<Nzpad_; ++nz) {

	// Copy data spread through memory into a stride-1 scratch array.
	for (int ny=0; ny<Ny_; ++ny) 
	  scratch_[ny] = rdata_[flatten(nx, ny, nz, i)];

	// Transform data
	fftw_execute_r2r(y_plan_, scratch_, scratch_);

	// Copy back to multi-d arrays, normalizing on the way
	// 0th elem is different because of reln btwn cos and Cheb transforms
	rdata_[flatten(nx, 0, nz, i)] = 0.5*nrm*scratch_[0]; 
	for (int ny=1; ny<Ny_; ++ny) 
	  rdata_[flatten(nx, ny, nz, i)] = nrm*scratch_[ny];
      }
  ystate_ = Spectral;
  return;
}

void FlowField::ichebyfft_y() {
  assert(ystate_ == Spectral);
  for (int i=0; i<Nd_; ++i) 
    for (int nx=0; nx<Nx_; ++nx) 
      for (int nz=0; nz<Nzpad_; ++nz) {

	// Copy data spread through memory into a stride-1 scratch array.
	// 0th elem is different because of reln btwn cos and Cheb transforms
	scratch_[0] = 2.0*rdata_[flatten(nx, 0, nz, i)];
	for (int ny=1; ny<Ny_; ++ny) 
	  scratch_[ny] = rdata_[flatten(nx, ny, nz, i)];
	
	// Transform data
	fftw_execute_r2r(y_plan_, scratch_, scratch_);

	// Copy transformed data back into main data array
	for (int ny=0; ny<Ny_; ++ny) 
	  rdata_[flatten(nx, ny, nz, i)] = 0.5*scratch_[ny];

      }
  ystate_ = Physical;
  return;
}

void FlowField::makeSpectral_xz() {if (xzstate_==Physical) realfft_xz();}
void FlowField::makePhysical_xz() {if (xzstate_==Spectral) irealfft_xz();}
void FlowField::makeSpectral_y()  {if (ystate_==Physical) chebyfft_y();}
void FlowField::makePhysical_y()  {if (ystate_==Spectral) ichebyfft_y();}
void FlowField::makeSpectral()  {makeSpectral_xz(); makeSpectral_y();}
void FlowField::makePhysical()  {makePhysical_y(); makePhysical_xz();}
void FlowField::makeState(fieldstate xzstate, fieldstate ystate)  {
  (xzstate == Physical) ? makePhysical_xz() : makeSpectral_xz();
  (ystate  == Physical) ? makePhysical_y()  : makeSpectral_y();
}

Complex FlowField::Dx(int mx, int n) const {
  Complex rot(0.0, 0.0);
  switch(n % 4) {
  case 0: rot = Complex( 1.0, 0.0); break;
  case 1: rot = Complex( 0.0, 1.0); break;
  case 2: rot = Complex(-1.0, 0.0); break;
  case 3: rot = Complex( 0.0,-1.0); break;
  default: cferror("FlowField::Dx(mx,n) : impossible: n % 4 > 4 !!");
  }
  int kx_ = kx(mx);
  return rot*(pow(2*pi*kx_/Lx_, n)*zero_last_mode(kx_,kxmax(),n));
}
Complex FlowField::Dz(int mz, int n) const {
  Complex rot(0.0, 0.0);
  switch(n % 4) {
  case 0: rot = Complex( 1.0, 0.0); break;
  case 1: rot = Complex( 0.0, 1.0); break;
  case 2: rot = Complex(-1.0, 0.0); break;
  case 3: rot = Complex( 0.0,-1.0); break;
  default: cferror("FlowField::Dx(mx,n) : impossible: n % 4 > 4 !!");
  }
  int kz_ = kz(mz);
  return rot*(pow(2*pi*kz_/Lz_, n)*zero_last_mode(kz_,kzmax(),n));
}

Complex FlowField::Dx(int mx) const {
  int kx_ = kx(mx);
  return Complex(0.0, 2*pi*kx_/Lx_*zero_last_mode(kx_,kxmax(),1));
}
Complex FlowField::Dz(int mz) const {
  int kz_ = kz(mz);
  return Complex(0.0, 2*pi*kz_/Lz_*zero_last_mode(kz_,kzmax(),1));
}


void FlowField::addPerturbation(int kx, int kz, Real mag, Real decay) {
  assertState(Spectral, Spectral);
  if (mag == 0.0)
    return;

  // Add a dive-free perturbation to the base flow.
  ComplexChebyCoeff u(Ny_, a_, b_, Spectral);
  ComplexChebyCoeff v(Ny_, a_, b_, Spectral);
  ComplexChebyCoeff w(Ny_, a_, b_, Spectral);
  randomProfile(u,v,w, kx, kz, Lx_, Lz_, mag, decay);
  int m_x = mx(kx);
  int m_z = mz(kz);
  for (int ny=0; ny<Ny_; ++ny) {
    cmplx(m_x, ny, m_z, 0) += u[ny];
    cmplx(m_x, ny, m_z, 1) += v[ny];
    cmplx(m_x, ny, m_z, 2) += w[ny];
  }
  if (kz==0 && kx!=0) {
    int m_x = mx(-kx);
    int m_z = mz(0); // -kz=0
    for (int i=0; i<Nd_; ++i)
      for (int ny=0; ny<Ny_; ++ny) {
	cmplx(m_x, ny, m_z, 0) += conj(u[ny]);
	cmplx(m_x, ny, m_z, 1) += conj(v[ny]);
	cmplx(m_x, ny, m_z, 2) += conj(w[ny]);
      }
  }
  return;
}

void FlowField::perturb(Real mag, Real decay) {
  addPerturbations(kxmax(), kzmax(), mag, decay);
}

void FlowField::addPerturbations(int Kx, int Kz, Real mag, Real decay) {
  assertState(Spectral, Spectral);
  if (mag == 0.0)
    return;

  int Kxmin = Greater(-Kx, kxmin());
  int Kxmax = lesser(Kx, kxmax()-1);
  int Kzmax = lesser(Kz, kzmax()-1);

  // Add a div-free perturbation to the base flow.
  for (int kx=Kxmin; kx<=Kxmax; ++kx) 
    for (int kz=0; kz<=Kzmax; ++kz) {
      //Real norm = pow(10.0, -(abs(2*pi*kx/Lx_) + abs(2*pi*kz/Lz_)));
      //Real norm = pow(decay, 2*(abs(kx) + abs(kz)));
      Real norm = pow(decay, abs(2*pi*kx/Lx_) + abs(2*pi*kz/Lz_));
      if (!(kx==0 && kz==0))
	addPerturbation(kx, kz, mag*norm, decay);
    }
  makePhysical();
  makeSpectral();
  return;
}

void FlowField::addPerturbations(Real mag, Real decay) {
  assertState(Spectral, Spectral);
  if (mag == 0.0)
    return;

  // Add a div-free perturbation to the base flow.
  for (int mx=0; mx<Mx(); ++mx) {
    int kx_ = kx(mx);
    for (int mz=0; mz<Mz(); ++mz) {
      int kz_ = kz(mz);
      Real norm = pow(decay, 2*(abs(kx_) + abs(kz_)));
      addPerturbation(kx_, kz_, mag*norm, decay);
    }
  }
  makePhysical();
  makeSpectral();
  return;
}
void FlowField::dealiasIO(bool b) {dealiasIO_ = b;}

void FlowField::setToZero() {
  int Ntotal = Nx_*Ny_*Nzpad_*Nd_;
  for (int i=0; i<Ntotal; ++i)
    rdata_[i] = 0.0;
}

void FlowField::print() const {
  cout << Nx_ << " x " << Ny_ << " x " << Nz_ << endl;
  cout << "[0, " << Lx_ << "] x [-1, 1] x [0, " << Lz_ << "]" << endl;
  cout << xzstate_ << " x " << ystate_ << " x " << xzstate_ << endl;
  cout << xzstate_ << " x " << ystate_ << " x " << xzstate_ << endl;
  if (xzstate_ == Spectral) {
  cout << "FlowField::print() real view " << endl;
    for (int i=0; i<Nd_; ++i) {
      for (int ny=0; ny<Ny_; ++ny) {
	for (int nx=0; nx<Nx_; ++nx) {
 	  cout << "i=" << i << " ny=" << ny << " nx= " << nx << ' ';
 	  int nz; // MSVC++ FOR-SCOPE BUG
	  for (nz=0; nz<Nz_; ++nz) 
	    cout << rdata_[flatten(nx,ny,nz,i)] << ' ';
	  cout << " pad : ";
	  for (nz=Nz_; nz<Nzpad_; ++nz) 
	    cout << rdata_[flatten(nx,ny,nz,i)] << ' ';
	  cout << endl;
	}
      }
    }
  }
  else {
    cout << "complex view " << endl;
    for (int i=0; i<Nd_; ++i) {
       for (int ny=0; ny<Ny_; ++ny) {
	 for (int nx=0; nx<Nx_; ++nx) {
	   cout << "i=" << i << " ny=" << ny << " nx= " << nx << ' ';
	   for (int nz=0; nz<Nz_/2; ++nz) 
	     cout << cdata_[complex_flatten(nx,ny,nz,i)] << ' ';
	   cout << endl;
	 }
       }    
    }
  }
}

// k == direction of normal    (e.g. k=0 means a x-normal slice in yz plane.
// i == component of FlowField (e.g. i=0 means u-component)
// n == nth gridpoint along k direction
void FlowField::saveSlice(int k, int i, int nk, const string& filebase) const {
  assert(k>=0 && k<3);
  assert(i>=0 && i<Nd_);

  string filename(filebase);
  filename += string(".asc");
  ofstream os(filename.c_str());

  FlowField& u = (FlowField&) *this;  // cast away constness
  fieldstate xzstate = xzstate_;
  fieldstate ystate = ystate_;

  u.makePhysical();

  switch (k) {
  case 0: 
    os << "% yz slice\n";
    os << "% (i,j)th elem is field at (x_n, y_i, z_j)\n";
    for (int ny=0; ny<Ny_; ++ny) {
      for (int nz=0; nz<Nz_; ++nz) 
	os << u(nk, ny, nz, i) << ' ';
      os << '\n';
    }
    break;
  case 1: // xz slice
    os << "% xz slice\n";
    os << "% (i,j)th elem is field at (x_j, y_n, z_i)\n";
    for (int nz=0; nz<Nz_; ++nz) {
      for (int nx=0; nx<Nx_; ++nx) 
	os << (*this)(nx, nk, nz, i) << ' ';
      os << '\n';
    }
    break;
  case 2: 
    os << "% yz slice\n";
    os << "% (i,j)th elem is field at (x_j, y_i, z_n)\n";
    for (int ny=0; ny<Ny_; ++ny) {
      for (int nx=0; nx<Nx_; ++nx) 
	os << (*this)(nx, ny, nk, i) << ' ';
      os << '\n';
    }
    break;
  }
  u.makeState(xzstate, ystate);
}

void FlowField::saveProfile(int mx, int mz, const string& filebase) const {
  ChebyTransform trans(Ny_);
  saveProfile(mx,mz,filebase, trans);
}

void FlowField::saveProfile(int mx, int mz, const string& filebase, const ChebyTransform& trans) const {
  assert(xzstate_ == Spectral);
  string filename(filebase);
  if (Nd_ == 3)
    filename += string(".bf");  // this convention is unfortunate, need to fix
  else 
    filename += string(".asc");

  ofstream os(filename.c_str());
  os << setprecision(REAL_DIGITS);

  if (ystate_ == Physical) {
    for (int ny=0; ny<Ny_; ++ny) {
      for (int i=0; i<Nd_; ++i) {
	Complex c = (*this).cmplx(mx, ny, mz, i);
	os << Re(c) << ' ' << Im(c) << ' ';
      }
      os << '\n';
    }
  }
  else {
    ComplexChebyCoeff* f = new ComplexChebyCoeff[Nd_];
    for (int i=0; i<Nd_; ++i) {
      f[i] = ComplexChebyCoeff(Ny_, a_, b_, Spectral);
      for (int ny=0; ny<Ny_; ++ny) 
	f[i].set(ny, (*this).cmplx(mx, ny, mz, i));
      f[i].makePhysical(trans);
    }
    for (int ny=0; ny<Ny_; ++ny) {
      for (int i=0; i<Nd_; ++i) 
	os << f[i].re[ny] << ' ' << f[i].im[ny] << ' ';
      os << '\n';
    }
  }
}
void FlowField::saveSpectrum(const string& filebase, int i, int ny, bool pretty) const {
  string filename(filebase);
  filename += string(".asc");
  ofstream os(filename.c_str());

  bool sum = (ny == -1) ? true : false;
  assert(xzstate_ == Spectral);
  
  if (pretty) {
    for (int kx=kxmin(); kx<kxmax(); ++kx) {
      for (int kz=kzmin(); kz<kzmax(); ++kz) {
	if (sum) 
	  os << sqrt(energy(mx(kx),mz(kz))) << ' ';
	else {
	  Complex f = this->cmplx(mx(kx), ny, mz(kz), i);
	  os << Re(f) << ' ' << Im(f) << ' ';
	}
      }
      os << endl;
    }
  }
  else {
    for (int mx=0; mx<Mx(); ++mx) {
      for (int mz=0; mz<Mz(); ++mz) {
	if (sum) 
	  os << sqrt(energy(mx,mz)) << ' ';
	else {
	  Complex f = this->cmplx(mx, ny, mz, i);
	  os << Re(f) << ' ' << Im(f) << ' ';
	}
      }
      os << endl;
    }
  }
  os.close();
}

void FlowField::saveSpectrum(const string& filebase, bool pretty) const {
  string filename(filebase);
  filename += string(".asc");
  ofstream os(filename.c_str());

  assert(xzstate_ == Spectral && ystate_ == Spectral);
 
  ComplexChebyCoeff u(Ny_,a_,b_,Spectral);

  if (pretty) {
    for (int kx=kxmin(); kx<kxmax(); ++kx) {
      for (int kz=kzmin(); kz<kzmax(); ++kz) {
	Real e = 0.0;
	for (int i=0; i<Nd_; ++i) {
	  for (int ny=0; ny<Ny_; ++ny) 
	    u.set(ny,this->cmplx(mx(kx),ny,mz(kz),i));
	  e += L2Norm2(u);
	}
	os << sqrt(e) << ' ';
      }
      os << endl;
    }
  }
  else {
    for (int mx=0; mx<Mx(); ++mx) {
      for (int mz=0; mz<Mz(); ++mz) {
	Real e = 0.0;
	for (int i=0; i<Nd_; ++i) {
	  for (int ny=0; ny<Ny_; ++ny) 
	    u.set(ny,this->cmplx(mx,ny,mz,i));
	  e += L2Norm2(u);
	}
	os << sqrt(e) << ' ';
      }
      os << endl;
    }
  }
  os.close();
}

/*******************************************************************
void FlowField::saveDissSpectrum(const string& filebase, Real nu) const {
  string filename(filebase);
  filename += string(".asc");
  ofstream os(filename.c_str());

  assert(xzstate_ == Spectral && ystate_ == Spectral);
  BasisFunc prof(Ny_, 0, 0, Lx_, Lz_, a_, b_, Spectral);
  for (int mx=0; mx<Mx(); ++mx) {
    for (int mz=0; mz<Mz(); ++mz) {
      os << dissipation(mx,mz,nu) << ' ';
    }
    os << endl;
  }
  os.close();
}
*******************************************************************/

void FlowField::saveDivSpectrum(const string& filebase, bool pretty) const {
  string filename(filebase);
  filename += string(".asc");
  ofstream os(filename.c_str());

  assert(xzstate_ == Spectral && ystate_ == Spectral);
  
  //assert(congruent(div));
  assert(Nd_ >= 3);

  ComplexChebyCoeff v(Ny_, a_, b_, Spectral);
  ComplexChebyCoeff vy(Ny_, a_, b_, Spectral);

  // Rely on compiler to pull loop invariants out of loops
  if (pretty) {

    for (int kx=kxmin(); kx<kxmax(); ++kx) {
      Complex d_dx(0.0, 2*pi*kx/Lx_*zero_last_mode(kx,kxmax(),1));

      for (int kz=kzmin(); kz<kzmax(); ++kz) {
	Complex d_dz(0.0, 2*pi*kz/Lz_*zero_last_mode(kz,kzmax(),1));

	for (int ny=0; ny<Ny_; ++ny) 
	  v.set(ny, cmplx(mx(kx),ny,mz(kz),1));
	diff(v,vy);

	Real div = 0.0;
	for (int ny=0; ny<Ny_; ++ny) {
	  Complex ux = d_dx*cmplx(mx(kx),ny,mz(kz),0);
	  Complex wz = d_dz*cmplx(mx(kx),ny,mz(kz),2);
	  div += abs2(ux + vy[ny] + wz);
	}
	os << sqrt(div) << ' ';
      }
      os << endl;
    }
  }
  else {
    for (int mx=0; mx<Mx(); ++mx) {
      Complex d_dx(0.0, 2*pi*kx(mx)/Lx_*zero_last_mode(kx(mx),kxmax(),1));
      
      for (int mz=0; mz<Mz(); ++mz) {
	Complex d_dz(0.0, 2*pi*kz(mz)/Lz_*zero_last_mode(kz(mz),kzmax(),1));
	  
	for (int ny=0; ny<Ny_; ++ny) 
	  v.set(ny, cmplx(mx,ny,mz,1));
	diff(v,vy);

	Real div = 0.0;
	for (int ny=0; ny<Ny_; ++ny) {
	  Complex ux = d_dx*cmplx(mx,ny,mz,0);
	  Complex wz = d_dz*cmplx(mx,ny,mz,2);
	  div += abs2(ux + vy[ny] + wz);
	}
	os << sqrt(div) << ' ';
      }
      os << endl;
    }
  }
  os.close();
}

void FlowField::asciiSave(const string& filebase) const {
  string filename(filebase);
  filename += ".asc";

  ofstream os(filename.c_str());
  os << scientific << setprecision(REAL_DIGITS);

  const char s = ' ';
  const char nl = '\n';
  const int w = REAL_IOWIDTH;
  os << "% Channelflow FlowField data" << nl;
  os << "% xzstate == " << xzstate_ << nl;
  os << "%  ystate == " << ystate_ << nl;
  os << "% Nx Ny Nz Nd == " << Nx_ <<s<< Ny_ <<s<< Nz_ <<s<< Nd_ << " gridpoints\n";
  os << "% Mx My Mz Nd == " << Mx() <<s<< My() <<s<< Mz() <<s<< Nd_ << " spectral modes\n";
  os << "% Lx Lz == " << setw(w) << Lx_ <<s<< setw(w) << Lz_ << nl;
  os << "% Lx Lz == " << setw(w) << Lx_ <<s<< setw(w) << Lz_ << nl;
  os << "% a  b  == " << setw(w) << a_ <<s<< setw(w) << b_ << nl;
  os << "% loop order:\n";
  os << "%   for (int i=0; i<Nd; ++i)\n";
  os << "%     for(long ny=0; ny<Ny; ++ny) // note: Ny == My\n";
  if (xzstate_ == Physical) {
    os << "%       for (int nx=0; nx<Nx; ++nx)\n";
    os << "%         for (int nz=0; nz<Nz; ++nz)\n";
    os << "%           os << f(nx, ny, nz, i) << newline;\n";
  } 
  else {
    os << "%       for (int mx=0; mx<Mx; ++mx)\n";
    os << "%         for (int mz=0; mz<Mz; ++mz)\n";
    os << "%           os << Re(f.cmplx(mx, ny, mz, i) << ' ' << Im(f.cmplx(mx, ny, mz, i) << newline;\n";
  }
  if (xzstate_ == Physical) {
    for (int i=0; i<Nd_; ++i) 
      for(long ny=0; ny<Ny_; ++ny)
	for (int nx=0; nx<Nx_; ++nx)
	  for (int nz=0; nz<Nz_; ++nz)
	    os << setw(w) << (*this)(nx, ny, nz, i) << nl;
  }
  else {
    for (int i=0; i<Nd_; ++i) 
      for(long ny=0; ny<Ny_; ++ny)
	for (int mx=0; mx<Mx(); ++mx)
	  for (int mz=0; mz<Mz(); ++mz) 
	    os << setw(w) << Re(cmplx(mx, ny, mz, i)) << s
	       << setw(w) << Im(cmplx(mx, ny, mz, i)) << nl;
  }
  os.close();
}
      
void FlowField::binarySave(const string& filebase) const {
  string filename(filebase);
  filename += string(".ff"); // "flow field"
  ofstream os(filename.c_str(), ios::out | ios::binary);
  if (!os.good()) {
    cerr << "FlowField::binarySave(filebase) : can't open file " << filename << endl;
    abort();
  }

  write(os, CHANNELFLOW_MAJOR_VERSION);
  write(os, CHANNELFLOW_MINOR_VERSION);
  write(os, CHANNELFLOW_UPDATE_VERSION);
  write(os, Nx_);
  write(os, Ny_);
  write(os, Nz_);
  write(os, Nd_);
  write(os, xzstate_);
  write(os, ystate_);
  write(os, Lx_);
  write(os, Lz_);
  write(os, a_);
  write(os, b_);
  write(os, dealiasIO_);

  // Write data only for non-aliased modes.
  if (dealiasIO_ && xzstate_ == Spectral) {
    int Nxd=2*(Nx_/6);
    int Nzd=2*(Nz_/3)+1;

    // In innermost loop, array index is (nz + Nzpad2_*(nx + Nx_*(ny + Ny_*i))),
    // which is the same as the FlowField::flatten function.
    for (int i=0; i<Nd_; ++i) {
      for (int ny=0; ny<Ny_; ++ny) {

	for (int nx=0; nx<=Nxd; ++nx) {
	  for (int nz=0; nz<=Nzd; ++nz)
	    write(os, rdata_[flatten(nx,ny,nz,i)]);
	}
	for (int nx=Nx_-Nxd; nx<Nx_; ++nx) {
	  for (int nz=0; nz<=Nzd; ++nz)
	    write(os, rdata_[flatten(nx,ny,nz,i)]);
	}
      }
    }
  }
  else {
    int Ntotal = Nd_* Nx_ * Ny_ * Nzpad_;
    for (int i=0; i<Ntotal; ++i)
      write(os, rdata_[i]);
  }
}

void FlowField::dump() const {
  for (int i=0; i<Nx_ * Ny_ * Nzpad_ * Nd_; ++i)
    cout << rdata_[i] << ' ';
  cout << endl;
}

Real FlowField::energy(bool normalize) const {
  assert(xzstate_ == Spectral && ystate_ == Spectral);
  return L2Norm2(*this, normalize);
}
Real FlowField::energy(int mx, int mz, bool normalize) const {
  assert(xzstate_ == Spectral && ystate_ == Spectral);
  ComplexChebyCoeff u(Ny_,a_,b_,Spectral);
  Real e = 0.0;
  for (int i=0; i<Nd_; ++i) {
    for (int ny=0; ny<Ny_; ++ny) 
      u.set(ny,this->cmplx(mx,ny,mz,i));
    e += L2Norm2(u, normalize);
  }
  if (!normalize)
    e *= Lx_*Lz_;
  return e;
}

Real FlowField::dudy_a() const {
  assert(ystate_ == Spectral);
  BasisFunc prof = profile(0,0);
  ChebyCoeff dudy = diff(Re(prof.u()));
  return dudy.eval_a();
}
Real FlowField::dudy_b() const {
  assert(ystate_ == Spectral);
  BasisFunc prof = profile(0,0);
  ChebyCoeff dudy = diff(Re(prof.u()));
  return dudy.eval_b();
}

Real FlowField::CFLfactor() const {
  assert(Nd_ == 3);
  FlowField& u = (FlowField&) *this;
  fieldstate xzstate = xzstate_;
  fieldstate ystate = ystate_;
  u.makePhysical();
  Vector y = chebypoints(Ny_, a_, b_);
  Real cfl = 0.0;
  Vector dx(3);
  dx[0] = Lx_/Nx_;
  dx[2] = Lz_/Nz_;
  for (int i=0; i<Nd_; ++i)
    for (int ny=0; ny<Ny_; ++ny) { 
      dx[1] = (ny==0 || ny==Ny_-1) ? y[0]-y[1] : (y[ny-1]-y[ny+1])/2.0;
      for (int nx=0; nx<Nx_; ++nx) 
	for (int nz=0; nz<Nz_; ++nz) 
	  cfl = Greater(cfl, abs(u(nx,ny,nz,i)/dx[i]));
  }
  u.makeState(xzstate,ystate);

  return cfl;
}

Real FlowField::CFLfactor(const ChebyCoeff& Ubase) const {
  if (Ubase.N() == 0)
    return CFLfactor();

  assert(Nd_ == 3);
  assert(Ubase.state() == Physical);
  FlowField& u = (FlowField&) *this;
  fieldstate xzstate = xzstate_;
  fieldstate ystate = ystate_;
  u.makePhysical();
  Vector y = chebypoints(Ny_, a_, b_);
  Real cfl = 0.0;
  Vector dx(3);
  dx[0] = Lx_/Nx_;
  dx[2] = Lz_/Nz_;
  for (int i=0; i<3; ++i)
    for (int ny=0; ny<Ny_; ++ny) {
      dx[1] = (ny==0 || ny==Ny_-1) ? y[0]-y[1] : (y[ny-1]-y[ny+1])/2.0;
      Real U = (i==0) ? Ubase(ny) : 0;
      for (int nx=0; nx<Nx_; ++nx) 
	for (int nz=0; nz<Nz_; ++nz) 
	  cfl = Greater(cfl, (u(nx,ny,nz,i)+U)/dx[i]);
    }
  u.makeState(xzstate,ystate);

  return cfl;
}

/********************************************************
Real FlowField::dissipation(int mx, int mz, Real nu) const {
  assert(xzstate_ == Spectral && ystate_== Spectral);
  BasisFunc prof;
  Real d=0.0;
  for (int _=0; _<Nd_; ++_) {
    prof = profile(mx, mz);
    d += nu*Re(L2InnerProduct(prof, ::laplacian(prof), false));
  }
  return d;
}
***********************************************************/

void FlowField::setState(fieldstate xz, fieldstate y) {
  xzstate_ = xz;
  ystate_ = y;
}
void FlowField::assertState(fieldstate xz, fieldstate y) const {
  assert(xzstate_ == xz && ystate_ == y);
}

void swap(FlowField& f, FlowField& g) {
  assert(f.congruent(g));
  Real* tmp = f.rdata_;
  f.rdata_ = g.rdata_;
  g.rdata_ = tmp;

  f.cdata_ = (Complex*)f.rdata_;
  g.cdata_ = (Complex*)g.rdata_;

  // We must swap the FFTW3 plans, as well, since they're tied to specific 
  // memory locations. This burned me when I upgraded FFTW2->FFTW3.
  fftw_plan tmp_plan;
  tmp_plan = f.xz_plan_;
  f.xz_plan_ = g.xz_plan_;
  g.xz_plan_ = tmp_plan;

  tmp_plan = f.xz_iplan_;
  f.xz_iplan_ = g.xz_iplan_;
  g.xz_iplan_ = tmp_plan;
  
  tmp_plan = f.y_plan_;
  f.y_plan_ = g.y_plan_;
  g.y_plan_ = tmp_plan;
}
  


Real bcNorm2(const FlowField& f, bool normalize) {
  assert(f.xzstate() == Spectral);
  Real bc2 = 0.0;
  int Mx = f.Mx();
  int My = f.My();
  int Mz = f.Mz();
  ComplexChebyCoeff prof(f.Ny(), f.a(), f.b(), f.ystate());
  for (int i=0; i<f.Nd(); ++i) 
    for (int mx=0; mx<Mx; ++mx) {
      for (int mz=0; mz<Mz; ++mz) {
	for (int my=0; my<My; ++my)
	  prof.set(my, f.cmplx(mx,my,mz,i));
	bc2 += abs2(prof.eval_a());
	bc2 += abs2(prof.eval_b());
      }
    }
  if (!normalize)
    bc2 *= f.Lx()*f.Lz();
  return bc2;
}

Real bcDist2(const FlowField& f, const FlowField& g, bool normalize) {
  assert(f.congruent(g));
  assert(f.xzstate() == Spectral && g.xzstate() == Spectral);
  //assert(f.ystate() == Spectral && g.ystate() == Spectral);

  Real bc2 = 0.0;
  int Nd = f.Nd();
  int Mx = f.Mx();
  int My = f.My();
  int Mz = f.Mz();
  ComplexChebyCoeff diff(My, f.a(), f.b(), f.ystate());

  for (int i=0; i<Nd; ++i) 
    for (int mx=0; mx<Mx; ++mx) {
      for (int mz=0; mz<Mz; ++mz) {
	for (int my=0; my<My; ++my)
	  diff.set(my, f.cmplx(mx,my,mz,i)-g.cmplx(mx,my,mz,i));
	bc2 += abs2(diff.eval_a());
	bc2 += abs2(diff.eval_b());
      }
    }
  if (!normalize)
    bc2 *= f.Lx()*f.Lz();
  return bc2;
}


Real bcNorm(const FlowField& f, bool normalize) {
  return sqrt(bcNorm2(f, normalize));
}

Real bcDist(const FlowField& f, const FlowField& g, bool normalize) {
  return sqrt(bcDist2(f,g, normalize));
}

Real divNorm(const FlowField& f, bool normalize)  {
  return sqrt(divNorm2(f,normalize));
}

Real divNorm2(const FlowField& f, bool normalize)  {
  assert(f.xzstate() == Spectral && f.ystate() == Spectral);
  assert(f.Nd() == 3);

  Real div2=0.0;
  int Mx = f.Mx();
  int Mz = f.Mz();
  BasisFunc prof;
  for (int mx=0; mx<Mx; ++mx) 
    for (int mz=0; mz<Mz; ++mz) 
      div2 += divNorm2(f.profile(mx,mz), normalize);
  return div2;
}
Real divDist2(const FlowField& f, const FlowField& g, bool normalize)  {
  assert(f.xzstate() == Spectral && f.ystate() == Spectral);
  assert(f.congruent(g));
  assert(f.Nd() == 3);

  Real div2=0.0;
  int Mx = f.Mx();
  int Mz = f.Mz();
  for (int mx=0; mx<Mx; ++mx)
    for (int mz=0; mz<Mz; ++mz) 
      div2 += divDist2(f.profile(mx,mz), g.profile(mx,mz), normalize);
  return div2;
}

Real divDist(const FlowField& f, const FlowField& g, bool normalize) {
  return sqrt(divDist2(f,g,normalize));
}

Real L1Norm(const FlowField& u, bool normalize) {
  fieldstate xstate = u.xzstate();
  fieldstate ystate = u.ystate();
  ((FlowField&) u).makePhysical_xz();

  Real sum = 0.0;
  
  ChebyCoeff prof; 
  ChebyTransform trans(u.Ny());
  for (int i=0; i<u.vectorDim(); ++i)
    for (int nx=0; nx<u.Nx(); ++nx) 
      for (int nz=0; nz<u.Nz(); ++nz) {
	prof = Re(u.profile(nx,nz,i));
	prof.makePhysical(trans);
	sum += L1Norm(prof);
      }

  sum /= u.Nx()*u.Nz();

  if (!normalize)
    sum *= u.Lx()*u.Lz();

  ((FlowField&) u).makeState(xstate, ystate);

  return sum;
}

Real L1Dist(const FlowField& f, const FlowField& g, bool normalize) {
  assert(f.congruent(g));
  fieldstate fxstate = f.xzstate();
  fieldstate gxstate = g.xzstate();
  fieldstate fystate = f.ystate();
  fieldstate gystate = g.ystate();

  ((FlowField&) f).makePhysical_xz();
  ((FlowField&) g).makePhysical_xz();

  Real sum = 0.0;
  
  ChebyCoeff prof; 
  ChebyTransform trans(f.Ny());
  for (int i=0; i<f.vectorDim(); ++i)
    for (int nx=0; nx<f.Nx(); ++nx) 
      for (int nz=0; nz<f.Nz(); ++nz) {
	prof = Re(f.profile(nx,nz,i));
	prof -= Re(g.profile(nx,nz,i));
	prof.makePhysical(trans);
	sum += L1Norm(prof, normalize);
      }

  sum /= f.Nx()*f.Nz();

  if (!normalize)
    sum *= f.Lx()*f.Lz();

  ((FlowField&) f).makeState(fxstate, fystate);
  ((FlowField&) g).makeState(gxstate, gystate);

  return sum;
}

Real LinfNorm(const FlowField& u) {
  fieldstate xstate = u.xzstate();
  fieldstate ystate = u.ystate();
  ((FlowField&) u).makePhysical_xz();

  Real rtn = 0.0;
  
  ChebyCoeff prof; 
  ChebyTransform trans(u.Ny());
  for (int i=0; i<u.vectorDim(); ++i)
    for (int nx=0; nx<u.Nx(); ++nx) 
      for (int nz=0; nz<u.Nz(); ++nz) {
	prof = Re(u.profile(nx,nz,i));
	prof.makePhysical(trans);
	rtn = Greater(rtn, LinfNorm(prof));
      }

  ((FlowField&) u).makeState(xstate, ystate);

  return rtn;
}

Real LinfDist(const FlowField& f, const FlowField& g) {
  assert(f.congruent(g));
  fieldstate fxstate = f.xzstate();
  fieldstate gxstate = g.xzstate();
  fieldstate fystate = f.ystate();
  fieldstate gystate = g.ystate();

  ((FlowField&) f).makePhysical_xz();
  ((FlowField&) g).makePhysical_xz();

  Real rtn = 0.0;
  
  ChebyCoeff prof; 
  ChebyTransform trans(f.Ny());
  for (int i=0; i<f.vectorDim(); ++i)
    for (int nx=0; nx<f.Nx(); ++nx) 
      for (int nz=0; nz<f.Nz(); ++nz) {
	prof = Re(f.profile(nx,nz,i));
	prof -= Re(g.profile(nx,nz,i));
	prof.makePhysical();
	rtn = Greater(rtn, LinfNorm(prof));
      }

  ((FlowField&) f).makeState(fxstate, fystate);
  ((FlowField&) g).makeState(gxstate, gystate);

  return rtn;
}

Real L2Dist(const FlowField& u,const FlowField& v, bool normalize) {
  return sqrt(L2Dist2(u,v,normalize));
}

Real L2Dist2(const FlowField& u, const FlowField& v, bool normalize) {
  Real sum = 0.0;
  assert(u.congruent(v));
  assert(u.xzstate()==Spectral && u.ystate()==Spectral);
  ComplexChebyCoeff u_v(u.Ny(), u.a(), u.b(), Spectral);
  for (int i=0; i<u.vectorDim(); ++i)
    for (int mx=0; mx<u.Mx(); ++mx) {
      int cz = 1;
      for (int mz=0; mz<u.Mz(); ++mz) {
	for (int ny=0; ny<u.Ny(); ++ny) 
	  u_v.set(ny, u.cmplx(mx,ny,mz,i)-v.cmplx(mx,ny,mz,i));
	sum += cz*L2Norm2(u_v, normalize);
	cz = 2;
      }
    }
  if (!normalize)
    sum *= u.Lx()*u.Lz();
  return sum;
}

Real L2Norm(const FlowField& u, bool normalize) {
  return sqrt(L2Norm2(u, normalize));
}

Real L2Norm2(const FlowField& u, bool normalize) {
  assert(u.ystate() == Spectral);
  assert(u.xzstate()==Spectral);
  Real sum = 0.0;
  ComplexChebyCoeff prof(u.Ny(), u.a(), u.b(), Spectral);
  for (int i=0; i<u.vectorDim(); ++i)
    for (int mx=0; mx<u.Mx(); ++mx) {
      Real cz = 1.0; // cz = 2 for kz>0 to take account of kz<0 ghost modes
      for (int mz=0; mz<u.Mz(); ++mz) {
	for (int ny=0; ny<u.Ny(); ++ny) 
	  prof.set(ny, u.cmplx(mx,ny,mz,i));
	sum += cz*L2Norm2(prof, normalize);
	cz = 2.0;
      }
    }
  if (!normalize)
    sum *= u.Lx()*u.Lz();
  return sum;
}

Real L2InnerProduct(const FlowField& f, const FlowField& g, bool normalize) {
  assert(f.ystate() == Spectral);
  assert(g.ystate() == Spectral);
  assert(f.xzstate() == Spectral);
  assert(g.xzstate() == Spectral);
  assert(f.congruent(g));

  Real sum = 0.0;
  ComplexChebyCoeff fprof(f.Ny(), f.a(), f.b(), Spectral);
  ComplexChebyCoeff gprof(g.Ny(), g.a(), g.b(), Spectral);
  for (int i=0; i<f.vectorDim(); ++i)
    for (int mx=0; mx<f.Mx(); ++mx) {
      Real cz = 1.0; // cz = 2 for kz>0 to take account of kz<0 ghost modes
      for (int mz=0; mz<f.Mz(); ++mz) {
	for (int ny=0; ny<f.Ny(); ++ny) {
	  fprof.set(ny, f.cmplx(mx,ny,mz,i));
	  gprof.set(ny, g.cmplx(mx,ny,mz,i));
	}
	sum += cz*Re(L2InnerProduct(fprof, gprof, normalize));
	cz = 2.0;
      }
    }
  if (!normalize)
    sum *= f.Lx()*f.Lz();
  return sum;
}

Real L2Norm2(const FlowField& u, int kxmax, int kzmax, bool normalize) {
  assert(u.ystate()==Spectral);
  assert(u.xzstate()==Spectral);
  Real sum = 0.0;
  
  // EFFICIENCY: would be better with smarter looping, but prob not worth trouble.
  ComplexChebyCoeff prof(u.Ny(), u.a(), u.b(), Spectral);
  for (int i=0; i<u.vectorDim(); ++i)
    for (int mx=0; mx<u.Mx(); ++mx) {
      if (abs(u.kx(mx)) > kxmax)
	continue;
      Real cz = 1.0; // cz = 2 for kz>0 to take account of kz<0 ghost modes
      for (int mz=0; mz<u.Mz(); ++mz) {
	if (abs(u.kz(mz)) > kzmax)
	  continue;
	for (int ny=0; ny<u.Ny(); ++ny) 
	  prof.set(ny, u.cmplx(mx,ny,mz,i));
	sum += cz*L2Norm2(prof, normalize);
	cz = 2.0;
      }
    }
  if (!normalize)
    sum *= u.Lx()*u.Lz();
  return sum;
}

Real L2Norm(const FlowField& u, int kxmax, int kzmax, bool normalize) {
  return sqrt(L2Norm2(u, kxmax, kzmax, normalize));
}
Real L2Dist(const FlowField& u, const FlowField& v, int kxmax, int kzmax, bool normalize) {
  return sqrt(L2Dist2(u, v, kxmax, kzmax, normalize));
}

Real L2Dist2(const FlowField& u, const FlowField& v, int kxmax, int kzmax, bool normalize) {
  Real sum = 0.0;
  assert(u.congruent(v));
  assert(u.xzstate()==Spectral && u.ystate()==Spectral);
  ComplexChebyCoeff u_v(u.Ny(), u.a(), u.b(), Spectral);
  for (int i=0; i<u.vectorDim(); ++i)
    for (int mx=0; mx<u.Mx(); ++mx) {
      if (abs(u.kx(mx)) > kxmax)
	continue;
      int cz = 1;
      for (int mz=0; mz<u.Mz(); ++mz) {
	if (abs(u.kz(mz)) > kzmax)
	  continue;
	for (int ny=0; ny<u.Ny(); ++ny) 
	  u_v.set(ny, u.cmplx(mx,ny,mz,i)-v.cmplx(mx,ny,mz,i));
	sum += cz*L2Norm2(u_v, normalize);
	cz = 2;
      }
    }
  if (!normalize)
    sum *= u.Lx()*u.Lz();
  return sum;
}

Complex L2InnerProduct(const FlowField& u, const BasisFunc& phi,
		       bool normalize) {
  assert(u.xzstate()==Spectral && u.ystate()==Spectral);
  assert(u.congruent(phi));
  int kx = phi.kx();
  int kz = phi.kz();
  int mx = u.mx(kx);
  int mz = u.mz(kz);
  int Ny = u.Ny();
  BasisFunc profile(Ny, kx, kz, u.Lx(), u.Lz(), u.a(), u.b(), Spectral);
  
  
  for (int i=0; i<u.Nd(); ++i) 
    for (int ny=0; ny<Ny; ++ny)
      profile[i].set(ny, u.cmplx(mx,ny,mz, i));

  return L2InnerProduct(profile, phi, normalize);
}


void randomUprofile(ComplexChebyCoeff& u, Real mag, Real decay) {
  // Set a random u(y)
  int N = u.length();
  u.setState(Spectral);
  int n; // MSVC++ FOR-SCOPE BUG
  for (n=0; n<N; ++n) {
    u.set(n, mag*randomComplex());
    mag *= decay;
  }
  ChebyTransform trans(N);
  // Adjust u(y) so that u(+-1) == 0
  //cout << "before random u(a) == " << u.eval_a() << endl;
  //cout << "before random u(b) == " << u.eval_b() << endl;
  Complex u0 = (u.eval_b() + u.eval_a())/2.0;
  Complex u1 = (u.eval_b() - u.eval_a())/2.0;
  u.sub(0, u0);
  u.sub(1, u1);
  //cout << "after random u(a) == " << u.eval_a() << endl;
  //cout << "after random u(b) == " << u.eval_b() << endl;

  assert(abs(u.eval_b()) < EPSILON);
  assert(abs(u.eval_a()) < EPSILON);
}

void randomVprofile(ComplexChebyCoeff& v, Real mag, Real decay) {
  int N = v.length();
  v.setState(Spectral);
  // Assign a random v(y).
  v.set(0, 0.0);
  v.set(1, 0.0);
  v.set(2, 0.0);
  v.set(3, 0.0);
  int n; // MSVC++ FOR-SCOPE BUG
  for (n=0; n<4; ++n) 
    v.set(n, 0.0);
  for (n=4; n<N-2; ++n) {
    v.set(n, mag*randomComplex());
    mag *= decay;
  }

  for (n=Greater(N-2, 0); n<N; ++n) 
    v.set(n, 0.0);

  // Adjust v so that v(+-1) == v'(+/-1) == 0, by subtracting off 
  // s0 T0(x) + s1 T1(x) + s2 T2(x) + s3 T3(x), with s's chosen to
  // have same BCs as v. 
  ComplexChebyCoeff vy = diff(v);

  Complex a = v.eval_a();
  Complex b = v.eval_b(); 
  Complex c = vy.eval_a();
  Complex d = vy.eval_b(); 
    
  // The numercial coeffs are inverse of the matrix (values found with Maple)
  // T0(-1)  T1(-1)  T2(-1)  T3(-1)     s0      a
  // T0(1)   T1(1)   T2(1)   T3(1)      s1      b
  // T0'(-1) T1'(-1) T2'(-1) T3'(-1)    s2  ==  c
  // T0'(1)  T1'(1)  T2'(1)  T3'(1)     s3      d

  Complex s0 = 0.5*(a + b) + 0.125*(c - d);
  Complex s1 = 0.5625*(b - a) - 0.0625*(c + d);
  Complex s2 = 0.125*(d - c);
  Complex s3 = 0.0625*(a - b + c + d);

  ComplexChebyCoeff adj(v.numModes(), v.a(), v.b(), Spectral);
  adj.set(0, s0);
  adj.set(1, s1);
  adj.set(2, s2);
  adj.set(3, s3);
  ComplexChebyCoeff adjy = diff(adj);
  

  // Subtract off the coeffs 
  v.sub(0, s0);
  v.sub(1, s1);
  v.sub(2, s2);
  v.sub(3, s3);

  diff(v,vy);

  //cout << "random v(a)  == " << v.eval_a() << endl;
  //cout << "random v(b)  == " << v.eval_b()  << endl;
  //cout << "random v'(a) == " << vy.eval_a() << endl;
  //cout << "random v'(b) == " << vy.eval_b() << endl;

}


void chebyUprofile(ComplexChebyCoeff& u, int n, Real decay) {
  // Set a random u(y)
  int N = u.length();
  u.setToZero();
  u.setState(Spectral);
  Real theta = pi*randomReal();
  u.set(n, (cos(theta) + I*sin(theta))*pow(decay,n));
  
  ChebyTransform trans(N);
  
  // Adjust u(y) so that u(+-1) == 0
  Complex u0 = (u.eval_b() + u.eval_a())/2.0;  // 2.0 is correct for genl a,b
  Complex u1 = (u.eval_b() - u.eval_a())/2.0;  // 2.0 is correct for genl a,b
  u.sub(0, u0);
  u.sub(1, u1);
  //cout << "random u(a) == " << u.eval_a() << endl;
  //cout << "random u(b) == " << u.eval_b() << endl;
  assert(abs(u.eval_b()) < EPSILON);
  assert(abs(u.eval_a()) < EPSILON);
}

void chebyVprofile(ComplexChebyCoeff& v, int n, Real decay) {
  v.setToZero();
  v.setState(Spectral);
  Real theta = pi*randomReal();
  v.set(n, (cos(theta) + I*sin(theta))*pow(decay,n));

  // Adjust v so that v(+-1) == v'(+/-1) == 0, by subtracting off 
  // s0 T0(x) + s1 T1(x) + s2 T2(x) + s3 T3(x), with s's chosen to
  // have same BCs as v. 
  ComplexChebyCoeff vy = diff(v);

  Complex a = v.eval_a();
  Complex b = v.eval_b(); 
  Complex c = vy.eval_a();
  Complex d = vy.eval_b(); 
    
  // The numercial coeffs are inverse of the matrix (values found with Maple)
  // T0(-1)  T1(-1)  T2(-1)  T3(-1)     s0      a
  // T0(1)   T1(1)   T2(1)   T3(1)      s1      b
  // T0'(-1) T1'(-1) T2'(-1) T3'(-1)    s2  ==  c
  // T0'(1)  T1'(1)  T2'(1)  T3'(1)     s3      d

  // The above matrix is 
  // 1  -1   1  -1 
  // 1   1   1   1
  // 0   1  -4   9
  // 0   1   4   9

  Complex s0 = 0.5*(a + b) + 0.125*(c - d);
  Complex s1 = 0.5625*(b - a) - 0.0625*(c + d);
  Complex s2 = 0.125*(d - c);
  Complex s3 = 0.0625*(a - b + c + d);

  //ComplexChebyCoeff adj(v.numModes(), v.a(), v.b(), Spectral);
  //adj.set(0, s0);
  //adj.set(1, s1);
  //adj.set(2, s2);
  //adj.set(3, s3);
  //ComplexChebyCoeff adjy = diff(adj);
  
  // Subtract off the coeffs 
  v.sub(0, s0);
  v.sub(1, s1);
  v.sub(2, s2);
  v.sub(3, s3);

  //diff(v,vy);

  //cout << "random v(a)  == " << v.eval_a() << endl;
  //cout << "random v(b)  == " << v.eval_b()  << endl;
  //cout << "random v'(a) == " << vy.eval_a() << endl;
  //cout << "random v'(b) == " << vy.eval_b() << endl;

}


void randomProfile(ComplexChebyCoeff& u, ComplexChebyCoeff& v, ComplexChebyCoeff& w, int kx, int kz, Real Lx, Real Lz, Real mag, Real decay) {
  
  int N = u.length();
  ChebyTransform trans(N);
  u.setState(Spectral);
  v.setState(Spectral);
  w.setState(Spectral);
  //u.setToZero();
  //v.setToZero();
  //w.setToZero();
  if (kx == 0 && kz == 0) {
    // Assign an odd perturbation to u, so as not to change mean(U).
    // Just set even modes to zero.
    
    randomUprofile(w, mag, decay);
    w.im.setToZero();

    randomUprofile(u, mag, decay);
    u.im.setToZero();

    v.setToZero();

  }
  else {
    // Other kx,kz cases are based on a random v(y). 

    randomVprofile(v, mag, decay);
    ComplexChebyCoeff vy = diff(v);

    if (kx == 0) {
      randomUprofile(u, mag, decay);
      u.im.setToZero();
      w = vy;
      w *= -Lz/((2*pi*kz)*I);
    }
    else if (kz == 0) {
      randomUprofile(w, mag, decay);
      w.im.setToZero();
      u = vy;
      u *= -Lx/((2*pi*kx)*I);
    }
    else {

      ComplexChebyCoeff v0(v);
      ComplexChebyCoeff v1(v);
      randomVprofile(v0, mag, decay);
      randomVprofile(v1, mag, decay);

      ComplexChebyCoeff v0y = diff(v0);
      ComplexChebyCoeff v1y = diff(v1);

      // Finally, the general case, where kx, kz != 0 and u,v,w are nonzero
      // Set a random u(y)
      ComplexChebyCoeff u0(v.numModes(), v.a(), v.b(), Spectral);
      ComplexChebyCoeff w0(v.numModes(), v.a(), v.b(), Spectral);
      ComplexChebyCoeff u1(v.numModes(), v.a(), v.b(), Spectral);
      ComplexChebyCoeff w1(v.numModes(), v.a(), v.b(), Spectral);
      
      randomUprofile(u0, mag, decay);

      // Calculate w0 from div u0 == u0x + v0y + w0z == 0.
      ComplexChebyCoeff u0x(u0);
      u0x *= (2*pi*kx/Lx)*I;
      w0 = v0y;
      w0 += u0x;
      w0 *= -Lz/((2*pi*kz)*I);       // Set w = -Lz/(2*pi*I*kz) * (ux + vy);


      //randomUprofile(w1, mag, decay);

      // Calculate u0 from div u0 == u0x + v0y + w0z == 0.
      ComplexChebyCoeff w1z(w1);
      w1z *= (2*pi*kz/Lz)*I;
      u1 = v1y;
      u1 += w1z;
      u1 *= -Lx/((2*pi*kx)*I);       // Set w = -Lz/(2*pi*I*kz) * (ux + vy);

      u = u0;
      v = v0;
      w = w0;
      u += u1;
      v += v1;
      w += w1;
    }
  }

  // Check divergence
  ComplexChebyCoeff ux(u);
  ux *= (2*pi*kx/Lx)*I;
  ComplexChebyCoeff wz(w);
  wz *= (2*pi*kz/Lz)*I;
  ComplexChebyCoeff vy = diff(v);
  
  ComplexChebyCoeff div(ux);
  div += vy;
  div += wz;

  Real divNorm = L2Norm(div);
  Real ubcNorm = abs(u.eval_a()) + abs(u.eval_b());
  Real vbcNorm = abs(v.eval_a()) + abs(v.eval_b());
  Real wbcNorm = abs(w.eval_a()) + abs(w.eval_b());
  assert(divNorm < EPSILON);
  assert(ubcNorm < EPSILON);
  assert(vbcNorm < EPSILON);
  assert(wbcNorm < EPSILON);
  // supress unused-variable compiler warnings...
  wbcNorm += divNorm + ubcNorm + vbcNorm;

}

void chebyProfile(ComplexChebyCoeff& u, ComplexChebyCoeff& v, ComplexChebyCoeff& w, int un, int vn, int kx, int kz, Real Lx, Real Lz, Real decay) {
  
  int N = u.length();
  ChebyTransform trans(N);
  u.setState(Spectral);
  v.setState(Spectral);
  w.setState(Spectral);
  //u.setToZero();
  //v.setToZero();
  //w.setToZero();
  if (kx == 0 && kz == 0) {
    chebyUprofile(u, un, decay);
    chebyUprofile(w, vn, decay); // yes, vn
    u.im.setToZero();
    w.im.setToZero();
    v.setToZero();
  }
  else {
    // Other kx,kz cases are based on a random v(y). 

    chebyVprofile(v, vn, decay);
    ComplexChebyCoeff vy = diff(v);

    if (kx == 0) {
      chebyUprofile(u, un, decay);
      u.im.setToZero();
      w = vy;
      w *= -Lz/((2*pi*kz)*I);
    }
    else if (kz == 0) {
      chebyUprofile(w, un, decay); // yes, un
      w.im.setToZero();
      u = vy;
      u *= -Lx/((2*pi*kx)*I);
    }
    else {

      ComplexChebyCoeff v0(v);
      ComplexChebyCoeff v1(v);
      chebyVprofile(v0, vn, decay);
      chebyVprofile(v1, vn, decay);

      ComplexChebyCoeff v0y = diff(v0);
      ComplexChebyCoeff v1y = diff(v1);

      // Finally, the general case, where kx, kz != 0 and u,v,w are nonzero
      // Set a random u(y)
      ComplexChebyCoeff u0(v.numModes(), v.a(), v.b(), Spectral);
      ComplexChebyCoeff w0(v.numModes(), v.a(), v.b(), Spectral);
      ComplexChebyCoeff u1(v.numModes(), v.a(), v.b(), Spectral);
      ComplexChebyCoeff w1(v.numModes(), v.a(), v.b(), Spectral);
      
      chebyUprofile(u0, un, decay);

      // Calculate w0 from div u0 == u0x + v0y + w0z == 0.
      ComplexChebyCoeff u0x(u0);
      u0x *= (2*pi*kx/Lx)*I;
      w0 = v0y;
      w0 += u0x;
      w0 *= -Lz/((2*pi*kz)*I);       // Set w = -Lz/(2*pi*I*kz) * (ux + vy);


      //randomUprofile(w1, mag, decay);

      // Calculate u0 from div u0 == u0x + v0y + w0z == 0.
      ComplexChebyCoeff w1z(w1);
      w1z *= (2*pi*kz/Lz)*I;
      u1 = v1y;
      u1 += w1z;
      u1 *= -Lx/((2*pi*kx)*I);       // Set w = -Lz/(2*pi*I*kz) * (ux + vy);

      u = u0;
      v = v0;
      w = w0;
      u += u1;
      v += v1;
      w += w1;
    }
  }

  // Check divergence
  ComplexChebyCoeff ux(u);
  ux *= (2*pi*kx/Lx)*I;
  ComplexChebyCoeff wz(w);
  wz *= (2*pi*kz/Lz)*I;
  ComplexChebyCoeff vy = diff(v);
  
  ComplexChebyCoeff div(ux);
  div += vy;
  div += wz;

  Real divNorm = L2Norm(div);
  Real ubcNorm = abs(u.eval_a()) + abs(u.eval_b());
  Real vbcNorm = abs(v.eval_a()) + abs(v.eval_b());
  Real wbcNorm = abs(w.eval_a()) + abs(w.eval_b());
  assert(divNorm < EPSILON);
  assert(ubcNorm < EPSILON);
  assert(vbcNorm < EPSILON);
  assert(wbcNorm < EPSILON);
  // supress unused-variable compiler warnings...
  wbcNorm += divNorm + ubcNorm + vbcNorm;

}


void assignOrrSommField(FlowField& u, FlowField& P,
			Real t, Real Reynolds, Complex omega,
			const ComplexChebyCoeff& ueig, 
			const ComplexChebyCoeff& veig,
			const ComplexChebyCoeff& peig) {

  int Ny=u.Ny();

  // Reconstruct velocity field (Poisseuille plus OS perturbation) from 
  // y-profile of (kx,kz) == (1,0) Spectral mode of pertubation (ueig & veig).
  u.setState(Spectral, Physical);
  u.setToZero();
  Complex c = exp((-t*omega)*I);
  int n=u.Mx()-1;
  int ny; // MSVC++ FOR-SCOPE BUG
  for (ny=0; ny<Ny; ++ny) {    
    Complex uc = c*ueig[ny];
    Complex vc = c*veig[ny];
    u.cmplx(0, ny, 0, 0) = Complex(1.0-square(u.y(ny)));
    u.cmplx(1, ny, 0, 0) = uc;
    u.cmplx(1, ny, 0, 1) = vc;
    u.cmplx(n, ny, 0, 0) = conj(uc);
    u.cmplx(n, ny, 0, 1) = conj(vc);
  }

  // Assign pressure perturbation p to P field.
  P.setState(Spectral, Physical);
  P.setToZero();
  for (ny=0; ny<Ny; ++ny) {    
    Complex pc = c*peig[ny];
    P.cmplx(1, ny, 0, 0) = pc;
    P.cmplx(n, ny, 0, 0) = conj(pc);
  }

  // Add velocity contrib to get modified pressure P = p + 1/2 |u|^2
  u.irealfft_xz();
  P.irealfft_xz();
  
  for (ny=0; ny<Ny; ++ny)
    for (int nx=0; nx<u.Nx(); ++nx)
      for (int nz=0; nz<u.Nz(); ++nz)
	P(nx,ny,nz,0) += 0.5*(square(u(nx,ny,nz,0)) + 
			      square(u(nx,ny,nz,1)) + 
			      square(u(nx,ny,nz,2)));

  u.realfft_xz();
  P.realfft_xz();
  u.chebyfft_y();
  P.chebyfft_y();
}


