/* poissonsolver.cpp: solve possion equation 
 * Channelflow-0.9
 *
 * Copyright (C) 2001-2005  John F. Gibson
 *
 * John F. Gibson
 * Center for Nonlinear Sciences
 * School of Physics
 * Georgia Institute of Technology
 * Atlanta, GA 30332-0430
 *
 * gibson@cns.physics.gatech.edu
 * jfg@member.fsf.org
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, U
 */

#include "poissonsolver.h"

PoissonSolver::PoissonSolver() 
  :
  Mx_(0),
  My_(0),
  Mz_(0),
  Nd_(0),
  Lx_(0),
  Lz_(0),
  a_(0),
  b_(0),
  helmholtz_(0)
{}

  
PoissonSolver::PoissonSolver(int Nx, int Ny, int Nz, int Nd, Real Lx, Real Lz,
			     Real a, Real b) 
  :
  Mx_(Nx),
  My_(Ny),
  Mz_(Nz/2+1),
  Nd_(Nd),
  Lx_(Lx),
  Lz_(Lz),
  a_(a),
  b_(b),
  helmholtz_(0)
{
  // Allocate Mx x Mz array of HelmholtzSolvers
  helmholtz_ = new HelmholtzSolver*[Mx_]; // new #1
  for (int mx=0; mx<Mx_; ++mx)
    helmholtz_[mx] = new HelmholtzSolver[Mz_];  // new #2
  
  // Assign Helmholtz solvers into array. Each solves u'' - lambda u = f
  for (int mx=0; mx<Mx_; ++mx) { 
    int kx = (mx <= Nx/2) ? mx : mx - Nx; // same as FlowField::kx(mx)
    for (int mz=0; mz<Mz_; ++mz) {
      int kz = mz; // same as FlowField::kz(mz)
      Real lambda = 4.0*pi*pi*(square(kx/Lx_) + square(kz/Lz_));
      helmholtz_[mx][mz] = HelmholtzSolver(My_, a_, b_, lambda);
    }
  }
}

PoissonSolver::PoissonSolver(const PoissonSolver& ps) 
  :
  Mx_(ps.Mx_),
  My_(ps.My_),
  Mz_(ps.Mz_),
  Nd_(ps.Nd_),
  Lx_(ps.Lx_),
  Lz_(ps.Lz_),
  a_(ps.a_),
  b_(ps.b_),
  helmholtz_(0) 
{

  // Allocate Mx x Mz array of HelmholtzSolvers
  helmholtz_ = new HelmholtzSolver*[Mx_]; // new #1
  for (int mx=0; mx<Mx_; ++mx)
    helmholtz_[mx] = new HelmholtzSolver[Mz_];  // new #2
  
  // Assign Helmholtz solvers into array. Each solves p'' - lambda p = f
  for (int mx=0; mx<Mx_; ++mx) 
    for (int mz=0; mz<Mz_; ++mz) 
      helmholtz_[mx][mz] = ps.helmholtz_[mx][mz];
}

PoissonSolver::PoissonSolver(const FlowField& u) 
  :
  Mx_(u.Mx()),
  My_(u.My()),
  Mz_(u.Mz()),
  Nd_(u.Nd()),
  Lx_(u.Lx()),
  Lz_(u.Lz()),
  a_(u.a()),
  b_(u.b()),
  helmholtz_(0)
{

  // Allocate Mx x Mz array of HelmholtzSolvers
  helmholtz_ = new HelmholtzSolver*[Mx_]; // new #1
  for (int mx=0; mx<Mx_; ++mx)
    helmholtz_[mx] = new HelmholtzSolver[Mz_];  // new #2
  
  // Assign Helmholtz solvers into array. Each solves p'' - lambda p = f
  for (int mx=0; mx<Mx_; ++mx) { 
    int kx = u.kx(mx);
    for (int mz=0; mz<Mz_; ++mz) {
      int kz = u.kz(mz);
      Real lambda = 4.0*pi*pi*(square(kx/Lx_) + square(kz/Lz_));
      helmholtz_[mx][mz] = HelmholtzSolver(My_, a_, b_, lambda);
    }
  }
}

PoissonSolver::~PoissonSolver() {
  for (int mx=0; mx<Mx_; ++mx)
    delete[] helmholtz_[mx]; // undo new #2
  delete[] helmholtz_;       // undo new #1
}

bool PoissonSolver::geomCongruent(const FlowField& u) const {
  return 
    (u.Mx() == Mx_) && (u.My() == My_) && (u.Mz()== Mz_) && 
    (u.Lx() == Lx_) && (u.Lz() == Lz_) && (u.a() == a_)  && (u.b() == b_);
}
bool PoissonSolver::congruent(const FlowField& u) const {
  return 
    (u.Mx() == Mx_) && (u.My() == My_) && (u.Mz()== Mz_) && (u.Nd()== Nd_) &&
    (u.Lx() == Lx_) && (u.Lz() == Lz_) && (u.a() == a_)  && (u.b() == b_);
    (u.Lx() == Lx_) && (u.Lz() == Lz_) && (u.a() == a_) && (u.b() == b_);
}

void PoissonSolver::solve(FlowField& u, const FlowField& f) const {
  assert(this->congruent(u));
  assert(this->congruent(f));
  f.assertState(Spectral, Spectral);
  u.setToZero();
  u.setState(Spectral, Spectral);
 
  ComplexChebyCoeff fk(My_, a_, b_, Spectral);
  ComplexChebyCoeff uk(My_, a_, b_, Spectral);
 
  Real zero = 0.0; // dirichlet BC
  for (int i=0; i<Nd_; ++i) 
    for (int mx=0; mx<Mx_; ++mx) 
      for (int mz=0; mz<Mz_; ++mz) {
	for (int my=0; my<My_; ++my) 
	  fk.set(my, f.cmplx(mx,my,mz,i));
	helmholtz_[mx][mz].solve(uk.re, fk.re, zero, zero);
	helmholtz_[mx][mz].solve(uk.im, fk.im, zero, zero);
	for (int my=0; my<My_; ++my)
	  u.cmplx(mx,my,mz,i) = uk[my];
      }
}

void PoissonSolver::solve(FlowField& u, const FlowField& f, const FlowField& bc) const {
  assert(this->congruent(f));
  assert(this->congruent(u));
  assert(this->congruent(bc));
  assert(bc.xzstate() == Spectral);
  f.assertState(Spectral, Spectral);
  u.setToZero();
  u.setState(Spectral, Spectral);
 
  ComplexChebyCoeff fk(My_, a_, b_, Spectral);
  ComplexChebyCoeff bk(My_, a_, b_, Spectral);
  ComplexChebyCoeff uk(My_, a_, b_, Spectral);
 
  for (int i=0; i<Nd_; ++i) 
    for (int mx=0; mx<Mx_; ++mx)
      for (int mz=0; mz<Mz_; ++mz) {
      	for (int my=0; my<My_; ++my) {
	  fk.set(my, f.cmplx(mx,my,mz,i));
	  bk.set(my, bc.cmplx(mx,my,mz,i));
	}
	helmholtz_[mx][mz].solve(uk.re, fk.re, bk.re.eval_a(), bk.re.eval_b());
	helmholtz_[mx][mz].solve(uk.im, fk.im, bk.im.eval_a(), bk.im.eval_b());
	for (int my=0; my<My_; ++my)
	  u.cmplx(mx,my,mz,i) = uk[my];
      }
}

// verify that lapl u = f and u=0 on BC
void PoissonSolver::verify(const FlowField& u, const FlowField& f) const {
  assert(this->congruent(u));
  assert(this->congruent(f));
  u.assertState(Spectral, Spectral);
  f.assertState(Spectral, Spectral);

  FlowField lapl_u;
  lapl(u, lapl_u);
  FlowField diff(f);
  diff -= lapl_u;
  
  /***********************************************
  // Debugging output
  f.saveSpectrum("f");  
  lapl_u.saveSpectrum("lapl_u");  
  diff.saveSpectrum("diff");  

  for (int mx=0; mx<=3; ++mx) {
    for (int mz=0; mz<=3; ++mz) {
      string lbl = i2s(mx)+i2s(mz);
      u.saveProfile(mx,mz, "u"+lbl);
      f.saveProfile(mx,mz, "f"+lbl);  
      lapl_u.saveProfile(mx,mz, "lapl_u"+lbl);
    }
  }
  ***********************************************/

  cerr << "PoissonSolver::verify(u, f) {\n";
  cerr << "  L2Norm(u)         == " << L2Norm(u) << endl;
  cerr << "  L2Norm(f)         == " << L2Norm(f) << endl;
  cerr << "  L2Norm(lapl u)    == " << L2Norm(lapl_u) << endl;
  cerr << "  L2Dist(lapl u, f) == " << L2Dist(lapl_u, f) << endl;
  cerr << "  bcNorm(u)         == " << bcNorm(u) << endl;
  cerr << "} // PoissonSolver::verify(u, f)\n";
}

void PoissonSolver::verify(const FlowField& u, const FlowField& f, const FlowField& bc) const {
  assert(this->congruent(u));
  assert(this->congruent(f));
  u.assertState(Spectral, Spectral);
  f.assertState(Spectral, Spectral);

  FlowField lapl_u;
  lapl(u, lapl_u);
  cerr << "PoissonSolver::verify(u, f) {\n";
  cerr << "  L2Norm(u)         == " << L2Norm(u) << endl;
  cerr << "  L2Norm(f)         == " << L2Norm(f) << endl;
  cerr << "  L2Norm(lapl u)    == " << L2Norm(lapl_u) << endl;
  cerr << "  L2Dist(lapl u, f) == " << L2Dist(lapl_u, f) << endl;
  cerr << "  bcNorm(u)         == " << bcNorm(u) << endl;
  cerr << "  bcNorm(bc)        == " << bcNorm(bc) << endl;
  cerr << "  bcDist(u,bc)      == " << bcDist(u,bc) << endl;
  cerr << "} // PoissonSolver::verify(u, f)\n";
}

// ========================================================================

PressureSolver::PressureSolver() 
  :
  PoissonSolver(),
  U_(),
  trans_(0),
  nonl_(),
  tmp_(),
  div_nonl_(),
  nu_(0)
{}


PressureSolver::PressureSolver(int Nx, int Ny, int Nz, Real Lx, Real Lz,
			       Real a, Real b, const ChebyCoeff& U, Real nu, 
			       NonlinearMethod nonl)
  :
  PoissonSolver(Nx, Ny, Nz, 1, Lx, Lz, a, b),
  U_(U),
  trans_(Ny),
  nonl_(Nx, Ny, Nz, 3, Lx, Lz, a, b),
  tmp_(), // geom will be set in call to navierstokesNL 
  div_nonl_(Nx, Ny, Nz, 1, Lx, Lz, a, b),
  nu_(nu),
  nonl_method_(nonl)
{
  assert(U_.N() == Ny);
  U_.makeSpectral(trans_);
}

PressureSolver::PressureSolver(const FlowField& u, Real nu, NonlinearMethod nl)
  :
  PoissonSolver(u.Nx(), u.Ny(), u.Nz(), 1, u.Lx(), u.Lz(), u.a(), u.b()),
  U_(u.Ny(), u.a(), u.b(), Spectral),
  trans_(u.Ny()),
  nonl_(u.Nx(), u.Ny(), u.Nz(), 3, u.Lx(), u.Lz(), u.a(), u.b()),
  tmp_(), // geom will be set in call to navierstokesNL 
  div_nonl_(u.Nx(), u.Ny(), u.Nz(), 1, u.Lx(), u.Lz(), u.a(), u.b()),
  nu_(nu),
  nonl_method_(nl)
{}


PressureSolver::PressureSolver(const FlowField& u, const ChebyCoeff& U, 
			       Real nu, NonlinearMethod nonl_method)
  :
  PoissonSolver(u.Nx(), u.Ny(), u.Nz(), 1, u.Lx(), u.Lz(), u.a(), u.b()),
  U_(U),
  trans_(u.Ny()),
  nonl_(u.Nx(), u.Ny(), u.Nz(), 3, u.Lx(), u.Lz(), u.a(), u.b()),
  tmp_(), // geom will be set in call to navierstokesNL 
  div_nonl_(u.Nx(), u.Ny(), u.Nz(), 1, u.Lx(), u.Lz(), u.a(), u.b()),
  nu_(nu),
  nonl_method_(nonl_method)
{
  assert(U_.N() == u.Ny());
  U_.makeSpectral(trans_);
}
  
PressureSolver::~PressureSolver() {}

void PressureSolver::solve(FlowField& p, const FlowField& u) {
  assert(u.xzstate() == Spectral && u.ystate() == Spectral);
  assert(congruent(p));
  assert(geomCongruent(u));
  
  p.setToZero();
  p.setState(Spectral, Spectral);

  // I. Get particular solution satisfying lapl p = div(nonl(u)), BCs p=0
  navierstokesNL(u, U_, nonl_, tmp_, nonl_method_);
  div(nonl_, div_nonl_);
  div_nonl_ *= -1.0;

  PoissonSolver::solve(p, div_nonl_);

  // II. Adjust solution to match von Neumann BCs 
  //   dp/dy ==  nu d^2 v/dy^2  at y=a and y=b

  // Derivation of calculation in 5/19/05 notes, jfg. Idea is that 
  //   g = c exp(lambda y) + d exp(-lambda y) 
  // satisfies
  //   g" - lambda g = 0 with nonzero BCs.
  // Find the particular values of c and d such that adding g to p leaves
  //   p" - lambda p = f unchanged but sets dp/dy = nu d^v/dy^2 at y= a,b.

  ComplexChebyCoeff vk(My_, a_, b_, Spectral);
  ComplexChebyCoeff vkyy(My_, a_, b_, Spectral);
  ComplexChebyCoeff pk(My_, a_, b_, Spectral);
  ComplexChebyCoeff pky(My_, a_, b_, Spectral);
  ComplexChebyCoeff gk(My_, a_, b_, Physical);

  Vector y = p.ygridpts();
  for (int mx=0; mx<Mx_; ++mx)
    for (int mz=0; mz<Mz_; ++mz) {

      // Don't modify the mx=mz=0 (kx=kz=0) solution, dirichlet BCs are fine
      if (mx != 0 || mz != 0) {

	Real lambda = helmholtz_[mx][mz].lambda();
	assert(lambda > 0);
	Real gamma = sqrt(lambda); 

	for (int my=0; my<My_; ++my) {
	  pk.set(my, p.cmplx(mx,my,mz,0));
	  vk.set(my, u.cmplx(mx,my,mz,1));
	}
	diff(pk,pky);
	diff2(vk,vkyy);

	Complex alpha = nu_*vkyy.eval_a() - pky.eval_a();
	Complex  beta = nu_*vkyy.eval_b() - pky.eval_b();
	Real delta = gamma*(exp(gamma*(a_-b_)) - exp(gamma*(b_-a_)));

	Complex c = (alpha*exp(-gamma*b_) - beta*exp(-gamma*a_))/delta;
	Complex d = (alpha*exp(gamma*b_) - beta*exp(gamma*a_))/delta;

	gk.setState(Physical);
	for (int my=0; my<My_; ++my) 
	  gk.set(my, c*exp(gamma*y[my]) + d*exp(-gamma*y[my]));
	gk.makeSpectral(trans_);
      
	for (int my=0; my<My_; ++my) 
	  p.cmplx(mx,my,mz,0) += gk[my];
      }
    }
  return;
}
      
      
void PressureSolver::verify(const FlowField& p, const FlowField& u) {
  assert(u.xzstate() == Spectral && u.ystate() == Spectral);
  assert(congruent(p));
  assert(geomCongruent(u));
  
  // I. Verify that solution satisfies lapl p = -div(nonl(u))
  navierstokesNL(u, U_, nonl_, tmp_, nonl_method_);
  div(nonl_, div_nonl_);
  div_nonl_ *= -1.0;

  PoissonSolver::verify(p, div_nonl_);

  FlowField lapl_p;
  lapl(p, lapl_p);

  cerr << "PressureSolver::verify(p,u) {\n";
  cerr << "  L2Norm(u)           == " << L2Norm(u) << endl;
  cerr << "  L2Norm(div(nonl(u)) == " << L2Norm(div_nonl_) << endl;
  cerr << "  L2Norm(lapl p)      == " << L2Norm(lapl_p) << endl;
  cerr << "  L2Dist(lapl p, div(nonl(u))) == " << L2Dist(lapl_p, div_nonl_) << endl;

  // I. Verify that solution satisfies dpdy = d^2 /dy^2 (u+U) on boundary
  FlowField dpdy;
  ydiff(p, dpdy);
  
  FlowField v(u.Nx(), u.Ny(), u.Nz(), 1, u.Lx(), u.Lz(), u.a(), u.b(), Spectral, Spectral);
  for (int my=0; my<Mx_; ++my)
    for (int mx=0; mx<Mx_; ++mx)
      for (int mz=0; mz<Mz_; ++mz)
	v.cmplx(mx,my,mz,0) = u.cmplx(mx,my,mz,1);

  FlowField nu_vyy;
  ydiff(v, nu_vyy, 2); 
  nu_vyy *= nu_;
  cerr << "  bcNorm(dpdy)         == " << bcNorm(dpdy) << endl;
  cerr << "  bcNorm(nu_vyy)       == " << bcNorm(nu_vyy) << endl;
  cerr << "  bcDist(dpdy, nu_vyy) == " << bcDist(dpdy, nu_vyy) << endl;

  ComplexChebyCoeff vk(My_, a_, b_, Spectral);
  ComplexChebyCoeff vkyy(My_, a_, b_, Spectral);
  ComplexChebyCoeff pk(My_, a_, b_, Spectral);
  ComplexChebyCoeff pky(My_, a_, b_, Spectral);

  return;
}
