/* chebyshev.cpp: Real- and Complex-valued Chebyshev expansion classes.
 * Channelflow-0.9
 *
 * Copyright (C) 2001  John F. Gibson  
 *  
 * jgibson@mail.sjcsf.edu  
 * John F. Gibson 
 * St. John's College
 * 1160 Camino de la Cruz Blanca
 * Santa Fe, NM 87501
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, U
 */

// Wordexp is used in fftw_savewisdom and load, to expand possible home
// dir symbol ~. Comment it out if it causes you trouble.

#include <wordexp.h>

#include <fstream>
#include <iomanip>
#include "chebyshev.h"

const Real QUADRATURE_EPSILON=1e-17;

// Integral_(-1)^1 Tm(y) Tn(y) dy
Real chebyIP(int m, int n) { 
  return ((m+n) % 2 == 1) ? 0 :
    (1.0-m*m-n*n)/((1.0+m-n)*(1.0-m+n)*(1.0+m+n)*(1.0-m-n));
}

Real legendre(int n, Real x) {
  Real p=1.0;
  Real q=0.0;
  for (int m=0; m<n; ++m) {
    Real r = q;
    q = p;
    p = ((2*m+1)*x*q-m*r)/(m+1);
  }
  return p;
}

Real chebyshev(int n, Real x) {
  return cos(n*acos(x));
}

ChebyCoeff legendre(int N, int n, bool normalize) {
  assert(n<N);
  ChebyCoeff Pn(N, -1 , 1, Physical);
  Real piN=pi/(N-1);
  for (int q=0; q<N; ++q) 
    Pn[q] = legendre(n, cos(q*piN));
  ChebyTransform trans(N);
  Pn.makeSpectral(trans);
  if (normalize)
    Pn *= sqrt((double) (2*n+1));
  return Pn;
}

ChebyCoeff chebyshev(int N, int n, bool normalize) {
  assert(n<N);
  ChebyCoeff Tn(N, -1 , 1, Spectral);
  if (!normalize)
    Tn[n] = 1;
  else
    Tn[n] = sqrt(2/(pi*cheby_c(n)));
  return Tn;
}  

void gaussLegendreQuadrature(int N, Real a, Real b, Vector& x, Vector& w) {
  x = Vector(N);
  w = Vector(N);
  int M = (N+1)/2; 
  Real middle = 0.5*(b+a);
  Real radius = 0.5*(b-a);
  Real pp;
  Real z,z1;

  for (int m=0; m<M; ++m) {
    z = cos(M_PI*(m+0.75)/(N+0.5));
    Real p1,p2,p3;
    int iterations=0;
    do {
      p1 = 1.0;
      p2 = 0.0;
      for (int n=0; n<N; ++n) {
	p3 = p2;
	p2 = p1;
	p1 = ((2*n+1)*z*p2-n*p3)/(n+1);
      }
      pp = N*(z*p1-p2)/(z*z-1.0);
      z1=z;
      z=z1-p1/pp;
    } while (fabs(z-z1) > QUADRATURE_EPSILON && ++iterations < 1024);

    x(m) = middle - radius*z;
    x(N-1-m) = middle + radius*z;
    w(m) = 2.0*radius/((1.0-z*z)*pp*pp);
    w(N-1-m) = w(m);
  }
}

void fftw_loadwisdom(const char* filename_) {
  const char* filename = (filename_ ==0) ? "~/.fftw_wisdom" : filename_;
  wordexp_t w;
  w.we_offs=0;
  int flags = 0;
  wordexp(filename, &w, flags);
  if (w.we_wordc > 0)
    filename = w.we_wordv[0];  

  FILE* input_file;
  input_file = fopen(filename, "r");
  if (FFTW_FAILURE == fftw_import_wisdom_from_file(input_file)) 
    cerr << "Error reading fftw-wisdom file " << filename << ", proceding without fftw-wisdom.\n";
  else
    fclose(input_file);
} 
void fftw_savewisdom(const char* filename_) {
  const char* filename = (filename_ ==0) ? "~/.fftw_wisdom" : filename_;
  wordexp_t w;
  w.we_offs=0;
  int flags = 0;
  wordexp(filename, &w, flags);
  if (w.we_wordc > 0)
    filename = w.we_wordv[0];
  
  FILE* output_file;
  output_file = fopen(filename, "w");
  if (output_file ==0)
    cerr << "Can't write to fftw-wisdom file " << filename << "\n";

  fftw_export_wisdom_to_file(output_file);
  fclose(output_file);
}       

ChebyCoeff::ChebyCoeff()
  :
  Vector(),
  a_(0),
  b_(0),
  state_(Spectral)
{}

ChebyCoeff::ChebyCoeff(int N, Real a, Real b, fieldstate s)
  :
  Vector(N),
  a_(a),
  b_(b),
  state_(s)
{
  assert( b_ > a_);
}

ChebyCoeff::ChebyCoeff(const Vector& v, Real a, Real b, fieldstate s)
  :
  Vector(v),
  a_(a),
  b_(b),
  state_(s)
{
  assert( b_ > a_);
}

ChebyCoeff::ChebyCoeff(int N, const ChebyCoeff& u) 
  :
  Vector(N),
  a_(u.a_),
  b_(u.b_),
  state_(u.state_)
{
  assert(b_ > a_);
  assert(state_ == Spectral);
  int Ncommon = lesser(N, u.N_);
  int i; // MSVC++ FOR-SCOPE BUG
  for (i=0; i<Ncommon; ++i)
    data_[i] = u.data_[i];
  for (i=Ncommon; i<N_; ++i)
    data_[i] = 0.0;
  
}

ChebyCoeff::ChebyCoeff(const string& filebase) 
  :
  Vector(0),
  a_(0),
  b_(0),
  state_(Spectral)
{
  string filename(filebase);
  filename += string(".asc");
  ifstream is(filename.c_str());
  if (!is.good()) {
    cerr << "ChebyCoeff::ChebyCoeff(filebase) : can't open file " << filename << endl;
    abort();
  }
 
  // Read in header. Form is "%N a b s"
  char c;
  int N;
  is >> c;
  if (c != '%') {
    string message("ChebyCoeff(filebase): bad header in file ");
    message += filename;
    cerr << message << endl;
    assert(false);
  }
  is >> N >> a_ >> b_ >> state_;
  resize(N);
  assert(is.good());
  for (int i=0; i<N; ++i) {
    is >> (*this)[i];
    assert(is.good());
  }

  is.close();
}

/******************
ChebyCoeff::ChebyCoeff(istream& is) 
  :
  Vector(0),
  a_(0),
  b_(0),
  state_(Spectral)
{
  binaryLoad(is);
}
******************/

ChebyCoeff::~ChebyCoeff() {}

void ChebyCoeff::chebyfft() {
  ChebyTransform t(N_);
  t.chebyfft(*this);
}
void ChebyCoeff::ichebyfft(){
  ChebyTransform t(N_);
  t.ichebyfft(*this);
}
void ChebyCoeff::makeSpectral() {
  ChebyTransform t(N_);
  t.makeSpectral(*this);
}
void ChebyCoeff::makePhysical() {
  ChebyTransform t(N_);
  t.makePhysical(*this);
}
void ChebyCoeff::makeState(fieldstate s){ 
  if (state_ != s) {       // need to change state?
    ChebyTransform t(N_);
    if (state_==Physical)  // state is Physical; change to Spectral
      t.chebyfft(*this);
    else
      t.ichebyfft(*this);  // state is Spectral; change to Physical
  }
}


void ChebyCoeff::randomize(Real decay, bool a_dirichlet, bool b_dirichlet) {
  fieldstate startState = state_;
  state_ = Spectral;
  Real pow_decay_n = 1.0;
  for (int n=0; n<N_; ++n) {
    data_[n] = pow_decay_n * randomReal();
    pow_decay_n *= decay;
  }
  
  if (a_dirichlet && b_dirichlet) {
    data_[1] -= 0.5*(eval_b() - eval_a());
    data_[0] -= 0.5*(eval_b() + eval_a());
  }
  else if (a_dirichlet)
    data_[0] -= eval_a();
  else if (a_dirichlet)
    data_[0] -= eval_b();

  if (startState == Physical) {
    ChebyTransform t(N_);
    t.ichebyfft(*this);
  }
  
}

void ChebyCoeff::setToZero() {
  for (int i=0; i<N_; ++i) 
    data_[i] = 0.0;
}

void ChebyCoeff::fill(const ChebyCoeff& v) {
  assert(v.state_ == Spectral);
  assert(state_ == Spectral);
  int Ncommon = lesser(N_, v.N_);
  int i; // MSVC++ FOR-SCOPE BUG
  for (i=0; i<Ncommon; ++i) 
    data_[i] = v.data_[i];
  for (i=Ncommon; i<N_; ++i) 
    data_[i] = 0.0;
}
void ChebyCoeff::interpolate(const ChebyCoeff& v) {
  assert(a_ >= v.a_ && b_ <= v.b_);
  assert(v.state_ == Spectral);
  state_ = Physical;
  Real piN = pi/(N_-1);
  Real width = (b_-a_)/2;
  Real center = (b_+a_)/2;
  for (int n=0; n<N_; ++n) 
    data_[n] = v.eval(center + width*cos(n*piN));
  //makeSpectral();
}

void ChebyCoeff::reflect(const ChebyCoeff& v, parity p) {
  assert((a_+b_)/2 == v.a() && b_ <= v.b() && b_ > v.a());
  assert(v.state_ == Spectral);
  state_ = Physical;
  Real piN = pi/(N_-1);
  Real width = (b_-a_)/2;
  Real center = (b_+a_)/2;
  int N2=N_/2;
  int N1=N_-1;
  int sign = (p==Odd) ? -1 : 1;
  for (int n=0; n<N2; ++n) {
    Real tmp = v.eval(center + width*cos(n*piN));
    data_[n] = tmp;
    data_[N1-n] = sign*tmp;
  }
  ChebyTransform t(N_);
  makeSpectral(t);
  for (int n=2*N_/3; n<N_; ++ n)
    data_[n] = 0.0;
  makePhysical(t);
}

void ChebyCoeff::setBounds(Real a, Real b) {
  a_ = a;
  b_ = b;
  assert(b_ > a_);
}

void ChebyCoeff::setState(fieldstate s) {
  assert(s == Physical || s == Spectral);
  state_=s;
}


Real ChebyCoeff::eval_b() const {
  if (state_ == Spectral) {
    Real sum = 0.0;
    for (int n=N_-1; n>=0; --n)
      sum += data_[n];
    return sum;
  }
  else 
    return data_[0];
}

Real ChebyCoeff::eval_a()  const {
  if (state_ == Spectral) {
    Real sum = 0.0;
    for (int n=N_-1; n>=0; --n) 
      sum += data_[n] * ((n%2 == 0) ? 1 : -1);
    return sum;
  }
  else
    return data_[N_-1];
}

Real ChebyCoeff::eval(Real x) const {
  assert(state_ == Spectral);
  Real y = (2*x-a_-b_)/(b_-a_);
  Real y2 = 2*y;
  Real d=0.0;
  Real dd=0.0;
  for (int j=N_-1; j>0; --j) {
    Real sv=d;
    d = y2*d - dd + data_[j];
    dd=sv;
  }
  return (y*d - dd + data_[0]); // NR has 0.5*c[0], but that gives wrong results!
}

ChebyCoeff ChebyCoeff::eval(const Vector& x)  const {
  ChebyCoeff f(N_, a_, b_, Physical);
  eval(x, f);
  return f;
}

// Numerical Recipes Clenshaw evaluation of Spectral expansion
void ChebyCoeff::eval(const Vector& x, ChebyCoeff& f) const {
  assert(state_ == Spectral);
  int N=x.length();
  if (f.length() != N)
    f.resize(N);
  f.setBounds(a_, b_);
  f.setState(Physical);

  int M=N_;
  for (int i=0; i<N; ++i) {
    Real y = (2*x[i]-a_-b_)/(b_-a_);
    Real y2 = 2*y;
    Real d=0.0;
    Real dd=0.0;
    for (int j=M-1; j>0; --j) {
      Real sv=d;
      d = y2*d - dd + data_[j];
      dd=sv;
    }
    f[i] = y*d - dd + data_[0]; // NR has 0.5*c[0], but that gives wrong results!
  }
}

Real ChebyCoeff::mean() const {
  assert(state_ == Spectral);
  Real sum = data_[0];    
  for (int n=2; n<N_; n+=2)
    sum -= data_[n]/(n*n-1); // *2
  return sum;                // /2
}

//Real ChebyCoeff::energy() const {return L2Norm2(*this, false);}

void ChebyCoeff::save(const string& filebase) const {
  string filename(filebase);
  filename += string(".asc");
  ofstream os(filename.c_str());
  os << setprecision(REAL_DIGITS);
  if (MATLABVERSION == 5)
    os << "% " << N_ << ' ' << a_ << ' ' << b_ << ' ' << state_ << '\n';
  for (int i=0; i<N_; ++i) 
    os << data_[i] << '\n';
  os.close();
}

/**************
void ChebyCoeff::m4save(const string& filebase) const {
  string filename(filebase);
  filename += string(".asc");
  ofstream os(filename.c_str());
  os << setprecision(REAL_DIGITS);
  for (int i=0; i<N_; ++i) 
    os << data_[i] << '\n';
  os.close();
}
******************/

void ChebyCoeff::binaryDump(ostream& os) const {
  write(os, N_);
  write(os, a_);
  write(os, b_);
  write(os, state_);
  for (int i=0; i<N_; ++i) 
    write(os, data_[i]);
}
void ChebyCoeff::binaryLoad(istream& is) {
  if (!is.good()) {
    cerr << "ChebyCoeff::binaryLoad(istream& is) : input error\n";
    abort();
  }
  int newN_;
  read(is, newN_);
  read(is, a_);
  read(is, b_);
  read(is, state_);
  resize(newN_);
  for (int i=0; i<N_; ++i) {
    if (!is.good()) {
      cerr << "ChebyCoeff::binaryLoad(istream& is) : input error\n";
      abort();
    }
    read(is, data_[i]);
  }
}


ChebyCoeff& ChebyCoeff::operator*=(Real c) {
  for (int i=0; i<N_; ++i)
    data_[i] *= c;
  return *this;
}

ChebyCoeff& ChebyCoeff::operator+=(const ChebyCoeff& a) {
  assert(congruent(a));
  for (int i=0; i<N_; ++i)
    data_[i] += a.data_[i];
  return *this;
}

ChebyCoeff& ChebyCoeff::operator-=(const ChebyCoeff& a) {
  assert(congruent(a));
  for (int i=0; i<N_; ++i)
    data_[i] -= a.data_[i];
  return *this;
}

ChebyCoeff operator*(Real c, const ChebyCoeff& v) {
  ChebyCoeff rtn(v);
  rtn *= c;
  return rtn;
}

ChebyCoeff& ChebyCoeff::operator*=(const ChebyCoeff& a) {
  assert(congruent(a));
  assert(state_ == Physical);
  for (int i=0; i<N_; ++i)
    data_[i] *= a.data_[i];
  return *this;
}

bool ChebyCoeff::congruent(const ChebyCoeff& v)  const {
  return (v.N_==N_  && v.a_==a_ && v.b_==b_ &&  v.state_==state_) 
    ? true : false;
}

ChebyCoeff operator+(const ChebyCoeff& u, const ChebyCoeff& v) {
  ChebyCoeff rtn(u);
  rtn += v;
  return rtn;
}

ChebyCoeff operator-(const ChebyCoeff& u, const ChebyCoeff& v) {
  ChebyCoeff rtn(u);
  rtn -= v;
  return rtn;
}
bool operator==(const ChebyCoeff& u, const ChebyCoeff& v) {
  if (!u.congruent(v))
    return false;
  for (int i=0; i<u.numModes(); ++i)
    if (u[i] != v[i])
      return false;
  return true;
}

void swap(ChebyCoeff& f, ChebyCoeff& g) {
  Real rtmp = f.a_;
  f.a_ = g.a_;
  g.a_ = rtmp;
  rtmp = f.b_;
  f.b_ = g.b_;
  g.b_ = rtmp;

  fieldstate stmp = f.state_;
  f.state_ = g.state_;
  g.state_ = stmp;

  int itmp = f.N_;
  f.N_ = g.N_;
  g.N_ = itmp;

  Real* dtmp = f.data_;
  f.data_ = g.data_;
  g.data_ = dtmp;
}
    

void integrate(const ChebyCoeff& dudy, ChebyCoeff& u) {
  assert(dudy.state() == Spectral);
  int N=dudy.numModes();
  if (u.numModes() != N)
    u.resize(N);
  u.setBounds(u.a(), u.b());
  u.setState(Spectral);
  
  Real h2 = (dudy.b()-dudy.a())/2;
  switch (N) {
  case 0:
    break;
  case 1:
    u[0] = 0.0;
    break;
  case 2:
    u[0] = 0;
    u[1] = h2*dudy[0];
    break;
  default:
    u[1] = h2*(dudy[0] - dudy[2]/2);
    for (int n=2; n<N-1; ++n) 
      u[n] = h2*(dudy[n-1] - dudy[n+1])/(2*n);
    u[N-1] = h2*dudy[N-2]/(2*(N-1));
    u[0] -= u.mean();  // const term is arbitrary, set to zero.
  }
  return;

}

ChebyCoeff integrate(const ChebyCoeff& dudy) {
  ChebyCoeff u(dudy.length(), dudy.a(), dudy.b(), Spectral);
  integrate(dudy, u);
  return u;
}


void diff(const ChebyCoeff& u, ChebyCoeff& dudy) {
  assert(u.state() == Spectral);
  if (dudy.numModes() != u.numModes())
    dudy.resize(u.numModes());
  dudy.setBounds(u.a(), u.b());
  dudy.setState(Spectral);

  int Nb = u.numModes() - 1;
  if (Nb==-1)
    return;
  if (Nb==0) {
    dudy[0] = 0.0;
    return;
  }
  Real scale=4.0/u.L();
  dudy[Nb] = 0.0;
  dudy[Nb-1] = scale*Nb*u[Nb];
  for (int n=Nb-2; n>=0; --n)
    dudy[n] = dudy[n+2] + scale*(n+1)*u[n+1];
  dudy[0] /= 2.0;
  
  //dudy *= 2.0/u.L();
}

void diff2(const ChebyCoeff& u, ChebyCoeff& d2udy2) {
  ChebyCoeff dudy(u.length(), u.a(), u.b(), Spectral);
  diff(u, dudy);
  diff(dudy, d2udy2);
}

void diff2(const ChebyCoeff& u, ChebyCoeff& d2udy2, ChebyCoeff& tmp) {
  diff(u, tmp);
  diff(tmp, d2udy2);
}

void diff(const ChebyCoeff& f, ChebyCoeff& df, int n) {
  assert(n>=0);
  assert(f.state() == Spectral);
  df = f;
  ChebyCoeff tmp;
  for (int k=0; k<n; ++k) {
    diff(df,tmp);
    swap(df,tmp);
  }
  return;
}

ChebyCoeff diff(const ChebyCoeff& u) {
  ChebyCoeff du(u.length(), u.a(), u.b(), Spectral);
  diff(u, du);
  return du;
}

ChebyCoeff diff2(const ChebyCoeff& u) {
  ChebyCoeff du2(u.length(), u.a(), u.b(), Spectral);
  diff2(u, du2);
  return du2;
}

ChebyCoeff diff(const ChebyCoeff& f, int n) {
  assert(n>=0);
  ChebyCoeff  g = f;
  ChebyCoeff gy(f.N(), f.a(), f.b(), Spectral);

  for (int n_=0; n_<n; ++n_) {
    diff(g,gy);
    swap(g,gy);
  }
  return g;
}

Vector chebypoints(int N, Real a, Real b) {
  Vector xcheb(N);
  Real piN = pi/(N-1);
  Real radius = (b-a)/2;
  Real center = (b+a)/2;
  for (int j=0; j<N; ++j)
    xcheb[j] = center + radius*cos(j*piN);
  return xcheb;
}
Vector periodicpoints(int N, Real L) {
  Vector x(N);
  Real dx=L/N;
  for (int n=0; n<N; ++n)
    x[n] = n*dx;
  return x;
}

// ===========================================================
// L2 norms
Real L2Norm2(const ChebyCoeff& u,  bool normalize) {
  assert(u.state() == Spectral);
  int N=u.numModes();
  Real sum=0.0;
  Real e=1.0;
  for (int m=N-1; m>=0; --m) {
    Real um = u[m];
    Real psum = 0.0;
    for (int n=m%2; n<N; n+=2) 
      psum += um*u[n]*(e-m*m-n*n)/((e+m-n)*(e-m+n)*(e+m+n)*(e-m-n));
    sum += psum;
  }
  if (!normalize)
    sum *= u.b()-u.a();
  return sum;
}

Real L2Dist2(const ChebyCoeff& u, const ChebyCoeff& v, bool normalize) {
  ChebyCoeff tmp(u);
  tmp -= v;
  return L2Norm2(tmp, normalize);
}

Real L2Norm(const ChebyCoeff& u, bool normalize) {
  return sqrt(L2Norm2(u, normalize));
}

Real L2Dist(const ChebyCoeff& u, const ChebyCoeff& v, bool normalize) {
  return sqrt(L2Dist2(u,v, normalize));
}

Real L2InnerProduct(const ChebyCoeff& u, const ChebyCoeff& v, bool normalize) {
  assert(u.state()==Spectral && v.state()==Spectral);
  assert(u.a()==v.a() && u.b()==v.b());
  assert(u.numModes() ==  v.numModes());
  int N = u.numModes();
  Real sum=0.0;
  Real e=1.0;
  for (int m=N-1; m>=0; --m) {
    Real um = u[m];
    Real psum = 0.0;
    for (int n=m%2; n<N; n+=2) 
      psum += um*v[n]*(e-m*m-n*n)/((e+m-n)*(e-m+n)*(e+m+n)*(e-m-n));
    sum += psum;
  }
  if (!normalize)
    sum *= u.b()-u.a();
  return sum;
}

// ===========================================================
// cheby norms
Real chebyNorm2(const ChebyCoeff& u,  bool normalize) {
  assert(u.state() == Spectral);
  int N=u.numModes();
  Real sum=0.0;
  for (int m=N-1; m>0; --m) 
    sum += square(u[m]);
  if (N>0)
    sum += 2*square(u[0]);  // coeff of T_0(x) has prefactor of 2 in norm
  if (!normalize)
    sum *= u.b()-u.a();
  sum *= pi/2;
  return sum;
}

Real chebyDist2(const ChebyCoeff& u, const ChebyCoeff& v, bool normalize) {
  assert(u.state()==Spectral && v.state()==Spectral);
  assert(u.a()==v.a() && u.b()==v.b());
  assert(u.numModes() ==  v.numModes());
  int N=u.numModes();
  Real sum=0.0;
  for (int m=N-1; m>0; --m) 
    sum += square(u[m]-v[m]);
  if (N>0)
    sum += 2*square(u[0]-v[0]);  // coeff of T_0(x) has prefactor of 2 in norm
  if (!normalize)
    sum *= u.b()-u.a();
  return sum*pi/2;
}

Real chebyNorm(const ChebyCoeff& u, bool normalize) {
  return sqrt(chebyNorm2(u, normalize));
}

Real chebyDist(const ChebyCoeff& u, const ChebyCoeff& v, bool normalize) {
  return sqrt(chebyDist2(u,v, normalize));
}

Real chebyInnerProduct(const ChebyCoeff& u, const ChebyCoeff& v, bool normalize) {
  assert(u.state()==Spectral && v.state()==Spectral);
  assert(u.a()==v.a() && u.b()==v.b());
  assert(u.numModes() ==  v.numModes());
  int N=u.numModes();
  Real sum=0.0;
  for (int m=N-1; m>0; --m) 
    sum += u[m]*v[m];
  if (N>0)
    sum += 2*u[0]*v[0];  // coeff of T_0(x) has prefactor of 2 in norm
  if (!normalize)
    sum *= u.b()-u.a();
  return sum*pi/2;
}

// ===========================================================
// switchable norms
Real norm2(const ChebyCoeff& u, NormType n, bool normalize) {
  return (n==L2NORM) ? L2Norm2(u,normalize) : chebyNorm2(u,normalize);
}
; 
Real dist2(const ChebyCoeff& u, const ChebyCoeff& v, NormType n,
	   bool normalize) {
  return (n==L2NORM) ? L2Dist2(u,v,normalize) : chebyDist2(u,v,normalize);
}

Real norm(const ChebyCoeff& u, NormType n, bool normalize) {
  return (n==L2NORM) ? L2Norm(u,normalize) : chebyNorm(u,normalize);
}
Real dist(const ChebyCoeff& u, const ChebyCoeff& v, NormType n, 
	  bool normalize) {
  return (n==L2NORM) ? L2Dist(u,v,normalize) : chebyDist(u,v,normalize);
}
Real innerProduct(const ChebyCoeff& u, const ChebyCoeff& v, 
		  NormType n,  bool normalize) {
  return (n==L2NORM) ? L2InnerProduct(u,v,normalize):chebyInnerProduct(u,v,normalize);
}

//=======================================================================
// ComplexChebyCoeff

ComplexChebyCoeff::ComplexChebyCoeff() 
  :
  re(),
  im()
{}

ComplexChebyCoeff::ComplexChebyCoeff(int N, Real a, Real b, fieldstate s) 
  :
  re(N,a,b,s),
  im(N,a,b,s) 
{}
ComplexChebyCoeff::ComplexChebyCoeff(int N, const ComplexChebyCoeff& u) 
  :
  re(N, u.re),
  im(N, u.im) 
{}

ComplexChebyCoeff::ComplexChebyCoeff(const ChebyCoeff& r, const ChebyCoeff& i)
  :
  re(r),
  im(i) 
{}

ComplexChebyCoeff::ComplexChebyCoeff(const string& filebase) 
  :
  re(),
  im()
{
  string filename(filebase);
  filename += string(".asc");
  
  //ifstream sis(sizefile.c_str());
  ifstream is(filename.c_str());

  // Read in header. Form is "%N a b s"
  char c;
  int N;
  is >> c;
  if (c != '%') {
    string message("ComplexChebyCoeff(filebase): bad header in file ");
    message += filename;
    cerr << message << endl;
    assert(false);
  }
  Real a,b;
  fieldstate s;
  is >> N >> a >> b >> s;
  assert(is.good());
  
  re.resize(N);
  im.resize(N);
  re.setBounds(a,b);
  im.setBounds(a,b);
  re.setState(s);
  im.setState(s);

  for (int i=0; i<N; ++i) {
    is >> re[i] >> im[i];
    assert(is.good());
  }
  is.close();
}

/****************
ComplexChebyCoeff::ComplexChebyCoeff(istream& is) 
:
re(is),
im(is)
{
if (!is.good()) {
cerr << "ComplexChebyCoeff::ComplexChebyCoeff(istream& is) : input error\n";
abort();
}}
*****************/
void ComplexChebyCoeff::resize(int N) {re.resize(N); im.resize(N);}

void ComplexChebyCoeff::chebyfft() {
  ChebyTransform t(re.numModes());
  t.chebyfft(*this);
}
void ComplexChebyCoeff::ichebyfft(){
  ChebyTransform t(re.numModes());
  t.ichebyfft(*this);
}
void ComplexChebyCoeff::makeSpectral() {
  ChebyTransform t(re.numModes());
  t.makeSpectral(*this);
}
void ComplexChebyCoeff::makePhysical() {
  ChebyTransform t(re.numModes());
  t.makePhysical(*this);
}
void ComplexChebyCoeff::makeState(fieldstate s){ 
  assert(re.state() == im.state());
  if (re.state() != s) {       // need to change state
    ChebyTransform t(re.numModes());
    if (re.state()==Physical)  // state is Physical, change to Spectral
      t.chebyfft(*this);
    else
      t.ichebyfft(*this);      // state is Spectral, change to Physical
  }
}

void ComplexChebyCoeff::randomize(Real decay, bool a_dirichlet, bool b_dirichlet) {
  re.randomize(decay,a_dirichlet,b_dirichlet); 
  im.randomize(decay,a_dirichlet,b_dirichlet); 
}
void ComplexChebyCoeff::setToZero(){re.setToZero(); im.setToZero();}
void ComplexChebyCoeff::fill(const ComplexChebyCoeff& v) {
  re.fill(v.re);
  im.fill(v.im);
}
void ComplexChebyCoeff::interpolate(const ComplexChebyCoeff& v) {
  re.interpolate(v.re);
  im.interpolate(v.im);
}
void ComplexChebyCoeff::reflect(const ComplexChebyCoeff& v, parity p) {
  re.reflect(v.re, p);
  im.reflect(v.im, p);
}

void ComplexChebyCoeff::setState(fieldstate s) {
  re.setState(s);
  im.setState(s);
}
void ComplexChebyCoeff::setBounds(Real a, Real b) {
  re.setBounds(a,b);
  im.setBounds(a,b);
}



Complex ComplexChebyCoeff::eval_a() const {
  return Complex(re.eval_a(), im.eval_a());
}
Complex ComplexChebyCoeff::eval_b() const {
  return Complex(re.eval_b(), im.eval_b());
}
Complex ComplexChebyCoeff::eval(Real x) const {
  return Complex(re.eval(x), im.eval(x));
}
ComplexChebyCoeff& ComplexChebyCoeff::operator+=(const ComplexChebyCoeff& u) {
  re += u.re;
  im += u.im;
  return *this;
}
ComplexChebyCoeff& ComplexChebyCoeff::operator-=(const ComplexChebyCoeff& u) {
  re -= u.re;
  im -= u.im;
  return *this;
}

ComplexChebyCoeff& ComplexChebyCoeff::operator*=(Real c) {
  re *= c;
  im *= c;
  return *this;
}

ComplexChebyCoeff& ComplexChebyCoeff::operator*=(const ComplexChebyCoeff& u) {
  assert(congruent(u));
  assert(re.state() == Physical);
  assert(im.state() == Physical);
  Real r;
  Real i;
  for (int ny=0; ny<im.numModes(); ++ny) {
    r = re[ny]*u.re[ny] - im[ny]*u.im[ny];
    i = re[ny]*u.im[ny] + im[ny]*u.re[ny];
    re[ny] = r;
    im[ny] = i;
  }
  return *this;
}
void ComplexChebyCoeff::conjugate() {
  im *= -1.0;
}

void ComplexChebyCoeff::save(const string& filebase) const {
  string filename(filebase);
  filename += string(".asc");
  ofstream os(filename.c_str());

  os << setprecision(REAL_DIGITS);
  if (MATLABVERSION == 5)
    os << "% " << re.length() << ' ' << re.a() << ' ' << re.b() << ' ' 
       << re.state() << '\n';
  for (int n=0; n<re.length(); ++n) 
    os << re[n] << ' ' << im[n] << endl;
  os.close();
}
/*****************************
void ComplexChebyCoeff::m4save(const string& filebase) const {
  string filename(filebase);
  filename += string(".asc");
  ofstream os(filename.c_str());
  os << setprecision(REAL_DIGITS);
  for (int n=0; n<re.length(); ++n) 
    os << re[n] << ' ' << im[n] << endl;
  os.close();
}
******************************/

void ComplexChebyCoeff::binaryDump(ostream& os) const {
  re.binaryDump(os);
  im.binaryDump(os);
}
void ComplexChebyCoeff::binaryLoad(istream& is) {
  re.binaryLoad(is);
  im.binaryLoad(is);
}

Complex ComplexChebyCoeff::mean() const {
  return Complex(re.mean(), im.mean());
}

//Real ComplexChebyCoeff::energy() const {
//return re.energy() + im.energy();
//}

ComplexChebyCoeff& ComplexChebyCoeff::operator *= (Complex c) {
  Real cr = Re(c);
  Real ci = Im(c);
  Real ur;
  Real ui;
  for (int n=0; n<re.numModes(); ++n) {
    ur = re[n];
    ui = im[n];
    re[n] = ur*cr - ui*ci;
    im[n] = ur*ci + ui*cr;
  }
  return *this;
}

bool ComplexChebyCoeff::congruent(const ComplexChebyCoeff& v) const {
  assert(re.congruent(im));
  return (re.congruent(v.re) && im.congruent(v.im));
}


bool operator==(const ComplexChebyCoeff& u, const ComplexChebyCoeff& v) {
  return u.re == v.re && u.im == v.im;
}

void swap(ComplexChebyCoeff& f, ComplexChebyCoeff& g) {
  swap(f.re,g.re);
  swap(f.im,g.im);
}
    

ComplexChebyCoeff operator*(Real c, const ComplexChebyCoeff& v) {
  ComplexChebyCoeff rtn(v);
  rtn *= c;
  return rtn;
}

ComplexChebyCoeff operator+(const ComplexChebyCoeff& u, const ComplexChebyCoeff& v) {
  ComplexChebyCoeff rtn(u);
  rtn += v;
  return rtn;
}

ComplexChebyCoeff operator-(const ComplexChebyCoeff& u, const ComplexChebyCoeff& v) {
  ComplexChebyCoeff rtn(u);
  rtn -= v;
  return rtn;
}

void diff(const ComplexChebyCoeff& u, ComplexChebyCoeff& du) {
  diff(u.re, du.re);
  diff(u.im, du.im);
}

void diff2(const ComplexChebyCoeff& u, ComplexChebyCoeff& d2u) {
  diff2(u.re, d2u.re);
  diff2(u.im, d2u.im);
}
void diff2(const ComplexChebyCoeff& u, ComplexChebyCoeff& d2u, ComplexChebyCoeff& tmp) {
  diff2(u.re, d2u.re, tmp.re);
  diff2(u.im, d2u.im, tmp.im);
}

void diff(const ComplexChebyCoeff& u, ComplexChebyCoeff& du, int n) {
  diff(u.re, du.re, n);
  diff(u.im, du.im, n);
}
ComplexChebyCoeff diff(const ComplexChebyCoeff& u) {
  ComplexChebyCoeff du(u.numModes(), u.a(), u.b(), Spectral);
  diff(u.re, du.re);
  diff(u.im, du.im);
  return du;
}
ComplexChebyCoeff diff2(const ComplexChebyCoeff& u) {
  ComplexChebyCoeff du2(u.numModes(), u.a(), u.b(), Spectral);
  diff2(u.re, du2.re);
  diff2(u.im, du2.im);
  return du2;
}
ComplexChebyCoeff diff(const ComplexChebyCoeff& u, int n) {
  ComplexChebyCoeff du(u.numModes(), u.a(), u.b(), Spectral);
  diff(u.re, du.re, n);
  diff(u.im, du.im, n);
  return du;
}

void integrate(const ComplexChebyCoeff& du, ComplexChebyCoeff& u) {
  integrate(du.re, u.re);
  integrate(du.im, u.im);
}

ComplexChebyCoeff integrate(const ComplexChebyCoeff& du) {
  ComplexChebyCoeff u(du.numModes(), du.a(), du.b(), Spectral);
  integrate(du.re, u.re);
  integrate(du.im, u.im);
  return u;
}

// ====================================================================
ChebyTransform::ChebyTransform(int N, int flags) 
  : N_(N),
    sin_table_(new Real[N-1]),
    cos_table_(new Real[N-1]),
    q_(new Real[N-1]),
    Q_(new fftw_complex[(N-1)/2 +1]),
    realfft_plan_(0){
  assert(N_ >0);
  assert(N_%2 == 1);

  int Nb = N_-1;
  Real piNb = pi/Nb;
  for (int j=0; j<Nb; ++j) {
    sin_table_[j] = sin(j*piNb);
    cos_table_[j] = cos(j*piNb);
  }
  int rank=1;
  int n[1];
  n[0] = Nb;
  realfft_plan_  = rfftwnd_create_plan(rank, n, FFTW_REAL_TO_COMPLEX, flags);
}

ChebyTransform::~ChebyTransform() {
  delete[] sin_table_;
  delete[] cos_table_;
  delete[] q_;
  delete[] Q_;
  rfftwnd_destroy_plan(realfft_plan_);
}

void ChebyTransform::save(const ChebyCoeff& u, const string& name, int deriv) const {
  ChebyCoeff tmp(u.length(), u.a(), u.b(), u.state());
  if (deriv == 0) 
    tmp = u;
  else if (deriv == 1)
    diff(u,tmp);
  else if (deriv == 2)
    diff2(u,tmp);

  string cfilebase(name);
  cfilebase += string("c");
  tmp.save(cfilebase.c_str());
  ichebyfft(tmp);
  tmp.save(name);
}
  
void ChebyTransform::save(const ComplexChebyCoeff& u, const string& name, int deriv) const {
  ComplexChebyCoeff tmp(u.length(), u.a(), u.b(), u.state());
  if (deriv == 0) 
    tmp = u;
  else if (deriv == 1) {
    diff(u.re, tmp.re);
    diff(u.im, tmp.im);
  }
  else if (deriv == 2) {
    diff2(u.re, tmp.re);
    diff2(u.im, tmp.im);
  }
  else 
    assert(false);

  string cfilebase(name);
  cfilebase += string("c");
  tmp.save(cfilebase.c_str());
  ichebyfft(tmp);
  tmp.save(name);
}
  

void ChebyTransform::chebyfft(ChebyCoeff& u) const {
  assert(u.length() == N_);
  assert(u.state() == Physical);
  //int rank = 1;
  int Nb = N_-1;
  int n[1];
  n[0] = Nb;

  double c = 2.0/Real(Nb);

  Real a;
  Real b;

  int j; // MSVC++ FOR-SCOPE BUG
  for (j=0; j<N_; ++j) 
    u[j] *= c;
  
  // Transform data according to NR formula 
  for (j=0; j<Nb; ++j) {
    a = u[j];
    b = u[Nb-j];
    q_[j] = 0.5*(a+b) - sin_table_[j]*(a-b);
  }

  rfftwnd_one_real_to_complex(realfft_plan_, q_, Q_);

  // Initialize recursion for unpacking transform
  Real sum = 0.5*(u[0] - u[Nb]);
  u[0] = Q_[0].re;

  for (j=1; j<Nb; ++j)
    sum += u[j]*cos_table_[j];
  u[1] = sum;
	
  for (int k=1; k<Nb/2; ++k) {      // N is even 
    u[2*k]   = Q_[k].re;
    u[2*k+1] = (sum -= Q_[k].im); 
  }
  u[Nb] = Q_[Nb/2].re;

  // Copy transformed data back into main data array

  // 0th elem is different due to reln between cheby and cos transforms.
  u[0] *= 0.5; // can fold into initial assigment for efficiency.
  u.setState(Spectral);
  return;

}

void ChebyTransform::ichebyfft(ChebyCoeff& u) const {
  assert(u.length() == N_);
  assert(u.state() == Spectral);
  //int rank =1;
  int Nb = N_-1;
  int n[1];
  n[0] = Nb;
  Real a;
  Real b;

  // Copy data spread through memory into a stride-1 scratch array.
  // 0th elem is different due to reln between cheby and cos transforms.
  u[0] *= 2.0;
	
  // Transform data according to NR formula 
  int j; // MSVC++ FOR-SCOPE BUG
  for (j=0; j<Nb; ++j) {
    a = u[j];
    b = u[Nb-j];
    q_[j] = 0.5*(a+b) - sin_table_[j]*(a-b);
  }

  rfftwnd_one_real_to_complex(realfft_plan_, q_, Q_);

  // Initialize recursion for unpacking transform
  Real sum = 0.5*(u[0] - u[Nb]);
  u[0] = Q_[0].re;

  for (j=1; j<Nb; ++j)
    sum += u[j]*cos_table_[j];
  u[1] = sum;
	
  for (int k=1; k<Nb/2; ++k) {      // N is even 
    u[2*k]   = Q_[k].re;
    u[2*k+1] = (sum -= Q_[k].im); 
  }
  u[Nb] = Q_[Nb/2].re;
  u.setState(Physical);
  return;
}

void ChebyTransform::chebyfft(ComplexChebyCoeff& u) const {
  chebyfft(u.re);
  chebyfft(u.im);
}

void ChebyTransform::ichebyfft(ComplexChebyCoeff& u) const {
  ichebyfft(u.re);
  ichebyfft(u.im);
}
void ChebyTransform::makeSpectral(ChebyCoeff& u) const {
  if (u.state() == Physical)
    chebyfft(u);
  return;
}
void ChebyTransform::makeSpectral(ComplexChebyCoeff& u) const {
  if (u.state() == Physical)
    chebyfft(u);
  return;
}
void ChebyTransform::makePhysical(ChebyCoeff& u) const {
  if (u.state() == Spectral)
    ichebyfft(u);
  return;
}
void ChebyTransform::makePhysical(ComplexChebyCoeff& u) const {
  if (u.state() == Spectral)
    ichebyfft(u);
  return;
}

ostream& operator<<(ostream& os, const ComplexChebyCoeff& u) {
  for (int i=0; i<u.length(); ++i)
    os << '(' << u.re[i] << ", " << u.im[i] << ")\n";
  return os;
}

//Real L1Norm(const ComplexChebyCoeff& u) {return L1Norm(u.re) + L1Norm(u.im);}
//Real L1Dist(const ComplexChebyCoeff& u, const ComplexChebyCoeff& v) {
//return L1Dist(u.re, v.re) + L1Dist(u.im, v.im);
//}


// =======================================================================
// L2 norms
Real L2Norm2(const ComplexChebyCoeff& u, bool normalize) {
  return L2Norm2(u.re, normalize) + L2Norm2(u.im, normalize);
}

Real L2Dist2(const ComplexChebyCoeff& u, const ComplexChebyCoeff& v, 
	     bool normalize) {
  return L2Dist2(u.re, v.re, normalize) + L2Dist2(u.im, v.im, normalize);
}

Real L2Norm(const ComplexChebyCoeff& u, bool normalize) {
  return sqrt(L2Norm2(u.re, normalize) + L2Norm2(u.im, normalize));
}

Real L2Dist(const ComplexChebyCoeff& u, const ComplexChebyCoeff& v, 
	    bool normalize) {
  return sqrt(L2Dist2(u.re, v.re, normalize) + L2Dist2(u.im, v.im, normalize));
}

Complex L2InnerProduct(const ComplexChebyCoeff& u, const ComplexChebyCoeff& v,
		       bool normalize){
  assert(u.state()==Spectral && v.state()==Spectral);
  assert(u.numModes() == v.numModes());
  int N = u.numModes();

  // Unroll the complex operations, for efficiency.
  // Profiling an L2InnerProduct-intensive program reveals that 
  // complex * double is expensive
  Real e=1.0;
  Real sum_r=0.0;
  Real sum_i=0.0;
  Real psum_r;
  Real psum_i;
  Real um_r;
  Real um_i;
  Real vn_r;
  Real vn_i;
  Real k;
  for (int m=N-1; m>=0; --m) {
    um_r = u.re[m];
    um_i = u.im[m];
    psum_r = 0.0;
    psum_i = 0.0;
    for (int n=m%2; n<N; n+=2) {
      k = (e-m*m-n*n)/((e+m-n)*(e-m+n)*(e+m+n)*(e-m-n));
      vn_r = v.re[n];
      vn_i = v.im[n];

      psum_r += k*(um_r*vn_r + um_i*vn_i);
      psum_i += k*(um_i*vn_r - um_r*vn_i);
    }
    sum_r += psum_r;
    sum_i += psum_i;
  }
  if (!normalize) {
    sum_r *= u.b()-u.a();
    sum_i *= u.b()-u.a();
  }
  return Complex(sum_r, sum_i);
}


// =======================================================================
// cheby norms 
Real chebyNorm2(const ComplexChebyCoeff& u, bool normalize) {
  return chebyNorm2(u.re, normalize) + chebyNorm2(u.im, normalize);
}

Real chebyDist2(const ComplexChebyCoeff& u, const ComplexChebyCoeff& v, 
	     bool normalize) {
  return chebyDist2(u.re, v.re, normalize) + chebyDist2(u.im, v.im, normalize);
}

Real chebyNorm(const ComplexChebyCoeff& u, bool normalize) {
  return sqrt(chebyNorm2(u.re, normalize) + chebyNorm2(u.im, normalize));
}

Real chebyDist(const ComplexChebyCoeff& u, const ComplexChebyCoeff& v, 
	    bool normalize) {
  return sqrt(chebyDist2(u.re, v.re, normalize) + chebyDist2(u.im, v.im, normalize));
}

Complex chebyInnerProduct(const ComplexChebyCoeff& u, 
			  const ComplexChebyCoeff& v,
			  bool normalize){
  assert(u.state()==Spectral && v.state()==Spectral);
  assert(u.numModes() == v.numModes());
  int N = u.numModes();

  // Unroll the complex operations, for efficiency.
  // Profiling an L2InnerProduct-intensive program reveals that 
  // complex * double is expensive
  Real sum_r = 0.0; 
  Real sum_i = 0.0; 
  for (int m=N-1; m>0; --m) {
    sum_r += u.re[m]*v.re[m] + u.im[m]*v.im[m];
    sum_i += u.im[m]*v.re[m] - u.re[m]*v.im[m];
  }
  if (N>0) {
    sum_r += 2*(u.re[0]*v.re[0] + u.im[0]*v.im[0]);
    sum_i += 2*(u.im[0]*v.re[0] - u.re[0]*v.im[0]);
  }
  if (!normalize) {
    sum_r *= u.b()-u.a();
    sum_i *= u.b()-u.a();
  }
  return Complex(sum_r*pi/2, sum_i*pi/2);
}


// ===========================================================
// switchable norms
Real norm2(const ComplexChebyCoeff& u, NormType n, bool normalize) {
  return (n == L2NORM) ? L2Norm2(u,normalize) : chebyNorm2(u,normalize);
}

Real dist2(const ComplexChebyCoeff& u, const ComplexChebyCoeff& v,
	   NormType n, bool normalize) {
  return (n == L2NORM) ? L2Dist2(u,v,normalize) : chebyDist2(u,v,normalize);
}

Real norm(const ComplexChebyCoeff& u, NormType n, bool normalize) {
  return (n == L2NORM) ? L2Norm(u,normalize) : chebyNorm(u,normalize);
}

Real dist(const ComplexChebyCoeff& u, const ComplexChebyCoeff& v, 
	  NormType n, bool normalize) {
  return (n == L2NORM) ? L2Dist(u,v,normalize) : chebyDist(u,v,normalize);
}
Complex innerProduct(const ComplexChebyCoeff& u, const ComplexChebyCoeff& v, 
		  NormType n,  bool normalize) {
  return (n == L2NORM) ? L2InnerProduct(u,v,normalize) : chebyInnerProduct(u,v,normalize);
}



/***************************
ComplexChebyCoeff chebyEval(const ComplexChebyCoeff& c, const Vector& x, 
			    Real a, Real b) {
  ComplexChebyCoeff rtn(c.length(), );
  chebyEval(rtn.re, c.re, x, a, b);
  chebyEval(rtn.im, c.im, x, a, b);
  return rtn;
}

Complex chebyEval(const ComplexChebyCoeff& c, Real x, Real a, Real b) {
  return chebyEval(c.re, x, a, b) + I*chebyEval(c.im, x, a, b);
}
******************************/
