/* mathdefs.h: some small mathematical conveniences.
 * Channelflow-0.9
 *
 * Copyright (C) 2001  John F. Gibson  
 *  
 * jgibson@mail.sjcsf.edu  
 * John F. Gibson 
 * St. John's College
 * 1160 Camino de la Cruz Blanca
 * Santa Fe, NM 87501
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, U
 */

#ifndef MATHDEFS_H
#define MATHDEFS_H

#include <stdlib.h>
#include <assert.h>
#include <math.h>
#include <iostream>
#include <complex>
#include <string>

using namespace std;
typedef double Real;
typedef std::complex<double> Complex;
enum fieldstate { Physical, Spectral };
enum parity { Even, Odd };

const int REAL_DIGITS = 17;
const int MATLABVERSION = 5;
const Real LOG2 = log (2.0);
const Complex I (0.0, 1.0);

#ifdef WIN32
const Real pi = 3.14159265358979323846;
#else
const Real pi = M_PI;
#endif

string i2s (int n);

void cferror (const string & message);
void cfpause();

inline int kronecker(int m, int n) {return (m==n)? 1 : 0;}

//inline Real abs(Real x) {return fabs(x);}
inline Real log2 (Real x) {
  return log (x) / LOG2;
}
inline Real square (Real x) {
  return x * x;
}
inline Real cube (Real x) {
  return x * x * x;
}
inline Real nr_sign (Real a, Real b) {
  return (b >= 0.0) ? fabs (a) : -fabs (a);
}

inline int square (int x) {
  return x * x;
}
inline int cube (int x) {
  return x * x * x;
}
inline void swap (int &a, int &b) {
  int tmp = a;
  a = b;
  b = tmp;
}
inline void swap (Real & a, Real & b) {
  Real tmp = a;
  a = b;
  b = tmp;
}

// Inline replacements for >? and <? GNUisms.
inline int Greater (int a, int b) {
  return (a > b) ? a : b;
}
inline int lesser (int a, int b) {
  return (a < b) ? a : b;
}
inline Real Greater (Real a, Real b) {
  return (a > b) ? a : b;
}
inline Real lesser (Real a, Real b) {
  return (a < b) ? a : b;
}

//Real pow (Real x, int n);
int pow (int n, int p);

Real pythag (Real a, Real b);	// sqrt(a^2 + b^2) avoiding over/underflow
bool isPowerOfTwo (int n);

void save (Real c, const string & filebase);
void load (Real & c, const string & filebase);
void save (Complex c, const string & filebase);
void load (Complex & c, const string & filebase);

void write (ostream & os, Complex z);
void write (ostream & os, Real x);
void write (ostream & os, int n);
void write (ostream & os, bool b);
void write (ostream & os, fieldstate s);

void read (istream & is, Complex & z);
void read (istream & is, Real & x);
void read (istream & is, int &n);
void read (istream & is, bool &b);
void read (istream & os, fieldstate & s);

inline Real Re (const Complex & z) {
  return z.real ();
}
inline Real Im (const Complex & z) {
  return z.imag ();
}
inline Complex conjugate (Complex z) {
  return Re (z) - I * Im (z);
}
inline Real norm2 (const Complex& z) {
  return square (Re (z)) + square (Im (z));
}

// The following functions are defined in complexdefs.h, but not here, in
// order to avoid name clashes with Octave.

//inline Real norm (const Complex& z)
//inline Real phase (const Complex& z)
//inline Complex exp (const Complex& z)
//inline Complex log (const Complex& z)

Real randomReal ();	    // uniform in [-1,1]
Complex randomComplex ();   // gaussian distribution about zero.

ostream & operator << (ostream & os, Complex z);

ostream & operator << (ostream & os, fieldstate f);
istream & operator >> (istream & is, fieldstate & f);

//#ifdef WIN32
//#undef assert
//#define assert(expr) (if(!expr) {cout << "Assertion failed. "<<endl; abort();})
//#endif

#endif				/* MATHDEFS_H */
