#include <iostream>
#include <fstream>
#include <sstream>
#include <iomanip>
#include "flowfield.h"
#include "cfvector.h"
#include "chebyshev.h"
#include "tausolver.h"
#include "nsintegrator.h"

// Start a pressure-driven Poisseuille flow with either 
// U(y) == 0      and u(x,y,z) == 1-y^2 or
// U(y) == 1-y^20 and u(x,y,z) == 0
// Integrate and verify that nothing changes
// 
// Or start U and u from zero and check convergence onto utot = 1-y^2 

int main() {

  const int Nx=8;
  const int Ny=33;
  const int Nz=8;

  const int Nd=3;
  const Real Lx=2*pi;
  const Real Lz=2*pi;
  const Real a= -1.0;
  const Real b=  1.0;
  const Real Ly= b-a;
  const Real Reynolds = 400.0;
  const Real nu = 1.0/Reynolds;
  const Real dPdx = -2.0*nu;    // mean streamwise pressure gradient.
  const Real dt = 0.1;
  const int  n = 10;
  const Real T0 = 0.0;
  const Real T1 = 100.0;

  // utot = Ubase + un
  const bool baseflow = false;  // true => Ubase==parabola, false => Ubase=0

  DNSFlags flags;
  flags.constraint = BulkVelocity;
  flags.dealiasing = DealiasXZ;
  flags.nonlinearity = Rotational;

  // sinusoid perturbation params.
  const Real eps = 0.10;
  const int ky = 1;

  char s = ' ';
  cout << "Nx  Ny  Nz == " << Nx << s << Ny << s << Nz << endl;

  cout << setprecision(14);
  cout << "Nx Ny Nz Nd == " << Nx << ' ' << Ny << ' ' << Nz << ' ' << Nd << endl;
  cout << "Lx Ly Lz == " << Lx << " 2 "  << Lz << ' ' << endl;

  Vector y = chebypoints(Ny,a,b);
  y.save("y");
  ChebyTransform trans(Ny);
  
  // Build DNS
  ChebyCoeff sinusoid(Ny,a,b,Physical);
  ChebyCoeff parabola(Ny,a,b,Physical);
  ChebyCoeff parabola2(Ny,a,b,Physical);
  ChebyCoeff zero(Ny,a,b,Physical);
  for (int ny=0; ny<Ny; ++ny) {
    sinusoid[ny] = sin(2*pi*ky*y[ny]/Ly);
    parabola[ny] = 1.0 - square(y[ny]);
    parabola2[ny] = 0.5*(1.0 - square(y[ny]));
  }
  trans.makeSpectral(sinusoid);
  trans.makeSpectral(parabola);
  trans.makeSpectral(parabola2);
  trans.makeSpectral(zero);

  FlowField un(Nx,Ny,Nz,Nd,Lx,Lz,a,b);

  ChebyCoeff Ubase = (baseflow) ? parabola : zero;
  ChebyCoeff uprof = (baseflow) ? zero : parabola;
  un.addProfile(uprof);

  FlowField ut = un;
  FlowField Pn(Nx,Ny,Nz,1,Lx,Lz,a,b);
  FlowField vortn(Nx,Ny,Nz,Nd,Lx,Lz,a,b);
  FlowField fn1(Nx,Ny,Nz,Nd,Lx,Lz,a,b);
  FlowField fn(Nx,Ny,Nz,Nd,Lx,Lz,a,b);
  
  cout << "building DNS..." << flush;
  NSIntegrator dns(un, Ubase, nu, dt, flags, T0);
  dns.resetdPdx(dPdx);
  cout << "done" << endl;

  cout << "========================================================" << endl;
  cout << "Initial data, prior to time stepping" << endl;
  cout << "L2Norm(un)    == " << L2Norm(un) << endl;
  cout << "divNorm(un)   == " << divNorm(un) << endl;  
  cout << "L2Norm(Pn)    == " << L2Norm(Pn) << endl;
  cout << "........................................................" << endl;
  cout << endl;

  cout << "======================================================" << endl;
  cout << "Starting time stepping..." << endl;

  trans.makePhysical(Ubase);
  Ubase.save("Ubase");

  for (Real t=T0; t<T1; t += n*dt) {
    cout << "================================================" << endl;
    cout << "t == " << t << endl;
    cout << "CFL == " << dns.CFL() << endl;
    cout << "L2Norm(un)    == " << L2Norm(un) << endl;
    cout << "L2Dist(un,ut) == " << L2Dist(un,ut) << endl;
    cout << "dPdx  == " << dns.dPdx() << endl;
    cout << "Ubulk == " << dns.Ubulk() << endl;

    un.saveProfile(0,0, "un", trans);

    dns.advance(un, Pn, n);
  }
}

