/* mathdefs.cpp: some small mathematical conveniences.
 * Channelflow-0.9
 *
 * Copyright (C) 2001  John F. Gibson  
 *  
 * jgibson@mail.sjcsf.edu  
 * John F. Gibson 
 * St. John's College
 * 1160 Camino de la Cruz Blanca
 * Santa Fe, NM 87501
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, U
 */

#include <stdlib.h>
#include <stdio.h>
#include <fstream>
#include <iomanip>
#include "mathdefs.h"

string i2s (int n) {
  const int Nbuf = 32;
  char buff[Nbuf];
  if (n >= 0)
    sprintf (buff, "%d", n);
  else {
    n = abs (n);
    sprintf (buff, "_%d", n);
  }
  return string (buff);
}

void cferror (const string & message) {
  cerr << message << endl;
  exit (-1);
}

void cfpause() {
  cout << "cfpause..." << flush;
  char s[10];
  cin.getline(s,10);
}

Real pythag (Real a, Real b) {
  Real absa = fabs (a);
  Real absb = fabs (b);

  if (absa > absb)
    return absa * (sqrt (1.0 + square (absb / absa)));
  else
    return (absb ==
	    0.0) ? 0.0 : absb * (sqrt (1.0 + square (absa / absb)));
}

bool isPowerOfTwo (int N) {
  int onbits = 0;
  for (unsigned int i = 0; i < int (8 * sizeof (int)); ++i) {
    onbits += N % 2;
    N = N >> 1;
    if (onbits > 1)
      return false;
  }
  return true;
}

/****************
Real pow (Real x, int n) {
  if (n < 0) {
    x = 1.0 / x;
    n = -n;
  }
  switch (n) {
  case 0:
    return 1;
  case 1:
    return x;
  case 2:
    return x * x;
  case 3:
    return x * x * x;
  case 4:{
    Real xx = x * x;
    return xx * xx;
  }
  case 5:{
    Real xx = x * x;
    return x * xx * xx;
  }
  case 6:{
    Real xx = x * x;
    return xx * xx * xx;
  }
  case 7:{
    Real xxx = x * x * x;
    return xxx * xxx * x;
  }
  case 8:{
    Real xx = x * x;
    Real xxxx = xx * xx;
    return xxxx * xxxx;
  }
  default:{
    int m = n / 2;
    return pow (x, m) * pow (x, n - m);	// I can't remember the while() algorithm.
  }
  }
}
***************************************/
int pow (int x, int n) {
  if (n < 0)
    cferror
      ("pow(int, int) : can't do negative exponents, use pow(Real,int)");
  switch (n) {
  case 0:
    return 1;
  case 1:
    return x;
  case 2:
    return x * x;
  case 3:
    return x * x * x;
  case 4:{
    int xx = x * x;
    return xx * xx;
  }
  case 5:{
    int xx = x * x;
    return x * xx * xx;
  }
  case 6:{
    int xx = x * x;
    return xx * xx * xx;
  }
  case 7:{
    int xxx = x * x * x;
    return xxx * xxx * x;
  }
  case 8:{
    int xx = x * x;
    int xxxx = xx * xx;
    return xxxx * xxxx;
  }
  default:{
    if ((n % 2) == 0)
      return square(pow(x,n/2));
    else 
      return x*pow(x,n-1);
  }
  }
}

void save (Complex c, const string & filebase) {
  string extension (".cmplx");
  string filename = filebase + extension;
  ofstream os (filename.c_str ());
  os << setprecision (17);
  os << Re (c) << ' ' << Im (c) << '\n';	// format can be read by matlab.
  os.close ();
}

void load (Complex & c, const string & filebase) {
  string filename = filebase + string (".cmplx");
  ifstream is (filename.c_str ());
  Real r, i;
  is >> r >> i;
  c = r + I * i;
}

void save (Real c, const string & filebase) {
  string extension (".real");
  string filename = filebase + extension;
  ofstream os (filename.c_str ());
  os << setprecision (17);
  os << c << '\n';	// format can be read by matlab.
  os.close ();
}

void load (Real & c, const string & filebase) {
  string filename = filebase + string (".real");
  ifstream is (filename.c_str ());
  Real r;
  is >> r;
  c = r;
}

void write (ostream & os, int n) {
  const char *s = (const char *) &n;
  for (unsigned int i = 0; i < sizeof (int); ++i)
    os.put(s[i]);
}

void read (istream & is, int &n) {
  char *s = (char *) &n;
  for (unsigned int i = 0; i < sizeof (int); ++i)
    is.get(s[i]);
}

void write (ostream & os, bool b) {
  (b) ? os.put('1') : os.put('0');
}

void read (istream & is, bool &b) {
  char s;
  is.get(s);
  b = (s=='1') ? true : false;
}

void write (ostream & os, fieldstate s) {
  os << s;
}

void read (istream & is, fieldstate & s) {
  is >> s;
}

void write (ostream & os, Real x) {
  char *s = (char *) &x;
  for (unsigned int i = 0; i < sizeof (Real); ++i)
    os.put (s[i]);
}

void read (istream & is, Real & x) {
  char *s = (char *) &x;
  for (unsigned int i = 0; i < sizeof (Real); ++i)
    is.get (s[i]);
}

void write (ostream & os, Complex z) {
  Real x = Re (z);
  write (os, x);
  x = Im (z);
  write (os, x);
}

void read (istream & is, Complex & z) {
  Real r, i;
  read (is, r);
  read (is, i);
  z = Complex (r, i);
}

Real randomReal () {
#ifdef WIN32
  return Real (rand ()) / 1073741823.5 - 1.0;
#else
  return 2 * drand48 () - 1.0;
#endif
}

Complex randomComplex () {
  Real a;
  Real b;
  Real r2;
  do {
    a = randomReal ();
    b = randomReal ();
    r2 = a * a + b * b;
  } while (r2 >= 1.0 || r2 == 0.0);
  return sqrt (-log (r2) / r2) * (a+b*I);
}

ostream & operator << (ostream & os, Complex z) {
  os << '(' << Re (z) << ", " << Im (z) << ')';
  return os;
}

ostream & operator << (ostream & os, fieldstate f) {
  switch (f) {
  case Physical:
    os << 'P';
    break;
  default:
  case Spectral:
    os << 'S';
    break;
  }
  return os;
}

istream & operator >> (istream & is, fieldstate & f) {
  char c = ' ';
  while (c == ' ')
    is >> c;
  switch (c) {
  case 'P':
    f = Physical;
    break;
  case 'S':
    f = Spectral;
    break;
  default:
    cerr << "read fieldstate error: unknown fieldstate " << c <<
      endl;
    f = Spectral;
    assert (false);
  }
  return is;
}

void alwaysAssert_fail (const char *__assertion, const char *__file,
			unsigned int __line, const char *__function) {
  cerr << "Assertion failed: " << __assertion << " in function " <<
    __function << endl;
  cerr << "file " << __file << " line " << __line << endl;
  abort ();
}

void assertfail () {
  cout << "Assertion failed. " << endl;
}
