/* flowfield.h: Class for N-dim Fourier x Chebyshev x Fourier expansions
 * Channelflow-0.9
 *
 * Copyright (C) 2001  John F. Gibson  
 *  
 * jgibson@mail.sjcsf.edu  
 * John F. Gibson 
 * St. John's College
 * 1160 Camino de la Cruz Blanca
 * Santa Fe, NM 87501
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, U
 */

#ifndef FLOWFIELD_H
#define FLOWFIELD_H

#ifdef WIN32
#include <rfftw.h>
#else
#include <drfftw.h>
#endif
#include "mathdefs.h"
#include "cfvector.h"
#include "chebyshev.h"
#include "basisfunc.h"

// Field is a class for storing big arrays of multidimensional
// data. It's hard-coded for vector fields in three dimensions. 
// I carved out such a huge chunk of class namespace because I
// don't expect this code to be used outside of fluids simulations.

class FlowField {
  
public:
  FlowField();
  FlowField(int Nx, int Ny, int Nz, int Nd, Real Lx, Real Lz, Real a, Real b,
	    fieldstate xzstate=Spectral, fieldstate ystate=Spectral, 
	    int fftw_flags = FFTW_ESTIMATE | FFTW_USE_WISDOM);

  FlowField(const FlowField& u);
  FlowField(const string& filename); // distngshs ascii and binary by extension
  ~FlowField();

  FlowField& operator = (const FlowField& u); // assign identical copy of U
  
  void resize(int Nx, int Ny, int Nz, int Nd, Real Lx, Real Lz, Real a, Real b,
	      int fftw_flags = FFTW_ESTIMATE | FFTW_USE_WISDOM);

  void rescale(Real Lx, Real Lz);
  void interpolate(const FlowField& u); // interpolate U onto this grid.
  void reflect(const FlowField& u);     // reflect U symmetrically about U.a()
  void optimizeFFTW();

  // Real    operator()  (nx,ny,nz,i):  0<=nx<Nx, 0<=ny<Ny, 0<=nz<Nz
  // Complex cmplx(nx,ny,nz,i):  0<=nx<Nx, 0<=ny<Ny, 0<=nz2<=Nz/2
  inline Real&        operator()(int nx, int ny, int nz, int i);
  inline const Real&  operator()(int nx, int ny, int nz, int i) const;
  inline Complex&       cmplx(int nx, int ny, int nz2, int i);
  inline const Complex& cmplx(int nx, int ny, int nz2, int i) const;
  inline Complex&       coeff(int nx, int ny, int nz2, int i);
  inline const Complex& coeff(int nx, int ny, int nz2, int i) const;

  
  //Real eval(Real x, Real y, Real z, int d) const;     
  //Complex eval(int nx, Real y, Real nz, int d) const; 

  void addProfile(const ChebyCoeff& profile, int i=0); // assume kx=kz=0
  void addProfile(const ComplexChebyCoeff& profile, int kx, int kz, int i=0, bool addconj=false);
  void addProfile(const BasisFunc& profile, bool addconj=false);
  ComplexChebyCoeff profile(int kx, int kz, int i) const;
  BasisFunc profile(int kx, int kz) const;

  // Destructive transforms.
  void realfft_xz();
  void irealfft_xz();
  void chebyfft_y();
  void ichebyfft_y();

  void makeSpectral_xz();
  void makePhysical_xz();
  void makeSpectral_y();
  void makePhysical_y();
  void makeSpectral();
  void makePhysical();
  void makeState(fieldstate xzstate, fieldstate ystate);

  void setToZero(); 
  void addPerturbation(int kx, int kz, Real mag, Real spectralDecay);
  void addPerturbations(int kxmax, int kzmax, Real mag, Real spectralDecay);
  void addPerturbations(Real magnitude, Real spectralDecay);
  void addPoisseuille(Real Ucenter);

  inline int numXmodes() const; 
  inline int numYmodes() const; 
  inline int numZmodes() const; 
  inline int numXgridpts() const; 
  inline int numYgridpts() const; 
  inline int numZgridpts() const; 
  inline int vectorDim() const; 
  inline int Nx() const; // same as numXgridpts()
  inline int Ny() const; // same as numYgridpts()
  inline int Nz() const; // same as numZgridpts()
  inline int Nd() const; 
  inline int nx(int kx) const;  // where, in the array, is kx?
  inline int nz(int kz) const;
  inline int kx(int nx) const;  // the wavenumber of the nxth array elem
  inline int kz(int nz) const;
  inline int kxmax() const;     // the largest value kx takes on
  inline int kzmax() const;     
  inline int kxmin() const;     // the smallest value kx takes on
  inline int kzmin() const;     
  inline int kxmaxDealiased() const; // kx > kxmaxDealiased is aliased mode
  inline int kzmaxDealiased() const; 
  inline bool isAliased(int kx, int kz) const;
  

  inline Real x(int nx) const;  // the x coord of the nxth gridpoint
  inline Real y(int ny) const;
  inline Real z(int nz) const;
  inline Real Lx() const;
  inline Real Ly() const;
  inline Real Lz() const;
  inline Real a() const;
  inline Real b() const;

  FlowField& operator *= (Real x);
  FlowField& operator *= (Complex x);
  FlowField& operator += (const FlowField& u);
  FlowField& operator -= (const FlowField& u);
  FlowField& operator *= (const FlowField& u);
  FlowField& operator += (const BasisFunc& profile);
  FlowField& operator -= (const BasisFunc& profile);

  bool congruent(const BasisFunc& phi) const;
  bool congruent(const FlowField& v) const;
  friend void swap(FlowField& u, FlowField& v); // exchange data of two congruent fields.

  // save methods add extension .aff or .bff ("ascii/binary flow field")
  void asciiSave(const string& filebase) const; 
  void binarySave(const string& filebase) const;

  void saveSlice(int ny, int i, const string& filebase) const;
  void saveSliceXY(int nz, int i, const string& filebase) const;
  void saveCrossSection(int nx, int i, const string& filebase) const;
  void saveProfile(int nx, int nz, const string& filebase) const;
  void saveProfile(int nx, int nz, const string& filebase,
		   const ChebyTransform& t) const; 
  
  void saveSpectrum(const string& filebase, int i, int ny=-1) const; 
  void saveSpectrum(const string& filebase) const; 
  void saveDissSpectrum(const string& filebase, Real nu=1.0) const; 
  void saveDivSpectrum(const string& filebase) const; 
  void print() const;
  void dump() const; 

  Real energy(bool normalize=true) const; 
  Real energy(int kx, int kz, bool normalize=true) const; 
  Real dissipation(int kx, int kz, Real nu=1.0) const; // Int u lapl(u) dy, 
  Real dudy_a() const;  
  Real dudy_b() const;
  Real CFLfactor() const;

  //inline Real& rawData(int n);
  //inline const Real& rawData(int n) const;
  //inline int rawDataLength() const;

  void setState(fieldstate xz, fieldstate y);
  void assertState(fieldstate xz, fieldstate y) const;

  inline fieldstate xzstate() const;
  inline fieldstate ystate() const;

  Real bcnorm() const;   // sqrt(bcnorm2)
  Real bcnorm2() const;  // 1/2hLxLz integral L2Norm2(u) over boundary
  Real divnorm() const;  // sqrt(divnorm2)
  Real divnorm2() const; // 1/2hLxLz integral L2Norm2(div(u)) over domain 
  Real divergence() const; // same as divnorm2

  void divergence(FlowField& div) const;
  void divComps(FlowField& div) const;
  void laplacian(FlowField& lap) const;
  void gradient(FlowField& grad) const;
  //void vorticity(FlowField& omega, const ChebyCoeff& Ubasey) const;
  //void vorticity2(FlowField& omega, const ChebyCoeff& Ubasey) const; 
  void vorticity2(FlowField& omega) const; // del x u, where u == *this
  void normsquared(FlowField& u2) const; 

  // f = omega x u + Omega x u + omega x U  
  // CFLfactor = max(|u|/dx + |v|/dy + |w|/dz).
  // nonlinearity() is logically const, but it does internally transform u to 
  // physical and back. 
  void nonlinearityRotational(const ChebyCoeff& Ubase, 
			      const ChebyCoeff& Ubasey,
			      const ChebyCoeff& UbaseyT, 
			      FlowField& omega_tot, 
			      FlowField& f) const;

  // f = (omega + Omega) x (u + U). 
  // CFLfactor = max(|u|/dx + |v|/dy + |w|/dz).
  // nonlinearity() is logically const, but it does internally transform u to 
  // physical and back. 
  void nonlinearityRotational2(const ChebyCoeff& Ubase, 
			       const ChebyCoeff& Ubasey,
			       const ChebyCoeff& UbaseyT, 
			       FlowField& omega_tot, 
			       FlowField& f) const;

  // Calculate vorticity omega = del x u and NS eqn nonlinearity f.
  // f = (omega + Omega) x (u + U). The omega returned is omega + Omega.
  // nonlinearity() is logically const, but it does internally transform u to 
  // physical and back. CFLfactor = max(|u|/dx + |v|/dy + |w|/dz).
  void nonlinearityDealiased(const ChebyCoeff& Ubase, 
			     const ChebyCoeff& Ubasey,
			     const ChebyCoeff& UbaseyT, 
			     FlowField& omega_tot, 
			     FlowField& f) const;

  // Nonlinearity linearized about base flow. Returns
  // f = omega x U + Omega x u + Omega x U. Returned omega is omega + Omega.
  void nonlinearityLinearized(const ChebyCoeff& Ubase, 
			      const ChebyCoeff& Ubasey,
			      const ChebyCoeff& UbaseyT, 
			      FlowField& omega_tot, 
			      FlowField& f) const;

  // Nonlinearity linearized about base flow, and delaliased. Returns
  // f = omega x U + Omega x u + Omega x U. Returned omega is omega + Omega.
  void nonlinearityLinearizedDealiased(const ChebyCoeff& Ubase, 
				       const ChebyCoeff& Ubasey,
				       const ChebyCoeff& UbaseyT, 
				       FlowField& omega_tot, 
				       FlowField& f) const;

  // Computes 
  // f =  1/2 [u grad u + U du/dx + v Uy e_x + div ((u + U e_x)(u + U e_x))]
  void nonlinearitySkewSymmetric(const ChebyCoeff& Ubase, 
				 const ChebyCoeff& Ubasey,
				 const ChebyCoeff& UbaseyT,
				 FlowField& tmp, 
				 FlowField& f) const;
  

  // Computes 
  // f =  1/2 div ((u + U e_x)(u + U e_x))
  void nonlinearityDivergence(const ChebyCoeff& Ubase, 
			      const ChebyCoeff& Ubasey,
			      const ChebyCoeff& UbaseyT,
			      FlowField& tmp, 
			      FlowField& f) const;

  // Computes 
  // f =  1/2 [u grad u + U du/dx + v Uy e_x]
  void nonlinearityConvection(const ChebyCoeff& Ubase, 
			      const ChebyCoeff& Ubasey,
			      const ChebyCoeff& UbaseyT,
			      FlowField& tmp, 
			      FlowField& f) const;
  

  void dealiasIO(bool b); // if true, don't write aliased modes in IO

private:
  int Nx_;     // number of X gridpoints and modes
  int Ny_;     // number of Y gridpoints and modes
  int Nz_;     // number of Z gridpoints
  int Nzpad_;  // 2*(Nz/2+1)
  int Nzpad2_; // number of Z modes == Nz/2+1.
  int Nd_;

  Real Lx_;
  Real Lz_;
  Real a_;
  Real b_;

  bool dealiasIO_; // if true, binary IO leaves out aliased modes
  
  Real* rdata_;    // stored with indices in order d, Ny, Nx, Nz
  Complex* cdata_; // Complex alias for rdata_ (cdata_ = (Complex*) rdata_).

  Real* q_;         // scratch space for y transforms. Q = realfftw(q)
  fftw_complex* Q_; // q and Q are meaningless names for temp variables.
  Real* scratch_;   // scratch space for y transforms.

  Real* sin_table_; // sin_table[j] = sin(j*pi/(Ny-1));
  Real* cos_table_; // cos_table[j] = cos(j*pi/(Ny-1));

  fieldstate xzstate_;
  fieldstate ystate_;
  
  rfftwnd_plan xz_plan_;
  rfftwnd_plan xz_iplan_;
  rfftwnd_plan   y_plan_;
  rfftwnd_plan  y_iplan_;
  
  inline int flatten(int nx, int ny, int nz, int i) const;
  inline int complex_flatten(int nx, int ny, int nz2, int i) const;
  void fftw_initialize(int fftw_flags = FFTW_ESTIMATE | FFTW_USE_WISDOM );
};

void assignOrrSommField(FlowField& u, FlowField& P, 
			Real t, Real Reynolds, Complex omega, 
			const ComplexChebyCoeff& ueig, 
			const ComplexChebyCoeff& veig,
			const ComplexChebyCoeff& peig);

Real BoundaryL1Norm(const FlowField& u);
Real L1Norm(const FlowField& u);
Real L1Dist(const FlowField& u, const FlowField& v);
Real L2Norm2(const FlowField& u, bool normalize=true);
Real L2Dist2(const FlowField& u, const FlowField& v, bool normalize=true);
Real L2Norm(const FlowField& u, bool normalize=true);
Real L2Dist(const FlowField& u, const FlowField& v, bool normalize=true);

// Restrict sums in norm computation to |kx|<=kxmax, |kz|<=kzmax, 
Real L2Norm2(const FlowField& u, int kxmax, int kzmax, bool normalize=true);
Real L2Dist2(const FlowField& u, const FlowField& v, int kxmax, int kzmax, bool normalize=true);
Real L2Norm(const FlowField& u, int kxmax, int kzmax, bool normalize=true);
Real L2Dist(const FlowField& u, const FlowField& v, int kxmax, int kzmax, bool normalize=true);
Real BCNorm(const FlowField& u);

Vector vectorL1Dist(const FlowField& u, const FlowField& v);
Complex innerProduct(const FlowField& u, const FlowField& v);
Complex L2InnerProduct(const FlowField& u, const BasisFunc& phi0, 
		       bool normalize=false);

void swap(FlowField& u, FlowField& v); // exchange data of two flow fields.


inline int FlowField::flatten(int nx, int ny, int nz, int i) const {
  assert(nx>=0 && nx<Nx_);    assert(ny>=0 && ny<Ny_); 
  assert(nz>=0 && nz<Nzpad_); assert(i>=0  && i<Nd_);
  return nz + Nzpad_*(nx + Nx_*(ny + Ny_*i));
}

inline int FlowField::complex_flatten(int nx, int ny, int nz, int i) const {
  assert(nx>=0 && nx<Nx_);     assert(ny>=0 && ny<Ny_); 
  assert(nz>=0 && nz<Nzpad2_); assert(i>=0  && i<Nd_);
  return (nz + Nzpad2_*(nx + Nx_*(ny + Ny_*i)));
}

inline const Real& FlowField::operator()(int nx, int ny, int nz, int i) const {
  assert(xzstate_==Physical);
  return rdata_[flatten(nx,ny,nz,i)];
}
inline Real& FlowField::operator()(int nx, int ny, int nz, int i) {
  assert(xzstate_==Physical);
  return rdata_[flatten(nx,ny,nz,i)];
}
inline Complex& FlowField::cmplx(int nx, int ny, int nz2, int i) {
  assert(xzstate_==Spectral);
  return cdata_[complex_flatten(nx,ny,nz2,i)];
}
inline const Complex& FlowField::cmplx(int nx, int ny, int nz2, int i) const{
  assert(xzstate_==Spectral);
  return cdata_[complex_flatten(nx,ny,nz2,i)];
}
inline Complex& FlowField::coeff(int nx, int ny, int nz2, int i) {
  assert(xzstate_==Spectral);
  return cdata_[complex_flatten(nx,ny,nz2,i)];
}
inline const Complex& FlowField::coeff(int nx, int ny, int nz2, int i) const{
  assert(xzstate_==Spectral);
  return cdata_[complex_flatten(nx,ny,nz2,i)];
}
  
//inline Real& FlowField::rawData(int n) {return rdata_[n];}
//inline const Real& FlowField::rawData(int n) const{return rdata_[n];}
//inline int FlowField::rawDataLength() const {return Nx_*Ny_*Nz_*Nd_;}

inline Real FlowField::Lx() const {return Lx_;}
inline Real FlowField::Ly() const {return b_ - a_;}
inline Real FlowField::Lz() const {return Lz_;} 
inline Real FlowField::a() const {return a_;}
inline Real FlowField::b() const {return b_;}

inline int FlowField::kx(int nx) const { 
  //assert(xzstate_ == Spectral);
  assert(nx >= 0 && nx <Nx_);
  return (nx <= Nx_/2) ? nx : nx - Nx_;
}

inline int FlowField::kz(int nz2) const { 
  //assert(zstate_ == Spectral);
  assert(nz2>= 0 && nz2 <Nzpad2_);
  return nz2;
}

inline int FlowField::nx(int kx) const { 
  //assert(xzstate_ == Spectral);
  assert(kx >= 1-Nx_/2 && kx <= Nx_/2);
  return (kx >= 0) ? kx : kx + Nx_;
}

inline int FlowField::nz(int kz) const { 
  //assert(zstate_ == Spectral);
  assert(kz>= 0 && kz <Nzpad2_);
  return kz;
}

inline Real FlowField::x(int nx) const {
  //assert(xzstate_ == Physical);
  return nx*Lx_/Nx_;
}
inline Real FlowField::y(int ny) const {
  //assert(ystate_ == Physical);
  return cos(pi*ny/(Ny_-1));
}

inline Real FlowField::z(int nz) const {
  //assert(zstate_ == Physical);
  return nz*Lz_/Nz_;
}

inline int FlowField::numXmodes() const {return Nx_;} 
inline int FlowField::numYmodes() const {return Ny_; }
inline int FlowField::numZmodes() const {return Nzpad2_;}

inline int FlowField::numXgridpts() const {return Nx_;} 
inline int FlowField::numYgridpts() const {return Ny_; }
inline int FlowField::numZgridpts() const {return Nz_;}

inline int FlowField::Nx() const {return Nx_;}
inline int FlowField::Ny() const {return Ny_;}
inline int FlowField::Nz() const {return Nz_;} 

inline int FlowField::kxmax() const {return Nx_/2;}
inline int FlowField::kzmax() const {return Nz_/2;}
inline int FlowField::kxmin() const {return Nx_/2+1-Nx_;}
inline int FlowField::kzmin() const {return 0;}
inline int FlowField::kxmaxDealiased() const {return Nx_/3-1;}  // not Nx_/3 
inline int FlowField::kzmaxDealiased() const {return Nz_/3-1;}  // not Nz_/3
inline bool FlowField::isAliased(int kx, int kz) const {
  return (abs(kx) > kxmaxDealiased() || kz > kzmaxDealiased()) ? true : false;
}
inline int FlowField::vectorDim() const {return Nd_;} 
inline int FlowField::Nd() const {return Nd_;} 

inline fieldstate FlowField::xzstate() const {return xzstate_;}
inline fieldstate FlowField::ystate() const {return ystate_;}

void randomUprofile(ComplexChebyCoeff& u, Real mag, Real spectralDecay);
void randomVprofile(ComplexChebyCoeff& v, Real mag, Real spectralDecay);
void randomProfile(ComplexChebyCoeff& u, ComplexChebyCoeff& v, ComplexChebyCoeff& w, 
		   int kx, int kz, Real Lx, Real Lz, Real mag, Real spectralDecay);

void chebyUprofile(ComplexChebyCoeff& u, int n, Real decay);
void chebyVprofile(ComplexChebyCoeff& v, int n, Real decay);
void chebyProfile(ComplexChebyCoeff& u, ComplexChebyCoeff& v, ComplexChebyCoeff& w, 
		   int un, int vn, int kx, int kz, Real Lx, Real Lz, Real decay);

#endif
