#include <iostream>
#include <fstream>
#include <sstream>
#include <iomanip>
#include "flowfield.h"
#include "cfvector.h"
#include "chebyshev.h"
#include "tausolver.h"
#include "nsintegrator.h"
#include "complexdefs.h"

// This program compares the growth of OrrSommerfeld eigenfuncs integrated 
// with three methods:
// (1) direct numerical simulation (fully 3d with nonlinearity)
// (2) analytic expression exp(-i omega t) * OS eigenfunction 
// (3) an ODE integration of the linearized equations

// The DNS is represented by  FlowField un (u nonlinear) 
// The analytic expression by FlowField ul (u linear)
// The ODE is represented by a Complex ark (a runge-kutta)
// Admittedly the names of these variables are poorly chosen.

// To compare the DNS and analytic expression to the ODE integration,
// we compute the inner product of un and ul with the OS eigenfunction:
// Complex an = L2InnerProduct(un, u_oseig);
// Complex al = L2InnerProduct(ul, u_oseig);
// If the three integration methods are identical, an == al == ark.

const Real EPSILON=1e-12;
void RK3update(Complex& an, Complex lambda_v, Complex lambda_b, Real dt, 
		int nsteps);

ComplexChebyCoeff Ptotal(const ComplexChebyCoeff& p, const BasisFunc& u);



int main() {
  fftw_loadwisdom();

  const int Nx=16;
  const int Ny=65;
  const int Nz=16;
  const int Nd=3;
  const int Npod=128;
  const int kxmask=0;
  const int kzmask=0;
  const Real Lx=2*pi;
  const Real Lz=8*pi;
  const Real a= -1.0;
  const Real b=  1.0;
  const Real Reynolds = 7500.0;
  const Real nu = 1.0/Reynolds;
  const Real dPdx = -2.0*nu;    // mean streamwise pressure gradient.
  const Real dt = 0.1;
  const Real T = 100;
  const int plotmod = 1;
  const bool pause = true;
  const int nSteps = int(T/dt);
  DNSFlags flags;
  Real scale =  0.01;
  string outfileNum = string("osNum") + i2s(Ny);
  string outfileLin = string("osLin") + i2s(Ny);
  const char s = ' ';
  
  cout << setprecision(17);
  cout << "Nx Ny Nz Nd == " << Nx << s << Ny << s << Nz << s << Nd << endl;
  cout << "Lx Ly Lz == " << Lx << " 2 "  << Lz << s << endl;

  Vector y = chebypoints(Ny,a,b);
  y.save("y");

  // Get Orr-Sommerfeld eigenfunction from file. The data is the 
  // y-dependence of the eigenfunction's (u,v) components. x dependence
  // is exp(2 pi i x / Lx). Eigfunction is const in z. Reconstruct
  // w from velocity field, pressure from OS eqns.
  ChebyTransform trans(Ny);

  BasisFunc u_oseig("oseig10_65");
  ComplexChebyCoeff u_peig("ospeig10_65");
  u_peig.makeSpectral(trans);
  u_oseig.makeSpectral(trans);

  u_peig  *=  1/L2Norm(u_oseig);
  u_oseig *=  1/L2Norm(u_oseig);

  Complex omega = 0.24989153647208251 + I*0.0022349757548207664;
  Complex lambda = -1.0*I*omega;
  Complex lambda_v(-0.006564724662891, 0.0);
  Complex lambda_b = lambda - lambda_v; // ( 0.008799700331459, -0.2498915365425);
  cout << "   omega = " << omega << endl;
  cout << "i lambda = " << I*lambda << endl;
  cout << "  lambda = " << lambda << endl;
  cout << "-i omega = " << -1.0*I*omega << endl;

  ChebyCoeff Ubase(Ny,a,b,Physical);
  for (int ny=0; ny<Ny; ++ny) 
    Ubase[ny] = 1.0 - square(y[ny]);

  FlowField un(Nx,Ny,Nz,Nd,Lx,Lz,a,b);
  FlowField ul(Nx,Ny,Nz,Nd,Lx,Lz,a,b);
  FlowField pn(Nx,Ny,Nz,1,Lx,Lz,a,b);
  FlowField pl(Nx,Ny,Nz,1,Lx,Lz,a,b);
  
  // Clauclate L2Norm of poisseuille flow.
  un.setState(Spectral, Physical);
  for (int ny=0; ny<Ny; ++ny) 
    un.cmplx(0, ny, 0, 0) = Complex(1- square(y[ny]));
  un.chebyfft_y();
  Real uPoissNorm = L2Norm(un);
  un.setState(Spectral, Spectral);
  un.setToZero();

  // Set t = 0, time step 0. Assign un, vortn, fn, and t counterparts
  BasisFunc s_oseig(u_oseig);
  ComplexChebyCoeff s_peig(u_peig);
  un.addProfile(s_oseig, true);
  scale *= uPoissNorm/L2Norm(un);
  s_oseig *= scale;
  s_peig *= scale;
  
  cout << "building DNS..." << flush;

  // Set DNS with un(-dt) for correct initialization of nonlinear term fn1.
  un.setToZero();
  s_oseig *= exp(lambda*(-dt));
  un.addProfile(s_oseig, true);
  s_oseig *= exp(lambda*dt);
  NSIntegrator dns(un, Ubase, nu, dt, flags);
  cout << "done" << endl;
 
  un.setToZero();
  ul.setToZero();
  pn.setToZero();
  pl.setToZero();
  un.addProfile(s_oseig, true);
  ul.addProfile(s_oseig, true);
  pn.addProfile(s_peig,1,0,0, true);
  pl.addProfile(s_peig,1,0,0, true);

  Real u0norm = L2Norm(un);

  // =================================================================
  // Get everybody ready for timestepping
  cout << "========================================================" << endl;
  cout << "Initial data, prior to time stepping" << endl;
  cout << "L2Norm(u_oseig)        == " << L2Norm(u_oseig) << endl;
  cout << "L2Norm(ul)             == " << L2Norm(ul) << endl;
  cout << "L2Norm(un)             == " << L2Norm(un) << endl;
  cout << "L2Norm(pn)             == " << L2Norm(pn) << endl;
  cout << "L2Dist(un,ul)/L2Norm(ul) == " << L2Dist(un, ul)/L2Norm(ul) <<endl;
  cout << "L2Dist(pn,pl)/L2Norm(pl) == " << L2Dist(pn,pl)/L2Norm(pl) << endl;

  Complex L2IP0 = L2InnerProduct(un, u_oseig);
  Complex ark = L2InnerProduct(ul, u_oseig);

  for (int step=0; step<nSteps; step += plotmod) {
    Real t = dt*step;
    cout << "================================================" << endl;
    cout << "step " << step << "  t == " << t << endl;
    Real unnorm = L2Norm(un);
    Real ulnorm = L2Norm(ul);
    Real plnorm = L2Norm(pl);

    Real CFL = dns.CFL();
    if (CFL > 2.0 || unnorm > 10) {
      cerr << "DNS trouble: CFL==" << CFL << " L2Norm(un)==" << unnorm << endl;
      exit(1);
    }

    for (int kx=-2; kx<=2; ++kx) {
      for (int kz=-2; kz<=2; ++kz) {
      ul.saveProfile(ul.nx(kx), ul.nz(kz), string("ul")+i2s(kx)+i2s(kz), trans);
      un.saveProfile(un.nx(kx), ul.nz(kz), string("un")+i2s(kx)+i2s(kz), trans);
      
      //pl.saveProfile(pl.nx(kx), pl.nz(kz), string("pl")+i2s(kx)+i2s(kz), trans);
      //pn.saveProfile(pn.nx(kx), pn.nz(kz), string("pn")+i2s(kx)+i2s(kz), trans);

      // Extract the 0-component of kx,kz pressure profile, transform, and save
      ComplexChebyCoeff pn_profile = pn.profile(pn.nx(kx), pn.nz(kz), 0);
      pn_profile.makePhysical(trans);
      pn_profile.save(string("pn")+i2s(kx)+i2s(kz));
      }
    }
    cout << "CFL == " << CFL << endl;
    cout << "div(un)/L2Norm(un) == " << un.divergence()/L2Norm(un) << endl;
    cout << "L2Norm(un)/L2Norm(uPoiss) == " << unnorm/uPoissNorm << endl;
    cout << "L2Norm(ul)/L2Norm(uPoiss) == " << ulnorm/uPoissNorm << endl;
    cout << "L2Dist(un,ul)/L2Norm(ul)  == " << L2Dist(un, ul)/ulnorm <<endl;
    cout << "L2Dist(pn,pl)/L2Norm(pl)  == " << L2Dist(pn, pl)/plnorm <<endl;
    
    Complex an = L2InnerProduct(un, u_oseig);
    Complex al = L2InnerProduct(ul, u_oseig);
    cout << "al  == " << al << endl;
    cout << "ark == " << ark << endl;
    cout << "an  == " << an << endl;
    cout << "phase al  == " << phase(al) << endl;
    cout << "phase ark == " << phase(ark) << endl;
    cout << "phase an  == " << phase(an) << endl;
    cout << "(phase(an)-phase(al))/phase(al) == " << (phase(an)-phase(al))/phase(al) << endl;
    cout << "norm al  == " << norm(al) << endl;
    cout << "norm ark == " << norm(ark) << endl;
    cout << "norm an  == " << norm(an) << endl;
    cout << "(norm(an)-norm(al))/norm(al) == " << (norm(an)-norm(al))/norm(al) << endl;
    Complex L2IP = L2InnerProduct(un, u_oseig);
    cout << "L2IP(u,os)/L2(u0,os) == " << L2IP/L2IP0  << endl;
    cout << "exp(lambda*t)        == " << exp(lambda*t) << endl;
    cout << "lambda      == " << lambda <<endl;
    cout << "lambda appr == " << log(L2IP/L2IP0)/t <<endl;
    cout << "growth rate ==  "<< log(unnorm/u0norm)/t << endl;

    // time integration for everybody
    if (pause) {
      cout << "hit return to continue..." << flush;
      char buff[10];
      cin.getline(buff,10);
    }
    ul.setToZero();
    pl.setToZero();

    BasisFunc e_oseig(s_oseig);
    e_oseig *= exp(lambda*(t+dt*plotmod));
    ul.addProfile(e_oseig, true);
      
    ComplexChebyCoeff e_peig(s_peig);
    e_peig *= exp(lambda*(t+dt*plotmod));

    pl.addProfile(e_peig, 1, 0, 0, true);

    RK3update(ark, lambda_v, lambda_b, dt, plotmod);
    dns.advance(un, pn, plotmod);
  }    
  fftw_savewisdom();
}

Real alpha[] = {29.0/96.0, -3.0/40.0, 1.0/6.0};
Real beta[]  = {37.0/160.0, 5.0/24.0, 1.0/6.0};
Real gamm[] = {8.0/15.0, 5.0/12.0, 3.0/4.0};
Real zeta[]  = {0.0, -17.0/60.0, -5.0/12.0};

void RK3update(Complex& an, Complex lambda_v, Complex lambda_b, Real dt, int steps) {
  for (int n=0; n<steps; ++n) {
    Complex an1(0.0, 0.0);
    Complex tmp;
    for (int i=0; i<3; ++i) {
      tmp = an;
      an = 
	((1.0+dt*alpha[i]*lambda_v)*an + dt*lambda_b*(gamm[i]*an + zeta[i]*an1))
	/ (1.0-dt*beta[i]*lambda_v);
      an1 = tmp;
    }
  }
}

  
    


  
