#include <iostream>
#include <fstream>
#include <sstream>
#include <iomanip>
#include "flowfield.h"
#include "vector.h"
#include "chebyshev.h"
#include "tausolver.h"
#include "nsintegrator.h"
const Real EPSILON=1e-12;

int main() {

  cout << "=================================================================\n";
  cout << "This program simulates a flow with a sinusoidal initial condition\n";
  cout << "and compares the DNS solution to the known analytic solution.\n";
  cout << "Take base flow U(y) == 0 and constrain the system to have zero\n";
  cout << "pressure gradient. For the initial condition \n\n";
  cout << "  u(x,y,z,0) = eps*sin(2*pi*ky*y[ny]/Ly),\n\n";
  cout << "the exact solution of the Navier-Stokes equation is\n\n";
  cout << "  u(x,y,z,t) = u(x,y,z,0)*exp(-nu*t*square(2*pi*ky/Ly))\n\n";
  cout << "In the following output,\n\n";
  cout << "  un == the numerical solution.\n";
  cout << "  ut == the true solution.\n\n";


  const int Nx=32;
  const int Ny=65;
  const int Nz=8;
  const int Nxd=(2*Nx)/3;
  const int Nyd=(2*Ny)/3;
  const int Nzd=(2*Nz)/3;

  char s = ' ';
  cout << "Nx  Ny  Nz == " << Nx << s << Ny << s << Nz << endl;
  cout << "Nxd Nyd Nzd == " << Nxd << s << Nyd << s << Nzd << endl;

  const int Nd=3;
  const Real Lx=2*pi;
  const Real Lz=2*pi;
  const Real a= -1.0;
  const Real b=  1.0;
  const Real Ly= b-a;
  const Real Reynolds = 7500.0;
  const Real nu = 1.0/Reynolds;
  //const Real dPdx = -2.0*nu;    // mean streamwise pressure gradient.
  //const Real dPdx = 0.0;
  const Real dt = 0.10;
  const Real T = 5000;
  const int plotmod = 10;
  const int nSteps = int(T/dt);
  DNSFlags flags;
  flags.timestepping = RK3;
  
  // sinusoid perturbation params.
  const Real eps = 0.10;
  const int ky = 1;

  cout << setprecision(14);
  cout << "Nx Ny Nz Nd == " << Nx << ' ' << Ny << ' ' << Nz << ' ' << Nd << endl;
  cout << "Lx Ly Lz == " << Lx << " 2 "  << Lz << ' ' << endl;

  Vector y = chebypoints(Ny,a,b);
  y.save("y");
  ChebyTransform trans(Ny);
  
  ChebyCoeff sinusoid(Ny,a,b,Physical);
  ChebyCoeff parabola(Ny,a,b,Physical);
  ChebyCoeff zero(Ny,a,b,Physical);
  for (int ny=0; ny<Ny; ++ny) {
    sinusoid[ny] = eps*sin(2*pi*ky*y[ny]/Ly);
    parabola[ny] = 1.0 - square(y[ny]);
  }
  trans.makeSpectral(sinusoid);
  ChebyCoeff Ubase = zero;
  
  
  FlowField un(Nx,Ny,Nz,Nd,Lx,Lz,a,b);
  for (int ny=0; ny<Ny; ++ny) 
    un.cmplx(0,ny,0,0) = sinusoid[ny];
  
  FlowField Pn(Nx,Ny,Nz,1,Lx,Lz,a,b);
  FlowField vortn(Nx,Ny,Nz,Nd,Lx,Lz,a,b);
  FlowField fn1(Nx,Ny,Nz,Nd,Lx,Lz,a,b);
  FlowField fn(Nx,Ny,Nz,Nd,Lx,Lz,a,b);

  cout << "=================================================" << endl;
  cout << "Initial data, prior to time stepping" << endl;
  cout << "L2Norm(un)  == " << L2Norm(un) << endl;
  cout << "L2Norm(Pn)  == " << L2Norm(Pn) << endl;
  cout << "div(un)     == " << un.divergence() << endl;  
  cout << endl;
  NSIntegrator dns(un, Ubase, nu, dt, flags);

  cout << "=================================================" << endl;
  cout << "Starting time stepping..." << endl;
  ofstream ns("osnorm.asc");
  ns << setprecision(14);

  Real u0norm = L2Norm(un);
  for (int step=0; step<nSteps; step += plotmod) {
    Real t = dt*step;
    cout << "step " << step << "  t == " << t << endl;
    Real unnorm = L2Norm(un);
    Real utnorm = exp(-nu*t *square(2*pi*ky/Ly))*u0norm;
    Real CFL = dns.CFL();
    ns << utnorm << endl;
    cout << "======================================================" << endl;
    cout << "CFL == " << CFL << endl;
    cout << "L2Norm(un) == " << unnorm << endl;
    cout << "L2Norm(ut) == " << utnorm << endl;
    Real err = abs(unnorm - utnorm);
    cout << "L2Norm(un)-L2norm(ut)              == " << err << endl;
    cout << "(L2Norm(un)-L2norm(ut))/L2Norm(ut) == " << err/utnorm << endl;

    un.saveProfile(0,0, "un", trans);
    if (CFL > 2.0 || unnorm > 1) {
      cerr << "Problem!" << endl;
      cerr << "CFL  == " << CFL << endl;
      cerr << "norm == " << unnorm << endl;
      break;
    }
    dns.advance(un, Pn, plotmod);
  }
  un.binarySave("uOS");
}

