#include <iostream>
#include <fstream>
#include <sstream>
#include <iomanip>
#include "flowfield.h"
#include "vector.h"
#include "chebyshev.h"
#include "tausolver.h"
#include "nsintegrator.h"

#include "helperfuncs.h"

int main() {

  const int Nx=32;
  const int Ny=65;
  const int Nz=32;

  const Real Lx=4*pi/3;
  const Real a= -1.0;
  const Real b=  1.0;
  const Real Lz=4*pi/3;

  const Real Reynolds = 4000.0;
  const Real nu = 1.0/Reynolds;

  const Real CFLmin = 0.80;
  const Real CFLmax = 1.1;
  const Real dtmin  = 0.005;
  const Real dtmax  = 0.10;
  const Real T0 = 0;    // start time
  const Real T1 = 200;  // end time
  const Real dT = 0.25; // plotting and statistics interval

  DNSFlags flags;
  flags.timestepping = RK3;

  const Real perturbMag = 0.02;
  const Real decay = 0.7; // cheb spectral decay of perturb profiles
  const int kxmax=5;      // maximum Fourier mode for perturbations
  const int kzmax=5;
 
  const char sp= ' ';
  const char nl= '\n';

  cout << setprecision(4);
  Vector x = periodicpoints(Nx, Lx);
  Vector y = chebypoints(Ny,a,b);
  Vector z = periodicpoints(Nz, Lz);
  x.save("x");
  y.save("y");
  z.save("z");
  
  ChebyTransform trans(Ny);

  ChebyCoeff Ubase(Ny,a,b,Physical);
  for (int ny=0; ny<Ny; ++ny) 
    Ubase[ny] = 1.0 - square(y[ny]);
  Ubase.save("Ubase");
  Ubase.makeSpectral(trans);

  FlowField u(Nx,Ny,Nz,3,Lx,Lz,a,b);
  FlowField q(Nx,Ny,Nz,1,Lx,Lz,a,b);
  u.addPerturbations(kxmax,kzmax,perturbMag,decay);
  //FlowField u("u80");
  //FlowField q("q80");
  cout << "div(u)/L2Norm(u)  == " << u.divergence()/L2Norm(u) << endl;  
  
  FlowField omega(Nx,Ny,Nz,3,Lx,Lz,a,b);

  cout << "optimizing FFTW..." << flush;
  fftw_loadwisdom();
  u.optimizeFFTW();
  fftw_savewisdom();
  cout << "done" << endl;

  cout << "constructing NSIntegrator..." << flush;
  TimeStep dt((dtmin+dtmax)/2, dtmin, dtmax, dT, CFLmin, CFLmax); 
  NSIntegrator dns(u, Ubase, nu, dt, flags, T0);
  dns.resetUbulk(2.0/3.0);
  cout << "done" << endl;
  
  // print header info in fmodes.asc file

  ofstream modestream("fmodes.asc");
  ofstream dragstream("drags.asc");

  modestream << "% ";
  for (int kx=-kxmax; kx<=kxmax; ++kx) 
    for (int kz=0; kz<=kzmax; ++kz) 
      modestream << kx << ',' << kz << sp;
  modestream << nl;
  
  for (Real t=T0; t<=T1; t += dT) {
    cout << "===============================================" << endl;
    cout << "         t == " << t << endl;
    cout << "        dt == " << dt << endl;
    cout << "       CFL == " << dns.CFL() << endl;
    cout << "L2Norm2(u) == " << L2Norm2(u) << endl;
    cout << "     Ubulk == " << dns.Ubulk() << endl;
    cout << "     ubulk == " << Re(u.profile(0,0,0)).mean()/2 << endl;
    cout << "      dPdx == " << dns.dPdx() << endl;
    

    if (int(t) % 40 == 0 && t != T0) {
      cout << "saving flowfields..." << endl;
      u.binarySave("u"+i2s(int(t)));
      q.binarySave("q"+i2s(int(t)));
      cout << "done" << endl;
    }

    u.makePhysical();
    u.saveSliceXY(0,0,"uside");
    u.saveSliceXY(0,1,"vside");
    u.saveSliceXY(0,2,"wside");
    u.saveSliceXY(0,2,"wside");
    u.saveCrossSection(0,0,"usec");
    u.saveCrossSection(0,1,"vsec");
    u.saveCrossSection(0,2,"wsec");
    u.makeSpectral();

    u.vorticity2(omega);
    omega.saveCrossSection(0,0,"omsec0");
    omega.saveCrossSection(0,1,"omsec1");
    omega.saveCrossSection(0,2,"omsec2");

    for (int kx=-kxmax; kx<=kxmax; ++kx) {
      int nx = u.nx(kx);
      for (int kz=0; kz<=kzmax; ++kz) {
	int nz = u.nz(kz);
	BasisFunc u_kxkz = u.profile(nx,nz);
	modestream << L2Norm(u_kxkz) << sp;
      }
    }
    modestream << endl;
    ChebyCoeff dudy = diff(Re(u.profile(0,0,0)));
    dragstream << nu*dudy.eval_a() << sp << nu*dudy.eval_b() << endl;

    if (dt.adjust(dns.CFL())) {
      cerr << "Resetting dt to " << dt << ", new CFL == " << dt.CFL() << endl;
      dns.reset(nu, dt);
    }
    cout << "timestepping..." << endl;
    dns.advance(u, q, dt.n());
    cout << "done" << endl;
  }
  cout << "done!" << endl;
}
