"""Command line generic functionality"""

# Copyright 2003 Iustin Pop
#
# This file is part of cfvers.
#
# cfvers is free software; you can redistribute it and/or modify it
# under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# cfvers is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
# General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with cfvers; if not, write to the Free Software Foundation,
# Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA

import os, struct, stat, os.path, re, commands, sys
import types
import random
import errno
from mx import DateTime
from ConfigParser import SafeConfigParser
import tarfile
from cStringIO import StringIO

import cfvers
import cfvers.repository
import cfvers.main
from cfvers.main import *

__all__ = [
    "AdminCommands", "Commands",
    ]

class CLIScript(object):
    """Base class for command line handlers"""
    
    def __init__(self, options):
        self.readconfig(options)

    def readconfig(options):
        if options.repository is not None and \
           options.area is not None:
            return
        env_repo = os.environ.get("CFVERS_REPO")
        if env_repo is not None and options.repository is None:
            options.repository = env_repo
        env_area = os.environ.get("CFVERS_AREA")
        if env_area is not None and options.area is None:
            options.area = env_area
        if options.repository is not None and \
           options.area is not None:
            return
        cp = SafeConfigParser()
        cp.read(["/etc/cfvers.conf", os.path.expanduser("~/.cfvers"),])
        if options.repository is None and \
               cp.has_option("repositories", "default"):
            options.repository = cp.get("repositories", "default")
        if options.area is None and \
               cp.has_option("repositories", "default_area"):
            options.area = cp.get("repositories", "default_area")
        return
        
    readconfig = staticmethod(readconfig)

    def get_version():
        nv = "%d.%d.%d (%s release)" % cfvers.version_info[0:4]
        version="""%%prog - cfvers %s
Copyright (C) 2003 Iustin Pop

This is free software; see the source for copying conditions.
There is NO warranty ; not even for MERCHANTABILITY or FITNESS
FOR A PARTICULAR PURPOSE.""" % nv
        return version

    get_version = staticmethod(get_version)

    def parserev(revstring):
        """Tries to parse a revision string of form num[:num]"""
        
        if revstring is None:
            return None, None
        # match and extract from glob rev[:rev]
        m=re.match("^((?P<start>[0-9]+):)?(?P<end>[0-9]+)$", revstring)
        if m is None:
            raise ValueError, "Invalid syntax in revision specification %s" % revstring
        r1 = m.group('start')
        r2 = m.group('end')
        if r1 is not None:
            r1 = int(r1)
        if r2 is not None:
            r2 = int(r2)
        return r1, r2

    parserev = staticmethod(parserev)
    
class AdminCommands(CLIScript):
    def __init__(self, options):
        """Constructor for the Commands class"""

        super(AdminCommands, self).__init__(options)
        self.repo = cfvers.repository.open(options.repository)
        
    def close(self):
        self.repo.close()

    def create_area(self, cmdoptions, name):
        a = Area(name=name, description=cmdoptions.description,
                 root=cmdoptions.root)
        self.repo.addArea(a)
        self.repo.commit()
        return

    def init_repo(options, createopts):
        CLIScript.readconfig(options)
        cfvers.repository.open(create=True, cnxargs=options.repository,
                               createopts=createopts)
        return

    init_repo = staticmethod(init_repo)
    
class Commands(CLIScript):
    def __init__(self, options):
        """Constructor for the Commands class"""

        super(Commands, self).__init__(options)
        if options.repository is None:
            raise cfvers.ConfigException, "no repository specified"
        self.repo = cfvers.repository.open(cnxargs=options.repository)
        if options.area is None:
            raise cfvers.ConfigException, "no area specified"
        self.area = self.repo.getArea(options.area)
        if self.area is None:
            raise cfvers.ConfigException, "can't find area named '%s'" % options.area
        return
    
    def close(self):
        self.repo.close()
        return
    
    def _map_files(self, area, files=None, norecurse=False):
        """Maps a (possibly empty) list of files to a list of targets"""
        
        def helper(arg, dirname, names):
            mylist, path = arg
            # Translate from real root to area root
            if path == "/":
                realdir = dirname
            elif dirname == path:
                realdir = "/"
            elif dirname.startswith(path):
                realdir = dirname[len(path):]
            else:
                raise ValueError, "Path doesn't lie in its area!"
            mylist.extend([os.path.join(realdir, fname) for fname in names])
            return

        repo = self.repo
        if len(files) == 0:
            for item in repo.items(area=area):
                latest = repo.getEntry(item, None, do_payload=False)
                if latest is not None:
                    files.append(latest.filename)
                else:
                    files.append(item.name)
            
        files = map(os.path.abspath, files)
        files = dict.fromkeys(files).keys()
        targets = []
        errors = []
        for virtf in files:
            realf = forcejoin(area.root, virtf)
            try:
                st = os.lstat(realf)
            except EnvironmentError, e:
                errors.append(Result(Result.STORED_IOERROR, exc=e))
                continue
            # If we got here, the file at least exists,
            # so we'll mark to process
            targets.append(virtf)
            if not norecurse and stat.S_ISDIR(st.st_mode):
                os.path.walk(realf, helper, (targets, area.root))
        targets = dict.fromkeys(targets).keys()
        targets.sort()
        return (targets, errors)
        
    def store(self, files=[], options=None):
        """Stored a set of files or all the items already in the repository.

        Parameters:
          - files: a list of filename, or empty if all the items should be (re)commited
          - logmsg: the log message

        A new revision won't be stored if no items have been stored (e.g. nothing changed,
        or no items could be read).
        """

        logmsg = options.logmsg
        norecurse = options.norecurse
        repo = self.repo
        a = self.area
        (targets, errors) = self._map_files(a, files=files, norecurse=norecurse)
        ar = AreaRev(area=a, logmsg=logmsg, commiter=options.commiter)
        newrev = ar.revno
        stored = 0
        for name in targets:
            item = repo.getItemByName(a, name)
            if item is None:
                item = Item(area=a, name=name,
                            metadata_only=options.nopayload)
                repo.addItem(item)
                item = repo.getItemByName(a, name)
                item.metadata_only = options.nopayload
            retcode = self._store_newrev(item, newrev)
            if retcode.code == Result.STORED_OK:
                stored += 1
            errors.append(retcode)

        if stored > 0:
            repo.putAreaRev(ar)
            repo.commit()
            store_done = True
        else:
            repo.rollback()
            store_done = False
        return errors, store_done, newrev

    def _store_newrev(self, item, revno):
        try:
            rev = RevEntry(item, revno)
        except EnvironmentError, e:
            return Result(Result.STORED_IOERROR, item=item, exc=e)
        
        old = self.repo.getEntry(item, None)
        if (not old is None) and rev == old:
            return Result(Result.STORED_NOTCHANGED, item=item, rev=old)
        else:
            self.repo.addEntry(rev)
        return Result(Result.STORED_OK, item=item, rev=rev)

    def diff(self, options=None, files=None):
        rev1, rev2 = self.parserev(options.rev)
        if files is None or len(files) == 0:
            for vi in self.repo.items(area=self.area):
                self._diff_file(vi, rev1, rev2, options)
        else:
            for fn in files:
                vi = self.repo.getItemByName(self.area, os.path.abspath(fn))
                if vi is None:
                    print >>sys.stderr, "Item '%s' is not in the area!" % fn
                    return
                self._diff_file(vi, rev1, rev2, options)
        return
                
    def _diff_file(self, vi, rev1, rev2, options):
        """Execute a diff between two revisions.

        Parameters:
          - vi: the item which to analyze
          - rev1: the source revision
          - rev2: the target revision

        """
        rlist = self.repo.getRevList(vi)
        if len(rlist) == 0:
            if options.list:
                print vi.name
            else:
                print "===== Item %s does not have any revisions" % vi.name
            return
        # rev1 is older (source)
        # rev2 is newer (target)
        if rev1 is None:
            rev1 = rev2
            rev2 = None
        e1 = self.repo.getEntry(vi, rev1)
        if e1 is None:
            if options.list:
                print vi.name
            else:
                print "===== Item %s doesn't have revision entry %d" % (vi.name, rev1)
            return
        if rev2 is None:
            # Build from filesystem, if possible
            if cfvers.rexists(vi.name):
                try:
                    e2 = RevEntry(vi)
                except IOError, e:
                    if options.list:
                        print vi.name
                    else:
                        print "==================================================================="
                        print "Can't read current status of %s, error: %s" % (vi.name, e)
                    return
                except OSError, e:
                    if options.list:
                        print vi.name
                    else:
                        print "==================================================================="
                        print "Can't read current status of %s, error: %s" % (vi.name, e)
                    return
            else:
                if options.list:
                    print vi.name
                else:
                    print "==================================================================="
                    print "%s has been deleted" % vi.name 
                return
        else:
            e2 = self.repo.getEntry(vi, rev2)
            if e2 is None:
                raise ValueError, "Item %s doesn't have revision entry %d" % (vi.name, rev1)
        output = e2.diff(e1, options=options)
        if len(output) > 0:
            if options.list:
                print vi.name
            else:
                print "===== Item %s (rev %s -> %s)" % (vi.name, e1.revno, e2.revno or 'current')
                print output
        return

    def retrieve(self, files=None, options=None):
        results = []
        if len(files) != 0:
            mlist = []
            for filename in files:
                af = os.path.abspath(filename)
                item = self.repo.getItemByName(self.area, af)
                if item is None:
                    results.append(Result(Result.RETR_NTRACK, fname=af))
                    continue
                mlist.append(item)
                if not options.norecurse:
                    mlist += self._recur_build_list(mlist, [])
        else:
            mlist = self.repo.items(area=self.area)
        self._retrieve_list(mlist, options, results)
        return results

    def _recur_build_list(self, litems, alsoskip):
        nlist = []
        for item in litems:
            others = self.repo.getItemsByDirname(self.area, item.name)
            for it in others:
                if it not in [x.name for x in litems] and \
                   it not in [x.name for x in alsoskip] and \
                   it not in [x.name for x in nlist]:
                    nlist.append(it)
                    nlist += self._recur_build_list([it], nlist + litems)
        return nlist

    def _retrieve_file(self, vi, rev, options=None):
        return rev.to_filesys(destdir=options.destdir, use_dirs=options.use_dirs)

    def _retrieve_list(self, ilist, options, results):
        for item in ilist:
            rev = self.repo.getEntry(item, options.revno)
            if rev is None:
                revlist = self.repo.getRevList(item)
                if len(revlist) == 0:
                    ncod = Result.RETR_NOREVS
                else:
                    ncod = Result.RETR_NOXREV
                results.append(Result(ncod, item=item))
            else:
                results.append(self._retrieve_file(item, rev, options=options))
        return

    def show(self, filename, rev=None):
        vi = self.repo.getItemByName(self.area, os.path.abspath(filename))
        if vi is None:
            print >>sys.stderr, "Item '%s' is not in the area!" % filename
            return
        entry = self.repo.getEntry(vi, rev)
        if entry is None:
            rlist = self.repo.getRevList(vi)
            if len(rlist) == 0:
                print >> sys.stderr, "Item '%s' doesn't have any revisions!" % vi.name
            else:
                print >> sys.stderr, "Can't find revision '%s' for item '%s'!" % (rev, vi.name)
            return
        if entry.filecontents is None:
            print >> sys.stderr, "File contents has not been stored for file '%s'!" % entry.filename
            return
        print entry.filecontents,
        return

    def stat(self, options, files=None):
        if options.rev is not None:
            version = int(options.rev)
        else:
            version = None
            
        if len(files) != 0:
            mlist = []
            for filename in files:
                af = os.path.abspath(filename)
                item = self.repo.getItemByName(self.area, af)
                if item is None:
                    print "Skipped: file '%s' is not being tracked!" % af
                    print
                    continue
                mlist.append(item)
        else:
            mlist = self.repo.items(area=self.area)
        self._stat_list(mlist, version)
        return

    def _stat_list(self, ilist, version):
        for item in ilist:
            rev = self.repo.getEntry(item, None, do_payload=False)
            if rev is None:
                revlist = self.repo.getRevList(item)
                if len(revlist) == 0:
                    print "Item '%s': doesn't have any revisions!" % item.name
                else:
                    print "Item '%s': can't find revision %d!" % (item.name, version)
                print
            else:
                self._stat_file(rev)
        return

    def _stat_file(self, rev):
        print rev.stat()
        return

    def log(self):
        arearevs = self.repo.getAreaRevs(self.area)
        return arearevs

    def exportcksum(self, cmdopts, cmdargs):
        if cmdopts.output == "-":
            f = sys.stdout
            close = False
        else:
            f = file(cmdopts.output, "w")
            close = True
        for rev in self.repo.getEntries(self.area, cmdopts.r2, do_payload=False):
            if rev is not None and cmdopts.r1 is not None and rev.revno < cmdopts.r1:
                continue
            if rev is not None and rev.isreg():
                f.write("%s  %s\n" % (rev.sha1sum, rev.filename[1:]))
        if close:
            f.close()
        return

    def exporttar(self, cmdopts, cmdargs):
        if cmdopts.output == "-":
            tarh = tarfile.open(fileobj=sys.stdout, mode="w|")
        else:
            tarh = tarfile.open(name=cmdopts.output, mode="w")
        for rev in self.repo.getEntries(self.area, None):
            if rev is not None:
                if rev.isreg():
                    fdata = StringIO(rev.filecontents)
                else:
                    fdata = None
                tarh.addfile(self._genfakefile(rev), fdata)
        tarh.close()
        return

    def _genfakefile(self, rev):
        """Generate a fake TarInfo object from a revision entry.

        Parameters:
        rev - the revision entry from which to create the archive member.
        
        """
        ti = tarfile.TarInfo()
        if rev.filename.startswith("/"):
            ti.name = rev.filename[1:]
        else:
            ti.name = rev.filename
        ti.uname = ""
        ti.gname = ""
        ti.uid = rev.uid
        ti.gid = rev.gid
        ti.mtime = rev.mtime
        ti.mode = rev.mode
        if rev.isreg():
            ti.chksum = tarfile.calc_chksum(rev.filecontents)
            ti.size = rev.size
            ti.type = tarfile.REGTYPE
        elif rev.isdir():
            ti.type = tarfile.DIRTYPE
        elif rev.islnk():
            ti.type = tarfile.SYMTYPE
            ti.linkname = rev.filecontents
        elif rev.isblk():
            ti.type = tarfile.BLKTYPE
        elif rev.ischr():
            ti.type = tarfile.CHRTYPE
        elif rev.ififo():
            ti.type = tarfile.FIFOTYPE
        elif rev.isock():
            return None
        return ti
