"""Command line generic functionality"""

# Copyright 2003 Iustin Pop
#
# This file is part of cfvers.
#
# cfvers is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# cfvers is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with cfvers; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

import os, struct, stat, os.path, re, commands, sys
import types
import random
from mx import DateTime
from ConfigParser import SafeConfigParser

import __init__
from main import *

#Return codes for store operation
STORED_OK = 0
STORED_NOTCHANGED = 1
STORED_IOERROR = 2

__all__ = ["STORED_OK", "STORED_NOTCHANGED", "STORED_IOERROR",
           "AdminCommands", "Commands",
           ]

class CLIScript(object):
    """Base class for command line handlers"""
    
    def __init__(self, options):
        self.fixpath(options)

    def fixpath(options):
        """Reads path option from configuration files"""
        if not hasattr(options, "path") or options.path is None:
            # Parse default repo
            for fn in (os.path.expanduser("~/.cfvers"), "/etc/cfvers.conf"):
                p = CLIScript._parsefile(fn)
                if p is not None:
                    options.path = p
                    break
            else:
                raise ValueError, "Must specify repository path!"
        return

    fixpath = staticmethod(fixpath)

    def _parsefile(filename):
        if not os.path.exists(filename):
            return None
        cp = SafeConfigParser()
        cp.read(filename)
        if cp.has_option("repositories", "default"):
            return cp.get("repositories", "default")
        return None

    _parsefile = staticmethod(_parsefile)

    def get_version():
        nv = "%d.%d.%d (%s release)" % __init__.version_info[0:4]
        version="""%%prog - cfvers %s
Copyright (C) 2003 Iustin Pop

This is free software; see the source for copying conditions.
There is NO warranty ; not even for MERCHANTABILITY or FITNESS
FOR A PARTICULAR PURPOSE.""" % nv
        return version

    get_version = staticmethod(get_version)

    def parserev(revstring):
        """Tries to parse a revision string of form num[:num]"""
        
        if revstring is None:
            return None, None
        # match and extract from glob rev[:rev]
        m=re.match("^(?P<start>[0-9]+)(:(?P<end>[0-9]+))?$", revstring)
        if m is None:
            raise ValueError, "Invalid syntax in revision specification %s" % revstring
        r1 = m.group('start')
        r2 = m.group('end')
        if r1 is not None:
            r1 = int(r1)
        if r2 is not None:
            r2 = int(r2)
        return r1, r2

    parserev = staticmethod(parserev)
    
class AdminCommands(CLIScript):
    def __init__(self, options):
        """Constructor for the Commands class"""

        super(AdminCommands, self).__init__(options)
        self.repo = __init__.getRepo(options.path)
        
    def close(self):
        self.repo.close()

    def create_area(self, cmdoptions, name):
        a = Area(server=cmdoptions.server, name=name,
                 description=cmdoptions.description, root=cmdoptions.root)
        self.repo.addArea(a)
        self.repo.commit()
        return

    def init_repo(options):
        CLIScript.fixpath(options)
        __init__.getRepo(create=True, cnxargs=options.path)
        return

    init_repo = staticmethod(init_repo)
    
class Commands(CLIScript):
    def __init__(self, options):
        """Constructor for the Commands class"""

        super(Commands, self).__init__(options)
        self.repo = __init__.getRepo(cnxargs=options.path)
        self.area = Area.parse(self.repo, options.area)
        if self.area is None:
            raise ValueError, "Can't find area named '%s'" % area
        return
    
    def close(self):
        self.repo.close()
        return
    
    def _map_files(self, area, files=None, norecurse=0):
        """Maps a (possibly empty) list of files to a list of targets"""
        
        def helper(arg, dirname, names):
            mylist, path = arg
            # Translate from real root to area root
            if path == "/":
                realdir = dirname
            elif dirname == path:
                realdir = "/"
            elif dirname.startswith(path):
                realdir = dirname[len(path):]
            else:
                raise ValueError, "Path doesn't lie in its area!"
            #print path, realdir
            mylist.extend([os.path.join(realdir, fname) for fname in names])
            return

        repo = self.repo
        if len(files) == 0:
            subfiles = []
            for item in repo.items():
                latest = repo.getEntry(item, None)
                if latest is not None:
                    subfiles.append(latest.filename)
                else:
                    subfiles.append(item.name)
            files = subfiles
            
        files = map(os.path.abspath, files)
        targets = []
        errors = []
        for f in files:
            targets.append(f)
            realf = forcejoin(area.root, f)
            if not norecurse:
                try:
                    isdir = stat.S_ISDIR(os.lstat(realf).st_mode)
                except IOError, e:
                    errors.append((STORED_IOERROR, e, None, None))
                    continue
                except OSError, e:
                    errors.append((STORED_IOERROR, e, None, None))
                    continue
                if isdir:
                    os.path.walk(realf, helper, (targets, area.root))
        return (targets, errors)
        
    def store(self, files=[], logmsg=None, norecurse=0):
        """Stored a set of files or all the items already in the repository.

        Parameters:
          - files: a list of filename, or empty if all the items should be (re)commited
          - logmsg: the log message

        A new revision won't be stored if no items have been stored (e.g. nothing changed,
        or no items could be read).
        """

        repo = self.repo
        a = self.area
        (targets, errors) = self._map_files(a, files=files, norecurse=norecurse)
        ar = AreaRev(a, logmsg)
        newrev = ar.revno
        stored = 0
        for name in targets:
            item = repo.getItemByName(a.id, name)
            if item is None:
                item = Item(area=a, name=name)
                repo.addItem(item)
                item = repo.getItemByName(a.id, name)
            retcode = self._store_newrev(item, newrev)
            if retcode[0] != STORED_OK:
                errors.append(retcode)
            else:
                stored += 1

        if stored > 0:
            a.revno = newrev
            repo.updArea(a)
            repo.putAreaRev(ar)
            repo.commit()
        else:
            repo.rollback()
        return errors

    def _store_newrev(self, item, revno):
        try:
            rev = RevEntry(item, revno)
        except IOError, e:
            return (STORED_IOERROR, e, item, None)
        except OSError, e:
            return (STORED_IOERROR, e, item, None)
        
        old = self.repo.getEntry(item, None)
        if (not old is None) and rev == old:
            return (STORED_NOTCHANGED, 'Item has not changed, not stored', item, old)
        else:
            self.repo.addEntry(rev)
        return (STORED_OK, '', item, rev)

    def diff(self, options=None, files=None):
        rev1, rev2 = self.parserev(options.rev)
        if files is None or len(files) == 0:
            self._diff_all(rev1, rev2)
        else:
            for fn in files:
                vi = self.repo.getItemByName(self.area.id, os.path.abspath(fn))
                if vi is None:
                    print >>sys.stderr, "Item '%s' is not in the area!" % fn
                    return
                self._diff_file(vi, rev1, rev2)

    def _diff_all(self, rev1, rev2):
        for vi in self.repo.items():
            self._diff_file(vi, rev1, rev2)
        return
                
    def _diff_file(self, vi, rev1, rev2):
        rlist = self.repo.getRevList(vi)
        e1 = self.repo.getEntry(vi, rev1)
        if e1 is None:
            raise ValueError, "Item %s doesn't have revision entry %d" % (vi.name, rev1)
        if rev2 is None:
            # Build from filesystem, if possible
            if os.path.exists(vi.name):
                try:
                    e2 = RevEntry(vi)
                except IOError, e:
                    print "==================================================================="
                    print "Can't read current status of %s, error: %s" % (vi.name, e)
                    return
                except OSError, e:
                    print "==================================================================="
                    print "Can't read current status of %s, error: %s" % (vi.name, e)
                    return
            else:
                print "==================================================================="
                print "%s has been deleted" % vi.name 
                return
        else:
            e2 = self.repo.getEntry(vi, rev2)
            if e2 is None:
                raise ValueError, "Item %s doesn't have revision entry %d" % (vi.name, rev1)
        output = e2.diff(e1)
        if len(output) > 0:
            print "==================================================================="
            print output
        return

    def retrieve(self, version=None, files=None, destdir=None, use_dirs=1):
        if version is not None:
            version = int(version)
            
        if len(files) != 0:
            for filename in files:
                af = os.path.abspath(filename)
                item = self.repo.getItemByName(self.area.id, af)
                if item is None:
                    print "Skipped: file '%s' is not being tracked!" % af
                    continue
                rev = self.repo.getEntry(item, version)
                if rev is None:
                    print "Can't find revision %d for item %s!" % (version, item.name)
                else:
                    self._retrieve_file(item, rev, destdir=destdir, use_dirs=use_dirs)
        else:
            self._retrieve_all(version, destdir=destdir, use_dirs=use_dirs)

    def _retrieve_file(self, vi, rev, destdir=None, use_dirs=1):
        if rev is None:
            print >>sys.stderr, "Can't find requested version!"
            return
        print "Retrieving file %s..." % vi.name
        rev.to_filesys(destdir=destdir, use_dirs=use_dirs)

    def _retrieve_all(self, version, destdir=None, use_dirs=1):
        for item in self.repo.items():
            rev = self.repo.getEntry(item, None)
            self._retrieve_file(item, rev, destdir=destdir, use_dirs=use_dirs)

##    def forget(self, items, pfx=None):
##        pfx = Prefix(pfx)
##        for name in items:
##            id = pfx.join(os.path.abspath(name))
##            self.repo.delVIE(id)

    def show(self, filename, rev=None):
        vi = self.repo.getItemByName(self.area.id, os.path.abspath(filename))
        if vi is None:
            print >>sys.stderr, "Item '%s' is not in the area!" % filename
            return
        entry = self.repo.getEntry(vi, rev)
        if entry is None:
            print >> sys.stderr, "Can't find revision %s!" % rev
            return
        print entry.filecontents,

    def log(self):
        arearevs = self.repo.getAreaRevs(self.area.id)
        return arearevs

    def stat(self, args):
        return
