# Catkin/Interface.pm
# Copyright (C) 2002-2003 colin z robertson
# 
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
# 
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
# 
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

package Catkin::Interface;

use strict;
use Template;
use Catkin::Index;
use Catkin::Entry;
use Catkin::Comment;
use Catkin::Config;
use Catkin::Session;
use Symbol;
use IO::File;
use Fcntl ':flock';
use CGI::Carp;
use Data::Dumper;
use Catkin::Formatter;
use Text::Convert;
use XML::Escape;
use CGI;
use HTML::Sloppy;
use File::Spec::Functions;
use DateTime;
use DateTime::TimeZone;

sub new {
    my $proto = shift;
    my $class = ref($proto) || $proto;
    my $self  = {};
    bless ($self, $class);
	
	my ($config_dir) = @_;

	$self->{LOCK} = gensym();
	$self->{LOCK_FILE} = catfile($config_dir,"lock");
	$self->{CONFIG_DIR} = $config_dir;
	
	return $self;
}

sub index {
	my $self = shift;
	return $self->{INDEX};
}

sub config {
	my $self = shift;
	return $self->{CONFIG};
}

sub formatter {
	my $self = shift;
	return $self->{FORMATTER};
}

sub init_dirs {
	my $self = shift;
	#FIXME: probably only works for one level.
	mkdir($self->config->key('html_dir'),0777)
	  unless -e $self->config->key('html_dir');
	mkdir($self->config->key('priv_dir'),0777)
	  unless -e $self->config->key('priv_dir');
}

sub setup_r {
	my $self = shift;
	if (!$self->lock_r) {
		warn "Unable to obtain shared lock";
		return;
	}
	if (!$self->setup_icf_fields) {
		warn "Failed to set up ICF";
		return;
	}
	if (!$self->init_dirs) {
		warn "Failed to initialise directories";
		return;
	}
	return 1;
}

sub setup_rw {
	my $self = shift;
	if (!$self->lock_rw) {
		warn "Unable to obtain exclusive lock";
		return;
	}
	if (!$self->setup_icf_fields) {
		warn "Failed to set up ICF";
		return;
	}
	if (!$self->init_dirs) {
		warn "Failed to initialise directories";
		return;
	}
	return 1;
}

sub config_setup_r {
	my $self = shift;
	$self->lock_r or return;
	$self->{CONFIG} = new Catkin::Config($self->{CONFIG_DIR}) or return;
	return 1;
}

sub unsetup {
	my $self = shift;
	$self->clear_icf_fields;
	$self->unlock;
}

sub config_unsetup {
	my $self = shift;
	delete $self->{CONFIG};
	$self->unlock;
}

# Sets up index, config and formatter fields.
sub setup_icf_fields {
	my $self = shift;
	$self->{CONFIG}    = new Catkin::Config($self->{CONFIG_DIR}) or return;
	$self->{INDEX}     = new Catkin::Index($self->config()) or return;
	$self->{FORMATTER} = new Catkin::Formatter($self->config(),$self->index()) or return;
	return 1;
}

# Clears index, config and formatter fields.
sub clear_icf_fields {
	my $self = shift;
	delete $self->{CONFIG};
	delete $self->{INDEX};
	delete $self->{FORMATTER};
}

sub lock_r {
	my $self = shift;
	my $fh = $self->{LOCK};
	my $lockfile = $self->{LOCK_FILE};
	unless (open($fh,"> $lockfile")) {
		warn "Could not open $lockfile\n";
		return;
	}
	unless (flock($fh,LOCK_SH)) {
		warn "Could get shared lock on $lockfile\n";
		return;
	}
	return 1;
}

sub lock_rw {
	my $self = shift;
	my $fh = $self->{LOCK};
	my $lockfile = $self->{LOCK_FILE};
	unless (open($fh,"> $lockfile")) {
		warn "Could not open $lockfile\n";
		return;
	}
	unless (flock($fh,LOCK_EX)) {
		warn "Could get shared lock on $lockfile\n";
		return;
	}
	return 1;
}

sub unlock {
	my $self = shift;
	my $fh = $self->{LOCK};
	return close $fh;
}

sub process_cgi {
	my $self = shift;
	my ($cgi) = @_;
	
	my $action = (grep(m/^action_/,$cgi->param()))[0] || "";
	$action =~ s/^action_//;
	if ($action eq "rebuild") {
		my $redirect;
		
		if (!$self->authenticate($cgi->cookie('session'))) {
			my $page = $self->login_page("Please log in.");
			return "Content-Type: text/html\n\n$page";
		}
		
		if (!($redirect = $self->rebuild())) {
			return $self->fail("Rebuild","Rebuild failed");
		}
		
		return "Status: 303\nLocation: $redirect\n\n";
	} elsif ($action eq "post_comment") {
		my $comment;
		my $redirect;
		
		if (!$cgi->param('text')) {
			return $self->fail("Post Comment","Please say something.");
		}
		
		my $text;
		if ($cgi->param('mode') eq 'sloppy_html') {
			my $sloppy = new HTML::Sloppy();
			$text = $sloppy->as_strict($cgi->param('text'));
		} else {
			$text = Text::Convert::html($cgi->param('text'));
		}
		
		$comment->{name} = $cgi->param('name');
		$comment->{email} = $cgi->param('email');
		$comment->{url} = $cgi->param('url');
		$comment->{title} = $cgi->param('title');
		$comment->{text} = $text;
		$comment->{original_text} = $cgi->param('text');
		$comment->{mode} = $cgi->param('mode');
		
		if (!($redirect = $self->post_comment($cgi->param('reply_to'),$comment))) {
			return $self->fail("Post Comment","Failed to add comment.");
		}
		
		return "Status: 303\nLocation: $redirect\n\n";
	} elsif ($action eq "reply_page") {
		my $comment;
		my $output;
		
		my $text;
		if ($cgi->param('mode') eq 'sloppy_html') {
			my $sloppy = new HTML::Sloppy();
			$text = $sloppy->as_strict($cgi->param('text'));
		} else {
			$text = Text::Convert::html($cgi->param('text'));
		}
		
		$comment->{name} = $cgi->param('name');
		$comment->{email} = $cgi->param('email');
		$comment->{url} = $cgi->param('url');
		$comment->{title} = $cgi->param('title');
		$comment->{text} = $text;
		$comment->{original_text} = $cgi->param('text');
		$comment->{mode} = $cgi->param('mode');
		$comment->{remember} = ($cgi->param('remember') eq 'on');
		
		if (!($output = $self->reply_page($cgi->param('reply_to'),$comment))) {
			return $self->fail("Reply Page","Failed to generate reply page.");
		}
		
		return "Content-Type: text/html\n\n$output";
	} elsif ($action eq "post_entry") {
		if (!$self->authenticate($cgi->cookie('session'))) {
			my $page = $self->login_page("Please log in.");
			return "Content-Type: text/html\n\n$page";
		}
		
		my $entry;
		my $redirect;
		
		$entry->{title} = $cgi->param('title');
		$entry->{text} = $cgi->param('text');
		$entry->{id} = $cgi->param('id');
		$entry->{mode} = $cgi->param('mode');
		
		if (!$entry->{text} || !$entry->{title}) {
			return $self->fail("Post Entry","Please supply both text and title.");
		}
		
		if (!($redirect = $self->post_entry($entry))) {
			return $self->fail("Post Entry","Failed to add entry.");
		}
		
		return "Status: 303\nLocation: $redirect\n\n";
	} elsif ($action eq "new_entry_page") {
		if (!$self->authenticate($cgi->cookie('session'))) {
			my $page = $self->login_page("Please log in.");
			return "Content-Type: text/html\n\n$page";
		}
		
		my $entry_data;
		
		$entry_data->{title} = $cgi->param('title');
		$entry_data->{text} = $cgi->param('text');
		$entry_data->{id} = $cgi->param('id');
		$entry_data->{mode} = $cgi->param('mode');
		
		my $page = $self->new_entry_page($entry_data);
		return "Content-Type: text/html\n\n$page";
	} elsif ($action eq "edit_entry") {
		if (!$self->authenticate($cgi->cookie('session'))) {
			my $page = $self->login_page("Please log in.");
			return "Content-Type: text/html\n\n$page";
		}
		
		my $entry;
		my $redirect;
		
		$entry->{title} = $cgi->param('title');
		$entry->{text} = $cgi->param('text');
		$entry->{id} = $cgi->param('id');
		$entry->{mode} = $cgi->param('mode');
		
		if (!$entry->{text} || !$entry->{title}) {
			return $self->fail("Edit Entry","Please supply both text and title.");
		}
		
		if (!($redirect = $self->edit_entry($entry))) {
			return $self->fail("Edit Entry","Failed to edit entry.");
		}
		
		return "Status: 303\nLocation: $redirect\n\n";
	} elsif ($action eq "edit_entry_page") {
		if (!$self->authenticate($cgi->cookie('session'))) {
			my $page = $self->login_page("Please log in.");
			return "Content-Type: text/html\n\n$page";
		}
		my $result;
		eval {
			my $entry_data;
			$entry_data->{title} = $cgi->param('title');
			$entry_data->{text} = $cgi->param('text');
			$entry_data->{id} = $cgi->param('id');
			$entry_data->{mode} = $cgi->param('mode');
			
			my $page = $self->edit_entry_page($cgi->param('id'),$entry_data);
			$result = "Content-Type: text/html\n\n$page";
		};
		if ($@) {
			return $self->fail('Edit Entry Page',"Failed to generate page: $@");
		} else {
			return $result;
		}
	} elsif ($action eq "delete_entry") {
		if (!$self->authenticate($cgi->cookie('session'))) {
			my $page = $self->login_page("Please log in.");
			return "Content-Type: text/html\n\n$page";
		}
		my $result;
		eval {
			my $id = $cgi->param('id');
			$self->delete_entry($id);
			$self->config_setup_r();
			$result = "Status: 303\nLocation: " . $self->config->key('catkin_url') . '?action_edit_blog_page=true' . "\n\n";
			$self->config_unsetup();
		};
		if ($@) {
			 return $self->fail('Delete Entry',"Failed to delete entry: $@");
		} else {
			return $result;
		}
	} elsif ($action eq "edit_comment") {
		if (!$self->authenticate($cgi->cookie('session'))) {
			my $page = $self->login_page("Please log in.");
			return "Content-Type: text/html\n\n$page";
		}
		
		my $result;
		eval {
			my $comment_data;
			$comment_data->{name}  = $cgi->param('name');
			$comment_data->{email} = $cgi->param('email');
			$comment_data->{url}   = $cgi->param('url');
			$comment_data->{title} = $cgi->param('title');
			$comment_data->{text}  = $cgi->param('text');
			$comment_data->{id}    = $cgi->param('id');
			$comment_data->{mode}  = $cgi->param('mode');
			
			if (!($comment_data->{text} || $comment_data->{title})) {
				die "Please supply text or a subject.\n";
			}
			
			my $redirect = $self->edit_comment($comment_data);
			$result = "Status: 303\nLocation: $redirect\n\n";
		};
		if ($@) {
			return $self->fail('Edit Comment',"Failed to edit comment: $@");
		} else {
			return $result;
		}
	} elsif ($action eq "edit_comment_page") {
		if (!$self->authenticate($cgi->cookie('session'))) {
			my $page = $self->login_page("Please log in.");
			return "Content-Type: text/html\n\n$page";
		}
		my $result;
		eval {
			my $comment_data;
			$comment_data->{name}  = $cgi->param('name');
			$comment_data->{email} = $cgi->param('email');
			$comment_data->{url}   = $cgi->param('url');
			$comment_data->{title} = $cgi->param('title');
			$comment_data->{text}  = $cgi->param('text');
			$comment_data->{id}    = $cgi->param('id');
			$comment_data->{mode}  = $cgi->param('mode');
			
			my $page = $self->edit_comment_page($cgi->param('id'),$comment_data);
			$result = "Content-Type: text/html\n\n$page";
		};
		if ($@) {
			return $self->fail('Edit Comment Page',"Failed to generate page: $@");
		} else {
			return $result;
		}
	} elsif ($action eq "delete_comment") {
		if (!$self->authenticate($cgi->cookie('session'))) {
			my $page = $self->login_page("Please log in.");
			return "Content-Type: text/html\n\n$page";
		}
		my $result;
		eval {
			my $id = $cgi->param('id');
			$self->delete_comment($id);
			$id =~ m:([^/]+)/[^/]+:;
			my $entry_id = $1;
			$self->config_setup_r();
			$result = "Status: 303\nLocation: " . $self->config->key('catkin_url') . "?action_edit_entry_page=true&id=$entry_id\n\n";
			$self->config_unsetup();
		};
		if ($@) {
			 return $self->fail('Delete Entry',"Failed to delete entry: $@");
		} else {
			return $result;
		}
	} elsif ($action eq "set_config") {
		if (!$self->authenticate($cgi->cookie('session'))) {
			my $page = $self->login_page("Please log in.");
			return "Content-Type: text/html\n\n$page";
		}
		my %keys;
		foreach my $param_name ($cgi->param()) {
			if ($param_name =~ /^config_(.+)/) {
				$keys{$1} = $cgi->param($param_name)
			}
		}
		$self->set_config(%keys) or warn "Something failed while setting config\n";
		
		$self->config_setup_r();
		my $result = "Status: 303\nLocation: " . $self->config->key('catkin_url') . '?action_settings_page=true' . "\n\n";
		$self->config_unsetup();
		return $result;
		
	} elsif ($action eq "set_password") {
		if (!$self->authenticate($cgi->cookie('session'))) {
			my $page = $self->login_page();
			return "Content-Type: text/html\n\n$page";
		}
		if (!$cgi->param('new_password1')) {
			return $self->fail("Change Password","Please supply a password.");
		}
		if ($cgi->param('new_password1') eq $cgi->param('new_password2')) {
			$self->set_password($cgi->param('new_password1'));
			$self->config_setup_r();
			my $result = "Status: 303\nLocation: " . $self->config->key('catkin_url') . "\n\n";
			$self->config_unsetup();
			return $result;
		} else {
			return $self->fail("Change Password","New passwords do not match.");
		}
	} elsif ($action eq "login") {
		if (my $session = $self->login($cgi->param('password'))) {
			my $page = $self->new_entry_page();
			my $cookie = "Set-Cookie: session=" . $session->id . "\n";
			return "Content-Type: text/html\n$cookie\n$page";
		} else {
			my $page = $self->login_page("Incorrect password.");
			return "Content-Type: text/html\n\n$page";
		}
	} elsif ($action eq "logout") {
		if (my $session = $self->authenticate($cgi->cookie('session'))) {
			$self->logout($session);
			my $page = $self->login_page("");
			return "Content-Type: text/html\n\n$page";
		}
	} elsif ($action eq "edit_blog_page") {
		if (!$self->authenticate($cgi->cookie('session'))) {
			my $page = $self->login_page("");
			return "Content-Type: text/html\n\n$page";
		}
		my $page = $self->edit_blog_page();
		return "Content-Type: text/html\n\n$page";
	} elsif ($action eq "settings_page") {
		if (!$self->authenticate($cgi->cookie('session'))) {
			my $page = $self->login_page("");
			return "Content-Type: text/html\n\n$page";
		}
		my $page = $self->settings_page();
		return "Content-Type: text/html\n\n$page";
	} elsif ($action eq "tools_page") {
		if (!$self->authenticate($cgi->cookie('session'))) {
			my $page = $self->login_page("");
			return "Content-Type: text/html\n\n$page";
		}
		my $page = $self->tools_page();
		return "Content-Type: text/html\n\n$page";
	} elsif (!$action) {
		if (!$self->authenticate($cgi->cookie('session'))) {
			my $page = $self->login_page("");
			return "Content-Type: text/html\n\n$page";
		}
		my $entry_data = {};
		my $page = $self->new_entry_page($entry_data,$cgi->param('password'));
		return "Content-Type: text/html\n\n$page";
	} else {
		return $self->fail("Unknown Action","Specified action ($action) was not recognised.");
	}
}

sub set_config {
	my $self = shift;
	my (%keys) = @_;
	$self->setup_rw();
	foreach my $key (keys(%keys)) {
		$self->config->key($key,$keys{$key}) if $self->config->key_exists($key);
	}
	$self->config->flush();
	$self->formatter->write($self->index->all_entries);
	$self->unsetup();
	return 1;
}

sub set_password {
	my $self = shift;
	my ($password) = @_;
	my @salt_chars = ('a','b','c','d','e','f','g','h','i','j','k','l','m','n','o','p','q','r','s','t','u','v','w','x','y','z','A','B','C','D','E','F','G','H','I','J','K','L','M','N','O','P','Q','R','S','T','U','V','W','X','Y','Z','0','1','2','3','4','5','6','7','8','9','.','/');
	my $salt = $salt_chars[int(rand(64))] . $salt_chars[int(rand(64))];
	my $passcrypt = crypt($password,$salt);
	$self->set_config(passcrypt => $passcrypt, password => '');
}

sub edit_blog_page {
	my $self = shift;
	$self->setup_r();
	my $cgi_url = $self->config->key('catkin_url');
	my $template_data = <<EOT;
<?xml version="1.0" encoding="utf-8"?>
<!DOCTYPE html PUBLIC "-//W3C//DTD XHTML 1.0 Strict//EN" "http://www.w3.org/TR/xhtml1/DTD/strict.dtd">
<html>
<head>
<title>Catkin: Edit Blog</title>
<style type="text/css">
[% INCLUDE css %]
form {
	margin: 0;
}
</style>
</head>
<body>
[% INCLUDE links %]
<div class="main">
<h1>Catkin: Edit Blog</h1>

<div class="section">
<h2>Entries</h2>
<table>
[% FOREACH entry = blog.entries.reverse %]
	<tr>
		<td>
			<a href="[% blog.config('blog_url') %][% entry.id %].[% blog.config('default_ext') %]">[% entry.title FILTER html %]</a>
		</td>
		<td>
			<form method="GET" action="[% blog.config('cgi_url') %]">
				<input type="hidden" name="id" value="[% entry.id %]"/>
				<input type="submit" name="action_edit_entry_page" value="Edit"/>
			</form>
		</td>
		<td>
			<form method="POST" action="[% blog.config('cgi_url') %]">
				<input type="hidden" name="id" value="[% entry.id %]"/>
				<input type="submit" name="action_delete_entry" value="Delete"/>
			</form>
		</td>
	</tr>
[% END %]
</table>
</div>

</div>
</body>
</html>
EOT
	my $template = new_tt();
	my $context = $self->generate_context();
	my $output;
	$template->process(\$template_data,$context,\$output) || warn "Error while processing template:\n" . $template->error;
	$self->unsetup();
	return $output;
}

sub settings_page {
	my $self = shift;
	$self->setup_r();
	my $cgi_url = $self->config->key('catkin_url');
	my $template_data = <<EOT;
<?xml version="1.0" encoding="utf-8"?>
<!DOCTYPE html PUBLIC "-//W3C//DTD XHTML 1.0 Strict//EN" "http://www.w3.org/TR/xhtml1/DTD/strict.dtd">
<html>
<head>
<title>Catkin: Settings</title>
<style type="text/css">
[% INCLUDE css %]
</style>
</head>
<body>
[% INCLUDE links %]
<div class="main">
<h1>Catkin: Settings</h1>

<div class="section">
<h2>Settings</h2>
<p>
Here you can alter some of the settings for your blog.
</p>
<form action="[% cgi_url FILTER html %]" method="post">
<p>Title:<br /> <input type="text" name="config_blog_title" value="[% config_blog_title FILTER html %]"/></p>
<p>Author:<br /> <input type="text" name="config_blog_author" value="[% config_blog_author FILTER html %]"/></p>
<p>Description:<br /> <input type="text" name="config_blog_description" value="[% config_blog_description FILTER html %]"/></p>
<p>Email:<br /> <input type="text" name="config_blog_email" value="[% config_blog_email FILTER html %]"/></p>
<p>URL:<br /> <input type="text" name="config_blog_url" value="[% config_blog_url FILTER html %]"/></p>
<p>Timezone:<br />
  <select name="config_timezone">
    [% FOR tz = timezones %]
	   <option value="[% tz %]"
	     [% IF tz == config_timezone %]
		   selected="selected"
		 [% END %]
	   >[% tz %]</option>
	[% END %]
  </select>
</p>
<p>
<input type="submit" name="action_set_config" value="Save Settings"/>
</p>
</form>
</div>

<div class="section">
<h2>Change password</h2>
<form action="[% cgi_url FILTER html %]" method="post">
<p>New Password:<br /> <input type="password" name="new_password1"/></p>
<p>Confirm New Password:<br /> <input type="password" name="new_password2"/></p>
<p><input type="submit" name="action_set_password" value="Change Password"/></p>
</form>
</div>

</div>
</body>
</html>
EOT
	my $template = new_tt();
	my $context = {};
	$context->{cgi_url}                 = $self->config->key('catkin_url');
	$context->{config_blog_title}       = $self->config->key('blog_title');
	$context->{config_blog_author}      = $self->config->key('blog_author');
	$context->{config_blog_description} = $self->config->key('blog_description');
	$context->{config_blog_email}       = $self->config->key('blog_email');
	$context->{config_blog_url}         = $self->config->key('blog_url');
	$context->{config_timezone}         = $self->config->key('timezone') || 'UTC';
	$context->{timezones}               = ['UTC',DateTime::TimeZone::all_names()];
	my $output;
	$template->process(\$template_data,$context,\$output) || warn "Error while processing template:\n" . $template->error;
	$self->unsetup();
	return $output;
}

sub tools_page {
	my $self = shift;
	$self->setup_r();
	my $cgi_url = $self->config->key('catkin_url');
	my $template_data = <<EOT;
<?xml version="1.0" encoding="utf-8"?>
<!DOCTYPE html PUBLIC "-//W3C//DTD XHTML 1.0 Strict//EN" "http://www.w3.org/TR/xhtml1/DTD/strict.dtd">
<html>
<head>
<title>Catkin: Tools</title>
<style type="text/css">
[% INCLUDE css %]
</style>
</head>
<body>
[% INCLUDE links %]
<div class="main">
<h1>Catkin: Tools</h1>

<div class="section">
<h2>Rebuild</h2>
<p>
After you have made changes to your templates or edited the data files by hand, you will need to rebuild the pages so that visitors to your site can see those changes.
</p>
<form action="[% cgi_url FILTER html %]" method="post">
<p>
<input type="submit" name="action_rebuild" value="Rebuild"/>
</p>
</form>
</div>

</div>
</body>
</html>
EOT
	my $template = new_tt();
	my $context = {};
	$context->{cgi_url} = $self->config->key('catkin_url');
	my $output;
	$template->process(\$template_data,$context,\$output) || warn "Error while processing template:\n" . $template->error;
	$self->unsetup();
	return $output;
}

sub login_page {
	my $self = shift;
	my ($error) = @_;
	$self->setup_r();
	my $cgi_url = $self->config->key('catkin_url');
	my $template_data = <<EOT;
<?xml version="1.0" encoding="utf-8"?>
<!DOCTYPE html PUBLIC "-//W3C//DTD XHTML 1.0 Strict//EN" "http://www.w3.org/TR/xhtml1/DTD/strict.dtd">
<html>
<head>
<title>Catkin: Login</title>
<style type="text/css">
[% INCLUDE css %]
</style>
</head>
<body>
[% INCLUDE links %]
<div class="main">
<h1>Catkin: Login</h1>

<div class="section">
<h2>Login</h2>
[% IF error %]
	<p class="error">$error</p>
[% END %]
<form action="[% cgi_url FILTER html %]" method="post">
<p>Password:<br /> <input type="password" name="password" value="[% password %]"/></p>
<p>
<input type="submit" name="action_login" value="Login"/>
</p>
</form>
</div>

</div>
</body>
</html>
EOT
	my $template = new_tt();
	my $context = {};
	$context->{cgi_url} = $self->config->key('catkin_url');
	$context->{error} = $error;
	my $output;
	$template->process(\$template_data,$context,\$output) || warn "Error while processing template:\n" . $template->error;
	$self->unsetup();
	return $output;
}

sub new_entry_page {
	my $self = shift;
	$self->setup_r();
	my ($entry_data) = @_;
	my $template_data = <<EOT;
<?xml version="1.0" encoding="utf-8"?>
<!DOCTYPE html PUBLIC "-//W3C//DTD XHTML 1.0 Strict//EN" "http://www.w3.org/TR/xhtml1/DTD/strict.dtd">
<html>
<head>
<title>Catkin: New Entry</title>
<style type="text/css">
[% INCLUDE css %]
</style>
</head>
<body>
[% INCLUDE links %]
<div class="main">
<h1>Catkin: New Entry</h1>

[% IF preview %]
<div class="section">
<h2>Preview</h2>
<p>
This is what your entry will look like when posted.
</p>
<div class="entry">
	<div class="header">
		<h3>[% preview.title FILTER html %]</h3>
		<p class="date">[% preview.date("%Y-%m-%d %H:%M:%S %Z") %]</p>
	</div>
	<div class="text">
		[% preview.text %]
	</div>
</div>

</div>
[% END %]

<div class="section">
<h2>New Entry</h2>
<form action="[% cgi_url FILTER html %]" method="post">
<p>Title:<br /> <input type="text" name="title" id="title" value="[% title FILTER html %]"/></p>
<p>Text:<br />
<textarea rows="30" cols="50" name="text" id="text">[% text FILTER html %]</textarea>
</p>
<p>Mode:<br />
[% FOREACH mode = modes %]
<input type="radio" name="mode" value="[% mode.name %]" id="mode_[% mode.name %]" [% IF mode.checked %]checked="checked" [% END %]/>[% mode.label FILTER html %]<br />
[% END %]
</p>
<p>
<input type="submit" name="action_new_entry_page" value="Preview Entry" />
<input type="submit" name="action_post_entry" value="Post Entry" />
</p>
</form>
</div>

</div>
</body>
</html>
EOT
	
	my $entry;
	if ($entry_data->{text} || $entry_data->{title}) {
		$entry = new Catkin::Entry();
		$entry->date(DateTime->now(time_zone => $self->config->key('timezone')));
		$entry->id($entry_data->{id});
		$entry->title($entry_data->{title});
		$entry_data->{mode} = 'verbatim' unless $entry_data->{mode};
		my $text;
		if ($entry_data->{mode} eq 'sloppy_html') {
			my $sloppy = new HTML::Sloppy();
			$text = $sloppy->as_strict($entry_data->{text});
		} elsif ($entry_data->{mode} eq 'text') {
			$text = Text::Convert::html($entry_data->{text});
		} else {
			$text = $entry_data->{text};
		}
		$entry->text($text);
	}
	
	$entry_data->{mode} = 'sloppy_html' unless $entry_data->{mode};
	my $template = new_tt();
	my $context = {};
	$context->{cgi_url}  = $self->config->key('catkin_url');
	$context->{id}       = $entry_data->{id};
	$context->{title}    = $entry_data->{title};
	$context->{text}     = $entry_data->{text};
	$context->{modes}    = [
		    {
			    name    => 'sloppy_html',
				label   => 'Sloppy HTML',
				checked => $entry_data->{mode} eq 'sloppy_html',
			},
		    {
			    name    => 'verbatim',
				label   => 'Verbatim',
				checked => $entry_data->{mode} eq 'verbatim',
			},
		    {
			    name    => 'text',
				label   => 'Text',
				checked => $entry_data->{mode} eq 'text',
			},
		];
	$context->{preview}  = $entry;
	my $output;
	$template->process(\$template_data,$context,\$output) || warn "Error while processing template:\n" . $template->error;
	$self->unsetup();
	return $output;
}

sub edit_entry_page {
	my $self = shift;
	$self->setup_r();
	my ($id,$entry_data) = @_;
	my $template_data = <<EOT;
[%- USE StripTags -%]
[%- BLOCK comment %]
	<tr>
		<td style="padding-left: [% level %]em">
			[% FILTER truncate(20) %]
				[%- FILTER collapse -%]
					[% IF comment.title %]
						[% comment.title FILTER html %]
					[% ELSE %]
						[% comment.text FILTER strip_tags %]
					[% END %]
				[%- END -%]
			[% END %]
		</td>
		<td>
			[% comment.name | html %]
		</td>
		<td>
			<form action="[% cgi_url FILTER html %]" method="GET">
				<input type="hidden" name="id" value="[% entry.id %]/[% comment.id %]"/>
				<input type="submit" name="action_edit_comment_page" value="Edit" />
			</form>
		</td>
		<td>
			<form action="[% cgi_url FILTER html %]" method="POST">
				<input type="hidden" name="id" value="[% entry.id %]/[% comment.id %]"/>
				<input type="submit" name="action_delete_comment" value="Delete" />
			</form>
		</td>
	</tr>
	[% FOREACH child = comment.comments; %]
		[% INCLUDE comment comment=child level=level+1 %]
	[% END %]
[% END -%]
<?xml version="1.0" encoding="utf-8"?>
<!DOCTYPE html PUBLIC "-//W3C//DTD XHTML 1.0 Strict//EN" "http://www.w3.org/TR/xhtml1/DTD/strict.dtd">
<html>
<head>
<title>Catkin: Edit Entry</title>
<style type="text/css">
[% INCLUDE css %]
</style>
</head>
<body>
[% INCLUDE links %]
<div class="main">
<h1>Catkin: Edit Entry</h1>

[% IF preview %]
<div class="section">
<h2>Preview</h2>
<p>
This is what your entry will look like when posted.
</p>
<div class="entry">
	<div class="header">
		<h3>[% preview.title FILTER html %]</h3>
		<p class="date">[% preview.date("%Y-%m-%d %H:%M:%S %Z") %]</p>
	</div>
	<div class="text">
		[% preview.text %]
	</div>
</div>

</div>
[% END %]

<div class="section">
<h2>Edit Entry</h2>
<form action="[% cgi_url FILTER html %]" method="post">
<p>Title:<br /> <input type="text" name="title" id="title" value="[% title FILTER html %]"/></p>
<p>Text:<br />
<textarea rows="30" cols="50" name="text" id="text">[% text FILTER html %]</textarea>
</p>
<p>Mode:<br />
[% FOREACH mode = modes %]
<input type="radio" name="mode" value="[% mode.name %]" id="mode_[% mode.name %]" [% IF mode.checked %]checked="checked" [% END %]/>[% mode.label FILTER html %]<br />
[% END %]
</p>
<p>
<input type="hidden" name="id" value="[% id %]" />
<input type="submit" name="action_edit_entry_page" value="Preview Entry" />
<input type="submit" name="action_edit_entry" value="Post Entry" />
</p>
</form>
</div>

<div class="section">
<h2>Edit Comments</h2>
<table>
[% FOREACH child = entry.comments; %]
	[% INCLUDE comment comment=child level=0 %]
[% END %]
</table>
</div>

</div>
</body>
</html>
EOT
	
	my $entry = $self->index->get_entry($id);
	if (!$entry) {
		warn "Could not find entry $id\n";
		return;
	}
	
	my $preview;
	if (defined($entry_data->{text}) || defined($entry_data->{title})) {
		$preview = new Catkin::Entry();
		$preview->date($entry->date);
		$preview->id($entry_data->{id});
		$preview->title($entry_data->{title});
		$entry_data->{mode} = 'verbatim' unless $entry_data->{mode};
		my $text;
		if ($entry_data->{mode} eq 'sloppy_html') {
			my $sloppy = new HTML::Sloppy();
			$text = $sloppy->as_strict($entry_data->{text});
		} elsif ($entry_data->{mode} eq 'text') {
			$text = Text::Convert::html($entry_data->{text});
		} else {
			$text = $entry_data->{text};
		}
		$preview->text($text);
	} else {
		$entry_data->{text} = $entry->text;
		$entry_data->{title} = $entry->title;
	}
	
	$entry_data->{mode} = 'sloppy_html' unless $entry_data->{mode};
	my $template = new_tt();
	my $context = {};
	$context->{cgi_url}  = $self->config->key('catkin_url');
	$context->{preview}  = $preview;
	$context->{entry}    = $entry;
	$context->{id}       = $id;
	$context->{title}    = $entry_data->{title};
	$context->{text}     = $entry_data->{text};
	$context->{modes}    = [
		    {
			    name    => 'sloppy_html',
				label   => 'Sloppy HTML',
				checked => $entry_data->{mode} eq 'sloppy_html',
			},
		    {
			    name    => 'verbatim',
				label   => 'Verbatim',
				checked => $entry_data->{mode} eq 'verbatim',
			},
		    {
			    name    => 'text',
				label   => 'Text',
				checked => $entry_data->{mode} eq 'text',
			},
		];
	my $output;
	$template->process(\$template_data,$context,\$output) or die "Error while processing template:\n" . $template->error;
	$self->unsetup();
	return $output;
}

sub edit_comment_page {
	my $self = shift;
	$self->setup_r();
	my ($id,$comment_data) = @_;
	my $template_data = <<EOT;
<?xml version="1.0" encoding="utf-8"?>
<!DOCTYPE html PUBLIC "-//W3C//DTD XHTML 1.0 Strict//EN" "http://www.w3.org/TR/xhtml1/DTD/strict.dtd">
<html>
<head>
<title>Catkin: Edit Comment</title>
<style type="text/css">
[% INCLUDE css %]
</style>
</head>
<body>
[% INCLUDE links %]
<div class="main">
<h1>Catkin: Edit Comment</h1>

[% IF preview %]
<div class="section">
<h2>Preview</h2>
<p>
This is what your comment will look like when posted.
</p>
<div class="entry">
	<div class="header">
		[% IF preview.title %]<h3>[% preview.title FILTER html %]</h3>[% END %]
		<p>
			[% IF preview.email %]
				<a href="mailto:[% preview.email FILTER html %]">[% preview.name FILTER html %]</a>
			[% ELSE %]
				[% preview.name FILTER html %]
			[% END %]
			[% IF preview.url %]
				- <a href="[% preview.url FILTER html %]">[% preview.url FILTER html %]</a>
			[% END %]
		</p>
		<p class="date">[% preview.date("%Y-%m-%d %H:%M:%S %Z") %]</p>
	</div>
	<div class="text">
		[% preview.text %]
	</div>
</div>

</div>
[% END %]

<div class="section">
<h2>Edit Entry</h2>
<form action="[% cgi_url FILTER html %]" method="post">
<p>Name:<br /> <input type="text" name="name" id="name" value="[% name FILTER html %]"/></p>
<p>Email:<br /> <input type="text" name="email" id="email" value="[% email FILTER html %]"/></p>
<p>URL:<br /> <input type="text" name="url" id="url" value="[% url FILTER html %]"/></p>
<p>Subject:<br /> <input type="text" name="title" id="title" value="[% title FILTER html %]"/></p>
<p>Text:<br />
<textarea rows="30" cols="50" name="text" id="text">[% text FILTER html %]</textarea>
</p>
<p>Mode:<br />
[% FOREACH mode = modes %]
<input type="radio" name="mode" value="[% mode.name %]" id="mode_[% mode.name %]" [% IF mode.checked %]checked="checked" [% END %]/>[% mode.label FILTER html %]<br />
[% END %]
</p>
<p>
<input type="hidden" name="id" value="[% id %]" />
<input type="submit" name="action_edit_comment_page" value="Preview Comment" />
<input type="submit" name="action_edit_comment" value="Post Comment" />
</p>
</form>
</div>

</div>
</body>
</html>
EOT
	
	my $comment = $self->get_comment($id);
	
	my $preview;
	if (defined($comment_data->{text}) || defined($comment_data->{title})) {
		$preview = new Catkin::Comment();
		$preview->date($comment->date);
		fill_comment_data($preview,$comment_data);
	} else {
		$comment_data->{name} = $comment->name;
		$comment_data->{url} = $comment->url;
		$comment_data->{email} = $comment->email;
		$comment_data->{text} = $comment->text;
		$comment_data->{title} = $comment->title;
	}
	
	$comment_data->{mode} = 'sloppy_html' unless $comment_data->{mode};
	my $template = new_tt();
	my $context = {};
	$context->{cgi_url}  = $self->config->key('catkin_url');
	$context->{preview}  = $preview;
	$context->{comment}  = $comment;
	$context->{id}       = $id;
	$context->{name}     = $comment_data->{name};
	$context->{email}    = $comment_data->{email};
	$context->{url}      = $comment_data->{url};
	$context->{title}    = $comment_data->{title};
	$context->{text}     = $comment_data->{text};
	$context->{modes}    = [
		    {
			    name    => 'sloppy_html',
				label   => 'Sloppy HTML',
				checked => $comment_data->{mode} eq 'sloppy_html',
			},
		    {
			    name    => 'verbatim',
				label   => 'Verbatim',
				checked => $comment_data->{mode} eq 'verbatim',
			},
		    {
			    name    => 'text',
				label   => 'Text',
				checked => $comment_data->{mode} eq 'text',
			},
		];
	my $output;
	$template->process(\$template_data,$context,\$output) or die "Error while processing template:\n" . $template->error;
	$self->unsetup();
	return $output;
}

sub generate_context {
	my $self = shift;
	my $context = {
		blog => {
			entries  => [$self->index->list],
			comments => [$self->index->comments],
			config   => sub { $self->config->key($_[0]) },
		},
	};
	return $context;
}

sub authenticate {
	my $self = shift;
	my ($id) = @_;
	$self->config_setup_r();
	my $session;
	# If we have a password or passcrypt, see if the user has logged in.
	# Otherwise log them in automatically.
	if ($self->config->key('password') || $self->config->key('passcrypt')) {
		$session = new Catkin::Session($self->config,$id);
	} else {
		$session = create Catkin::Session($self->config);
	}
	$self->config_unsetup();
	return $session;
}

sub login {
	my $self = shift;
	my ($password) = @_;
	$self->config_setup_r();
	my $session;
	if (my $crypt = $self->config->key('passcrypt')) {
		my $salt = substr($crypt,0,2);
		if (crypt($password,$salt) eq $crypt) {
			$session = create Catkin::Session($self->config);
		}
	} else {
		if ($self->config->key('password') eq $password) {
			$session = create Catkin::Session($self->config);
		}
	}
	$self->config_unsetup();
	return $session;
}

sub logout {
	my $self = shift;
	my ($session) = @_;
	return $session->delete();
}

sub fail {
	my $self = shift;
	my ($action,$message) = @_;
	$self->setup_r();
	my ($page,$content_type) = $self->formatter->failure_page($action,$message);
		
	my $result = "Content-Type: $content_type\n\n$page";
	$self->unsetup();
	return $result;
}

sub post_entry {
	my $self = shift;
	my ($entry_data) = @_;
	$self->setup_rw();
	my $id = $entry_data->{id};
	if (!$id) {
		$id = $self->idize(DateTime->now->strftime("%Y%m%d"),$entry_data->{title});
	}
	$id = $self->uniqify_id($id);
	my $entry = $self->index()->new_entry($id);
	fill_entry_data($entry,$entry_data);
	$self->index->commit_entry($entry);
	$self->index->flush;
	$self->formatter->write($self->index->affected_entries);
	$self->index->clear_affected_entries;
	my $result = $self->config->key('blog_url') . $id . "." . $self->config->key('default_ext');
	$self->unsetup();
	return $result;
}

sub edit_entry {
	my $self = shift;
	my ($entry_data) = @_;
	$self->setup_rw();
	my $id = $entry_data->{id};
	if (!$id) {
		warn "Can't edit entry without an id\n";
		return;
	}
	my $entry = $self->index()->get_entry($id);
	if (!$entry) {
		warn "Can't find entry $id\n";
		return;
	}
	fill_entry_data($entry,$entry_data);
	$self->index->commit_entry($entry);
	$self->index->flush;
	$self->formatter->write($self->index->affected_entries);
	$self->index->clear_affected_entries;
	my $result = $self->config->key('blog_url') . $id . "." . $self->config->key('default_ext');
	$self->unsetup();
	return $result;
}

sub delete_entry {
	my $self = shift;
	my ($id) = @_;
	$self->setup_rw();
	$self->index->delete_entry($id);
	$self->formatter->delete($id);
	$self->formatter->write($self->index->all_entries);
	$self->index->flush;
	my $result = $self->config->key('catkin_url') or die "Could not retrieve CGI url.\n";
	$self->unsetup();
	return $result;
}

sub edit_comment {
	my $self = shift;
	my ($comment_data) = @_;
	$self->setup_rw();
	my $id = $comment_data->{id} or die "Can't edit comment without an id\n";
	my $comment = $self->get_comment($id);
	my $entry = $comment->parent_entry();
	my $entry_id = $entry->id;
	fill_comment_data($comment,$comment_data);
	$self->index->commit_entry($entry);
	$self->index->flush;
	$self->formatter->write($self->index->affected_entries);
	$self->index->clear_affected_entries;
	my $result = $self->config->key('blog_url') . $entry_id . "." . $self->config->key('default_ext');
	$self->unsetup();
	return $result;
}

sub get_comment {
	my $self = shift;
	my ($id) = @_;
	my ($entry_id,$comment_id);
	if ($id =~ m: ([^/]+) / ([^/]+) :x) {
		$entry_id = $1;
		$comment_id = $2;
	} else {
		die "Invalid comment id $id\n";
	}
	my $entry = $self->index()->get_entry($entry_id) or die "Can't find entry $entry_id\n";
	my $comment = $entry->get_comment($comment_id) or die "Can't find comment $id\n";
	return $comment;
}

sub delete_comment {
	my $self = shift;
	my ($id) = @_;
	$self->setup_rw();
	my ($entry_id,$comment_id);
	if ($id =~ m:([^/]+)/([^/]+):) {
		$entry_id = $1;
		$comment_id = $2;
	} else {
		die "Failed to parse id: $id\n";
	}
	my $entry = $self->index->get_entry($entry_id) or die "Non-existent entry $entry_id";
	$entry->remove_comment($comment_id) or die "Failed to remove comment $comment_id";
	$self->index->commit_entry($entry);
	$self->formatter->write($self->index->affected_entries);
	$self->index->flush;
	my $result = $self->config->key('catkin_url') or die "Could not retrieve CGI url.\n";
	$self->unsetup();
	return $result;
}

sub fill_entry_data {
	my ($entry,$entry_data) = @_;
	my $text;
	if ($entry_data->{mode} && ($entry_data->{mode} eq 'sloppy_html')) {
		my $sloppy = new HTML::Sloppy();
		$text = $sloppy->as_strict($entry_data->{text});
	} elsif ($entry_data->{mode} && ($entry_data->{mode} eq 'text')) {
		$text = Text::Convert::html($entry_data->{text});
	} else {
		$text = $entry_data->{text};
	}
	$entry->text($text);
	$entry->title($entry_data->{title});
}

sub fill_comment_data {
	my ($comment,$comment_data) = @_;
	my $text;
	if ($comment_data->{mode} && ($comment_data->{mode} eq 'sloppy_html')) {
		my $sloppy = new HTML::Sloppy();
		$text = $sloppy->as_strict($comment_data->{text});
	} elsif ($comment_data->{mode} && ($comment_data->{mode} eq 'text')) {
		$text = Text::Convert::html($comment_data->{text});
	} else {
		$text = $comment_data->{text};
	}
	$comment->name($comment_data->{name} || '');
	$comment->email($comment_data->{email} || '');
	$comment->url($comment_data->{url} || '');
	$comment->title($comment_data->{title} || '');
	$comment->text($text);
}

sub post_comment {
	my $self = shift;
	my ($reply_to,$comment_data) = @_;
	$self->setup_rw();
	my $comment = new Catkin::Comment();
	$comment->date(DateTime->now(time_zone => $self->config->key('timezone')));
	fill_comment_data($comment,$comment_data);
	
	my ($reply_to_entry,$reply_to_comment);
	if ($reply_to =~ m:^([^/]+)(/([^/]+))?$:) {
		$reply_to_entry = $1;
		$reply_to_comment = $3;
	} else {
		carp "Invalid comment or entry id: $reply_to\n";
		return;
	}
	if ($self->index()->locate($reply_to_entry) == -1) {
		carp "No such entry $reply_to_entry\n";
		return;
	}
	my $parent_entry = $self->index()->get_entry($reply_to_entry);
	if ($reply_to_comment) {
		my $parent_comment = $parent_entry->get_comment($reply_to_comment);
		if (!$parent_comment) {
			carp "No such comment: $reply_to\n";
			return;
		}
	}
	$parent_entry->add_comment($comment,$reply_to_comment);
	$self->index->commit_entry($parent_entry);
	$self->index->flush;
	$self->formatter->write($self->index->affected_entries);
	$self->index->clear_affected_entries;
	my $result = $self->config->key('blog_url') . $reply_to_entry . "." . $self->config->key('default_ext');
	$self->unsetup();
	return $result;
}

sub reply_page {
	my $self = shift;
	my ($reply_to,$comment_data) = @_;
	$self->setup_r();
	
	$comment_data = undef unless ($comment_data->{original_text} or $comment_data->{title});
	
	my ($reply_to_entry,$reply_to_comment);
	if ($reply_to =~ m:^([^/]+)(/([^/]+))?$:) { # e.g. 20010203-foo/4
		$reply_to_entry = $1;
		$reply_to_comment = $3;
	} else {
		carp "Invalid comment or entry id: $reply_to\n";
		return;
	}
	
	my $parent;
	if ($self->index()->locate($reply_to_entry) == -1) {
		carp "No such entry $reply_to_entry\n";
		return;
	}
	my $parent_entry = $self->index()->get_entry($reply_to_entry);
	if ($reply_to_comment) {
		$parent = $parent_entry->get_comment($reply_to_comment);
		if (!$parent) {
			carp "No such comment: $reply_to\n";
			return;
		}
	} else {
		$parent = $parent_entry;
	} #FIXME: yuck. Can be redone now with $index->comments?
	
	my $result = $self->formatter->reply_page($parent,$comment_data,$reply_to);
	$self->unsetup();
	return $result;
}

sub rebuild {
	my $self = shift;
	$self->setup_rw();
	$self->formatter->write($self->index->all_entries);
	my $result = $self->config->key('blog_url');
	$self->unsetup();
	return $result;
}

sub idize {
	my $self = shift;
	my ($date,$id) = @_;
	$id = lc($id);
	$id =~ s/[^0-9a-z_]/_/g;
	while ($id =~ /__/) {
		$id =~ s/__/_/g;
	}
	$id =~ s/^_//;
	$id =~ s/_$//;
	$id = "$date-$id" if $date;
	return $id;
}

sub uniqify_id {
	my $self = shift;
	my ($orig_id) = @_;
	my $id = $orig_id;
	my $count = 0;
	while ($self->index->get_entry($id)) {
		$count++;
		$id = "${orig_id}_${count}";
	}
	return $id;
}

sub new_tt {
	my $template = new Template({
		BLOCKS => {
			css   => template_block_css(),
			links => template_block_links(),
		},
		PLUGIN_BASE => 'Catkin::Template::Plugin',
	});
}

sub template_block_links {
<<EOT;
<div class="links">
	<ul>
		<li><a href="[% cgi_url FILTER html %]?action_new_entry_page=true">New Entry</a></li>
		<li><a href="[% cgi_url FILTER html %]?action_edit_blog_page=true">Edit Blog</a></li>
		<li><a href="[% cgi_url FILTER html %]?action_settings_page=true">Settings</a></li>
		<li><a href="[% cgi_url FILTER html %]?action_tools_page=true">Tools</a></li>
		<li><a href="[% cgi_url FILTER html %]?action_logout=true">Logout</a></li>
	</ul>
</div>
EOT
}

sub template_block_css {
<<EOT;
body {
	background-color: #FFFFFF;
	color: #000000;
	margin: 0;
	padding: 0;
}

h1 {
	margin: 0;
	padding: 0.5em;
	font-size: 2em;
	background-color: #EEEEEE;
	border-bottom: dotted 2px #CCCCCC;
}

.section {
	background-color: #EEEEEE;
	border: dotted 2px #CCCCCC;
	padding: 1em;
	margin: 1.5em;
}

.entry {
	background-color: #FFFFFF;
	border: solid 1px #CCCCCC;
	padding: 1em;
}

.header h3 {
	margin-bottom: 0;
}

.header .date {
	margin-top: 0;
}

input[type="text"], input[type="password"] {
	width: 60%;
}

textarea {
	width: 100%;
	height: 30em;
}

.error {
	color: red;
}

.main {
	width: 80%;
	position: absolute;
	top: 0%;
	left: 20%;
}

.links {
	position: fixed;
	top: 0;
	bottom: 0;
	left: 0;
	width: 20%;
	background-color: #EEEEEE;
	border-right: dotted 2px #CCCCCC;
	padding-top: 4em;
}

.links ul {
	margin: 0;
	margin-left: 2em;
	padding: 0;
}

.links ul li {
	list-style-type: none;
	margin: 0;
	margin-bottom: 0.5em;
}
EOT
}

# subroutine _formatted_time
#
# Creates a string of the current date and time (UTC) in the format:
# "YYYY-MM-DD HH:MM:SS"
#
# Returns: date and time string
#
sub _formatted_time {
	my ($sec,$min,$hour,$mday,$mon,$year,$wday,$yday,$isdst) = gmtime;
	$year = $year + 1900;
	$mon = $mon + 1;
	return sprintf("%.4d-%.2d-%.2d %.2d:%.2d:%.2d", $year, $mon, $mday, $hour, $min, $sec);
}

1;
