# Catkin/Index.pm
# Copyright (C) 2002-2003 colin z robertson
# 
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
# 
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
# 
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

package Catkin::Index;

use strict;
use Data::Dumper;
use XML::Escape;# qw(escape);
use XML::DOM;
use Catkin::Entry;
use Catkin::Config;
use Catkin::Util;
use XML::DOMWrap; # qw(text_of get_child_elements get_child_element);
use Fcntl ':flock';
use CGI::Carp;
use Data::Dumper;
use File::Spec::Functions;
use DateTime;
use vars qw($AUTOLOAD);  # it's a package global

my %fields = (
	config => undef,
);

# subroutine new
# 
# Args: config object
# Returns: Index object
#
sub new {
    my $proto = shift;
    my $class = ref($proto) || $proto;
    my $self  = {
		_permitted => \%fields,
		%fields,
		entries => {},
		comments => {},
		affected_entries => {},
	};
    bless ($self, $class);
	
	my ($config) = @_;
	$self->config($config);
	
	$self->load;
    
	return $self;
}

sub load {
	my $self = shift;
	my $filename = catfile($self->config->key('priv_dir'),'data.xml');
	if (! -e $filename) {
		return;
	}; 
	my $dom = new XML::DOM::Parser();
	my $blog = get_child_element($dom->parsefile($filename),"blog");
	if (!$blog) { return }
	foreach my $entry_node (get_child_elements($blog,"entry")) {
		my $entry = new Catkin::Entry($entry_node);
		$self->insert_entry($entry);
	}
	return 1;
}

sub flush {
    my $self = shift;
	my $filename = catfile($self->config->key('priv_dir'),'data.xml');
	my $output = $self->as_xml;
	return Catkin::Util::atomic_write($filename, $output);
}

sub as_xml {
	my $self = shift;
	my $doc = XML::DOM::Document->new();
	my $root = $doc->createElement("blog");
	$doc->appendChild($root);
	$root->setAttribute('xmlns','http://rtnl.org.uk/catkin/markup');
	my @list = $self->list;
	foreach my $entry (@list) {
		$entry->node->setOwnerDocument($doc);
		$root->appendChild($entry->node);
	}
	return $doc->toString;
}

sub clear_caches {
	my $self = shift;
	$self->{cache_list} = undef;
	$self->{cache_comments} = undef;
}

sub all_entries {
	my $self = shift;
	my $size = $self->size;
	if ($size > 0) {
		return (0..$size-1);
	} else {
		return;
	}
}

sub size {
	my $self = shift;
	return scalar @{$self->list};
}

sub affected_entries {
	my $self = shift;
	my @entry_ids = keys(%{$self->{affected_entries}});
	return map {$_ = $self->locate($_)} @entry_ids;
}

sub clear_affected_entries {
	my $self = shift;
	$self->{affected_entries} = {};
}

sub remove_old_comments {
	my $self = shift;
	my ($id) = @_;
	my @old_comment_keys = grep /^\Q$id\E\/.*/, keys %{$self->{comments}};
	foreach my $key(@old_comment_keys) {
		delete $self->{comments}->{$key};
	}
}

sub commit_entry {
	my $self = shift;
	my ($entry) = @_;
	if (!$entry) {
		carp "Cannot commit null entry\n";
		return;
	}
	$self->insert_entry($entry);
	$self->{affected_entries}->{$entry->id} = 1;
}

sub insert_entry {
	my $self = shift;
	my ($entry) = @_;
	if (!$entry) {
		carp "Cannot insert null entry\n";
		return;
	}
	if (!$entry->id) {
		carp "Cannot insert entry with no id\n";
		return;
	}
	$self->remove_old_comments($entry->id);
	$self->{entries}->{$entry->id} = $entry;
	
	foreach my $comment ($entry->comments_flat_list) {
		$self->{comments}->{$entry->id."/".$comment->id} = $comment;
	}
	
	$self->clear_caches;
}

sub delete_entry {
	my $self = shift;
	my ($id) = @_;
	if (!$id) {
		die "No id supplied\n";
	}
	if (!$self->{entries}->{$id}) {
		die "Non-existent entry: $id\n";
	}
	$self->remove_old_comments($id);
	delete(${$self->{entries}}{$id});
	$self->clear_caches;
}

sub locate {
	my $self = shift;
	my ($id) = @_;
	my @list = $self->list;
	for (my $i = 0; $i != @list; $i++) {
		if ($list[$i]->id eq $id) {
			return $i;
		}
	}
	return -1;
}

sub get_entry {
	my $self = shift;
	my ($id) = @_;
	if ($self->{entries}->{$id}) {
		return $self->{entries}->{$id};
	} else {
		return;
	}
}

sub new_entry {
	my $self = shift;
	my ($id) = @_;
	if ($self->{entries}->{$id}) {
		warn "Cannot create new entry: $id already exists.\n";
		return;
	} else {
		my $entry = new Catkin::Entry();
		$entry->date(DateTime->now(time_zone => $self->config->key('timezone')));
		$entry->id($id);
		return $entry;
	}
}

sub filename {
	my $self = shift;
	return catfile($self->config->key('priv_dir'),"index.xml");
}

sub list {
	my $self = shift;
	if (!$self->{cache_list}) {
		my @list = sort {$a->date <=> $b->date} values(%{$self->{entries}});
		$self->{cache_list} = \@list;
	}
	return wantarray? @{$self->{cache_list}} : $self->{cache_list};
}

sub comments {
	my $self = shift;
	if (!$self->{cache_comments}) {
		my @list = sort {$a->date <=> $b->date} values(%{$self->{comments}});
		$self->{cache_comments} = \@list;
	}
	return wantarray? @{$self->{cache_comments}} : $self->{cache_comments};
}

sub AUTOLOAD {
	my $self = shift;
	my $type = ref($self) or croak "$self is not an object";

	my $name = $AUTOLOAD;
	$name =~ s/.*://;   # strip fully-qualified portion
	if ($name eq 'DESTROY') { return }

	unless (exists $self->{_permitted}->{$name} ) {
		croak "Can't access `$name' field in class $type";
	}

	if (@_) {
		return $self->{$name} = shift;
	} else {
		return $self->{$name};
	}
}

1;
