# Catkin/Interface.pm
# Copyright (C) 2002-2003 colin z robertson
# 
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
# 
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
# 
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

package Catkin::Interface;

use strict;
use Template;
use Catkin::Index;
use Catkin::Entry;
use Catkin::Comment;
use Catkin::Config;
use Symbol;
use IO::File;
use Fcntl ':flock';
use CGI::Carp;
use Data::Dumper;
use Catkin::Formatter;
use Text::Convert;
use XML::Escape;
use CGI;
use HTML::Sloppy;
use File::Spec::Functions;
use DateTime;
use DateTime::TimeZone;

sub new {
    my $proto = shift;
    my $class = ref($proto) || $proto;
    my $self  = {};
    bless ($self, $class);
	
	my ($config_dir) = @_;

	$self->{LOCK} = gensym();
	$self->{LOCK_FILE} = catfile($config_dir,"lock");
	$self->{CONFIG_DIR} = $config_dir;
	
	return $self;
}

sub index {
	my $self = shift;
	return $self->{INDEX};
}

sub config {
	my $self = shift;
	return $self->{CONFIG};
}

sub formatter {
	my $self = shift;
	return $self->{FORMATTER};
}

sub init_dirs {
	my $self = shift;
	#FIXME: probably only works for one level.
	mkdir($self->config->key('html_dir'),0777)
	  unless -e $self->config->key('html_dir');
	mkdir($self->config->key('priv_dir'),0777)
	  unless -e $self->config->key('priv_dir');
}

sub setup_r {
	my $self = shift;
	if (!$self->lock_r) {
		warn "Unable to obtain shared lock";
		return;
	}
	if (!$self->setup_icf_fields) {
		warn "Failed to set up ICF";
		return;
	}
	if (!$self->init_dirs) {
		warn "Failed to initialise directories";
		return;
	}
	return 1;
}

sub setup_rw {
	my $self = shift;
	if (!$self->lock_rw) {
		warn "Unable to obtain exclusive lock";
		return;
	}
	if (!$self->setup_icf_fields) {
		warn "Failed to set up ICF";
		return;
	}
	if (!$self->init_dirs) {
		warn "Failed to initialise directories";
		return;
	}
	return 1;
}

sub config_setup_r {
	my $self = shift;
	$self->lock_r or return;
	$self->{CONFIG} = new Catkin::Config($self->{CONFIG_DIR}) or return;
	return 1;
}

sub unsetup {
	my $self = shift;
	$self->clear_icf_fields;
	$self->unlock;
}

sub config_unsetup {
	my $self = shift;
	delete $self->{CONFIG};
	$self->unlock;
}

# Sets up index, config and formatter fields.
sub setup_icf_fields {
	my $self = shift;
	$self->{CONFIG}    = new Catkin::Config($self->{CONFIG_DIR}) or return;
	$self->{INDEX}     = new Catkin::Index($self->config()) or return;
	$self->{FORMATTER} = new Catkin::Formatter($self->config(),$self->index()) or return;
	return 1;
}

# Clears index, config and formatter fields.
sub clear_icf_fields {
	my $self = shift;
	delete $self->{CONFIG};
	delete $self->{INDEX};
	delete $self->{FORMATTER};
}

sub lock_r {
	my $self = shift;
	my $fh = $self->{LOCK};
	my $lockfile = $self->{LOCK_FILE};
	unless (open($fh,"> $lockfile")) {
		warn "Could not open $lockfile\n";
		return;
	}
	unless (flock($fh,LOCK_SH)) {
		warn "Could get shared lock on $lockfile\n";
		return;
	}
	return 1;
}

sub lock_rw {
	my $self = shift;
	my $fh = $self->{LOCK};
	my $lockfile = $self->{LOCK_FILE};
	unless (open($fh,"> $lockfile")) {
		warn "Could not open $lockfile\n";
		return;
	}
	unless (flock($fh,LOCK_EX)) {
		warn "Could get shared lock on $lockfile\n";
		return;
	}
	return 1;
}

sub unlock {
	my $self = shift;
	my $fh = $self->{LOCK};
	return close $fh;
}

sub process_cgi {
	my $self = shift;
	my ($cgi) = @_;
	
	my $action = (grep(m/^action_/,$cgi->param()))[0] || "";
	$action =~ s/^action_//;
	if ($action eq "rebuild") {
		my $redirect;
		
		if (!$self->authenticate($cgi->param('password'))) {
			return $self->fail("Rebuild","Incorrect password.");
		}
		
		if (!($redirect = $self->rebuild())) {
			return $self->fail("Rebuild","Rebuild failed");
		}
		
		return "Status: 303\nLocation: $redirect\n\n";
	} elsif ($action eq "post_comment") {
		my $comment;
		my $redirect;
		
		if (!$cgi->param('text')) {
			return $self->fail("Post Comment","Please say something.");
		}
		
		my $text;
		if ($cgi->param('mode') eq 'sloppy_html') {
			my $sloppy = new HTML::Sloppy();
			$text = $sloppy->as_strict($cgi->param('text'));
		} else {
			$text = Text::Convert::html($cgi->param('text'));
		}
		
		$comment->{name} = $cgi->param('name');
		$comment->{email} = $cgi->param('email');
		$comment->{url} = $cgi->param('url');
		$comment->{title} = $cgi->param('title');
		$comment->{text} = $text;
		$comment->{original_text} = $cgi->param('text');
		$comment->{mode} = $cgi->param('mode');
		
		if (!($redirect = $self->post_comment($cgi->param('reply_to'),$comment))) {
			return $self->fail("Post Comment","Failed to add comment.");
		}
		
		return "Status: 303\nLocation: $redirect\n\n";
	} elsif ($action eq "reply_page") {
		my $comment;
		my $output;
		
		my $text;
		if ($cgi->param('mode') eq 'sloppy_html') {
			my $sloppy = new HTML::Sloppy();
			$text = $sloppy->as_strict($cgi->param('text'));
		} else {
			$text = Text::Convert::html($cgi->param('text'));
		}
		
		$comment->{name} = $cgi->param('name');
		$comment->{email} = $cgi->param('email');
		$comment->{url} = $cgi->param('url');
		$comment->{title} = $cgi->param('title');
		$comment->{text} = $text;
		$comment->{original_text} = $cgi->param('text');
		$comment->{mode} = $cgi->param('mode');
		$comment->{remember} = ($cgi->param('remember') eq 'on');
		
		if (!($output = $self->reply_page($cgi->param('reply_to'),$comment))) {
			return $self->fail("Reply Page","Failed to generate reply page.");
		}
		
		return "Content-Type: text/html\n\n$output";
	} elsif ($action eq "post_entry") {
		my $entry;
		my $redirect;
		
		$entry->{title} = $cgi->param('title');
		$entry->{text} = $cgi->param('text');
		$entry->{id} = $cgi->param('id');
		$entry->{mode} = $cgi->param('mode');
		
		if (!$self->authenticate($cgi->param('password'))) {
			return $self->fail("Post Entry","Incorrect password.");
		}
		
		if (!$entry->{text} || !$entry->{title}) {
			return $self->fail("Post Entry","Please supply both text and title.");
		}
		
		if (!($redirect = $self->post_entry($entry))) {
			return $self->fail("Post Entry","Failed to add entry.");
		}
		
		return "Status: 303\nLocation: $redirect\n\n";
	} elsif ($action eq "new_entry_page") {
		my $entry_data;
		my $redirect;
		
		$entry_data->{title} = $cgi->param('title');
		$entry_data->{text} = $cgi->param('text');
		$entry_data->{id} = $cgi->param('id');
		$entry_data->{mode} = $cgi->param('mode');
		
		my $page = $self->new_entry_page($entry_data,$cgi->param('password'));
		return "Content-Type: text/html\n\n$page";
	} elsif ($action eq "set_config") {
		if (!$self->authenticate($cgi->param('password'))) {
			return $self->fail("Change Settings","Incorrect password.");
		}
		my %keys;
		foreach my $param_name ($cgi->param()) {
			if ($param_name =~ /^config_(.+)/) {
				$keys{$1} = $cgi->param($param_name)
			}
		}
		$self->set_config(%keys) or warn "Something failed while setting config\n";
		
		$self->config_setup_r();
		my $result = "Status: 303\nLocation: " . $self->config->key('catkin_url') . "\n\n";
		$self->config_unsetup();
		return $result;
		
	} elsif ($action eq "set_password") {
		if (!$self->authenticate($cgi->param('password'))) {
			return $self->fail("Change Password","Incorrect password.");
		}
		if ($cgi->param('new_password1') eq $cgi->param('new_password2')) {
			$self->set_config(password => $cgi->param('new_password1'));
			return "Status: 303\nLocation: " . $self->config->key('catkin_url') . "\n\n";
		} else {
			return $self->fail("Change Password","Supplied passwords do not match.");
		}
	} elsif ($action eq "") {
		my $page = $self->management_page();
		return "Content-Type: text/html\n\n$page";
	} else {
		return $self->fail("Unknown Action","Specified action ($action) was not recognised.");
	}
}

sub set_config {
	my $self = shift;
	my (%keys) = @_;
	$self->setup_rw();
	foreach my $key (keys(%keys)) {
		$self->config->key($key,$keys{$key}) if $self->config->key_exists($key);
	}
	$self->config->flush();
	$self->formatter->write($self->index->all_entries);
	$self->unsetup();
	return 1;
}

sub management_page {
	my $self = shift;
	$self->setup_r();
	my $cgi_url = $self->config->key('catkin_url');
	my $template_data = <<EOT;
<?xml version="1.0" encoding="utf-8"?>
<!DOCTYPE html PUBLIC "-//W3C//DTD XHTML 1.0 Strict//EN" "http://www.w3.org/TR/xhtml1/DTD/strict.dtd">
<html>
<head>
<title>Catkin: Manage your blog</title>
<style type="text/css">
body {
	background-color: #FFFFFF;
	color: #000000;
	margin: 0;
	padding: 0;
}

h1 {
	margin: 0 0 1em;
	padding: 0.5em;
	font-size: 2em;
	background-color: #EEEEEE;
	border-bottom: dotted 2px #CCCCCC;
}

.section {
	background-color: #EEEEEE;
	border: dotted 2px #CCCCCC;
	padding: 1em;
	margin: 2em 1em;
}

input[type="text"], input[type="password"] {
	width: 60%;
}

textarea {
	width: 100%;
	height: 30em;
}
</style>
</head>
<body>
<h1>Catkin: Manage your blog</h1>

<div class="section">
<p>
From here you can post new entries and edit various aspects of your blog. You will need to provide your password for every operation.
</p>
</div>

<div class="section">
<h2>New Entry</h2>
<form action="[% cgi_url FILTER html %]" method="post">
<p>Entry ID (optional):<br /> <input type="text" name="id"/></p>
<p>Title:<br /> <input type="text" name="title"/></p>
<p>Text:<br />
<textarea rows="30" cols="50" name="text" >[% text FILTER html %]</textarea>
</p>
<p>Mode:<br />
<input type="radio" name="mode" value="sloppy_html" checked="checked" />Sloppy HTML<br />
<input type="radio" name="mode" value="verbatim" />Verbatim<br />
<input type="radio" name="mode" value="text" />Text<br />
</p>
<p>Password:<br /> <input type="password" name="password" value="[% password FILTER html %]"/></p>
<p>
<input type="submit" name="action_new_entry_page" value="Preview Entry"/>
<input type="submit" name="action_post_entry" value="Post Entry"/>
</p>
</form>
</div>

<div class="section">
<h2>Rebuild</h2>
<p>
After you have made changes to your templates or edited the data files by hand, you will need to rebuild the pages so that visitors to your site can see those changes.
</p>
<form action="[% cgi_url FILTER html %]" method="post">
<p>Password:<br /> <input type="password" name="password"/></p>
<p>
<input type="submit" name="action_rebuild" value="Rebuild"/>
</p>
</form>
</div>

<div class="section">
<h2>Settings</h2>
<p>
Here you can alter some of the settings for your blog.
</p>
<form action="[% cgi_url FILTER html %]" method="post">
<p>Title:<br /> <input type="text" name="config_blog_title" value="[% config_blog_title FILTER html %]"/></p>
<p>Author:<br /> <input type="text" name="config_blog_author" value="[% config_blog_author FILTER html %]"/></p>
<p>Description:<br /> <input type="text" name="config_blog_description" value="[% config_blog_description FILTER html %]"/></p>
<p>Email:<br /> <input type="text" name="config_blog_email" value="[% config_blog_email FILTER html %]"/></p>
<p>URL:<br /> <input type="text" name="config_blog_url" value="[% config_blog_url FILTER html %]"/></p>
<p>Timezone:<br />
  <select name="config_timezone">
    [% FOR tz = timezones %]
	   <option value="[% tz %]"
	     [% IF tz == config_timezone %]
		   selected="selected"
		 [% END %]
	   >[% tz %]</option>
	[% END %]
  </select>
</p>
<p>Password:<br /> <input type="password" name="password"/></p>
<p>
<input type="submit" name="action_set_config" value="Save Settings"/>
</p>
</form>
</div>

<div class="section">
<h2>Change password</h2>
<form action="[% cgi_url FILTER html %]" method="post">
<p>Current Password:<br /> <input type="password" name="password"/></p>
<p>New Password:<br /> <input type="password" name="new_password1"/></p>
<p>Confirm New Password:<br /> <input type="password" name="new_password2"/></p>
<p><input type="submit" name="action_set_password" value="Change Password"/></p>
</form>
</div>

</body>
</html>
EOT
	my $template = new Template();
	my $context = {};
	$context->{cgi_url}                 = $self->config->key('catkin_url');
	$context->{config_blog_title}       = $self->config->key('blog_title');
	$context->{config_blog_author}      = $self->config->key('blog_author');
	$context->{config_blog_description} = $self->config->key('blog_description');
	$context->{config_blog_email}       = $self->config->key('blog_email');
	$context->{config_blog_url}         = $self->config->key('blog_url');
	$context->{config_timezone}         = $self->config->key('timezone');
	$context->{timezones}               = scalar DateTime::TimeZone::all_names();
	my $output;
	$template->process(\$template_data,$context,\$output) || warn "Error while processing template:\n" . $template->error;
	$self->unsetup();
	return $output;
}

sub new_entry_page {
	my $self = shift;
	$self->setup_r();
	my ($entry_data,$password) = @_;
	my $template_data = <<EOT;
<?xml version="1.0" encoding="utf-8"?>
<!DOCTYPE html PUBLIC "-//W3C//DTD XHTML 1.0 Strict//EN" "http://www.w3.org/TR/xhtml1/DTD/strict.dtd">
<html>
<head>
<title>Catkin: New Entry</title>
<style type="text/css">
body {
	background-color: #FFFFFF;
	color: #000000;
	margin: 0;
	padding: 0;
}

h1 {
	margin: 0 0 1em;
	padding: 0.5em;
	font-size: 2em;
	background-color: #EEEEEE;
	border-bottom: dotted 2px #CCCCCC;
}

.section {
	background-color: #EEEEEE;
	border: dotted 2px #CCCCCC;
	padding: 1em;
	margin: 2em 1em;
}

.entry {
	background-color: #FFFFFF;
	border: solid 1px #CCCCCC;
	padding: 1em;
}

.header h3 {
	margin-bottom: 0;
}

.header .date {
	margin-top: 0;
}

input[type="text"], input[type="password"] {
	width: 60%;
}

textarea {
	width: 100%;
	height: 30em;
}
</style>
</head>
<body>
<h1>Catkin: New Entry</h1>

[% IF preview %]
<div class="section">
<h2>Preview</h2>
<p>
This is what your entry will look like when posted.
</p>
<div class="entry">
	<div class="header">
		<h3>[% preview.title FILTER html %]</h3>
		<p class="date">[% preview.date("%Y-%m-%d %H:%M:%S %Z") %]</p>
	</div>
	<div class="text">
		[% preview.text %]
	</div>
</div>

</div>
[% END %]

<div class="section">
<h2>Post Entry</h2>
<form action="[% cgi_url FILTER html %]" method="post">
<p>Entry ID (optional):<br /> <input type="text" name="id" id="id" value="[% id FILTER html %]"/></p>
<p>Title:<br /> <input type="text" name="title" id="title" value="[% title FILTER html %]"/></p>
<p>Text:<br />
<textarea rows="30" cols="50" name="text" id="text">[% text FILTER html %]</textarea>
</p>
<p>Mode:<br />
[% FOREACH mode = modes %]
<input type="radio" name="mode" value="[% mode.name %]" id="mode_[% mode.name %]" [% IF mode.checked %]checked="checked" [% END %]/>[% mode.label FILTER html %]<br />
[% END %]
</p>
<p>Password:<br /> <input type="password" name="password" id="password" value="[% password FILTER html %]"/></p>
<p>
<input type="submit" name="action_new_entry_page" value="Preview Entry">
<input type="submit" name="action_post_entry" value="Post Entry">
</p>
</form>
</div>

</body>
</html>
EOT
	my $entry;
	if ($entry_data->{text} || $entry_data->{title}) {
		$entry = new Catkin::Entry();
		$entry->date(DateTime->now(time_zone => $self->config->key('timezone')));
		$entry->id($entry_data->{id});
		$entry->title($entry_data->{title});
		my $text;
		if ($entry_data->{mode} eq 'sloppy_html') {
			my $sloppy = new HTML::Sloppy();
			$text = $sloppy->as_strict($entry_data->{text});
		} elsif ($entry_data->{mode} eq 'text') {
			$text = Text::Convert::html($entry_data->{text});
		} else {
			$text = $entry_data->{text};
		}
		$entry->text($text);
	}
	
	my $template = new Template();
	my $context = {};
	$context->{cgi_url}  = $self->config->key('catkin_url');
	$context->{id}       = $entry_data->{id};
	$context->{title}    = $entry_data->{title};
	$context->{text}     = $entry_data->{text};
	$context->{modes}    = [
		    {
			    name    => 'sloppy_html',
				label   => 'Sloppy HTML',
				checked => $entry_data->{mode} eq 'sloppy_html',
			},
		    {
			    name    => 'verbatim',
				label   => 'Verbatim',
				checked => $entry_data->{mode} eq 'verbatim',
			},
		    {
			    name    => 'text',
				label   => 'Text',
				checked => $entry_data->{mode} eq 'text',
			},
		];
	$context->{preview}  = $entry;
	$context->{password} = $password;
	my $output;
	$template->process(\$template_data,$context,\$output) || warn "Error while processing template:\n" . $template->error;
	$self->unsetup();
	return $output;
}

sub authenticate {
	my $self = shift;
	my ($password) = @_;
	$self->config_setup_r();
	# Yes, I know that a one-way hash would be far better here.
	my $result = $self->config->key('password') eq $password;
	$self->config_unsetup();
	return $result;
}

sub fail {
	my $self = shift;
	my ($action,$message) = @_;
	$self->setup_r();
	my ($page,$content_type) = $self->formatter->failure_page($action,$message);
		
	my $result = "Content-Type: $content_type\n\n$page";
	$self->unsetup();
	return $result;
}

sub post_entry {
	my $self = shift;
	my ($entry_data) = @_;
	$self->setup_rw();
	my $id = $entry_data->{id};
	if (!$id) {
		$id = $self->idize(DateTime->now->strftime("%Y%m%d"),$entry_data->{title});
	}
	$id = $self->uniqify_id($id);
	my $entry = $self->index()->new_entry($id);
	my $text;
	if ($entry_data->{mode} && ($entry_data->{mode} eq 'sloppy_html')) {
		my $sloppy = new HTML::Sloppy();
		$text = $sloppy->as_strict($entry_data->{text});
	} elsif ($entry_data->{mode} && ($entry_data->{mode} eq 'text')) {
		$text = Text::Convert::html($entry_data->{text});
	} else {
		$text = $entry_data->{text};
	}
	$entry->text($text);
	$entry->title($entry_data->{title});
	$self->index->commit_entry($entry);
	$self->index->flush;
	$self->formatter->write($self->index->affected_entries);
	$self->index->clear_affected_entries;
	my $result = $self->config->key('blog_url') . $id . "." . $self->config->key('default_ext');
	$self->unsetup();
	return $result;
}

sub post_comment {
	my $self = shift;
	my ($reply_to,$comment_data) = @_;
	$self->setup_rw();
	my $comment = new Catkin::Comment();
	$comment->date(DateTime->now(time_zone => $self->config->key('timezone')));
	$comment->name($comment_data->{name} || '');
	$comment->email($comment_data->{email} || '');
	$comment->url($comment_data->{url} || '');
	$comment->title($comment_data->{title} || '');
	$comment->text($comment_data->{text} || '');
	
	my ($reply_to_entry,$reply_to_comment);
	if ($reply_to =~ m:^([^/]+)(/([^/]+))?$:) {
		$reply_to_entry = $1;
		$reply_to_comment = $3;
	} else {
		carp "Invalid comment or entry id: $reply_to\n";
		return;
	}
	if ($self->index()->locate($reply_to_entry) == -1) {
		carp "No such entry $reply_to_entry\n";
		return;
	}
	my $parent_entry = $self->index()->get_entry($reply_to_entry);
	if ($reply_to_comment) {
		my $parent_comment = $parent_entry->get_comment($reply_to_comment);
		if (!$parent_comment) {
			carp "No such comment: $reply_to\n";
			return;
		}
	}
	$parent_entry->add_comment($comment,$reply_to_comment);
	$self->index->commit_entry($parent_entry);
	$self->index->flush;
	$self->formatter->write($self->index->affected_entries);
	$self->index->clear_affected_entries;
	my $result = $self->config->key('blog_url') . $reply_to_entry . "." . $self->config->key('default_ext');
	$self->unsetup();
	return $result;
}

sub reply_page {
	my $self = shift;
	my ($reply_to,$comment_data) = @_;
	$self->setup_r();
	
	$comment_data = undef unless ($comment_data->{original_text} or $comment_data->{title});
	
	my ($reply_to_entry,$reply_to_comment);
	if ($reply_to =~ m:^([^/]+)(/([^/]+))?$:) { # e.g. 20010203-foo/4
		$reply_to_entry = $1;
		$reply_to_comment = $3;
	} else {
		carp "Invalid comment or entry id: $reply_to\n";
		return;
	}
	
	my $parent;
	if ($self->index()->locate($reply_to_entry) == -1) {
		carp "No such entry $reply_to_entry\n";
		return;
	}
	my $parent_entry = $self->index()->get_entry($reply_to_entry);
	if ($reply_to_comment) {
		$parent = $parent_entry->get_comment($reply_to_comment);
		if (!$parent) {
			carp "No such comment: $reply_to\n";
			return;
		}
	} else {
		$parent = $parent_entry;
	} #FIXME: yuck. Can be redone now with $index->comments?
	
	my $result = $self->formatter->reply_page($parent,$comment_data,$reply_to);
	$self->unsetup();
	return $result;
}

sub rebuild {
	my $self = shift;
	$self->setup_rw();
	$self->formatter->write($self->index->all_entries);
	my $result = $self->config->key('blog_url');
	$self->unsetup();
	return $result;
}

sub idize {
	my $self = shift;
	my ($date,$id) = @_;
	$id = lc($id);
	$id =~ s/[^0-9a-z_]/_/g;
	while ($id =~ /__/) {
		$id =~ s/__/_/g;
	}
	$id =~ s/^_//;
	$id =~ s/_$//;
	$id = "$date-$id" if $date;
	return $id;
}

sub uniqify_id {
	my $self = shift;
	my ($orig_id) = @_;
	my $id = $orig_id;
	my $count = 0;
	while ($self->index->get_entry($id)) {
		$count++;
		$id = "${orig_id}_${count}";
	}
	return $id;
}

# subroutine _formatted_time
#
# Creates a string of the current date and time (UTC) in the format:
# "YYYY-MM-DD HH:MM:SS"
#
# Returns: date and time string
#
sub _formatted_time {
	my ($sec,$min,$hour,$mday,$mon,$year,$wday,$yday,$isdst) = gmtime;
	$year = $year + 1900;
	$mon = $mon + 1;
	return sprintf("%.4d-%.2d-%.2d %.2d:%.2d:%.2d", $year, $mon, $mday, $hour, $min, $sec);
}

1;
