# Catkin/Config.pm
# Copyright (C) 2002 colin z robertson
# 
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
# 
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
# 
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

package Catkin::Config;

use strict;
use Catkin::Util;
use IO::File;
use CGI::Carp;
use XML::DOM;
use XML::DOMWrap;
use File::Spec::Functions;
use CGI;

my $catkin_url = "";
if ($ENV{SERVER_NAME} && $ENV{SCRIPT_NAME}) {
	$catkin_url = 'http://' . $ENV{SERVER_NAME} . $ENV{SCRIPT_NAME};
}

sub new {
    my $proto = shift;
    my $class = ref($proto) || $proto;
    my $self  = {
		keyset => {
			blog_author      => '',
			blog_email       => '',
			blog_title       => '',
			blog_description => '',
			blog_url         => '',
			html_dir         => '',
			reply_url        => '',
			default_ext      => 'html',
			password         => '',
			timezone         => 'UTC',
			catkin_url       => $catkin_url,
		},
		dirty_keys => {},
		dirty => 0,
		filename => "",
	};
    bless ($self, $class);
	
	my ($directory) = @_;
	$self->key('priv_dir',$directory);
	$self->{filename} = (catfile($directory,'config.xml'));
	$self->load_from_file($self->{filename});
	$self->mark_clean;
	return $self;
}

sub load_from_file {
	my $self = shift;
	my ($filename) = @_;
	if (! -e $filename) {
		carp "No config file " . $filename;
		return 0;
	}; 
	my $dom = new XML::DOM::Parser;
	my $doc = get_child_element($dom->parsefile($filename),"keys");
	if (!$doc) { return }
	foreach my $key (get_child_elements($doc,"key")) {
		$self->key($key->getAttribute("name"),text_of($key));
	}
	return 1;
}

sub save_to_file {
	my $self = shift;
	my $filename = $self->{filename};
	my $output = "<?xml version=\"1.0\"?>\n<keys>\n";
	foreach my $key (keys(%{$self->{dirty_keys}})) {
		$output .= "\t<key name=\"$key\">" . $self->key($key) . "</key>\n";
	}
	$output .= "</keys>\n";
	if (Catkin::Util::atomic_write($filename,$output)) {
		$self->mark_clean;
		return 1;
	} else {
		return 0;
	}
}

sub key {
	my $self = shift;
	if (@_ == 1) {
		my ($keyname) = @_;
		return $self->{keyset}->{$keyname};
	} elsif (@_ == 2) {
		my ($keyname,$value) = @_;
		$self->{keyset}->{$keyname} = $value;
		$self->{dirty_keys}->{$keyname} = 1;
		$self->{dirty} = 1;
		return $self->{keyset}->{$keyname};
	} else {
		carp "Incorrect number of arguments to key()";
	}
}

sub key_exists {
	my $self = shift;
	my ($key) = @_;
	return exists($self->{keyset}->{$key});
}

sub dirty {
	my $self = shift;
	return $self->{dirty};
}

sub mark_clean {
	my $self = shift;
	$self->{dirty} = 0;
}

sub flush {
	my $self = shift;
	if ($self->dirty) {
		$self->save_to_file() or carp("Could not save file");
	}
}

1;
