/* XmlField.java -- Represents a field serialized to XML.
 Copyright (C) 2005  The University of Sheffield.

 This file is part of the CASheW-s editor.

 The CASheW-s editor is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 2, or (at your option)
 any later version.
 
 The CASheW-s editor is distributed in the hope that it will be useful, but
 WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with The CASheW-s editor; see the file COPYING.  If not, write to the
 Free Software Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA
 02111-1307 USA.
*/

package nongnu.cashews.xml;

import java.io.ObjectStreamField;

/**
 * Represents a field that can be serialized to XML.  XML fields
 * allow the serialization process to be customised.
 *
 * @author Andrew John Hughes (gnu_andrew@member.fsf.org)
 */
public class XmlField
  extends ObjectStreamField
{

  /**
   * Flag to specify whether or not the field name is serialized
   * as an element.
   */
  private boolean fieldElement;

  /**
   * Flag to specify whether or not the class name is serialized
   * as an element.
   */
  private boolean classElement;

  /**
   * Constructs a new <code>XmlField</code> with the specified field
   * name and type.  The flags are set to a default of a shared field
   * with an element for the class name and none for the field name.
   *
   * @param name the name of the field.
   * @param type the type of the field.
   */
  public XmlField(String name, Class<?> type)
  {
    this(name, type, false);
  }

  /**
   * Constructs a new <code>XmlField</code> with the specified field
   * name and type.  The field is either shared or unshared depending
   * on the value supplied.  Shared fields are written using
   * <code>writeObject</code> and read using <code>readObject</code>,
   * while unshared fields are read and written using
   * <code>readUnshared</code> and <code>writeUnshared</code>
   * respectively.  The other flags are set to give the default
   * serialization of an element for the class name and none for the
   * field name.
   *
   * @param name the name of the field.
   * @param type the type of the field.
   * @param unshared a flag to specify whether or not this field is unshared.
   */
  public XmlField(String name, Class<?> type, boolean unshared)
  {
    this(name,type,unshared,false,true);
  }

  /**
   * Constructs a new <code>XmlField</code> with the specified field
   * name and type.  The field is either shared or unshared depending
   * on the value supplied.  Shared fields are written using
   * <code>writeObject</code> and read using <code>readObject</code>,
   * while unshared fields are read and written using
   * <code>readUnshared</code> and <code>writeUnshared</code>
   * respectively.  If a value of <code>true</code> is given for
   * <code>fieldElement</code>, then an element is included using the
   * name supplied here.  If a value of <code>true</code> is given for
   * <code>classElement</code>, then an element is included using the
   * name of the type.  This only is the default.
   *
   * @param name the name of the field.
   * @param type the type of the field.
   * @param unshared a flag to specify whether or not this field is unshared.
   * @param fieldElement a flag to specify whether to include an XML element
   *                     baring the name of the field.
   * @param classElement a flag to specify whether to include an XML element
   *                     baring the name of the class.
   */
  public XmlField(String name, Class<?> type, boolean unshared,
		  boolean fieldElement, boolean classElement)
  {
    super(name,type,unshared);
    setFieldElement(fieldElement);
    setClassElement(classElement);
  }

  /**
   * Constructs a new <code>XmlField</code> using the contents
   * of the specified <code>ObjectStreamField</code>.
   *
   * @param field the object stream field whose contents should
   *              be used to populate this <code>XmlField</code>.
   */
  public XmlField(ObjectStreamField field)
  {
    this(field.getName(), field.getType(), field.isUnshared());
  }

  /**
   * Sets the field element flag to the value supplied.  If this
   * is <code>true</code>, an XML element containing the field
   * name is included.
   *
   * @param fieldElement the new setting of the field element flag.
   */
  public void setFieldElement(boolean fieldElement)
  {
    this.fieldElement = fieldElement;
  }

  /**
   * Sets the class element flag to the value supplied.  If this
   * is <code>true</code>, an XML element containing the class
   * name is included.
   *
   * @param classElement the new setting of the class element flag.
   */
  public void setClassElement(boolean classElement)
  {
    this.classElement = classElement;
  }

  /**
   * Returns <code>true</code> if an XML element is included, which
   * bares the name of the field.
   *
   * @return <code>true</code> if the field name is serialized.
   */
  public boolean isFieldNameSerialized()
  {
    return fieldElement;
  }

  /**
   * Returns <code>true</code> if an XML element is included, which
   * bares the name of the class of which this field is an instance.
   *
   * @return <code>true</code> if the class name  is serialized.
   */
  public boolean isClassNameSerialized()
  {
    return classElement;
  }

  /**
   * Returns a <code>String</code> representation of this XML field.
   *
   * @return a textual representation.
   */
  public String toString()
  {
    return getClass().getName() +
      "[name=" +
      getName() +
      ",type=" +
      getType() +
      ",unshared=" +
      isUnshared() +
      ",fieldElement=" +
      fieldElement +
      ",classElement=" +
      classElement +
      "]";
  }

}
