/* Parser.java -- Basic parser for XML files.
 Copyright (C) 2005  The University of Sheffield.

 This file is part of the CASheW-s editor.

 The CASheW-s editor is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 2, or (at your option)
 any later version.
 
 The CASheW-s editor is distributed in the hope that it will be useful, but
 WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with The CASheW-s editor; see the file COPYING.  If not, write to the
 Free Software Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA
 02111-1307 USA.
*/

package nongnu.cashews.xml;

import java.io.File;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.util.logging.ConsoleHandler;
import java.util.logging.Handler;
import java.util.logging.Level;

import org.xml.sax.InputSource;
import org.xml.sax.SAXException;
import org.xml.sax.XMLReader;

/**
 * The parser handles the parsing of XML data from a variety of sources.
 * It is designed with extension points, so that more complex handlers
 * can be plugged in as necessary.
 * 
 * @author Andrew John Hughes (gnu_andrew@member.fsf.org)
 */
public class Parser
{

  /**
   * The reader, which interprets the XML file.
   *
   * @serial the XML reader.
   */
  private XMLReader reader;

  /**
   * The handler for the XML.
   *
   * @serial the XML handler.
   */
  private XmlBaseHandler xmlHandler;

  /**
   * Constructs a new XML-based parser, using the specified handler
   * for messages.
   *
   * @param logHandler the handler to use for log messages.
   * @throws SAXException if the parser fails to obtain an
   *         <code>XMLReader</code> instance.
   */
  public Parser(Handler logHandler)
    throws SAXException
  {
    this(logHandler, new XmlBaseHandler(logHandler));
  }

  /**
   * Constructs a new XML-based parser, using the specified handlers
   * for messages and XML parsing events.
   *
   * @param logHandler the handler to use for log messages.
   * @param xmlHandler the handler to use for XML parsing events.
   * @throws SAXException if the parser fails to obtain an
   *         <code>XMLReader</code> instance.
   */
  public Parser(Handler handler, XmlBaseHandler xmlHandler)
    throws SAXException
  {
    reader = org.xml.sax.helpers.XMLReaderFactory.createXMLReader();
    reader.setFeature("http://xml.org/sax/features/namespace-prefixes", true);
    System.out.println(reader.getFeature("http://xml.org/sax/features/namespace-prefixes"));
    reader.setContentHandler(xmlHandler);
    this.xmlHandler = xmlHandler;
  }

  /**
   * Parse an XML document from the specified system identifier (URI).
   *
   * @param systemId the URI of the XML document.
   * @throws SAXException if an error occurs in parsing.
   * @throws IOException if an error occurs in the underlying input.
   */
  public void parse(String systemId)
    throws IOException, SAXException
  {
    parse(new InputSource(systemId));
  }

  /**
   * Parse an XML document from the specified input source.
   *
   * @param source the source of the XML document.
   * @throws SAXException if an error occurs in parsing.
   * @throws IOException if an error occurs in the underlying input.
   */
  public void parse(InputSource source)
    throws IOException, SAXException
  {
    xmlHandler.setBaseURI(source.getSystemId());
    reader.parse(source);
  }

  /**
   * Parse an XML document from the specified file.
   *
   * @param file the file containing the XML document.
   * @throws SAXException if an error occurs in parsing.
   * @throws IOException if an error occurs in the underlying input.
   */
  public void parse(File file)
    throws IOException, SAXException
  {
    parse(file.toURI().toString());
  }

  /**
   * Retrieves the XML handler used by the parser.
   *
   * @return the XML event handler.
   */
  public XmlBaseHandler getHandler()
  {
    return xmlHandler;
  }

  /**
   * A simple test harness to parse the XML files supplied on the command
   * line.
   * 
   * @param args the command line arguments.
   * @throws SAXException if a <code>XMLReader</code> can't be obtained
   *         or an error occurs during parsing.
   * @throws IOException if an error occurs in the underlying input.
   * @throws FileNotFoundException if one of the specified files can't
   *         be found.
   */
  public static void main(String[] args)
    throws SAXException, IOException, FileNotFoundException
  {
    Handler handler = new ConsoleHandler();
    handler.setLevel(Level.FINE);
    Parser parser = new Parser(handler);
    for (int a = 0; a < args.length; ++a)
      parser.parse(args[a]);
  }

}
