/* WsdlHandler.java -- Handler for WSDL.
 Copyright (C) 2005  The University of Sheffield.

 This file is part of the CASheW-s editor.

 The CASheW-s editor is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 2, or (at your option)
 any later version.
 
 The CASheW-s editor is distributed in the hope that it will be useful, but
 WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with The CASheW-s editor; see the file COPYING.  If not, write to the
 Free Software Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA
 02111-1307 USA.
*/

package nongnu.cashews.wsdl;

import java.net.URISyntaxException;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.logging.Handler;
import java.util.logging.Logger;

import javax.xml.namespace.QName;

import nongnu.cashews.commons.PairMap;

import nongnu.cashews.language.grounding.MessagePart;
import nongnu.cashews.language.grounding.SoapMessage;
import nongnu.cashews.language.grounding.SoapOperation;

import nongnu.cashews.xml.XmlBaseHandler;

import org.xml.sax.Attributes;
import org.xml.sax.SAXException;

/**
 * This class deals with the parsing of WSDL, as defined by the <a
 * href="">WSDL 1.1</a> standard.  It turns an XML-based
 * representation of a WSDL service into a list of
 * <code>SoapOperation</code> objects.
 * 
 * @author Andrew John Hughes (gnu_andrew@member.fsf.org)
 * @see nongnu.cashews.language.grounding.SoapOperation
 */
public class WsdlHandler
  extends XmlBaseHandler
{
  
  /**
   * The WSDL namespace.
   */
  public static final String
    WSDL_NAMESPACE = "http://schemas.xmlsoap.org/wsdl/";

  /**
   * The WSDL SOAP namespace.
   */
  public static final String
    WSDL_SOAP_NAMESPACE = "http://schemas.xmlsoap.org/wsdl/soap/";
 
  /**
   * The resulting list of operations.
   *
   * @serial the operations described in the WSDL file.
   */
  private List<SoapOperation> operations;

  /**
   * A <code>Logger</code> instance to log events generated by the
     * parsing process.
     */
  private Logger wsdlLogger;

  /**
   * The current SOAP message being created.
   */
  private SoapMessage currentMessage;

  /**
   * A map of parsed SOAP messages.
   */
  private Map<String,SoapMessage> messages;

  /**
   * A map of parsed port types.
   */
  private Map<String,List<SoapOperation>> portTypeMap;

  /**
   * A map of parsed SOAP operations.
   */
  private PairMap<String,String,SoapOperation> operationMap;

  /**
   * The current port type name.
   */
  private String currentPortTypeName;

  /**
   * The current operation name.
   */
  private String currentOperationName;

  /**
   * The current SOAP operation.
   */
  private SoapOperation currentOperation;

  /**
   * Constructs a new <code>WSDLHandler</code>, using the specified
   * handler for log messages.
   *
   * @param handler the handler to use for log messages.
   */
  public WsdlHandler(Handler handler)
  {
    super(handler);
    wsdlLogger = Logger.getLogger("nongnu.cashews.wsdl.WsdlHandler");
    wsdlLogger.addHandler(handler);
    wsdlLogger.setLevel(handler.getLevel());
  }
  
  /**
   * Captures the start of the document and sets up the initial
   * state.
   */
  public void startDocument()
  {
    super.startDocument();
    operations = new ArrayList<SoapOperation>();
    currentMessage = null;
    messages = new HashMap<String,SoapMessage>();
    operationMap = new PairMap<String,String,SoapOperation>();
    portTypeMap = new HashMap<String,List<SoapOperation>>();
    currentPortTypeName = null;
    currentOperationName = null;
    currentOperation = null;
  }
  
  /**
   * Captures the start of an XML element.
   *
   * @param uri the namespace URI.
   * @param localName the local name of the element inside the namespace.
   * @param qName the local name qualified with the namespace URI.  This
   *              may or may not be provided, as we don't ask for namespace
   *              prefixes.
   * @param attributes the attributes of this element.
   * @throws SAXException if some error occurs in parsing.
   */
  public void startElement(String uri, String localName,
			   String qName, Attributes attributes)
    throws SAXException
  {
    super.startElement(uri, localName, qName, attributes);
    if (uri.equals(WSDL_NAMESPACE))
      {
	if (localName.equals("message"))
	  {
	    String name = attributes.getValue("","name");
	    if (name == null)
	      wsdlLogger.severe("Required message name not found.");
	    else
	      {
		currentMessage = new SoapMessage(null, name);
		messages.put(name, currentMessage);
	      }
	  }
	else if (localName.equals("part"))
	  {
	    String name = attributes.getValue("","name");
	    String type = attributes.getValue("","type");
	    if (name == null)
	      wsdlLogger.severe("Required part name not found.");
	    else
	      {
		try
		  {
		    MessagePart part = 
		      new MessagePart(name);
		    part.setName(null, name);
		    if (type != null)
		      part.setType(parseQName(type));
		    currentMessage.addPart(part);
		    wsdlLogger.fine("Added part: " + part);
		  }
		catch (URISyntaxException e)
		  {
		    wsdlLogger.severe("Couldn't interprete name " + name +
				      " as URI.");
		  }
	      }
	  }
	else if (localName.equals("portType"))
	  {
	    String value = attributes.getValue("","name");
	    if (value == null)
	      wsdlLogger.severe("Required port type name not found.");
	    else
	      {
		currentPortTypeName = value;
		portTypeMap.put(currentPortTypeName,
				new LinkedList<SoapOperation>());
		wsdlLogger.fine("Port type found: " + currentPortTypeName);
	      }
	  }
	else if (localName.equals("operation") && currentPortTypeName != null)
	  {
	    String value = attributes.getValue("","name");
	    if (value == null)
	      wsdlLogger.severe("Required port type operation name " +
				"not found.");
	    else
	      {
		currentOperationName = value;
		currentOperation = new SoapOperation();
		operationMap.put(currentPortTypeName, currentOperationName,
				 currentOperation);
		List<SoapOperation> operationList = 
		  portTypeMap.get(currentPortTypeName);
		if (operationList == null)
		  wsdlLogger.severe("Current port type name is not in the " +
				    "parsed port types map.");
		else
		  operationList.add(currentOperation);
		wsdlLogger.fine("Port type operation found: " +
				currentOperationName);
	      }
	  }
	else if (localName.equals("input") && currentOperationName != null)
	  {
	    QName message = parseQName(attributes.getValue("", "message"));
	    SoapMessage input = messages.get(message.getLocalPart());
	    if (input == null)
	      wsdlLogger.severe("Operation " + currentOperationName +
				" references non-existant message as input.");
	    else
	      currentOperation.setInputMessage(input);
	  }
	else if (localName.equals("output") && currentOperationName != null)
	  {
	    QName message = parseQName(attributes.getValue("", "message"));
	    SoapMessage output = messages.get(message.getLocalPart());
	    if (output == null)
	      wsdlLogger.severe("Operation " + currentOperationName +
				" references non-existant message as output.");
	    else
	      currentOperation.setOutputMessage(output);
	  }
	else if (localName.equals("port"))
	  {
	    String name = attributes.getValue("","name");
	    if (name == null)
	      wsdlLogger.severe("No port name specified in the service.");
	    else
	      {
		currentPortTypeName = name;
		wsdlLogger.fine("Port " + name + 
				" encountered in the service.");
	      }
	  }
      }
    else if (uri.equals(WSDL_SOAP_NAMESPACE))
      {
	if (localName.equals("address"))
	  {
	    String location = attributes.getValue("","location");
	    if (location == null)
	      wsdlLogger.severe("No location specified.");
	    else
	      {
		List<SoapOperation> operationList = 
		  portTypeMap.get(currentPortTypeName);
		if (operationList == null)
		  wsdlLogger.severe("Current port type name, " +
				    currentPortTypeName +
				    ", is not in the " +
				    "parsed port types map.");
		else
		  for (SoapOperation op : operationList)
		    {
		      try
			{
			  op.setEndpoint(location);
			  op.setNamespace(location);
			}
		      catch (URISyntaxException e)
			{
			  wsdlLogger.severe("Location URL, " + location +
					    ", is invalid.");
			}
		    }
	      }
	  }
      }
  }
  
  /**
   * Captures characters within an XML element.
   *
   * @param ch the array of characters.
   * @param start the start index of the characters to use.
   * @param length the number of characters to use from the start index on.
   * @throws SAXException if some error occurs in parsing.
   */
  public void characters(char[] ch, int start, int length)
    throws SAXException
  {
    super.characters(ch, start, length);
    String value = new String(ch, start, length).trim();
    if (value.length() == 0)
      return;
    wsdlLogger.finer("Characters: " + value);
  }
  
  /**
   * Captures the end of an XML element.
   *
   * @param uri the namespace URI.
   * @param localName the local name of the element inside the namespace.
   * @param qName the local name qualified with the namespace URI.  This
   *              may or may not be provided, as we don't ask for namespace
   *              prefixes.
   * @throws SAXException if some error occurs in parsing.
   */
  public void endElement(String uri, String localName,
			 String qName)
    throws SAXException
  {
    super.endElement(uri, localName, qName);
    if (uri.equals(WSDL_NAMESPACE))
      {
	if (localName.equals("message"))
	  wsdlLogger.fine("Created SOAP message: " + currentMessage);
	else if (localName.equals("portType"))
	  currentPortTypeName = null;
	else if (localName.equals("operation") && currentOperationName != null)
	  {
	    currentOperationName = null;
	    wsdlLogger.fine("Created SOAP operation: " + currentOperation);
	  }
      }
  }

  /**
   * Retrieves the graph created by the parsing process.
   *
   * @return the graph resulting from the parse.
   */
  public List<SoapOperation> getOperations()
  {
    return operations;
  }

  /**
   * Parses a <code>QName</code> from a string.
   *
   * @param qname the qualified name in <code>String</code> form to parse.
   * @return the parsed <code>QName</code>.
   */
  private QName parseQName(String qname)
  {
    String[] parts = qname.split(":");
    /* FIXME: Add namespace lookup */
    return new QName(null,parts[1],parts[0]);
  }

  /**
   * Captures the end of the document and sets up the final
   * state.
   */
  public void endDocument()
  {
    if (operationMap != null)
      for (SoapOperation op : operationMap.values())
	operations.add(op);
  }

}
