/* RDFURI.java -- Representation of a RDF URI.
 Copyright (C) 2005  The University of Sheffield.

 This file is part of the CASheW-s editor.

 The CASheW-s editor is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 2, or (at your option)
 any later version.
 
 The CASheW-s editor is distributed in the hope that it will be useful, but
 WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with The CASheW-s editor; see the file COPYING.  If not, write to the
 Free Software Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA
 02111-1307 USA.
*/

package nongnu.cashews.rdf;

import java.net.URI;
import java.net.URISyntaxException;

/**
 * <p>
 * This class represents an RDF URI.
 * </p>
 * 
 * @author Andrew John Hughes (gnu_andrew@member.fsf.org)
 * @see java.net.URI
 */
public class RDFURI 
  extends Node
  implements Predicate
{

  /**
   * The URI itself.
   * 
   * @see java.net.URI
   * @serial the uniform resource identifier.
   */
  private URI uri;

  /**
   * Constructs an <code>RDFURI</code> using the specified URI.
   *
   * @param uri the URI to use for this RDF URI.
   */
  public RDFURI(URI uri)
  {
    setURI(uri);
  }

  /**
   * Returns a textual representation of the RDF URI.
   *
   * @return a textual representation.
   */
  public String toString()
  {
    String superToString = super.toString();
    return superToString.substring(0, superToString.length() - 1) +
      ", uri = " +
      uri +
      "]";
  }

  /**
   * Returns a clone of the URI used by this RDF URI.
   *
   * @return a clone of the URI.
   */
  public URI getURI()
  {
    try
      {
	return new URI(uri.toString());
      }
    catch (URISyntaxException e)
      {
	throw new IllegalStateException("The URI is invalid.", e);
      }
  }

  /**
   * Returns a deep copy of this RDF URI.
   *
   * @return a clone of the RDF URI.
   */
  public RDFURI clone()
  {
    Object clonedObject = super.clone();
    RDFURI clone = (RDFURI) clonedObject;
    clone.setURI(getURI());
    return clone;
  }

  /**
   * Sets the URI used by this RDF URI to that specified.
   *
   * @param uri the new URI to use.
   */
  public void setURI(URI uri)
  {
    this.uri = uri;
  }

}
