/* Node.java -- Representation of a RDF node.
 Copyright (C) 2005  The University of Sheffield.

 This file is part of the CASheW-s editor.

 The CASheW-s editor is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 2, or (at your option)
 any later version.
 
 The CASheW-s editor is distributed in the hope that it will be useful, but
 WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with The CASheW-s editor; see the file COPYING.  If not, write to the
 Free Software Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA
 02111-1307 USA.
*/

package nongnu.cashews.rdf;

import java.net.URI;
import java.net.URISyntaxException;

/**
 * <p>
 * This interface represents an RDF node. A node can either be a URI reference
 * or a blank node.
 * </p>
 * 
 * @author Andrew John Hughes (gnu_andrew@member.fsf.org)
 * @see RDFURI
 * @see Blank
 */
public abstract class Node
  implements Subject, RDFObject
{

  /**
   * The optional type of this node.
   */
  private URI type;

  /**
   * Returns a deep copy of the node.
   *
   * @return a deep copy of the node.
   */
  public Node clone()
  {
    try
      {
	Object clonedObject = super.clone();
	Node clone = (Node) clonedObject;
	clone.setType(getType());
	return clone;
      }
    catch (CloneNotSupportedException e)
      {
	throw new IllegalStateException("Unexpected exception: " + e, e);
      }
  }
  /**
   * Sets the type of this node to that supplied.
   *
   * @param type the new type of this node.
   */
  public void setType(URI type)
  {
    this.type = type;
  }

  /**
   * Retrieves a clone of the current type of this node.
   *
   * @return a clone of the current type.
   */
  public URI getType()
  {
    try
      {
	return (type == null ? null : new URI(type.toString()));
      }
    catch (URISyntaxException e)
      {
	throw new IllegalStateException("The URI is invalid.", e);
      }
  }

  /**
   * Returns a textual representation of the node.
   *
   * @return a textual representation.
   */
  public String toString()
  {
    return getClass().getName() +
      "[type = " +
      type +
      "]";
  }

  /**
   * Sets the type of this node to the URI represented by the supplied
   * <code>String</code>.
   *
   * @param type the new type of the subject, in <code>String</code> form.
   */
  public void setType(String type)
  {
    try
      {
	setType(new URI(type));
      }
    catch (URISyntaxException e)
      {
	throw new IllegalArgumentException("Failed to parse URI: " + e, e);
      }
  }


}
