/* Produce.java -- Produces output for use in a composite process.
 Copyright (C) 2005  The University of Sheffield.

 This file is part of the CASheW-s editor.

 The CASheW-s editor is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 2, or (at your option)
 any later version.
 
 The CASheW-s editor is distributed in the hope that it will be useful, but
 WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with The CASheW-s editor; see the file COPYING.  If not, write to the
 Free Software Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA
 02111-1307 USA.
*/

package nongnu.cashews.language.process;

import java.net.URI;
import java.net.URISyntaxException;

import java.io.Serializable;

/**
 * Produces output for a <code>CompositeProcess</code> as a whole.
 * The producer performs data aggregation on the outputs of the internal
 * performances in order to produce one output for the process.
 *
 * @author Andrew John Hughes (gnu_andrew@member.fsf.org)
 * @see Performance
 */
public class Produce
  implements Serializable
{

  /**
   * The output of the composite process.
   *
   * @serial the composite process output.
   */
  private URI toOutput;

  /**
   * The internal performance where the output comes from.
   *
   * @serial the source performance.
   */
  private URI fromPerformance;

  /**
   * The output of the internal performance.
   *
   * @serial the internal performance input.
   */
  private URI fromOutput;

  /**
   * Construct a new <code>Produce</code> with the specified
   * source performance and the supplied source and destination
   * outputs.
   *
   * @param fromOutput the output on the composite process.
   * @param fromPerformance the performance to retrieve the output from.
   * @param toOutput the output on the destination performance.
   * @throws URISyntaxException if one of the supplied names is not a valid
   *                            URI.
   */
  public Produce(String fromOutput, String fromPerformance, String toOutput)
    throws URISyntaxException
  {
    setFromOutput(fromOutput);
    setFromPerformance(fromPerformance);
    setToOutput(toOutput);
  }

  /**
   * Construct a new <code>Produce</code> with the specified
   * source performance and the supplied source and destination
   * outputs.
   *
   * @param fromOutput the output on the composite process.
   * @param fromPerformance the performance to retrieve the output from.
   * @param toOutput the output on the destination performance.
   */
  public Produce(URI fromOutput, URI fromPerformance, URI toOutput)
  {
    setFromOutput(fromOutput);
    setFromPerformance(fromPerformance);
    setToOutput(toOutput);
  }

  /**
   * Sets the source output on the performance to that specified.
   *
   * @param fromOutput the new source output on the performance.
   * @throws URISyntaxException if the supplied output is not a valid URI.
   */
  public void setFromOutput(String fromOutput)
    throws URISyntaxException
  {
    setFromOutput(new URI(fromOutput));
  }

  /**
   * Sets the source output on the performance to that specified.
   *
   * @param fromOutput the new source output on the performance.
   */
  public void setFromOutput(URI fromOutput)
  {
    this.fromOutput = fromOutput;
  }

  /**
   * Sets the source performance to that specified.
   *
   * @param fromPerformance the new source performance.
   * @throws URISyntaxException if the supplied performance is not a valid URI.
   */
  public void setFromPerformance(String fromPerformance)
    throws URISyntaxException
  {
    setFromPerformance(new URI(fromPerformance));
  }

  /**
   * Sets the destination performance to that specified.
   *
   * @param fromPerformance the new destination performance.
   */
  public void setFromPerformance(URI fromPerformance)
  {
    this.fromPerformance = fromPerformance;
  }

  /**
   * Sets the destination output on the composite process to that specified.
   *
   * @param toOutput the new destination output on the composite process.
   * @throws URISyntaxException if the supplied output is not a valid URI.
   */
  public void setToOutput(String toOutput)
    throws URISyntaxException
  {
    setToOutput(new URI(toOutput));
  }

  /**
   * Sets the destination output on the composite process to that specified.
   *
   * @param toOutput the new destination input on the composite process.
   */
  public void setToOutput(URI toOutput)
  {
    this.toOutput = toOutput;
  }

  /**
   * Returns a <code>String</code> representation of this performance.
   *
   * @return a textual representation.
   */
  public String toString()
  {
    return getClass().getName() +
      "[fromOutput=" +
      fromOutput +
      ",fromPerformance=" +
      fromPerformance +
      ",toOutput=" +
      toOutput +
      "]";
  }

}

