/* Process.java -- An abstraction of the two process types.
 Copyright (C) 2005  The University of Sheffield.

 This file is part of the CASheW-s editor.

 The CASheW-s editor is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 2, or (at your option)
 any later version.
 
 The CASheW-s editor is distributed in the hope that it will be useful, but
 WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with The CASheW-s editor; see the file COPYING.  If not, write to the
 Free Software Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA
 02111-1307 USA.
*/

package nongnu.cashews.language.process;

import java.io.Serializable;

import java.net.URI;
import java.net.URISyntaxException;

/**
 * Takes a set of inputs, does some processing and returns a set of
 * outputs.  The internal makeup of the process is split between
 * <strong>atomic</strong> and <strong>composite</strong> process.
 * The former are exactly as the description implies, with some code
 * performing the operation on the inputs.  This code is accessed via
 * a <code>Grounding</code> to a web service.  In contrast, composite
 * processes appear the same to the casual observer, looking at the
 * interface, but the internal structure is the result of the
 * composition of several services.
 *
 * @author Andrew John Hughes (gnu_andrew@member.fsf.org)
 * @see AtomicProcess
 * @see CompositeProcess
 * @see Grounding
 */
public abstract class Process
  implements Cloneable, Serializable
{

  /**
   * The name of this process.
   *
   * @serial the process name.
   */
  private URI name;

  /**
   * Constructs a new <code>Process</code> with the specified name.
   * 
   * @param name the name of the process.
   * @throws URISyntaxException if the supplied name is not a valid URI.
   */
  public Process(String name)
    throws URISyntaxException
  {
    setName(name);
  }

  /**
   * Constructs a new <code>Process</code> with the specified name.
   * 
   * @param name the name of the process.
   */
  public Process(URI name)
  {
    setName(name);
  }

  /**
   * Sets the name of this process to that specified.
   *
   * @param name the name of the process.
   * @throws URISyntaxException if the supplied name is not a valid URI.
   */
  public void setName(String name)
    throws URISyntaxException
  {
    setName(new URI(name));
  }

  /**
   * Sets the name of this process to that specified.
   *
   * @param name the name of the process.
   */
  public void setName(URI name)
  {
    this.name = name;
  }

  /**
   * Returns a <code>String</code> representation of this process.
   *
   * @return a textual representation.
   */
  public String toString()
  {
    return getClass().getName() +
      "[name=" +
      name +
      "]";
  }

  /**
   * Returns a shallow clone of the process (the name is not cloned,
   * as <code>URI<code>s are immutable).
   *
   * @return a clone of the process.
   */
  public Process clone()
  {
    try
      {
	Object clone = super.clone();
	return (Process) clone;
      }
    catch (CloneNotSupportedException e)
      {
	throw new
	  IllegalStateException("Clone not allowed on Cloneable Process", e);
      }
  }

}

