/* Performance.java -- An instance of a process.
 Copyright (C) 2005  The University of Sheffield.

 This file is part of the CASheW-s editor.

 The CASheW-s editor is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 2, or (at your option)
 any later version.
 
 The CASheW-s editor is distributed in the hope that it will be useful, but
 WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with The CASheW-s editor; see the file COPYING.  If not, write to the
 Free Software Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA
 02111-1307 USA.
*/

package nongnu.cashews.language.process;

import java.net.URI;
import java.net.URISyntaxException;
import java.util.ArrayList;
import java.util.List;

import javax.xml.namespace.QName;

import nongnu.cashews.xml.Xmlizable;

/**
 * Represents an <em>instance</em> of an <strong>atomic</strong> or
 * <strong>composite</strong> process.  Each performance has a name
 * and an associated process, which it realises.  Each performance
 * also has the possibility of multiple <code>ValueData</code>s and
 * <code>ValueCollector</code>s which link inputs to constant data and
 * functions, respectively.
 * 
 * @author Andrew John Hughes (gnu_andrew@member.fsf.org)
 * @see Process
 * @see ValueData
 * @see ValueCollector
 */
public class Performance
  implements MultiPerformElement, Xmlizable
{

  /**
   * The name of this performance.
   *
   * @serial the performance name.
   */
  private URI name;

  /**
   * The process this performance references.
   *
   * @serial the referenced process.
   */
  private Process process;

  /**
   * The <code>ValueData</code>s for this performance.  A
   * <code>ValueData</code> links an input to some constant data.
   *
   * @serial the <code>ValueData</code>s.
   */
  private List<ValueData> valueDatas;

  /**
   * The <code>ValueCollector</code>s for this performance.  A
   * <code>ValueCollector</code> links an input to the result of
   * some function.
   *
   * @serial the <code>ValueCollector</code>s.
   */
  private List<ValueCollector> valueCollectors;

  /**
   * Constructs a new <code>Performance</code>.
   */
  private Performance()
  {
    valueDatas = new ArrayList<ValueData>();
    valueCollectors = new ArrayList<ValueCollector>();
  }

  /**
   * Constructs a new <code>Performance</code> with the
   * specified name.
   * 
   * @param name the performance name.
   * @throws URISyntaxException if the supplied name is not a valid URI.
   */
  public Performance(String name)
    throws URISyntaxException
  {
    this();
    setName(name);
  }

  /**
   * Constructs a new <code>Performance</code> with the specified name.
   * 
   * @param name the name of the process.
   */
  public Performance(URI name)
  {
    setName(name);
  }

  /**
   * Sets the name of this performance to that specified.
   *
   * @param name the name of the performance.
   * @throws URISyntaxException if the supplied name is not a valid URI.
   */
  public void setName(String name)
    throws URISyntaxException
  {
    setName(new URI(name));
  }

  /**
   * Sets the name of this performance to that specified.
   *
   * @param name the name of the performance.
   */
  public void setName(URI name)
  {
    this.name = name;
  }

  /**
   * Sets the process associated with this performance.
   *
   * @param process the new process.
   */
  public void setProcess(Process process)
  {
    this.process = process;
  }

  /**
   * Adds a new <code>ValueData</code> to this performance.
   *
   * @param valueData the value data to add.
   * @return <code>true</code> if the value data was added.
   */
  public boolean addValueData(ValueData valueData)
  {
    if (valueData == null)
      return false;
    valueDatas.add(valueData);
    return true;
  }

  /**
   * Adds a new <code>ValueCollector</code> to this performance.
   *
   * @param valueCollector the value collector to add.
   * @return <code>true</code> if the value collector was added.
   */
  public boolean addValueCollector(ValueCollector valueCollector)
  {
    if (valueCollector == null)
      return false;
    valueCollectors.add(valueCollector);
    return true;
  }

  /**
   * Retrieves the name of this performance.
   * 
   * @return the name of the performance.
   */
  public URI getName()
  {
   return name;   
  }

  /**
   * Retrieves a clone of this performance's process.
   *
   * @return the performance's process.
   */
  public Process getProcess()
  {
    return process.clone();
  }

  /**
   * Retrieves a shallow clone of the value datas for this performance.
   *
   * @return a shallow clone of the value data list.
   */
  public List<ValueData> getValueData()
  {
    return new ArrayList<ValueData>(valueDatas);
  }
  
  /**
   * Retrieves a shallow clone of the value collectors for this performance.
   *
   * @return a shallow clone of the value collector list.
   */
  public List<ValueCollector> getValueCollectors()
  {
    return new ArrayList<ValueCollector>(valueCollectors);
  }

  /**
   * Returns a <code>String</code> representation of this performance.
   *
   * @return a textual representation.
   */
  public String toString()
  {
    return getClass().getName() +
      "[name=" +
      name +
      ",process=" +
      process +
      ",valueDatas=" +
      valueDatas +
      ",valueCollectors=" +
      valueCollectors +
      "]";
  }

  /**
   * Returns "perform" as the element name for XML serialization.
   *
   * @return <code>perform</code>
   */
  public String getElementName()
  {
    return "perform";
  }

  /**
   * Returns null as this class needs no further namespace declarations.
   *
   * @return <code>null</code>.
   */
  public QName[] getDeclaredNamespaces()
  {
    return null;
  }

}
