/* Connection.java -- A connection between two performances for data flow.
 Copyright (C) 2005  The University of Sheffield.

 This file is part of the CASheW-s editor.

 The CASheW-s editor is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 2, or (at your option)
 any later version.
 
 The CASheW-s editor is distributed in the hope that it will be useful, but
 WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with The CASheW-s editor; see the file COPYING.  If not, write to the
 Free Software Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA
 02111-1307 USA.
*/

package nongnu.cashews.language.process;

import java.net.URI;
import java.net.URISyntaxException;

/**
 * Represents the connection between two performances, which provides
 * the necessary data flow from one to the other.  The output of
 * one performance is connected to a indexed input on another.
 *
 * @author Andrew John Hughes (gnu_andrew@member.fsf.org)
 */
public class Connection
  implements MultiPerformElement
{

  /**
   * The performance from which the output comes.
   *
   * @serial the source performance.
   */
  private URI fromPerformance;

  /**
   * The output on the source performance.
   *
   * @serial the source output.
   */
  private URI fromOutput;

  /**
   * The performance to where the output goes.
   *
   * @serial the destination performance.
   */
  private URI toPerformance;

  /**
   * The input on the destination performance.
   *
   * @serial the destination input.
   */
  private URI toInput;

  /**
   * The index of the input on the destination performance.
   *
   * @serial the index of the destination input.
   */
  private int toIndex;
  
  /**
   * Sets the name of the source performance to that specified.
   *
   * @param name the name of the source performance.
   * @throws URISyntaxException if the supplied name is not a valid URI.
   */
  public void setFromPerformance(String name)
    throws URISyntaxException
  {
    setFromPerformance(new URI(name));
  }
  
  /**
   * Sets the name of the source performance to that specified.
   *
   * @param name the name of the source performance.
   */
  public void setFromPerformance(URI name)
  {
    fromPerformance = name;
  }
  
  /**
   * Sets the name of the target performance to that specified.
   *
   * @param name the name of the target performance.
   * @throws URISyntaxException if the supplied name is not a valid URI.
   */
  public void setToPerformance(String name)
    throws URISyntaxException
  {
    setToPerformance(new URI(name));
  }
  
  /**
   * Sets the name of the target performance to that specified.
   *
   * @param name the name of the target performance.
   */
  public void setToPerformance(URI name)
  {
    toPerformance = name;
  }

}
