/* CompositeProcess.java -- A composite process which wraps 1+ performances.
 Copyright (C) 2005  The University of Sheffield.

 This file is part of the CASheW-s editor.

 The CASheW-s editor is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 2, or (at your option)
 any later version.
 
 The CASheW-s editor is distributed in the hope that it will be useful, but
 WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with The CASheW-s editor; see the file COPYING.  If not, write to the
 Free Software Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA
 02111-1307 USA.
*/

package nongnu.cashews.language.process;

import java.net.URI;
import java.net.URISyntaxException;

import java.util.LinkedList;
import java.util.List;

/**
 * <p>
 * Represents the amalgamation of one or more <code>Performance</code>s
 * using some control structure.  A set of <code>Performance</code>
 * can be run both serially and concurrently, and provision
 * is also made for looping and determinstic and non-deterministic
 * choice.
 * </p>
 * <p>
 * A <code>CompositeProcess</code> can also have one or more consumers,
 * and one or more producers.  These associate the inputs of the composite
 * processes with the inputs of the internal performances, and perform
 * aggregation of the output data, respectively.
 * </p>
 *
 * @author Andrew John Hughes (gnu_andrew@member.fsf.org)
 */
public class CompositeProcess
  extends Process
{

  /**
   * The control structure which defines how this process operates.
   *
   * @serial the control structure.
   */
  private CProcess controlStructure;

  /**
   * The consumers for this composite process.
   *
   * @serial the consumers.
   */
  private List<Consume> consumers;

  /**
   * The producers for this composite process.
   *
   * @serial the producers.
   */
  private List<Produce> producers;

  /**
   * Constructs a new <code>CompositeProcess</code> with the specified
   * name.
   *
   * @param name the name for this composite process.
   * @throws URISyntaxException if the supplied name is not a valid URI.
   */
  public CompositeProcess(String name)
    throws URISyntaxException
  {
    super(name);
    consumers = new LinkedList<Consume>();
    producers = new LinkedList<Produce>();
  }

  /**
   * Constructs a new <code>CompositeProcess</code> with the specified
   * name.
   *
   * @param name the name for this composite process.
   */
  public CompositeProcess(URI name)
  {
    super(name);
    consumers = new LinkedList<Consume>();
    producers = new LinkedList<Produce>();
  }

  /**
   * Sets the control structure used by this composite process.
   *
   * @param structure the structure to use.
   */
  public void setControlStructure(CProcess structure)
  {
    controlStructure = structure;
  }

  /**
   * Adds a consumer to this composite process.
   *
   * @param consumer the new consumer.
   */
  public boolean addConsumer(Consume consumer)
  {
    if (consumer == null)
      return false;
    consumers.add(consumer);
    return true;
  }

  /**
   * Adds a producer to this composite process.
   *
   * @param producer the new producer.
   */
  public boolean addProducer(Produce producer)
  {
    if (producer == null)
      return false;
    producers.add(producer);
    return true;
  }

  /**
   * Retrieves the control structure of this composite process.
   *
   * @return the control structure.
   */
  public CProcess getControlStructure()
  {
    return controlStructure;
  }

  /**
   * Returns a <code>String</code> representation of this process.
   *
   * @return a textual representation.
   */
  public String toString()
  {
    String superString = super.toString();
    return superString.substring(0, superString.length() - 1) +
      ", controlStructure=" +
      controlStructure +
      ", consumers=" +
      consumers +
      ", producers=" +
      producers +
      "]";
  }
     
}

