/* SoapMessage.java -- The message format for a SOAP envelope body.
 Copyright (C) 2005  The University of Sheffield.

 This file is part of the CASheW-s editor.

 The CASheW-s editor is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 2, or (at your option)
 any later version.
 
 The CASheW-s editor is distributed in the hope that it will be useful, but
 WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with The CASheW-s editor; see the file COPYING.  If not, write to the
 Free Software Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA
 02111-1307 USA.
*/

package nongnu.cashews.language.grounding;

import java.io.Serializable;

import java.util.LinkedList;
import java.util.List;

import javax.xml.namespace.QName;

/**
 * Specifies the format of the message body within a SOAP
 * envelope.
 *
 * @author Andrew John Hughes (gnu_andrew@member.fsf.org)
 */
public class SoapMessage
  implements Serializable
{

  /**
   * The name of the SOAP message.
   *
   * @serial the message name.
   */
  private QName name;

  /**
   * The separate parts which compose the message.
   *
   * @serial the message parts.
   */
  private List<MessagePart> parts;

  /**
   * Constructs a new SOAP message.
   */
  private SoapMessage()
  {
    parts = new LinkedList<MessagePart>();
  }

  /**
   * Constructs a new SOAP message with the specified qualified
   * name, constructed from the supplied namespace URI and
   * local part.
   *
   * @param namespaceURI the namespace URI of this SOAP message's name.
   * @param localPart the local part of this SOAP message's name.
   */
  public SoapMessage(String namespaceURI, String localPart)
  {
    this();
    setName(new QName(namespaceURI, localPart));
  }

  /**
   * Constructs a new SOAP message with the specified qualified
   * name, constructed from the supplied namespace URI, local part
   * and prefix.
   *
   * @param namespaceURI the namespace URI of this SOAP message's name.
   * @param localPart the local part of this SOAP message's name.
   * @param prefix the prefix of this SOAP message's name.
   */
  public SoapMessage(String namespaceURI, String localPart, String prefix)
  {
    this();
    setName(namespaceURI, localPart, prefix);
  }

  /**
   * Constructs a new SOAP message with the specified name.
   *
   * @param name the name of this SOAP message.
   */
  public SoapMessage(QName name)
  {
    this();
    setName(name);
  }

  /**
   * Sets the qualified name of this SOAP message to that constructed
   * from the supplied namespace URI and local part.
   *
   * @param namespaceURI the namespace URI of this SOAP message's name.
   * @param localPart the local part of this SOAP message's name.
   */
  public void setName(String namespaceURI, String localPart)
  {
    setName(new QName(namespaceURI, localPart));
  }

  /**
   * Sets the qualified name of this SOAP message to that constructed
   * from the supplied namespace URI, local part and prefix.
   *
   * @param namespaceURI the namespace URI of this SOAP message's name.
   * @param localPart the local part of this SOAP message's name.
   * @param prefix the prefix of this SOAP message's name.
   */
  public void setName(String namespaceURI, String localPart, String prefix)
  {
    setName(new QName(namespaceURI, localPart, prefix));
  }

  /**
   * Sets the name of this SOAP message to that specified.
   *
   * @param name the name of this SOAP message.
   */
  public void setName(QName name)
  {
    this.name = name;
  }

  /**
   * Adds a new part to the SOAP message.
   *
   * @param part the new part to add.
   */
  public boolean addPart(MessagePart part)
  {
    if (part == null)
      return false;
    parts.add(part);
    return true;
  }

  /**
   * Retrieves the qualified name of this SOAP message.
   *
   * @return the qualified name of this SOAP message.
   */
  public QName getName()
  {
    return name;
  }

  /**
   * Retrieves a shallow clone of the parts of this message.
   *
   * @return a shallow clone of the message parts.
   */
  public List<MessagePart> getParts()
  {
    return new LinkedList<MessagePart>(parts);
  }

  /**
   * Returns a <code>String</code> representation of this SOAP message.
   *
   * @return a textual representation.
   */
  public String toString()
  {
    return getClass().getName() +
      "[name=" +
      name +
      ",parts=" +
      parts +
      "]";
  }

}

