/* MessagePart.java -- Part of a SOAP message body.
 Copyright (C) 2005  The University of Sheffield.

 This file is part of the CASheW-s editor.

 The CASheW-s editor is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 2, or (at your option)
 any later version.
 
 The CASheW-s editor is distributed in the hope that it will be useful, but
 WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with The CASheW-s editor; see the file COPYING.  If not, write to the
 Free Software Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA
 02111-1307 USA.
*/

package nongnu.cashews.language.grounding;

import java.net.URI;
import java.net.URISyntaxException;

import javax.xml.namespace.QName;

import nongnu.cashews.xml.Xmlizable;

/**
 * Represents a single part of a SOAP message body.  This specifies
 * the name and type of the element.
 *
 * @author Andrew John Hughes (gnu_andrew@member.fsf.org)
 */
public class MessagePart
  implements Xmlizable
{

  /**
   * The URI which relates this part to the calculus.
   *
   * @serial the URI relating this part to the calculus.
   */
  private URI uri;

  /**
   * The name of the SOAP message part.
   *
   * @serial the message part name.
   */
  private QName name;

  /**
   * The type of the SOAP message part.
   *
   * @serial the message part's type.
   */
  private QName type;

  /**
   * Constructs a new message part with the specified URI.
   *
   * @param uri the uri of this message part.
   * @throws URISyntaxException if the supplied name is not a valid URI.
   */
  public MessagePart(String uri)
    throws URISyntaxException
  {
    setURI(uri);
  }

  /**
   * Constructs a new message part with the specified URI.
   *
   * @param uri the uri of this message part.
   */
  public MessagePart(URI uri)
  {
    setURI(uri);
  }

  /**
   * Sets the URI of this message part to that specified.
   *
   * @param uri the uri of this message part.
   * @throws URISyntaxException if the supplied name is not a valid URI.
   */
  public void setURI(String uri)
    throws URISyntaxException
  {
    setURI(new URI(uri));
  }

  /**
   * Sets the URI of this message part to that specified.
   *
   * @param uri the uri of this message part.
   */
  public void setURI(URI uri)
  {
    this.uri = uri;
  }

  /**
   * Sets the qualified name of this SOAP message to that constructed
   * from the supplied namespace URI and local part.
   *
   * @param namespaceURI the namespace URI of this SOAP message's name.
   * @param localPart the local part of this SOAP message's name.
   */
  public void setName(String namespaceURI, String localPart)
  {
    setName(new QName(namespaceURI, localPart));
  }

  /**
   * Sets the qualified name of this SOAP message to that constructed
   * from the supplied namespace URI, local part and prefix.
   *
   * @param namespaceURI the namespace URI of this SOAP message's name.
   * @param localPart the local part of this SOAP message's name.
   * @param prefix the prefix of this SOAP message's name.
   */
  public void setName(String namespaceURI, String localPart, String prefix)
  {
    setName(new QName(namespaceURI, localPart, prefix));
  }

  /**
   * Sets the name of this SOAP message to that specified.
   *
   * @param name the name of this SOAP message.
   */
  public void setName(QName name)
  {
    this.name = name;
  }

  /**
   * Sets the type of this SOAP message to that constructed
   * from the supplied namespace URI and local part.
   *
   * @param namespaceURI the namespace URI of this SOAP message's type.
   * @param localPart the local part of this SOAP message's type.
   */
  public void setType(String namespaceURI, String localPart)
  {
    setType(new QName(namespaceURI, localPart));
  }

  /**
   * Sets the qualified type of this SOAP message to that constructed
   * from the supplied namespace URI, local part and prefix.
   *
   * @param namespaceURI the namespace URI of this SOAP message's type.
   * @param localPart the local part of this SOAP message's type.
   * @param prefix the prefix of this SOAP message's type.
   */
  public void setType(String namespaceURI, String localPart, String prefix)
  {
    setType(new QName(namespaceURI, localPart, prefix));
  }

  /**
   * Sets the type of this SOAP message part to that specified.
   *
   * @param type the type of this SOAP message.
   */
  public void setType(QName type)
  {
    this.type = type;
  }

  /**
   * Returns the name of this SOAP message part.
   *
   * @return the name of this SOAP message part.
   */
  public QName getName()
  {
    return name;
  }

  /**
   * Returns a <code>String</code> representation of this SOAP message.
   *
   * @return a textual representation.
   */
  public String toString()
  {
    return getClass().getName() +
      "[uri=" +
      uri +
      ",name=" +
      name +
      ",type=" +
      type +
      "]";
  }

  /**
   * Returns "part" as the element name for XML serialization.
   *
   * @return <code>part</code>
   */
  public String getElementName()
  {
    return "part";
  }

  /**
   * Returns null as this class needs no further namespace declarations.
   *
   * @return <code>null</code>.
   */
  public QName[] getDeclaredNamespaces()
  {
    return null;
  }

}

