/* PairSet.java -- A set of heterogenous pairs.
 Copyright (C) 2005  The University of Sheffield.

 This file is part of the CASheW-s editor.

 The CASheW-s editor is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 2, or (at your option)
 any later version.
 
 The CASheW-s editor is distributed in the hope that it will be useful, but
 WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with The CASheW-s editor; see the file COPYING.  If not, write to the
 Free Software Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA
 02111-1307 USA.
*/

package nongnu.cashews.commons;

import java.util.HashSet;

/**
 * A set consisting of pairs of values.
 * 
 * @author Andrew John Hughes (gnu_andrew@member.fsf.org)
 * @see java.util.Set
 */
public class PairSet<A,B>
  extends HashSet<Pair<A,B>>
{

  /**
   * Serialization ID.
   */
  private static final long serialVersionUID = 8613032536466128835L;

  /**
   * Constructs an empty <code>PairSet</code> with the default initial
   * capacity (16) and the default load factor (0.75).
   */
  public PairSet()
  {
    this(16);
  }

  /**
   * Constructs an empty <code>PairSet</code> using the supplied initial
   * capacity and the default load factor (0.75).
   *
   * @param initialCapacity the initial capacity of the map.
   */
  public PairSet(int initialCapacity)
  {
    this(initialCapacity, 0.75f);
  }

  /**
   * Constructs an empty <code>PairSet</code> using the supplied initial
   * capacity and the supplied load factor.
   *
   * @param initialCapacity the initial capacity of the map.
   * @param loadFactor the load factor of the map.
   */
  public PairSet(int initialCapacity, float loadFactor)
  {
    super(initialCapacity, loadFactor);
  }

  /**
   * Constructs a <code>PairSet</code> using the contents of the supplied
   * pair set.
   *
   * @param s the set whose contents are to be used as the start of the
   *          contents for the new set.
   */
  public PairSet(PairSet<A, B> s)
  {
    super(s);
  }

  /**
   * Returns true if the set contains the specified value.
   * The value is constructed as the pair of the two supplied values.
   *
   * @param left the left-hand value of the key.
   * @param right the right-hand value of the key.
   * @return true if the set contains the specified value.
   */
  public boolean contains(A left, B right)
  {
    return super.contains(new Pair<A,B>(left,right));
  }

  /**
   * Adds the specified value to the set.  The value
   * is constructed as the pair of the two supplied values.
   *
   * @param left the left-hand value of the key.
   * @param right the right-hand value of the key.
   * @return true if the set did not contain the value.
   */
  public boolean add(A left, B right)
  {
    return super.add(new Pair<A,B>(left,right));
  }

  /**
   * Removes the specified value from the set.  The value
   * is constructed as the pair of the two supplied values.
   *
   * @param left the left-hand value of the key.
   * @param right the right-hand value of the key.
   * @return true if the set contained the value.
   */
  public boolean remove(A left, B right)
  {
    return super.remove(new Pair<A,B>(left,right));
  }

}
