/* Pair.java -- A heterogenous pair type.
 Copyright (C) 2005  The University of Sheffield.

 This file is part of the CASheW-s editor.

 The CASheW-s editor is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 2, or (at your option)
 any later version.
 
 The CASheW-s editor is distributed in the hope that it will be useful, but
 WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with The CASheW-s editor; see the file COPYING.  If not, write to the
 Free Software Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA
 02111-1307 USA.
*/

package nongnu.cashews.commons;

import java.io.Serializable;

/**
 * A small helper class to pair up two arbitrary heterogenously-typed
 * values.  This is where we wish we had a tuple data type.
 * 
 * @author Andrew John Hughes (gnu_andrew@member.fsf.org)
 */
public class Pair<A,B>
  implements Serializable
{

  /**
   * Serialization ID.
   */
  private static final long serialVersionUID = -1924824870778608765L;

  /**
   * The left-hand value.
   *
   * @serial the left-hand value.
   */
  private A left;
  
  /**
   * The right-hand value.
   *
   * @serial the right-hand value.
   */
  private B right;
  
  /**
   * Constructs a new pair using the values specified.
   *
   * @param left the left-hand value.
   * @param right the right-hand value.
   */
  public Pair(A left, B right)
  {
    setLeft(left);
    setRight(right);
  }
  
  /**
   * Retrieve a reference to the left-hand value.
   * 
   * @return the left-hand value.
   */
  public A getLeft()
  {
    return left;
  }
  
  /**
   * Retrieve a reference to the right-hand value.
   *
   * @return the right-hand value.
   */
  public B getRight()
  {
    return right;
  }

  /**
   * Set the left-hand value.
   *
   * @param left the new left-hand value.
   */
  public void setLeft(A left)
  {
    this.left = left;
  }

  /**
   * Set the right-hand value.
   *
   * @param right the new right-hand value.
   */
  public void setRight(B right)
  {
    this.right = right;
  }

  /**
   * Returns true if the supplied object is also a pair, with equivalent
   * left and right-hand values.
   *
   * @param obj the object to compare.
   */
  public boolean equals(Object obj)
  {
    if (obj == null)
      return false;
    if (obj == this)
      return true;
    if (obj.getClass() == getClass())
      {
	Pair pair = (Pair) obj;
	return (pair.getLeft().equals(left) &&
		pair.getRight().equals(right));
      }
    return false;
  }

  /**
   * Return a hashcode composed of the hashcodes of the two values.
   *
   * @return a hashcode for the pair.
   */
  public int hashCode()
  {
    return 13 * left.hashCode() + 17 * right.hashCode();
  }

  /**
   * Returns a <code>String</code> representation of the pair.
   *
   * @return a <code>String</code> representation.
   */
  public String toString()
  {
    return "(" + left + "," + right + ")";
  }

}
