/*******************************************************************************
 * capi20wrapper - Collection of functions and data structures for accessing
 *                 a shared library which implements the interface as described
 *                 in section 'Linux' in part two of CAPI 2.0
 * 
 * Written in 2014 by Swen Lünig.
 * 
 * To the extent possible under law, the author(s) have dedicated all copyright
 * and related and neighboring rights to this software to the public domain
 * worldwide. This software is distributed without any warranty.
 * 
 * You should have received a copy of the CC0 Public Domain Dedication along
 * with this software. If not, see
 * <http://creativecommons.org/publicdomain/zero/1.0/>.
 ******************************************************************************/

#include <stdio.h>
#include <string.h>
#include <capi20.h>

#undef DEBUG

#include "capi20wrapper.h"

#define capi20wrapper_MAXMSGSIZE 2048

#pragma pack(1)
typedef struct {
  capi20wrapper_Byte_t Total_length   [ 2 ];
  capi20wrapper_Byte_t ApplID         [ 2 ];
  capi20wrapper_Byte_t Command        [ 1 ];
  capi20wrapper_Byte_t Subcommand     [ 1 ];
  capi20wrapper_Byte_t Message_number [ 2 ];
} capi20wrapper_MsgHeader_t;

typedef struct {
  capi20wrapper_MsgHeader_t header;
  capi20wrapper_Byte_t parameter [ capi20wrapper_MAXMSGSIZE - sizeof ( capi20wrapper_MsgHeader_t ) ];
} capi20wrapper_Msg_t;
#pragma pack()

static
unsigned int
capi20wrapper_getWord ( const capi20wrapper_Byte_t const mem [ 2 ],
			capi20wrapper_Word_t * const value )
{
  if ( NULL == mem )
    {
      return 0;
    }
  if ( NULL == value )
    {
      return 0;
    }
  
  *value = mem [ 1 ];
  *value = *value << 8;
  *value += mem [ 0 ];

  return sizeof ( *value );
}

static
unsigned int
capi20wrapper_getDWord ( const capi20wrapper_Byte_t const mem [ 4 ],
			 capi20wrapper_DWord_t * const value )
{
  if ( NULL == mem )
    {
      return 0;
    }
  if ( NULL == value )
    {
      return 0;
    }
  
  *value = mem [ 3 ];
  *value = *value << 8;
  *value += mem [ 2 ];
  *value = *value << 8;
  *value += mem [ 1 ];
  *value = *value << 8;
  *value += mem [ 0 ];

  return sizeof ( *value );
}

static
unsigned int
capi20wrapper_getStruct ( const capi20wrapper_Byte_t const mem [],
			  capi20wrapper_Struct_t * const value )
{
  unsigned int numberOfBytes = 0;
  
  if ( NULL == mem )
    {
      return 0;
    }
  if ( NULL == value )
    {
      return 0;
    }
  
  switch ( mem [ 0 ] )
    {
    case 0x00:
      value->length = 0;
      value->data = NULL;
      numberOfBytes = 1;
      break;
      
    case 0xff:
      numberOfBytes = capi20wrapper_getWord ( & mem [ 1 ],
					      & value->length );
      if ( 0 == numberOfBytes )
	{
	  value->length = 0;
	  value->data = NULL;
	}
      else
	{
	  value->data = & mem [ 3 ];
	  numberOfBytes = 3 + value->length;
	}
      break;
      
    default:
      value->length = mem [ 0 ];
      value->data = & mem [ 1 ];
      numberOfBytes = 1 + value->length;
      break;
    }
  
  return numberOfBytes;
}

static
unsigned int
capi20wrapper_setByte ( capi20wrapper_Byte_t mem [ 1 ],
			capi20wrapper_Byte_t const value )
{
  if ( NULL == mem )
    {
      return 0;
    }
  
  mem [ 0 ] = value;
  
  return sizeof ( value );
}

static
unsigned int
capi20wrapper_setWord ( capi20wrapper_Byte_t mem [ 2 ],
			capi20wrapper_Word_t const value )
{
  capi20wrapper_Word_t tmp_value = value;
  
  if ( NULL == mem )
    {
      return 0;
    }
  
  mem [ 0 ] = tmp_value & 0xff;
  tmp_value = tmp_value >> 8;
  mem [ 1 ] = tmp_value & 0xff;
  
  return sizeof ( value );
}

static
unsigned int
capi20wrapper_setDWord ( capi20wrapper_Byte_t mem [ 4 ],
			 capi20wrapper_DWord_t const value )
{
  capi20wrapper_DWord_t tmp_value = value;
  
  if ( NULL == mem )
    {
      return 0;
    }
  
  mem [ 0 ] = tmp_value & 0xff;
  tmp_value = tmp_value >> 8;
  mem [ 1 ] = tmp_value & 0xff;
  tmp_value = tmp_value >> 8;
  mem [ 2 ] = tmp_value & 0xff;
  tmp_value = tmp_value >> 8;
  mem [ 3 ] = tmp_value & 0xff;
  
  return sizeof ( value );
}

static
unsigned int
capi20wrapper_setStruct ( capi20wrapper_Byte_t mem [],
			  const capi20wrapper_Struct_t * const value )
{
  unsigned int numberOfBytes = 0;
  
  if ( NULL == mem )
    {
      return 0;
    }
  
  if ( NULL == value )
    {
      mem [ 0 ] = 0x00;
      numberOfBytes = 1;
    }
  else
    {
      if ( 0 == value->length )
	{
	  mem [ 0 ] = 0x00;
	  numberOfBytes = 1;
	}
      else
	{
	  if ( 0xfe < value->length )
	    {
	      mem [ 0 ] = 0xff;
	      numberOfBytes = capi20wrapper_setWord ( & mem [ 1 ],
						      value->length );
	      if ( 0 == numberOfBytes )
		{
		  ;
		}
	      else
		{
		  memmove ( & mem [ 3 ],
			    value->data,
			    value->length );
		  numberOfBytes = 3 + value->length;
		}
	    }
	  else
	    {
	      mem [ 0 ] = value->length;
	      memmove ( & mem [ 1 ],
			value->data,
			value->length );
	      numberOfBytes = 1 + value->length;
	    }
	}
    }
  
  return numberOfBytes;
}

static capi20wrapper_Msg_t theMsg;

static
unsigned int
capi20wrapper_fill_header_and_put_message ( unsigned int ApplID,
					    capi20wrapper_Word_t Message_number,
					    capi20wrapper_Byte_t Command,
					    capi20wrapper_Byte_t Subcommand,
					    capi20wrapper_Word_t totalLength )
{
  totalLength += capi20wrapper_setWord ( theMsg.header.ApplID,
					 ApplID );
  totalLength += capi20wrapper_setByte ( theMsg.header.Command,
					 Command );
  totalLength += capi20wrapper_setByte ( theMsg.header.Subcommand,
					 Subcommand );
  totalLength += capi20wrapper_setWord ( theMsg.header.Message_number,
					 Message_number );
  
  totalLength += sizeof ( capi20wrapper_Word_t );
  capi20wrapper_setWord ( theMsg.header.Total_length,
			  totalLength );
  
#ifdef DEBUG
  {
    unsigned int index = 0;
    printf ( "capi20wrapper_fill_header_and_put_message: \n" );
    for ( ; index < totalLength; index++ )
      {
	printf ( " 0x%02x", ((unsigned char *)& theMsg) [ index ] );
      }
    printf ( "\n" );
  }
#endif
    
  return capi20_put_message ( ApplID,
			      (unsigned char *) & theMsg );
}

extern
unsigned int
capi20wrapper_put_connect_active_resp ( unsigned int ApplID,
					capi20wrapper_Word_t Message_number,
					capi20wrapper_DWord_t PLCI )
{
  capi20wrapper_Word_t totalLength = 0;

  totalLength += capi20wrapper_setDWord ( & theMsg.parameter [ totalLength ],
					  PLCI );
  
  return capi20wrapper_fill_header_and_put_message ( ApplID,
						     Message_number,
						     0x03,
						     0x83,
						     totalLength );
}

extern
unsigned int
capi20wrapper_put_connect_resp ( unsigned int ApplID,
				 capi20wrapper_Word_t Message_number,
				 capi20wrapper_DWord_t PLCI,
				 capi20wrapper_Word_t Reject,
				 const capi20wrapper_Struct_t * B_protocol,
				 const capi20wrapper_Struct_t * Connected_number,
				 const capi20wrapper_Struct_t * Connected_subaddress,
				 const capi20wrapper_Struct_t * LLC,
				 const capi20wrapper_Struct_t * Additional_Info )
{
  capi20wrapper_Word_t totalLength = 0;
  
  totalLength += capi20wrapper_setDWord ( & theMsg.parameter [ totalLength ],
					  PLCI );
  totalLength += capi20wrapper_setWord ( & theMsg.parameter [ totalLength ],
					 Reject );
  
  totalLength += capi20wrapper_setStruct ( & theMsg.parameter [ totalLength ],
					   B_protocol );
  totalLength += capi20wrapper_setStruct ( & theMsg.parameter [ totalLength ],
					   Connected_number );
  totalLength += capi20wrapper_setStruct ( & theMsg.parameter [ totalLength ],
					   Connected_subaddress );
  totalLength += capi20wrapper_setStruct ( & theMsg.parameter [ totalLength ],
					   LLC );
  totalLength += capi20wrapper_setStruct ( & theMsg.parameter [ totalLength ],
					   Additional_Info );
  
  return capi20wrapper_fill_header_and_put_message ( ApplID,
						     Message_number,
						     0x02,
						     0x83,
						     totalLength );
}

extern
unsigned int
capi20wrapper_put_connect_req ( unsigned int ApplID,
				capi20wrapper_Word_t Message_number,
				capi20wrapper_DWord_t Controller,
				capi20wrapper_Word_t CIP_Value,
				const capi20wrapper_Struct_t * Called_party_number,
				const capi20wrapper_Struct_t * Calling_party_number,
				const capi20wrapper_Struct_t * Called_party_subaddress,
				const capi20wrapper_Struct_t * Calling_party_subaddress,
				const capi20wrapper_Struct_t * B_protocol,
				const capi20wrapper_Struct_t * BC,
				const capi20wrapper_Struct_t * LLC,
				const capi20wrapper_Struct_t * HLC,
				const capi20wrapper_Struct_t * Additional_Info )
{
  capi20wrapper_Word_t totalLength = 0;

  totalLength += capi20wrapper_setDWord ( & theMsg.parameter [ totalLength ],
					  Controller );
  totalLength += capi20wrapper_setWord ( & theMsg.parameter [ totalLength ],
					 CIP_Value );
  
  totalLength += capi20wrapper_setStruct ( & theMsg.parameter [ totalLength ],
					   Called_party_number );
  totalLength += capi20wrapper_setStruct ( & theMsg.parameter [ totalLength ],
					   Calling_party_number );
  totalLength += capi20wrapper_setStruct ( & theMsg.parameter [ totalLength ],
					   Called_party_subaddress );
  totalLength += capi20wrapper_setStruct ( & theMsg.parameter [ totalLength ],
					   Calling_party_subaddress );
  totalLength += capi20wrapper_setStruct ( & theMsg.parameter [ totalLength ],
					   B_protocol );
  totalLength += capi20wrapper_setStruct ( & theMsg.parameter [ totalLength ],
					   BC );
  totalLength += capi20wrapper_setStruct ( & theMsg.parameter [ totalLength ],
					   LLC );
  totalLength += capi20wrapper_setStruct ( & theMsg.parameter [ totalLength ],
					   HLC );
  totalLength += capi20wrapper_setStruct ( & theMsg.parameter [ totalLength ],
					   Additional_Info );
  
  return capi20wrapper_fill_header_and_put_message ( ApplID,
						     Message_number,
						     0x02,
						     0x80,
						     totalLength );
}

extern
unsigned int
capi20wrapper_put_disconnect_req ( unsigned int ApplID,
				   capi20wrapper_Word_t Message_number,
				   capi20wrapper_DWord_t PLCI,
				   const capi20wrapper_Struct_t * Additional_Info )
{
  capi20wrapper_Word_t totalLength = 0;

  totalLength += capi20wrapper_setDWord ( & theMsg.parameter [ totalLength ],
					  PLCI );
  totalLength += capi20wrapper_setStruct ( & theMsg.parameter [ totalLength ],
					   Additional_Info );
  
  return capi20wrapper_fill_header_and_put_message ( ApplID,
						     Message_number,
						     0x04,
						     0x80,
						     totalLength );
}

extern
unsigned int
capi20wrapper_put_disconnect_resp ( unsigned int ApplID,
				    capi20wrapper_Word_t Message_number,
				    capi20wrapper_DWord_t PLCI )
{
  capi20wrapper_Word_t totalLength = 0;
  
  totalLength += capi20wrapper_setDWord ( & theMsg.parameter [ totalLength ],
					  PLCI );
  
  return capi20wrapper_fill_header_and_put_message ( ApplID,
						     Message_number,
						     0x04,
						     0x83,
						     totalLength );
}

extern
unsigned int
capi20wrapper_put_listen_req ( unsigned int ApplID,
			       capi20wrapper_Word_t Message_number,
			       capi20wrapper_DWord_t Controller,
			       capi20wrapper_DWord_t Info_mask,
			       capi20wrapper_DWord_t CIP_Mask,
			       capi20wrapper_DWord_t CIP_Mask_2,
			       const capi20wrapper_Struct_t * Calling_party_number,
			       const capi20wrapper_Struct_t * Calling_party_subaddress )
{
  capi20wrapper_Word_t totalLength = 0;
  
  totalLength += capi20wrapper_setDWord ( & theMsg.parameter [ totalLength ],
					  Controller );
  totalLength += capi20wrapper_setDWord ( & theMsg.parameter [ totalLength ],
					  Info_mask );
  totalLength += capi20wrapper_setDWord ( & theMsg.parameter [ totalLength ],
					  CIP_Mask );
  totalLength += capi20wrapper_setDWord ( & theMsg.parameter [ totalLength ],
					  CIP_Mask_2 );
  totalLength += capi20wrapper_setStruct ( & theMsg.parameter [ totalLength ],
					   Calling_party_number );
  totalLength += capi20wrapper_setStruct ( & theMsg.parameter [ totalLength ],
					   Calling_party_subaddress );
  
  return capi20wrapper_fill_header_and_put_message ( ApplID,
						     Message_number,
						     0x05,
						     0x80,
						     totalLength );
}

extern
unsigned int
capi20wrapper_get_message ( unsigned int ApplID,
			    const capi20wrapper_Callbacks_t * callbacks )
{
  unsigned int result = 0x1104;
  capi20wrapper_Msg_t * msg = NULL;
  unsigned int index = 0;
  
  result = capi20_get_message ( ApplID,
				(unsigned char **) & msg );
  if ( 0x0000 != result )
    {
      return result;
    }
  
#ifdef DEBUG
  {
    unsigned int index = 0;
    capi20wrapper_Word_t Total_length = 0;
    
    capi20wrapper_getWord ( msg->header.Total_length,
			    & Total_length );
    
    printf ( "capi20wrapper_get_message: \n" );
    for ( ; index < Total_length; index++ )
      {
	printf ( " 0x%02x", ((unsigned char *) msg) [ index ] );
      }
    printf ( "\n" );
  }
#endif
  
  /* I do not care about ApplID in the message so I do not read and check it.
   */
  
  capi20wrapper_Word_t Message_number = 0xffff;
  capi20wrapper_getWord ( msg->header.Message_number,
			  & Message_number );
  
  /* The result so far is 0x0000. For avoiding many default-cases in the following
   * switches, the result is set here. Because the value from here is the result in
   * case the received message contains an unknown Command or Subcommand, the value
   * is 0x1102. That value fits the situation.
   */
  result = 0x1102;
  
  int isKnown = 0;
  switch ( msg->header.Command [ 0 ] )
    {
    case 0x02:
      switch ( msg->header.Subcommand [ 0 ] )
	{
	case 0x81:
	  
	  isKnown = 1;
	  if ( NULL != callbacks->get_connect_conf )
	    {
	      capi20wrapper_DWord_t PLCI = 0x00000000;
	      index += capi20wrapper_getDWord ( & msg->parameter [ index ],
						& PLCI );
	      capi20wrapper_Word_t Info = 0x0000;
	      index += capi20wrapper_getWord ( & msg->parameter [ index ],
					       & Info );
	      
	      result = (*callbacks->get_connect_conf) ( ApplID,
							Message_number,
							PLCI,
							Info );
	    }
	  break;
	  
	case 0x82:
	  
	  isKnown = 1;
	  if ( NULL != callbacks->get_connect_ind )
	    {
	      capi20wrapper_DWord_t PLCI = 0x00000000;
	      index += capi20wrapper_getDWord ( & msg->parameter [ index ],
						& PLCI );
	      capi20wrapper_Word_t CIP_Value = 0x0000;
	      index += capi20wrapper_getWord ( & msg->parameter [ index ],
					       & CIP_Value );
	      capi20wrapper_Struct_t Called_party_number;
	      index += capi20wrapper_getStruct ( & msg->parameter [ index ],
						 & Called_party_number );
	      capi20wrapper_Struct_t Calling_party_number;
	      index += capi20wrapper_getStruct ( & msg->parameter [ index ],
						 & Calling_party_number );
	      capi20wrapper_Struct_t Called_party_subaddress;
	      index += capi20wrapper_getStruct ( & msg->parameter [ index ],
						 & Called_party_subaddress );
	      capi20wrapper_Struct_t Calling_party_subaddress;
	      index += capi20wrapper_getStruct ( & msg->parameter [ index ],
						 & Calling_party_subaddress );
	      capi20wrapper_Struct_t BC;
	      index += capi20wrapper_getStruct ( & msg->parameter [ index ],
						 & BC );
	      capi20wrapper_Struct_t LLC;
	      index += capi20wrapper_getStruct ( & msg->parameter [ index ],
						 & LLC );
	      capi20wrapper_Struct_t HLC;
	      index += capi20wrapper_getStruct ( & msg->parameter [ index ],
						 & HLC );
	      capi20wrapper_Struct_t Additional_Info;
	      index += capi20wrapper_getStruct ( & msg->parameter [ index ],
						 & Additional_Info );
	      capi20wrapper_Struct_t Second_Calling_party_number;
	      index += capi20wrapper_getStruct ( & msg->parameter [ index ],
						 & Second_Calling_party_number );
		
	      result = (*callbacks->get_connect_ind) ( ApplID,
						       Message_number,
						       PLCI,
						       CIP_Value,
						       & Called_party_number,
						       & Calling_party_number,
						       & Called_party_subaddress,
						       & Calling_party_subaddress,
						       & BC,
						       & LLC,
						       & HLC,
						       & Additional_Info,
						       & Second_Calling_party_number );
	    }
	  break;
	}
      break;
      
    case 0x03:
      switch ( msg->header.Subcommand [ 0 ] )
	{
	case 0x82:
	  
	  isKnown = 1;
	  if ( NULL != callbacks->get_connect_active_ind )
	    {
	      capi20wrapper_DWord_t PLCI = 0x00000000;
	      index += capi20wrapper_getDWord ( & msg->parameter [ index ],
						& PLCI );
	      capi20wrapper_Struct_t Connected_number;
	      index += capi20wrapper_getStruct ( & msg->parameter [ index ],
						 & Connected_number );
	      capi20wrapper_Struct_t Connected_subaddress;
	      index += capi20wrapper_getStruct ( & msg->parameter [ index ],
						 & Connected_subaddress );
	      capi20wrapper_Struct_t LLC;
	      index += capi20wrapper_getStruct ( & msg->parameter [ index ],
						 & LLC );
	      
	      result = (*callbacks->get_connect_active_ind) ( ApplID,
							      Message_number,
							      PLCI,
							      & Connected_number,
							      & Connected_subaddress,
							      & LLC );
	    }
	  break;
	}
      
    case 0x04:
      switch ( msg->header.Subcommand [ 0 ] )
	{
	case 0x82:
	  
	  isKnown = 1;
	  if ( NULL != callbacks->get_disconnect_ind )
	    {
	      capi20wrapper_DWord_t PLCI = 0x00000000;
	      index += capi20wrapper_getDWord ( & msg->parameter [ index ],
						& PLCI );
	      capi20wrapper_Word_t Reason = 0x0000;
	      index += capi20wrapper_getWord ( & msg->parameter [ index ],
					       & Reason );
	      
	      result = (*callbacks->get_disconnect_ind) ( ApplID,
							  Message_number,
							  PLCI,
							  Reason );
	    }
	  break;
	}
      break;
      
    case 0x05:
      switch ( msg->header.Subcommand [ 0 ] )
	{
	case 0x81:
	  
	  isKnown = 1;
	  if ( NULL != callbacks->get_listen_conf )
	    {
	      capi20wrapper_DWord_t Controller = 0x00000000;
	      index += capi20wrapper_getDWord ( & msg->parameter [ index ],
						& Controller );
	      capi20wrapper_Word_t Info = 0x0000;
	      index += capi20wrapper_getWord ( & msg->parameter [ index ],
					       & Info );
	      
	      result = (*callbacks->get_listen_conf) ( ApplID,
						       Message_number,
						       Controller,
						       Info );
	    }
	  break;
	}
      break;
    }
  
  if ( 0 == isKnown )
    {
      if ( NULL != callbacks->get_unknown_message )
	{
	  (*callbacks->get_unknown_message) ( ApplID,
					      Message_number,
					      msg->header.Command [ 0 ],
					      msg->header.Subcommand [ 0 ] );
	}
    }
  
  return result;
}

/******************************************************************************
 *****************************************************************************/

extern
void capi20wrapper_printf_Struct ( const char * name,
				   const capi20wrapper_Struct_t * Struct )
{
  printf ( "%s: \n", name );
  if ( NULL == Struct )
    {
      printf ( " <empty>\n" );
    }
  else
    {
      if ( NULL == Struct->data )
	{
	  printf ( " <empty>\n" );
	}
      else
	{
	  unsigned int index = 0;
	  for ( ; index < Struct->length; index++ )
	    {
	      printf ( " 0x%02x", Struct->data [ index ] );
	    }
	  printf ( "\n" );
	}
    }
}
