/*******************************************************************************
 * capi20wrapper - Collection of functions and data structures for accessing
 *                 a shared library which implements the interface as described
 *                 in section 'Linux' in part two of CAPI 2.0
 * 
 * Written in 2014 by Swen Lünig.
 * 
 * To the extent possible under law, the author(s) have dedicated all copyright
 * and related and neighboring rights to this software to the public domain
 * worldwide. This software is distributed without any warranty.
 * 
 * You should have received a copy of the CC0 Public Domain Dedication along
 * with this software. If not, see
 * <http://creativecommons.org/publicdomain/zero/1.0/>.
 ******************************************************************************/

/******************************************************************************
 * This little example accepts all incoming calls and just waits for the end
 * of the connection.
 * The sequence for one connection is as follows:
 * LISTEN_REQ, LISTEN_CONF, CONNECT_IND, CONNECT_RESP, DISCONNECT_IND, DISCONNECT_RESP
 *****************************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <capi20.h>

#include "capi20wrapper.h"

static
int didTheSequenceFinish = 0;

static
unsigned int get_listen_conf ( unsigned int ApplID,
			       capi20wrapper_Word_t Message_number,
			       capi20wrapper_DWord_t Controller,
			       capi20wrapper_Word_t Info )
{
  printf ( "--------------------------------\n" );
  printf ( "got a LISTEN_CONF\n" );
  printf ( "Controller = 0x%08lx\n",
	   Controller );
  printf ( "Info = 0x%04x\n",
	   Info );
  
  if ( 0x0000 != Info )
    {
      printf ( "Oops! Info is not 0x0000, terminating the process.\n" );
      didTheSequenceFinish = 1;
    }
  
  return 0x0000;
}

static
unsigned int get_connect_ind ( unsigned int ApplID,
			       capi20wrapper_Word_t Message_number,
			       capi20wrapper_DWord_t PLCI,
			       capi20wrapper_Word_t CIP_Value,
			       const capi20wrapper_Struct_t * Called_party_number,
			       const capi20wrapper_Struct_t * Calling_party_number,
			       const capi20wrapper_Struct_t * Called_party_subaddress,
			       const capi20wrapper_Struct_t * Calling_party_subaddress,
			       const capi20wrapper_Struct_t * BC,
			       const capi20wrapper_Struct_t * LLC,
			       const capi20wrapper_Struct_t * HLC,
			       const capi20wrapper_Struct_t * Additional_Info,
			       const capi20wrapper_Struct_t * Second_Calling_party_number )
{
  printf ( "--------------------------------\n" );
  printf ( "got a CONNECT_IND\n" );
  printf ( "PLCI = 0x%08lx\n",
	   PLCI );
  printf ( "CIP_Value = 0x%04x\n",
	   CIP_Value );
  capi20wrapper_printf_Struct ( "Called_party_number",
				Called_party_number );
  capi20wrapper_printf_Struct ( "Calling_party_number",
				Calling_party_number );
  capi20wrapper_printf_Struct ( "Called_party_subaddress",
				Called_party_subaddress );
  capi20wrapper_printf_Struct ( "Calling_party_subaddress",
				Calling_party_subaddress );
  capi20wrapper_printf_Struct ( "BC",
				BC );
  capi20wrapper_printf_Struct ( "LLC",
				LLC );
  capi20wrapper_printf_Struct ( "HLC",
				HLC );
  capi20wrapper_printf_Struct ( "Additional_Info",
				Additional_Info );
  capi20wrapper_printf_Struct ( "Second_Calling_party_number",
				Second_Calling_party_number );
  
  unsigned int result = capi20wrapper_put_connect_resp ( ApplID,
							 Message_number,
							 PLCI,
							 0x0000,
							 NULL,
							 Called_party_number,
							 Called_party_subaddress,
							 LLC,
							 Additional_Info );
  printf ( "capi20wrapper_put_connect_resp (...) = 0x%04x\n",
	   result );
  return result;
}

static
unsigned int get_disconnect_ind ( unsigned int ApplID,
				  capi20wrapper_Word_t Message_number,
				  capi20wrapper_DWord_t PLCI,
				  capi20wrapper_Word_t Reason )
{
  printf ( "--------------------------------\n" );
  printf ( "got a DISCONNECT_IND\n" );
  printf ( "PLCI = 0x%08lx\n",
	   PLCI );
  printf ( "Reason = 0x%04x\n",
	   Reason );
  
  unsigned int result = capi20wrapper_put_disconnect_resp ( ApplID,
							    Message_number,
							    PLCI );
  printf ( "capi20wrapper_put_disconnect_resp (...) = 0x%04x\n",
	   result );
  return result;
}

static
void get_unknown_message ( unsigned int ApplID,
			   capi20wrapper_Word_t Message_number,
			   capi20wrapper_Byte_t Command,
			   capi20wrapper_Byte_t Subcommand )
{
  printf ( "--------------------------------\n" );
  printf ( "got an unknown message\n" );
  printf ( "Command = 0x%02x\n",
	   Command );
  printf ( "Subommand = 0x%02x\n",
	   Subcommand );
}

static
capi20wrapper_Callbacks_t theCallbacks = {
  .get_connect_ind     = get_connect_ind,
  .get_disconnect_ind  = get_disconnect_ind,
  .get_listen_conf     = get_listen_conf,
  .get_unknown_message = get_unknown_message
};

int
main ()
{
  unsigned int const isinstalled = capi20_isinstalled ();

  printf ( "capi20_isinstalled () = 0x%04x\n",
	   isinstalled );
  if ( 0x0000 != isinstalled )
    {
      printf ( "Oops! capi20 is not installed, terminating the process.\n" );
      return 1;
    }

  unsigned int ApplID = 0;
  unsigned int result = 0x0000;
  
  result = capi20_register ( 1,
			     7,
			     2048,
			     & ApplID );
  printf ( "capi20_register (...) = 0x%04x\n",
	   result );
  if ( 0x0000 != result )
    {
      printf ( "Oops! capi20_register did not succeed, terminating the process.\n" );
      return 1;
    }
  printf ( "ApplID = %d\n",
	   ApplID );

  const char * valuestring = NULL;
  
  capi20wrapper_DWord_t Controller = 0x00000001;
  
  valuestring = getenv ( "CAPI20_Controller" );
  if ( NULL != valuestring )
    {
      Controller = strtoul ( valuestring,
			     NULL,
			     16 );
    }

  /* INFO_IND is not activated by default.
   */
  capi20wrapper_DWord_t Info_mask = 0x00000000;
  
  valuestring = getenv ( "CAPI20_Info_mask" );
  if ( NULL != valuestring )
    {
      Info_mask = strtoul ( valuestring,
			    NULL,
			    16 );
    }

  /* I want to see every CONNECT_IND by default.
   */
  capi20wrapper_DWord_t CIP_Mask = 0x00000001;
  
  valuestring = getenv ( "CAPI20_CIP_Mask" );
  if ( NULL != valuestring )
    {
      CIP_Mask = strtoul ( valuestring,
			   NULL,
			   16 );
    }
  
  result = capi20wrapper_put_listen_req ( ApplID,
					  0x0000,
					  Controller,
					  Info_mask,
					  CIP_Mask,
					  0x00000000,
					  NULL,
					  NULL );
  printf ( "capi20wrapper_put_listen_req (...) = 0x%04x\n",
	   result );
  
  while ( 0x0000 == result
	  && ! didTheSequenceFinish )
    {
      result = capi20_waitformessage ( ApplID,
				       0 );
      printf ( "capi20_waitformessage (...) = 0x%04x\n",
	       result );
      if ( 0x0000 == result )
	{
	  result = capi20wrapper_get_message ( ApplID,
					       & theCallbacks );
	  printf ( "capi20wrapper_get_message (...) = 0x%04x\n",
		   result );
	}
    }
  
  result = capi20_release ( ApplID );
  printf ( "capi20_release (...) = 0x%04x\n",
	   result );
  
  return 0;
}
