/************************************************************************\
 * C2py converts cpy code (with a C-like syntax) to Python code.        *
 * Copyright (C) 2019  Asher Gordon <AsDaGo@protonmail.ch>              *
 *                                                                      *
 * This file is part of c2py.                                           *
 *                                                                      *
 * C2py is free software: you can redistribute it and/or modify         *
 * it under the terms of the GNU General Public License as published by *
 * the Free Software Foundation, either version 3 of the License, or    *
 * (at your option) any later version.                                  *
 *                                                                      *
 * C2py is distributed in the hope that it will be useful,              *
 * but WITHOUT ANY WARRANTY; without even the implied warranty of       *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        *
 * GNU General Public License for more details.                         *
 *                                                                      *
 * You should have received a copy of the GNU General Public License    *
 * along with c2py.  If not, see <https://www.gnu.org/licenses/>.       *
\************************************************************************/

/* parse.y -- bison input to generate the parser for c2py */

%{
  #define _GNU_SOURCE

  #ifdef HAVE_CONFIG_H
  # include <config.h>
  #endif

  #include "python.h" // Should be included first

  #include <stdio.h>
  #include <stdlib.h>
  #include <stdarg.h>
  #include <string.h>
  #include <ctype.h>
  #include <assert.h>
  #include <errno.h> // For program_invocation_name

  #include "c2py.h"
  #include "llist.h"
  #include "input.h"

  #define TOKEN_STRING "()[]{},=<>+-/*?:!&|^.;" // Valid one-character tokens
  #define INDENT_SIZE 4
  #define INDENT_STRING "    "
  #define QUOTE_CHARS "\"'"

  /* Increment num and make sure it didn't wrap around */
  #define increment(num) do {						\
      if (!++(num)) {							\
	/* num has wrapped around to zero */				\
	fprintf(stderr, "%s: " __FILE__ ":" XSTR(__LINE__)		\
		": in %s(): " XSTR(num) " has overflowed\n",		\
		program_invocation_name, __func__);			\
	YYABORT;							\
      }									\
    } while (0)

  /* Make sure num is not 0 and decrement it */
  #define decrement(num) do {						\
      if (!(num)) {							\
	fprintf(stderr, "%s: " __FILE__ ":" XSTR(__LINE__)		\
		": in %s(): tried to decrement " XSTR(num)		\
		" below zero\n", program_invocation_name, __func__);	\
	YYABORT;							\
      }									\
									\
      (num)--;								\
    } while (0)

  #ifndef HAVE_LIBREADLINE
  /* input.c does not use this if we're compiled without libreadline,
     so we can define it here. End a sequence of conditionals
     (if/for/while, elif, ..., else). */
  # define end_conditional() do {			\
      if (!indent_count && wait_string) {		\
	wait_string = 0;				\
	output(""); /* Flush the buffer and run it */	\
      }							\
    } while (0)
  #endif /* !HAVE_LIBREADLINE */

  /* Add a new element to `empty_block' and set it to 1. */
  #define next_empty() do {			\
      if (empty_block) {			\
	llist_add_forward(empty_block);		\
	llist_next(empty_block);		\
      }						\
      else {					\
	llist_create(empty_block);		\
      }						\
						\
      empty_block->data = 1;			\
    } while (0)

  /* Set the current `empty_block' to 1 (used in `case' and
     `default'). */
  #define empty()				\
    if (empty_block)				\
      empty_block->data = 1
  
  /* Set the current element of `empty_block' to 0. */
  #define not_empty()				\
    if (empty_block)				\
      empty_block->data = 0

  /* Set the previous element of `empty_block' to 0. */
  #define prev_not_empty()			\
    if (empty_block && empty_block->prev)	\
      empty_block->prev->data = 0

  /* Set the previous previous element of `empty_block to 0. */
  #define prev_prev_not_empty()						\
    if (empty_block && empty_block->prev && empty_block->prev->prev)	\
      empty_block->prev->prev->data = 0

  /* Pop the last `empty_block'. */
  #define pop_empty() do {			\
      if (empty_block) {			\
	if (empty_block->prev) {		\
	  llist_prev(empty_block);		\
	  llist_remove(empty_block->next);	\
	}					\
	else {					\
	  llist_remove(empty_block);		\
	  empty_block = NULL;			\
	}					\
      }						\
    } while (0)

  /* Add a new element to a linked list */
  #define list_add_new(list, type) do {		\
      if (list) {				\
	type *new = malloc(sizeof(*new));	\
	new->prev = (list);			\
	(list) = new;				\
      }						\
      else {					\
	(list) = malloc(sizeof(*(list)));	\
	(list)->prev = NULL;			\
      }						\
    } while (0)

  /* Free an element from a linked list */
  #define list_free(list, type) do {		\
      type *old = (list);			\
      (list) = (list)->prev;			\
      free(old);				\
    } while (0)

  /* Add a new element to a doubly linked list */
  #define dlist_add_new(list) do {			\
      if (list) {					\
	(list)->next = malloc(sizeof(*((list)->next)));	\
	(list)->next->prev = (list);			\
	(list) = (list)->next;				\
	(list)->next = NULL;				\
      }							\
      else {						\
	(list) = malloc(sizeof(*(list)));		\
	(list)->prev = (list)->next = NULL;		\
      }							\
    } while (0)

  /* Free an element from a doubly linked list */
  #define dlist_free(list) do {			\
      if ((list)->prev) {			\
	(list) = (list)->prev;			\
	free((list)->next);			\
	(list)->next = NULL;			\
      }						\
      else {					\
	free(list);				\
	(list) = NULL;				\
      }						\
    } while (0)

  /* We have to define this here because we can't use preprocessor
     directives in the grammer rules section. */
  #define inc_indent_if_interactive()		\
    if (interactive) increment(indent_count)
  #define dec_indent_if_interactive()		\
    if (interactive) decrement(indent_count)

  struct backtrace {
    struct backtrace *prev, *next;
    enum { class, func, method } type;
    char *name;
    unsigned int line, column, indent_count;
  };

  struct indent_count {
    struct indent_count *prev;
    unsigned int count;
  };

  struct quoted_pos {
    struct quoted_pos *prev;
    size_t start, end;
  };

  struct str_list {
    struct str_list *prev;
    char *str;
  };
 
  static void yyerror(const char *);
  static int yylex(void);
  static int indent(const char *, char *, size_t, size_t *);
  #ifndef HAVE_LIBPYTHON
  /* If we're compiled without libpython3, this will not be defined in
     input.h. */
  static int output(const char *, ...);
  #endif

  /* This is for error handling. */
  static unsigned int last_line = 1, last_column = 1,
    line = 1, column = 0, column_save;
  static unsigned int do_count = 0;
  /* What to put at the end of a while loop */
  static struct str_list *while_end_expr = NULL;
  /* Keep track of the backtrace for error reporting. */
  static struct backtrace *backtrace = NULL;
  static struct indent_count
    /* This is used for C style for loops. See the `block_end' and
     `for' symbols. */
    *while_indent_count = NULL,
    *switch_indent_count = NULL,
    *do_indent_count = NULL;
  /* Positions between which we should not indent because they are
     quoted. */
  static struct quoted_pos *quoted_pos = NULL;
  static char output_no_indent = 0;
  /* Whether to output "pass" if we recieved an empty block. */
  static llist_t *empty_block = NULL;
%}

%define parse.error verbose

/*
 * string is a string that is always dynamically allocated (with
 * malloc()), static_string is a string that is always statically
 * allocated (on the stack), and opt_string is a string that, if it is
 * "", is statically allocated, otherwise it is dynamically allocated.
 */
%union {
  char *string, *static_string, *opt_string;
  char bool;
}

%token STRING		"name"
%token STRING_LITERAL	"string literal"

%token IMPORT	"import"
%token FROM	"from"
%token AS	"as"

%token DO	"do"
%token WHILE	"while"
%token FOR	"for"
%token IN	"in"
%token IF	"if"
%token ELSE	"else"
%token ELIF	"elif"

%token SWITCH	"switch"
%token CASE	"case"
%token DEFAULT	"default"

%token TRY	"try"
%token EXCEPT	"except"
%token FINALLY	"finally"

%token POWER	"**"
%token INT_DIV	"//"

%token PLUS_EQ	"+="
%token MINUS_EQ	"-*"
%token MUL_EQ	"*="
%token DIV_EQ	"/="

%token LESS_EQ		"<="
%token GREATER_EQ	">="
%token EQUAL		"=="
%token NOT_EQUAL	"!="

%token BOOL_AND	"&&"
%token BOOL_OR	"||"

%token DEL	"del"

%token GLOBAL	"global"

%token CLASS	"class"
%token RETURN	"return"

%token END 0	"end of file"

%token YYLEX_ERROR

/* Set the types of the tokens and symbols. */
%type <string> STRING STRING_LITERAL

%type <string> string_literal
%type <string> inheritance
%type <string> expr args_real

%type <opt_string> opt_expr except_expr args

%type <static_string> conditional_start_real conditional_start
%type <static_string> try_finally
%type <static_string> operation sign

/* $$ of input will be whether the symbol was block_end so we can
   determine whether we should wait for a `while' to end a `do' */
%type <bool> input

%destructor { free($$); } <string>
%destructor { if (*$$) free($$); } <opt_string>

%initial-action {
#ifdef HAVE_LIBREADLINE
		  indent_count =
#endif
		    wait_string = switch_count = 0;
		}

%%

/* This is to run `exec_str' when we're done parsing. */
start:		input_real { end_conditional(); }
	;

/* So we can execute something at the end of every input */
input_real:	%empty
	|	input_real input {
		  /* End a do-while loop and abort if we're not in
		     interactive mode and the symbol wasn't
		     `block_end' (if it was `block_end', $2 will be
		     1). */
		  if (!$2 && do_indent_count &&
		      do_indent_count->count == indent_count) {
		    list_free(do_indent_count, struct indent_count);

		    yyerror("syntax error, expected `while' to end `do'");
		    if (!interactive) YYABORT;
		  }
		}
	;

input:		import	  { end_conditional(); not_empty(); $$ = 0; }
	|	func_call { end_conditional(); not_empty(); $$ = 0; }
	|	func_def  { end_conditional(); prev_not_empty(); $$ = 0; }
	|	class_def { end_conditional(); prev_not_empty(); $$ = 0; }
	|	sline_for { not_empty(); $$ = 0; }
	|	sline_do  { not_empty(); $$ = 0; }
	|	sline_conditional { not_empty(); $$ = 0; }
	|	sline_while	  { not_empty(); $$ = 0; }
	|	sline_else	  { not_empty(); $$ = 0; }
	|	for		  { prev_not_empty(); $$ = 0; }
	|	do		  { prev_not_empty(); $$ = 0; }
	|	do_end_while {
		  $$ = 0;
		  prev_prev_not_empty();
		  /* Yes, you read that right. That's two
		     `prev's. Please don't ask me why. */
		}
	|	conditional	{ prev_not_empty(); $$ = 0; }
	|	while		{ prev_not_empty(); $$ = 0; }
	|	else		{ prev_not_empty(); $$ = 0; }
	|	switch	{
		  $$ = 0;
		  end_conditional();
		  prev_prev_not_empty(); // Yep, again
		}
	|	case		{ end_conditional(); prev_not_empty(); $$ = 0; }
	|	try_except	{ prev_not_empty(); $$ = 0; }
	|	expr ';' {
		  $$ = 0;
		  end_conditional();
		  
		  output("%s\n", $1);
		  free($1);

		  not_empty();
		}
	|	block_end { $$ = 1; }
	|	DEL expr ';' {
		  $$ = 0;
		  end_conditional();

		  output("del %s\n", $2);
	  
		  free($2);

		  not_empty();
		}
	|	RETURN opt_expr ';' {
		  $$ = 0;
		  end_conditional();
		  
		  output("return%s%s\n", *$2 ? " " : "", $2);

		  if (*$2) free($2);

		  not_empty();
		}
	|	GLOBAL args ';' {
		  $$ = 0;
		  end_conditional();

		  output("global %s\n", $2);

		  free($2);

		  not_empty();
		}
	|	STRING_LITERAL {
		  $$ = 0;

		  if (!minify)
		    output("%s\n", $1);

		  free($1);

		  not_empty();
		}
	|	yylex_error { $$ = 0; }
	|	error {
		  if (interactive)
		    $$ = 0;
		  else
		    YYABORT;
		}
	;

string_literal:	STRING_LITERAL
	|	string_literal string_literal {
		  size_t size = sizeof(*$$) *
		    (strlen($1) + strlen($2) + (minify ? 1 : 2));

		  $$ = malloc(size);
		  snprintf($$, size, minify ? "%s%s" : "%s %s", $1, $2);

		  free($1);
		  free($2);
		}
	;

import:		IMPORT STRING ';' {
		  output("import %s\n", $2);

		  free($2);
		}
	|	IMPORT STRING AS STRING ';' {
		  output("import %s as %s\n", $2, $4);

		  free($2);
		  free($4);
		}
	|	FROM STRING IMPORT STRING ';' {
		  output("from %s import %s\n", $2, $4);

		  free($2);
		  free($4);
		}
	|	FROM STRING IMPORT STRING AS STRING ';' {
		  output("from %s import %s as %s\n", $2, $4, $6);

		  free($2);
		  free($4);
		  free($6);
		}
	;

func_call:	STRING '(' args ')' ';' {
		  output("%s(%s)\n", $1, $3);

		  free($1);
		  if (*$3) free($3);
		}
	;

func_def:	STRING '(' args ')' block_start {
		  output("def %s(%s):\n", $1, $3);

		  if (*$3) free($3);

		  /* Add the function to the backtrace. */
		  if (backtrace) {
		    backtrace->next = malloc(sizeof(*backtrace));
		    backtrace->next->prev = backtrace;
		    backtrace = backtrace->next;
		    backtrace->next = NULL;
		    backtrace->type = (backtrace->prev->type == class) ?
		      method : func;
		  }
		  else {
		    backtrace = malloc(sizeof(*backtrace));
		    backtrace->prev = NULL;
		    backtrace->next = NULL;
		    backtrace->type = func;
		  }
		  backtrace->name = $1;
		  backtrace->indent_count = indent_count;
		  backtrace->line = last_line;
		  backtrace->column = last_column;

		  increment(indent_count);
		}
	;

/* Optional class inheritance. */
inheritance:	%empty { $$ = ""; }
	|	'(' args ')' {
		  if (*$2) {
		    size_t size = sizeof(*$$) * (strlen($2) + 3);

		    $$ = malloc(size);
		    snprintf($$, size, "(%s)", $2);

		    free($2);
		  }
		  else {
		    $$ = "";
		  }
		}
	;

class_def:	CLASS STRING inheritance block_start {
		  output("class %s%s%s:\n",
			 $2, (!minify && *$3) ? " " : "", $3);

		  if (*$3) free($3);

		  /* Add the class to the backtrace. */
		  if (backtrace) {
		    backtrace->next = malloc(sizeof(*backtrace));
		    backtrace->next->prev = backtrace;
		    backtrace = backtrace->next;
		    backtrace->next = NULL;
		  }
		  else {
		    backtrace = malloc(sizeof(*backtrace));
		    backtrace->prev = NULL;
		    backtrace->next = NULL;
		  }
		  backtrace->type = class;
		  backtrace->name = $2;
		  backtrace->indent_count = indent_count;
		  backtrace->line = last_line;
		  backtrace->column = last_column;

		  increment(indent_count);
		}
	;

conditional_start_real:
		IF	{ $$ = "if"; }
	|	ELSE IF	{ $$ = "elif"; }
	|	ELIF	{ $$ = "elif"; }
	;

conditional_start:
		conditional_start_real {
		  if (!indent_count)
		    wait_string = 1;

		  /* Increment `indent_count' so the prompt will be
		     properly indented. */
		  inc_indent_if_interactive();
		}
	;

while_start:	WHILE {
		  if (!indent_count)
		    wait_string = 1;

		  /* Increment `indent_count' so the prompt will be
		     properly indented. */
		  inc_indent_if_interactive();
		}

/* This is so we can end `do's */
do_end_while:	while_start '(' expr ')' ';' {
		  /* Decrement `indent_count' if we incremented it for
		     the prompt. */
		  dec_indent_if_interactive();

		  /* Check if we're actually ending a `do' */
		  if (do_indent_count &&
		      do_indent_count->count == indent_count) {
		    decrement(do_count);
		    list_free(do_indent_count, struct indent_count);

		    /* Output the while expression */
		    output("while __c2py_do_%1$u or (%2$s):\n"
			   INDENT_STRING "__c2py_do_%1$u%3$s\n",
			   do_count, $3, minify ? "=0" : " = False");

		    /* Clean up the temporary variable */
		    output("del __c2py_do_%u\n", do_count);
		  }
		  /* Otherwise, we're just a regular `while' without a
		     body */
		  else {
		    output(minify ? "while %s:pass\n" : "while %s:\n"
			   INDENT_STRING "pass\n", $3);
		  }

		  free($3);
		}
	;

/* A single line conditional. We need to separate this out, because it
   acts as a single statement and does NOT set `block_empty'. */
sline_conditional:
		conditional_start '(' expr ')' opt_expr ';' {
		  /* Decrement `indent_count' if we incremented it for
		     the prompt. */
		  dec_indent_if_interactive();

		  output(minify ? "%s %s:%s\n" : "%s %s:\n"
			 INDENT_STRING "%s\n", $1, $3, *$5 ? $5 : "pass");

		  free($3);
		  if (*$5) free($5);
		}
	;

conditional:	conditional_start '(' expr ')' block_start {
		  /* Decrement `indent_count' if we incremented it for
		     the prompt. */
		  dec_indent_if_interactive();

		  output("%s %s:\n", $1, $3);

		  increment(indent_count);

		  free($3);
		}
	;

sline_while:	while_start '(' expr ')' opt_expr ';' {
		  /* Decrement `indent_count' if we incremented it for
		     the prompt. */
		  dec_indent_if_interactive();

		  output(minify ? "while %s:%s\n" : "while %s:\n"
			 INDENT_STRING "%s\n", $3, *$5 ? $5 : "pass");

		  free($3);
		  if (*$5) free($5);
		}
	;

while:		while_start '(' expr ')' block_start {
		  /* Decrement `indent_count' if we incremented it for
		     the prompt. */
		  dec_indent_if_interactive();

		  output("while %s:\n", $3);

		  increment(indent_count);

		  free($3);
		}
	;

for_start:	FOR {
		  if (!indent_count)
		    wait_string = 1;

		  /* Increment `indent_count' so the prompt will be
		     properly indented. */
		  inc_indent_if_interactive();
		}
	;

do_start:	DO {
		  if (!indent_count)
		    wait_string = 1;

		  /* Increment `indent_count' so the prompt will be
		     properly indented. */
		  inc_indent_if_interactive();
		}

else_start:	ELSE {
		  if (!indent_count)
		    wait_string = 1;

		  /* Increment `indent_count' so the prompt will be
		     properly indented. */
		  inc_indent_if_interactive();
		}
	;

sline_else:	else_start opt_expr ';' {
		  /* Decrement `indent_count' if we incremented it for
		     the prompt. */
		  dec_indent_if_interactive();

		  output(minify ? "else:%s\n" : "else:\n"
			 INDENT_STRING "%s\n", *$2 ? $2 : "pass");

		  if (*$2) free($2);
		}
	;

else:		else_start block_start {
		  /* Decrement `indent_count' if we incremented it for
		     the prompt. */
		  dec_indent_if_interactive();

		  output("else:\n");
		  increment(indent_count);
		}
	;

sline_for:	/* Python style syntax */
		for_start '(' STRING IN expr ')' opt_expr ';' {
		  /* Decrement `indent_count' if we incremented it for
		     the prompt. */
		  dec_indent_if_interactive();

		  output(minify ? "for %s in %s:%s\n" : "for %s in %s:\n"
			 INDENT_STRING "%s\n", $3, $5, *$7 ? $7 : "pass");

		  free($3);
		  free($5);
		  if (*$7) free($7);
		}
	|	/* C style syntax */
		for_start '(' opt_expr ';' opt_expr ';' opt_expr ')'
		opt_expr ';' {
		  /* Decrement `indent_count' if we incremented it for
		     the prompt. */
		  dec_indent_if_interactive();

		  if (*$3) {
		    output("%s\n", $3);
		    free($3);
		  }

		  output("while %s:\n", *$5 ? $5 : "True");
		  
		  increment(indent_count);
		  
		  if (!(*$7 || *$9)) {
		    output("pass\n");
		  }
		  else if (minify) {
		    if (!(*$9 && *$7)) {
		      output("%s\n", *$9 ? $9 : $7);
		    }
		    else {
		      output("%s\n%s\n", $9, $7);
		    }

		    if (*$9) free($9);
		    if (*$7) free($7);
		  }
		  else {
		    if (*$9) {
		      output("%s\n", $9);
		      free($9);
		    }
		    if (*$7) {
		      output("%s\n", $7);
		      free($7);
		    }
		  }

		  decrement(indent_count);
		  
		  if (*$5) free($5);
		}
	;

for:		/* Python style syntax */
		for_start '(' STRING IN expr ')' block_start {
		  /* Decrement `indent_count' if we incremented it for
		     the prompt. */
		  dec_indent_if_interactive();

		  output("for %s in %s:\n", $3, $5);

		  increment(indent_count);

		  free($3);
		  free($5);
		}
	|	/* C style syntax */
		for_start '(' opt_expr ';' opt_expr ';' opt_expr ')'
		block_start {
		  /* Decrement `indent_count' if we incremented it for
		     the prompt. */
		  dec_indent_if_interactive();

		  if (*$3) {
		    /* Run this first. `wait_string' will be 2 greater
		       than indent_count since it is incremented in
		       `for_start' and `block_start' (I think). */
		    decrement(wait_string);
		    decrement(wait_string);
		    output("%s\n", $3);
		    increment(wait_string);
		    increment(wait_string);

		    free($3);
		  }
	  
		  output("while %s:\n", *$5 ? $5 : "True");

		  increment(indent_count);
	    
		  if (*$5) free($5);

		  if (*$7) {
		    list_add_new(while_end_expr, struct str_list);
		    while_end_expr->str = $7;

		    list_add_new(while_indent_count, struct indent_count);
		    while_indent_count->count = indent_count;
		  }
		}
	;

sline_do:	do_start expr ';' { dec_indent_if_interactive(); }
		WHILE '(' expr ')' ';' {
		  /* Set the `do' variable to True */
		  decrement(wait_string);
		  output("__c2py_do_%u%s\n",
			 do_count, minify ? "=1" : " = True");
		  increment(wait_string);

		  output("while __c2py_do_%1$u or (%2$s):\n"
			 INDENT_STRING "__c2py_do_%1$u%3$s\n"
			 INDENT_STRING "%4$s\n"
			 "del __c2py_do_%1$u\n",
			 do_count, $7, minify ? "=0" : " = False", $2);

		  free($2);
		  free($7);
		}
	;

do:	        do_start block_start {
		  /* Decrement `indent_count' if we incremented it for
		     the prompt. */
		  dec_indent_if_interactive();

		  list_add_new(do_indent_count, struct indent_count);
		  do_indent_count->count = indent_count;

		  /* Set the `do' variable to True */
		  decrement(wait_string);
		  decrement(wait_string);
		  output("__c2py_do_%u%s\n",
			 do_count, minify ? "=1" : " = True");
		  increment(wait_string);
		  increment(wait_string);

		  increment(do_count);

		  increment(indent_count);
		}
	;

/* Python does not natively support switch (or case) statements. This
   is a hack to add them in. */
switch:		SWITCH '(' expr ')' block_start {
		  list_add_new(switch_indent_count, struct indent_count);
		  switch_indent_count->count = indent_count;
		  
		  output("while %s:\n",
			 minify ? "1" : "True");
		  
		  increment(indent_count);
		  
		  output("__c2py_switch_val_%1$u%2$s%3$s\n"
			 "__c2py_switch_%1$u%4$s\n"
			 /* Don't execute anything until the first
			    `case' (which will automatically exit the
			    `if False'). This is the behavior of
			    switch statements in C. */
			 "if %5$s:\n",
			 switch_count, minify ? "=" : " = ", $3,
			 minify ? "=0" : " = False\n",
			 minify ? "0" : "False");

		  free($3);
		  
		  increment(switch_count);

		  increment(indent_count);

		  /* The `switch' block isn't empty... */
		  not_empty();
		  /* However, the `if False:' is (so far). */
		  next_empty();
		}
	;

case:		CASE expr ':' {
		  if (!switch_count) {
		    yyerror("syntax error, case outside of switch");

		    free($2);
		    YYABORT;
		  }

		  /* Output "pass" here because we don't use
		     `block_end' to end the block. */
		  if (empty_block && empty_block->data)
		    output("pass\n");

		  /* Decrement to get out of the previous case
		     statement or the initial `if False'. */
		  decrement(indent_count);

		  output("if __c2py_switch_%1$u or "
			 "__c2py_switch_val_%1$u%2$s(%3$s):\n",
			 switch_count - 1, minify ? "==" : " == ", $2);

		  free($2);

		  increment(indent_count);

		  output("__c2py_switch_%u%s\n",
			 switch_count - 1, minify ? "=1" : " = True");

		  not_empty();
		}
	|	DEFAULT ':' {
		  if (!switch_count) {
		    yyerror("syntax error, default outside of switch");
		    YYABORT;
		  }

		  /* Output "pass" here because we don't use
		     `block_end' to end the block. */
		  if (empty_block->data)
		    output("pass\n");

		  /* Decrement to get out of the previous case
		     statement or the initial `if False'. */
		  decrement(indent_count);

		  output("if True:\n");

		  increment(indent_count);

		  output("__c2py_switch_%u%sTrue\n",
			 switch_count - 1, minify ? "=" : " = ");

		  not_empty();
		}
	;

except_expr:	opt_expr
	|	expr AS STRING {
		  size_t size = sizeof(*$$) * (strlen($1) + strlen($3) + 5);

		  $$ = malloc(size);
		  snprintf($$, size, "%s as %s", $1, $3);

		  free($1);
		  free($3);
		}
	;

/* Either TRY or FINALLY since they have the same syntax. */
try_finally:	TRY { $$ = "try"; if (!indent_count) wait_string = 1; }
	|	FINALLY { $$ = "finally"; if (!indent_count) wait_string = 1; }
	;

except:		EXCEPT { if (!indent_count) wait_string = 1; }
	;

try_except:	try_finally opt_expr ';' {
		  output(minify ? "%s:%s\n" : "%s:\n" INDENT_STRING "%s\n",
			 $1, *$2 ? $2 : "pass");

		  if (*$2) free($2);
		}
	|	try_finally block_start {
		  output("%s:\n", $1);

		  increment(indent_count);
		}
	|	except except_expr opt_expr ';' {
		  if (*$2) {
		    if (*$3)
		      output(minify ? "except %s:%s\n" : "except %s:\n"
			     INDENT_STRING "%s\n", $2, $3);
		    else
		      /* The user probably meant "except: $2" rather
			 than "except $2: pass". */
		      output(minify ? "except:%s\n" : "except:\n"
			     INDENT_STRING "%s\n", $2);

		    free($2);
		  }
		  else {
		    output(minify ? "except:%s\n" : "except:\n"
			   INDENT_STRING "%s\n", *$3 ? $3 : "pass");
		  }

		  if (*$3) free($3);
		}
	|	except except_expr block_start {
		  if (*$2) {
		    output("except %s:\n", $2);
		    free($2);
		  }
		  else {
		    output("except:\n");
		  }

		  increment(indent_count);
		}
	;

block_start:	'{' {
		  /* Let the symbol that uses block_start increment
		     indent_count so it can print things unindented
		     (e.g. func_def) */
		  /* Tell output() to wait for the whole block */
		  increment(wait_string);

		  /* Set `empty_block' so we remember to output "pass"
		     if the block is empty. */
		  next_empty();
		}
	;

block_end:	'}' {
		  /* This is for while loops translated from c-like
		     for loops */
		  if (while_end_expr && while_indent_count->count == indent_count) {
		    output("%s\n", while_end_expr->str);

		    free(while_end_expr->str);
		    list_free(while_end_expr, struct str_list);
		    list_free(while_indent_count, struct indent_count);
		  }
		  /* Output "pass" if the block was empty. */
		  else if (empty_block && empty_block->data) {
		    output("pass\n");
		  }

		  /* Pop the last `empty_block'. */
		  pop_empty();

		  decrement(wait_string);
		  decrement(indent_count);

		  /* Check if we're inside a `switch' block. If so,
		     decrement `switch_count' and decrement
		     `indent_count' an extra time since we're in
		     another block (the initial `if False' or a
		     case/default statement) which is not ended with a
		     `block_end'. */
		  if (switch_indent_count &&
		      switch_indent_count->count == indent_count - 1) {
		    /* Break out of the `while True' for the switch. */
		    if (!minify)
		      /* Separate the `break' if we're not minifying. */
		      output("\n");
		    output("break\n");
		    
		    decrement(indent_count);
		    decrement(switch_count);

		    /* Clean up the temporary variables */
		    output("del __c2py_switch_%u\n", switch_count);
		    output("del __c2py_switch_val_%u\n", switch_count);

		    /* Output "pass" if the last block BEFORE the
		       `switch' statement was empty. (I know, it's
		       confusing, I hardly understand it myself! Tell
		       the Python developers to add a `switch'
		       statement to Python!)

		       I don't think this is necessary anymore, but
		       I'll keep it (commented out) just in case. */
		    /*
		    if (empty_block && empty_block->data)
		      output("pass\n");
		    */

		    /* Pop the last `empty_block' AGAIN (it seems like
		       the right thing to do, I hope). */
		    pop_empty();
		    
		    /* Pop switch_indent_count */
		    list_free(switch_indent_count, struct indent_count);
		  }

		  if (backtrace && backtrace->indent_count == indent_count) {
		    /* Pop the backtrace. */
		    free(backtrace->name);
		    if (backtrace->prev) {
		      backtrace = backtrace->prev;
		      free(backtrace->next);
		      backtrace->next = NULL;
		    }
		    else {
		      free(backtrace);
		      backtrace = NULL;
		    }
		  }
		  
  #ifdef HAVE_LIBPYTHON
		  /* This will cause output() to run the command since
		     we're done with the block. */
		  if (shebang && !wait_string)
		    output("");
  #endif
		}
	;

operation:	'+'        { $$ = "+"; }
	|	'-'        { $$ = "-"; }
	|	'*'        { $$ = "*"; }
	|	'/'        { $$ = "/"; }
	|	'='        { $$ = "="; }
	|	'<'        { $$ = "<"; }
	|	'>'        { $$ = ">"; }
	|	'&'        { $$ = "&"; }
	|	'|'        { $$ = "|"; }
	|	'^'        { $$ = "^"; }
	|	PLUS_EQ    { $$ = "+="; }
	|	MINUS_EQ   { $$ = "-="; }
	|	MUL_EQ     { $$ = "*="; }
	|	DIV_EQ     { $$ = "/="; }
	|	POWER      { $$ = "**"; }
	|	INT_DIV    { $$ = "//"; }
	|	LESS_EQ    { $$ = "<="; }
	|	GREATER_EQ { $$ = ">="; }
	|	NOT_EQUAL  { $$ = "!="; }
	|	EQUAL      { $$ = "=="; }
	|	BOOL_AND   { $$ = "and"; }
	|	BOOL_OR    { $$ = "or"; }
	;

sign:		'+' { $$ = "+"; }
	|	'-' { $$ = "-"; }
	|	/* '.' so .5 means 0.5 (. 5 will also mean that, but
		   whatever). */
		'.' { $$ = "."; }
	;

/* An optional expression (can be empty). */
opt_expr:	%empty { $$ = ""; }
		/* Set `$$' explicitly to get rid of the "type clash"
		   warning. */
	|	expr { $$ = $1; }
	;

/* A string or string literal with operations. */
expr:		STRING
	|	string_literal
	|	/* Lists */
		'[' args ']' {
		  size_t size = sizeof(*$$) * (strlen($2) + 3);

		  $$ = malloc(size);
		  snprintf($$, size, "[%s]", $2);

		  if (*$2) free($2);
		}
	|	/* Tuples */
		'(' args ')' {
		  size_t size = sizeof(*$$) * (strlen($2) + 3);

		  $$ = malloc(size);
		  snprintf($$, size, "(%s)", $2);

		  if (*$2) free($2);
		}
	|	/* Indices */
		expr '[' expr ']' {
		  size_t size = sizeof(*$$) * (strlen($1) + strlen($3) + 3);

		  $$ = malloc(size);
		  snprintf($$, size, "%s[%s]", $1, $3);

		  free($1);
		  free($3);
		}
	|	/* Variable declarations */
		STRING STRING {
		  size_t size = sizeof(*$$) *
		    (strlen($1) + strlen($2) + (minify ? 4 : 6));

		  $$ = malloc(size);
		  snprintf($$, size, minify ? "%s=%s()" : "%s = %s()",
			   $2, $1);

		  free($1);
		  free($2);
		}
	|	/* Parentheses for precedence */
		'(' expr ')' {
		  size_t size = sizeof(*$$) * (strlen($2) + 3);

		  $$ = malloc(size);
		  snprintf($$, size, "(%s)", $2);

		  free($2);
		}
	|	/* Function call */
		expr '(' args ')' {
		  size_t size = sizeof(*$$) * (strlen($1) + strlen($3) + 3);

		  $$ = malloc(size);
		  snprintf($$, size, "%s(%s)", $1, $3);

		  free($1);
		  if (*$3) free($3);
		}
	|	/* Class variable access */
		expr '.' STRING {
		  size_t size = sizeof(*$$) * (strlen($1) + strlen($3) + 2);

		  $$ = malloc(size);
		  snprintf($$, size, "%s.%s", $1, $3);

		  free($1);
		  free($3);
		}
	|	/* Method call */
		expr '.' STRING '(' args ')' {
		  size_t size = sizeof(*$$) *
		    (strlen($1) + strlen($3) + strlen($5) + 4);

		  $$ = malloc(size);
		  snprintf($$, size, "%s.%s(%s)", $1, $3, $5);

		  free($1);
		  free($3);
		  if (*$5) free($5);
		}
	|	/* Variable operation */
		expr operation expr {
		  size_t size = sizeof(*$$) *
		    (strlen($1) + strlen($2) + strlen($3)
		     + ((minify && !isalpha($2[0])) ? 1 : 3));

		  $$ = malloc(size);
		  snprintf($$, size, (minify && !isalpha($2[0])) ?
			   "%s%s%s" : "%s %s %s", $1, $2, $3);

		  free($1);
		  free($3);
		}
	|	/* Explicit sign */
		sign expr {
		  size_t size = sizeof(*$$) * (strlen($2) + 2);

		  $$ = malloc(size);
		  snprintf($$, size, "%c%s", *$1, $2);

		  free($2);
		}
	|	/* Variable declaration and definition */
		STRING STRING '=' expr {
		  size_t size = sizeof(*$$) *
		    (strlen($1) + strlen($2) + strlen($4)
		     + (minify ? 4 : 6));

		  $$ = malloc(size);
		  snprintf($$, size, minify ? "%s=%s(%s)" : "%s = %s(%s)",
			   $2, $1, $4);

		  free($1);
		  free($2);
		  free($4);
		}
	|	/* ? : syntax */
		expr '?' expr ':' expr {
		  size_t size = sizeof(*$$) *
		    (strlen($1) + strlen($3) + strlen($5) + 11);

		  $$ = malloc(size);
		  snprintf($$, size, "%s if %s else %s", $3, $1, $5);

		  free($1);
		  free($3);
		  free($5);
		}
	|	/* Boolean `not' */
		'!' expr {
		  size_t size = sizeof(*$$) * (strlen($2) + 5);

		  $$ = malloc(size);
		  snprintf($$, size, "not %s", $2);

		  free($2);
		}
	;

/* This is so we can have empty arguments without allowing things like
   `func(,,);' */
args:		%empty { $$ = ""; }
		/* Set `$$' explicitly to get rid of the "type clash"
		   warning. */
	|	args_real { $$ = $1; }
	|	/* Comma separated lists allow an optional comma at
		   the end. I guess so one-member tuples can be
		   distinguished from parentheses for precedence. And
		   for the other places where comma separated lists
		   are applied, just to be consistent with the tuple
		   syntax (is my guess). There is probably more info
		   in the official Python documentation. */
		args_real ',' {
		  size_t size = sizeof(*$$) * (strlen($1) + 2);

		  $$ = malloc(size);

		  snprintf($$, size, "%s,", $1);

		  free($1);
		}
	;

args_real:	expr
	|	args_real ',' args_real {
		  size_t size = sizeof(*$$) *
		    (strlen($1) + strlen($3) + (minify ? 2 : 3));

		  $$ = malloc(size);

		  snprintf($$, size, minify ? "%s,%s" : "%s, %s", $1, $3);

		  free($1);
		  free($3);
		}
	;

/* A lexical error. */
yylex_error:	YYLEX_ERROR {
		  if (!interactive)
		    YYABORT;
		}
	;

%%

/* We have to put this in the epilogue so we can use the token macros */
static const struct {char *str; int token;} multi_char_tokens[] =
  {
   {"import",  IMPORT },
   {"from",    FROM   },
   {"as",      AS     },
   {"for",     FOR    },
   {"do",      DO     },
   {"while",   WHILE  },
   {"in",      IN     },
   {"if",      IF     },
   {"elif",    ELIF   },
   {"else",    ELSE   },
   {"switch",  SWITCH },
   {"case",    CASE   },
   {"default", DEFAULT},
   {"try",     TRY    },
   {"except",  EXCEPT },
   {"finally", FINALLY},
   {"del",     DEL    },
   {"global",  GLOBAL },
   {"class",   CLASS  },
   {"return",  RETURN }
  };

static void yyerror(const char *message) {
  fprintf(stderr, "%s:%u:%u", *input_file_names, last_line, last_column);

  if (backtrace) {
    /* Save the `backtrace' because we will overwrite it. */
    struct backtrace *backtrace_save = backtrace;
    
    /* Go to the beginning of `backtrace' and print it from there to
       be more like Python's "traceback"s (most recent call last). */
    while (backtrace->prev)
      backtrace = backtrace->prev;
    
    /* Print the backtrace. */
    while (backtrace) {
      fprintf(stderr, "\n%u:%u in %s %s", backtrace->line, backtrace->column,
	      (backtrace->type == class) ? "class" :
	      (backtrace->type == func) ? "function" : "method",
	      backtrace->name);
      
      backtrace = backtrace->next;
    }

    /* Restore the backtrace. */
    backtrace = backtrace_save;

    /* Now print the error message. */
    fprintf(stderr, "\n%s\n", message);
  }
  else {
    /* No need for a newline since we didn't print a backtrace. */
    fprintf(stderr, ": %s\n", message);
  }

#ifdef ENABLE_POLYRUN
  /* Print a sorry message if polyrun is installed (see polyrun(1) and
     polygen(1)). */
  system("polyrun sorry >&2 2> /dev/null");
#endif
}

static int yylex(void) {
  int c, d;
  llist_t *input_chars;
  char quote = 0, mline_quote = 0;

  /* Skip white space and comments. */
  while (1) {
    d = 0;

    while (isspace(c = get_input())) {
      /* Update location in file. */
      if (c == '\n') {
	/* Save the column in case we unget_input() c. */
	column_save = column;
	column = 0;
	line++;
      }
      else {
	column++;
      }
      
      if (!minify) {
	if (c == '\n' && d == '\n')
	  output("\n");

	d = c;
      }
    }

    if (c == EOF) {
      /* Make sure that all {'s are matched. */
      if (indent_count) {
	yyerror("syntax error, unmatched `{'");
	return YYLEX_ERROR;
      }
      
      return 0;
    }

    /* Update location in file. */
    if (c == '\n') {
      column_save = column; /* Save the column in case we unget_input() c. */
      column = 0;
      line++;
    }
    else {
      column++;
    }
      
    last_line = line;
    last_column = column;

    /* Skip comments. Note that we can't use // since that is used for
       integer division. */
    /* Multi-line comment. */
    if (c == '/') {
      d = c;
      c = get_input();

      if (c == EOF)
	return '/';
      
      /* Update location in file. */
      if (c == '\n') {
	/* Save the column in case we unget_input() c. */
	column_save = column;
	column = 0;
	line++;
      }
      else {
	column++;
      }
      
      if (c == '*') {
	if (!minify)
	  output("#");

	while (1) {
	  d = c;
	  c = get_input();

	  /* Check for EOF. */
	  if (c == EOF) {
	    yyerror("syntax error, unterminated comment");

	    return YYLEX_ERROR;
	  }

	  /* Update location in file. */
	  if (c == '\n') {
	    /* Save the column in case we unget_input() c. */
	    column_save = column;
	    column = 0;
	    line++;
	  }
	  else {
	    column++;
	  }
      
	  /* Check for the end of the comment. */
	  if (c == '*') {
	    d = c;
	    c = get_input();

	    /* Check for EOF. */
	    if (c == EOF) {
	      yyerror("syntax error, unterminated comment");

	      return YYLEX_ERROR;
	    }

	    /* Update location in file. */
	    if (c == '\n') {
	      /* Save the column in case we unget_input() c. */
	      column_save = column;
	      column = 0;
	      line++;
	    }
	    else {
	      column++;
	    }
      
	    if (c == '/') {
	      break;
	    }
	    else {
	      unget_input(c);

	      /* Update location in file. */
	      if (c == '\n') {
		column = column_save;
		line--;
	      }
	      else {
		column--;
	      }
      
	      c = d;
	    }
	  }

	  /* We haven't reached the end yet. */
	  if (!minify) {
	    output_no_indent = 1;
	    output("%c", c);
	    output_no_indent = 0;
	    
	    if (c == '\n') {
	      output("#");
	    }
	  }
	}

	if (!minify)
	  output("\n");
      }
      else {
	unget_input(c);

	/* Update location in file. */
	if (c == '\n') {
	  column = column_save;
	  line--;
	}
	else {
	  column--;
	}
      
	c = d;
      }
    }
    /* Single-line comment. EOF's are ok for single line comments
       because a single line comment without a '\n' at the end of a
       file is legitimate. */
    else if (c == '#') {
      /* Indent it correctly. */
      if (!minify)
	output("#");
      
      do {
	c = get_input();

	if (c != EOF) {
	  /* Update location in file. */
	  if (c == '\n') {
	    /* Save the column in case we unget_input() c. */
	    column_save = column;
	    column = 0;
	    line++;
	  }
	  else {
	    column++;
	  }
      
	  if (!minify) {
	    output_no_indent = 1;
	    output("%c", c);
	    output_no_indent = 0;
	  }
	}
      } while (c != '\n' && c != EOF);

      if (c != EOF) {
	unget_input(c);

	/* Update location in file. */
	if (c == '\n') {
	  column = column_save;
	  line--;
	}
	else {
	  column--;
	}
      }
    }
    else {
      break;
    }
  }

  if (strchr(TOKEN_STRING, c)) {
    d = get_input();

    if (d == EOF) {
      return c;
    }
      
    /* Update location in file. */
    if (d == '\n') {
      column_save = column; /* Save the column in case we unget_input() c. */
      column = 0;
      line++;
    }
    else {
      column++;
    }
      
    /* Check for ==, +=, etc. */
    switch (c) {
    case '!':
      if (d == '=')
	return NOT_EQUAL;
      break;
    case '&':
      if (d == '&')
	return BOOL_AND;
      break;
    case '|':
      if (d == '|')
	return BOOL_OR;
      break;
    case '=':
      if (d == '=')
	return EQUAL;
      break;
    case '<':
      if (d == '=')
	return LESS_EQ;
      break;
    case '>':
      if (d == '=')
	return GREATER_EQ;
      break;
    case '+':
      switch (d) {
      case '=':
	return PLUS_EQ;
      case '+':
	/* Make `++' behave like `+= 1' */
	unget_input('1');

	/* Make sure we don't mess up the location. */
	column--;
      
	return PLUS_EQ;
      }
      break;
    case '-':
      switch (d) {
      case '=':
	return MINUS_EQ;
      case '-':
	/* Make `--' behave like `-= 1' */
	unget_input('1');

	/* Make sure we don't mess up the location. */
	column--;
      
	return MINUS_EQ;
      }
      break;
    case '*':
      switch (d) {
      case '=':
	return MUL_EQ;
      case '*':
	return POWER;
      }
      break;
    case '/':
      switch (d) {
      case '=':
	return DIV_EQ;
      case '/':
	return INT_DIV;
      }
      break;
    }

    unget_input(d);

    /* Update location in file. */
    if (d == '\n') {
      column = column_save;
      line--;
    }
    else {
      column--;
    }
      
    return c;
  }

  /* Check if we're quoted. */
  if (strchr(QUOTE_CHARS, c))
    quote = c;

  llist_create(input_chars);

  /* Get the input string */
  do {
    /* If c is '\0' (not EOF), that means the user was trying to
       confuse us. Obviously, the user mistook us for your average
       dumb program. We are smarter than that but still, not cool. */
    if (!c) {
      fputs("Hey! That's not nice! :(\n", stderr);
      
      return YYLEX_ERROR;
    }

    if (c == quote && input_chars->prev) {
      if (input_chars->prev->prev) {
	if (input_chars->prev->data == quote &&
	    input_chars->prev->prev->data == quote) {
	  if (mline_quote)
	    break;

	  mline_quote = 1;

	  /* Handle empty multi-line string literals. */
	  input_chars->data = c;
	  llist_add_forward(input_chars);
	  llist_next(input_chars);

	  c = get_input();

	  if (c != EOF) {
	    /* Update location in file. */
	    if (c == '\n') {
	      /* Save the column in case we unget_input() c. */
	      column_save = column;
	      column = 0;
	      line++;
	    }
	    else {
	      column++;
	    }
	  }
      
	  if (c == quote) {
	    c = get_input();

	    if (c != EOF) {
	      /* Update location in file. */
	      if (c == '\n') {
		/* Save the column in case we unget_input() c. */
		column_save = column;
		column = 0;
		line++;
	      }
	      else {
		column++;
	      }
	    }
      
	    input_chars->data = c;
	    llist_add_forward(input_chars);
	    llist_next(input_chars);
	    
	    if (c == quote) {
	      c = get_input();

	      if (c != EOF) {
		/* Update location in file. */
		if (c == '\n') {
		  /* Save the column in case we unget_input() c. */
		  column_save = column;
		  column = 0;
		  line++;
		}
		else {
		  column++;
		}
	      }
      
	      input_chars->data = c;
	      llist_add_forward(input_chars);
	      llist_next(input_chars);
	    
	      if (c == quote) {
		break;
	      }
	    }
	  }
	}
      
	if (!mline_quote && input_chars->prev->data != '\\')
	  break;
      }
      /* Handle empty single-line string literals. */
      else if (input_chars->prev->data == quote) {
	unget_input(c = get_input());

	if (c == EOF) {
	  yyerror("syntax error, unexpected EOF when scanning string literal");
	  llist_destroy(input_chars);

	  return YYLEX_ERROR;
	}

	if (c != quote) {
	  c = quote;

	  break;
	}

	c = quote;
      }
    }

    input_chars->data = c;
    llist_add_forward(input_chars);
    llist_next(input_chars);

    c = get_input();

    /* Update location in file. */
    if (c == '\n') {
      column_save = column; /* Save the column in case we unget_input() c. */
      column = 0;
      line++;
    }
    else {
      column++;
    }
      
    if (quote) {
      if (c == EOF) {
	if (mline_quote)
	  yyerror("syntax error, unexpected end of file "
		  "when scanning triple-quoted string literal");
	else
	  yyerror("syntax error, unexpected end of "
		  "file when scanning string literal");

	llist_destroy(input_chars);

	return YYLEX_ERROR;
      }
      else if (c == '\n' && input_chars->prev->data != '\\' && !mline_quote) {
	yyerror("syntax error, unexpected end of "
		"line when scanning string literal");
	llist_destroy(input_chars);
	
	return YYLEX_ERROR;
      }
    }
  } while (quote || (c != EOF && !isspace(c) &&
		      !strchr(TOKEN_STRING "\"'", c)));

  /* Add the quote character. */
  if (quote) {
    input_chars->data = c;
    llist_add_forward(input_chars);
    llist_next(input_chars);
  }
  /* Make sure we catch one-character tokens such as '(', ')', ',',
     and ';'. */
  else if (c != EOF && !isspace(c)) {
    unget_input(c);

    /* Update location in file. */
    if (c == '\n') {
      column = column_save;
      line--;
    }
    else {
      column--;
    }
      
    c = quote;
  }

  /* Convert the list to a string. */
  yylval.string = llist_to_array(input_chars);

  /* Free the list. */
  llist_destroy(input_chars);

  /* Check if the string is a multi-character token unless it's a
     string literal. */
  if (!quote) {
    for (int i = 0;
	 i < sizeof(multi_char_tokens) / sizeof(*multi_char_tokens); i++) {
      if (!strcmp(yylval.string, multi_char_tokens[i].str)) {
	free(yylval.string);
	yylval.string = NULL;

	return multi_char_tokens[i].token;
      }
    }
  }

  return quote ? STRING_LITERAL : STRING;
}

/* Initialize `quoted_pos' from a string

   FIXME: This is less efficient than it should be because we parse
   for quotes twice (once in yylex()) */
static void quoted_init(const char *str) {
  enum { none, single, Double, /* Capital D to distinguish from the
				  `double' type */
	 triple_single, triple_double } quote_type = none;
  size_t length = strlen(str);

  for (size_t i = 0; i < length; i++) {
    if (i && str[i-1] != '\\' && (str[i] == '\'' || str[i] == '"')) {
      if (quote_type == none) {
	/* Check if it's triple-quoted */
	if (i + 2 < length && str[i+1] == str[i] && str[i+2] == str[i])
	  quote_type = str[i] == '\'' ? triple_single : triple_double;
	else
	  quote_type = str[i] == '\'' ? single : Double;

	/* Add it to the list of quoted positions */
	list_add_new(quoted_pos, struct quoted_pos);
	quoted_pos->start = i;
      }
      /* Check if we're ending a quote */
      else if ((quote_type == single && str[i] == '\'') ||
	       (quote_type == Double && str[i] == '"') ||
	       (i + 2 < length &&
		((quote_type == triple_single &&
		  str[i] == '\'' && str[i+1] == '\'' && str[i+2] == '\'') ||
		 (quote_type == triple_double &&
		  str[i] == '"' && str[i+1] == '"' && str[i+2] == '"')))) {
	if (quote_type == triple_single || quote_type == triple_double)
	  quoted_pos->end = i + 2;
	else
	  quoted_pos->end = i;
      }
    }
  }
}

/* Check if `pos' is quoted (we shouldn't indent it) */
static int quoted(size_t pos) {
  /* The position we are currently checking */
  struct quoted_pos *quoted_pos_cur = quoted_pos;

  while (quoted_pos_cur) {
    if (pos >= quoted_pos_cur->start && pos <= quoted_pos_cur->end)
      return 1;

    quoted_pos_cur = quoted_pos_cur->prev;
  }

  /* If we get here, `pos' wasn't quoted */
  return 0;
}

/* This is used in indent() */
#define _indent_add_char(c) do {		\
    if (indent_str && index) {			\
      indent_str[(*index)++] = (c);		\
    }						\
    else {					\
      if (putc((c), output_file) == EOF)	\
	return -1;				\
    }						\
  } while (0)

/* Indent a string, returning the number of spaces added */
static int indent(const char *str, char *indent_str,
		  size_t size, size_t *index) {
  int ret = 0;

  for (size_t i = 0; i < size; i++) {
    /* If i == 0, we want to indent first. Wait until after we indent
       it. */
    if (i)
      _indent_add_char(str[i]);

    /* We should only indent it if there is a newline if the newline
       is not the end of the string. This is because following calls
       to output() will add to the indent and not indent correctly. We
       also should not indent if we are inside a quoted string. */
    if (!i || (i != size - 1 && str[i] == '\n' && !quoted(i))) {
      /* Now add the correct number of spaces. */
      for (int j = 0; j < indent_count * INDENT_SIZE; j++) {
	_indent_add_char(' ');
	ret++;
      }
    }

    /* Now copy the character if we hadn't before. */
    if (!i)
      _indent_add_char(str[i]);
  }

  /* Add the '\0' if we're outputing to a string */
  if (indent_str && index)
    indent_str[*index] = '\0';

  return ret;
}

/* Output the string to output_file or run it with the embedded
   interpreter if available, preceding with the appropriate
   indenting */
#ifndef HAVE_LIBPYTHON
/* get_input() does not use output() if we're not compiled with
   libreadline. */
static
#endif
int output(const char *format, ...) {
  int size;
  char *str = NULL, *do_str_run = NULL;
  size_t do_str_run_size = 0;
  int ret;
  va_list ap;
  static unsigned int do_str_count = 0;
  static struct str_size_list {
    struct str_size_list *prev;
    char *str;
    size_t size, index;
  } *do_str = NULL;

  /* Determine size of string (vsnprintf() returns the number of bytes
     that WOULD have been returned if it was not truncated. So we can
     use it with a size of 0 to see how much we should allocate.) */
  va_start(ap, format);
  size = vsnprintf(NULL, 0, format, ap);
  va_end(ap); /* You cannot reuse `ap' (I learned that the hard way). */

  /* Check for errors. */
  if (size < 0)
    return size;

  size++; /* For '\0' */

  /* Allocate and set the string. */
  if (!(str = malloc(size)))
    return -1;

  va_start(ap, format);
  ret = vsnprintf(str, size, format, ap);
  va_end(ap);

  /* This should not happen, but check for it anyway. */
  if (ret >= size)
    ret = size - 1;

  /* We don't need `size' anymore. We do, however, need the length of
     the string minus the terminating '\0' which just happens to be
     `ret'. However, since we are going to change `ret', we can set
     `size' to `ret' and use that instead. */
  size = ret;

  /* Parse the string for quotes */
  quoted_init(str);

  /* Add a new member to `do_str' if we need to */
  if (do_str_count < do_count) {
    do_str_count++;
    list_add_new(do_str, struct str_size_list);
    do_str->str = NULL;
    do_str->size = 1; // 1 for the terminating '\0'
    do_str->index = 0;
  }
  else if (do_str_count > do_count) {
    do_str_run = do_str->str;
    do_str_run_size = do_str->size - 1;

    do_str_count--;
    list_free(do_str, struct str_size_list);
  }
  assert(do_str_count == do_count);

  /* Add the string to the current `do_str' if we need to */
  if (do_str) {
    /* Calculate the size of the string */
    do_str->size += size + do_str_run_size;

    if (!output_no_indent) {
      for (char *pos = str; pos && *pos;) {
	if (!quoted((pos - str) / sizeof(*pos)))
	  do_str->size += indent_count * INDENT_SIZE;

	if ((pos = strchr(pos, '\n'))) pos++;
      }
    }

    if (!(do_str->str =
	  realloc(do_str->str, sizeof(*(do_str->str)) * do_str->size)))
      return -1;

    if (output_no_indent) {
      strncat(do_str->str, str, size);
      do_str->index += size;
    }
    else {
      ret += indent(str, do_str->str, size, &(do_str->index));
    }

    if (do_str_run) {
      /* Add `do_str_run' */
      strncat(do_str->str, do_str_run, do_str_run_size);
      do_str->index += do_str_run_size;
    }
  }
#ifdef HAVE_LIBPYTHON
  else if (shebang) {
    static char *exec_str = NULL; // The string to execute
    static size_t exec_str_size = 1; // 1 for the terminating '\0'
    static size_t index = 0;

    /* Calculate the size of the string */
    exec_str_size += size + do_str_run_size;

    if (!output_no_indent) {
      for (char *pos = str; pos && *pos;) {
	if (!quoted((pos - str) / sizeof(*pos)))
	  exec_str_size += indent_count * INDENT_SIZE;

	if ((pos = strchr(pos, '\n'))) pos++;
      }
    }

    if (!(exec_str = realloc(exec_str, sizeof(*exec_str) * exec_str_size))) {
      free(str);
      return -1;
    }

    if (output_no_indent) {
      strncat(exec_str, str, size);
      index += size;
    }
    else {
      /* Now copy `str' to `exec_str' but indent correctly. We can
	 safely assume that indent() did not return -1, because it
	 only does that when outputting to `output_file'. */
      ret += indent(str, exec_str, size, &index);
    }

    if (do_str_run) {
      /* Add `do_str_run' */
      strncat(exec_str, do_str_run, do_str_run_size);
      index += do_str_run_size;
    }

    /* Run the string if it's not the start of a block ("if foo:\n"
       for example) and it's not part of a block. This is because we
       need to run the whole block at the same time ("if foo:\n
       bar()\n") so PyRun_String() doesn't complain about an
       unexpected EOF. */
    if (!wait_string) {
      PyObject *pymain, *pyvars, *pyret;
      int error_occured = 0;
      
      assert(index == exec_str_size - 1);

      exec_str[index] = '\0';

      /* Get a reference to the __main__ module and get the variables
	 for `globals' and `locals' to pass to PyRun_String(). */
      if (!(pymain = PyImport_AddModule("__main__")) ||
	  !(pyvars = PyModule_GetDict(pymain))) {
	error_occured = 1;
      }
      /* Now run the string if it's not empty */
      else if (*exec_str) {
	if (!(pyret = PyRun_String(exec_str, interactive ?
				   Py_single_input : Py_file_input,
				   pyvars, pyvars))) {
	  PyErr_Print();
	
	  if (!interactive)
	    exit(1);
	}
	else {
	  Py_DECREF(pyret);
	}
      }

      /* Reset the static variables. */
      free(exec_str);
      exec_str = NULL;
      exec_str_size = 1;
      index = 0;

      /* Check if an error occured. */
      if (error_occured) {
	free(str);

	if (!interactive)
	  exit(1);

# ifdef HAVE_LIBREADLINE
	return -1;
# endif
      }
    }
  }
#endif /* HAVE_LIBPYTHON */
  else {
    /* Now print the string to the output file, indenting
       correctly if needed. */
    if (output_no_indent) {
      if (fputs(str, output_file) == EOF)
	return -1;
    }
    else {
      int tmp;

      if ((tmp = indent(str, NULL, size, NULL)) == -1) {
	if (do_str_run) free(do_str_run);
	return -1;
      }

      ret += tmp;
    }

    /* Print `do_str_run' */
    if (do_str_run && fputs(do_str_run, output_file) == EOF) {
      free(do_str_run);
      return -1;
    }
  }

  if (do_str_run) free(do_str_run);

  /* Free the list of quoted positions */
  while (quoted_pos)
    list_free(quoted_pos, struct quoted_pos);

  free(str);

  return ret;
}
