#####################################
# Blended Radiance brad_sun.py      #
#####################################

##############################
# (c) Francesco Anselmo 2005 #
##############################

#    This file is part of brad (Blended RADiance).
#
#    brad is free software; you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation; either version 2 of the License, or
#    (at your option) any later version.
#
#    brad is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
#
#    You should have received a copy of the GNU General Public License
#    along with Foobar; if not, write to the Free Software
#    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

import math
#import Blender

class SunTime:
    def __init__(self, latitude, longitude, meridian, month, day, hour, minute, timetype):
        self.latitude = latitude
        self.longitude = longitude
        self.meridian = meridian
        self.month = month
        self.day = day
        self.daynumber = self.getDayNumber(month, day)
        self.hour = hour
        self.minute = minute
        # timetype can be "solar" or "clock"
        self.timetype = timetype
        self.sunaltitude = 0
        self.sunazimuth = 0
        self.SUN_ANGLE = 9.3084E-03
        self.SUN_SOLID_ANGLE = 6.805E-05
        self.RAD = math.pi / 180.0
        self.DEG = 180.0 / math.pi
        self.R = 1

    #get day number
    def getDayNumber(self, month, day):
        days = [0,
        31, #jan
        28, #feb
        31, #mar
        30, #apr
        31, #may
        30, #jun
        31, #jul
        31, #aug
        30, #sep
        31, #oct
        30, #nov
        31 #dec
        ]
        dn = 0
        for m in range(month):
           dn = dn + days[m]
        dn = dn + day
        return dn

    # get day angle
    def getDayAngle(self):
        return ((self.daynumber / 365.25) * 360.0)

    # get local mean time
    def getLocalMeanTime(self):
        return self.hour + (self.minute / 60.0)

    # get equation of time
    def getEquationOfTime(self):
       da = self.getDayAngle()
       et = (-0.128 * math.sin((da - 2.8)*self.RAD)) - (0.165 * math.sin(((2.0*da) + 19.7)*self.RAD))
       return et

    # get local apparent time (no daylight saving!)
    def getLocalApparentTime(self):
       return self.getLocalMeanTime() + ((self.longitude - self.meridian) / 15.0) + self.getEquationOfTime()

    def getMeridianAdjustment(self):
       return ((self.longitude - self.meridian) / 15.0)

    # get solar declination
    def getSolarDeclination(self):
       da = self.getDayAngle()
       sd = math.asin(0.3978 * math.sin((da - 80.2 + (1.92 * math.sin((da - 2.8)*self.RAD)))*self.RAD))
       if self.timetype == "clock":
          return sd
       else:
          return 0.4093*math.sin(((2*math.pi)/365)*(284+self.daynumber))

    # get solar hour angle
    def getSolarHourAngle(self):
        if self.timetype == "clock":
           return (15.0 * (self.getLocalApparentTime() - 12.0) * self.RAD)
        else:
           return math.pi+self.getLocalMeanTime()*0.2618

    # get solar altitude
    def getSolarAltitude(self):
        sd = self.getSolarDeclination()
        sha = self.getSolarHourAngle()
        salt = math.asin((math.sin(self.latitude * self.RAD)*math.sin(sd))+(math.cos(self.latitude * self.RAD)*math.cos(sd)*math.cos(sha)))
        if salt > 0:
           return salt
        else:
           return 0.0

     # get solar altitude (degrees)
    def getSolarAltitudeDEG(self):
        return self.getSolarAltitude() * self.DEG

    # get solar azimuth
    def getSolarAzimuth(self):
      if self.timetype == "clock" or self.timetype == "solar":
        sa = self.getSolarAltitude()
        if sa>0:
           sd = self.getSolarDeclination()
           sha = self.getSolarHourAngle()
           caz = ((math.sin(self.latitude * self.RAD) * math.sin(sa)) - math.sin(sd)) / (math.cos(self.latitude * self.RAD) * math.cos(sa))
           saz = ((math.cos(sd) * math.sin(sha)) / math.cos(sa))
           if self.latitude<0:
               caz = -caz
           if saz<0:
               return -math.acos(caz)
           else:
               return math.acos(caz)
        else:
           return 0.0

    # get solar azimuth (degrees)
    def getSolarAzimuthDEG(self):
        return self.getSolarAzimuth() * self.DEG

    # get solar x coordinate
    def getX(self):
        return self.R * math.cos(self.getSolarAltitude()) * math.cos(self.getSolarAzimuth())

    # get solar y coordinate
    def getY(self):
        return self.R * math.cos(self.getSolarAltitude()) * math.sin(self.getSolarAzimuth())

    # get solar z coordinate
    def getZ(self):
        return self.R * math.sin(self.getSolarAltitude())
# class SunTime: end

# class SunClockTime: begin
class SunClockTime:
    #def __init__(self, latitude, longitude, meridian, daynumber, hour, minute):
    def __init__(self, latitude, longitude, meridian, month, day, hour, minute):
        self.latitude = latitude
        self.longitude = longitude
        self.meridian = meridian
        self.month = month
        self.day = day
        self.daynumber = self.getDayNumber(month, day)
        self.hour = hour
        self.minute = minute
        self.sunaltitude = 0
        self.sunazimuth = 0
        self.SUN_ANGLE = 9.3084E-03
        self.SUN_SOLID_ANGLE = 6.805E-05
        self.RAD = math.pi / 180.0
        self.DEG = 180.0 / math.pi
        self.R = 1

    #get day number
    def getDayNumber(self, month, day):
        days = [0,
        31, #jan
        28, #feb
        31, #mar
        30, #apr
        31, #may
        30, #jun
        31, #jul
        31, #aug
        30, #sep
        31, #oct
        30, #nov
        31 #dec
        ]
        dn = 0
        for m in range(month):
           dn = dn + days[m]
        dn = dn + day
        return dn

    # get day angle
    def getDayAngle(self):
        return ((self.daynumber / 365.25) * 360.0)

    # get local mean time
    def getLocalMeanTime(self):
        return self.hour + (self.minute / 60.0)

    # get equation of time
    def getEquationOfTime(self):
       da = self.getDayAngle()
       et = (-0.128 * math.sin((da - 2.8)*self.RAD)) - (0.165 * math.sin(((2.0*da) + 19.7)*self.RAD))
       return et

    # get local apparent time (no daylight saving!)
    def getLocalApparentTime(self):
       return self.getLocalMeanTime() + ((self.longitude - self.meridian) / 15.0) + self.getEquationOfTime()

    # get solar declination
    def getSolarDeclination(self):
       da = self.getDayAngle()
       sd = math.asin(0.3978 * math.sin((da - 80.2 + (1.92 * math.sin((da - 2.8)*self.RAD)))*self.RAD))
       return sd

    # get solar hour angle
    def getSolarHourAngle(self):
        return (15.0 * (self.getLocalApparentTime() - 12.0) * self.RAD)

    # get solar altitude
    def getSolarAltitude(self):
        sd = self.getSolarDeclination()
        sha = self.getSolarHourAngle()
        salt = math.asin((math.sin(self.latitude * self.RAD) * math.sin(sd)) + (math.cos(self.latitude * self.RAD) * math.cos(sd) * math.cos(sha)))
        if salt > 0:
           return salt
        else:
           return 0.0

    # get solar altitude (degrees)
    def getSolarAltitudeDEG(self):
        return self.getSolarAltitude() * self.DEG

    # get solar azimuth
    def getSolarAzimuth(self):
        sa = self.getSolarAltitude()
        if sa>0:
           sd = self.getSolarDeclination()
           sha = self.getSolarHourAngle()
           caz = ((math.sin(self.latitude * self.RAD) * math.sin(sa)) - math.sin(sd)) / (math.cos(self.latitude * self.RAD) * math.cos(sa))
           saz = ((math.cos(sd) * math.sin(sha)) / math.cos(sa))
           if self.latitude<0:
               caz = -caz
           if saz<0:
               return -math.acos(caz)
           else:
               return math.acos(caz)
        else:
           return 0.0

    # get solar azimuth (degrees)
    def getSolarAzimuthDEG(self):
        return self.getSolarAzimuth() * self.DEG

    # get solar x coordinate
    def getX(self):
        return self.R * math.cos(self.getSolarAltitude()) * math.cos(self.getSolarAzimuth())

    # get solar y coordinate
    def getY(self):
        return self.R * math.cos(self.getSolarAltitude()) * math.sin(self.getSolarAzimuth())

    # get solar z coordinate
    def getZ(self):
        return self.R * math.sin(self.getSolarAltitude())
# class SunClockTime: end

# class SunSolarTime: begin
class SunSolarTime:
    #def __init__(self, latitude, longitude, meridian, daynumber, hour, minute):
    def __init__(self, latitude, longitude, meridian, month, day, hour, minute):
        self.latitude = latitude
        self.longitude = longitude
        self.meridian = meridian
        self.month = month
        self.day = day
        self.daynumber = self.getDayNumber(month, day)
        self.hour = hour
        self.minute = minute
        self.sunaltitude = 0
        self.sunazimuth = 0
        self.SUN_ANGLE = 9.3084E-03
        self.SUN_SOLID_ANGLE = 6.805E-05
        self.RAD = math.pi / 180.0
        self.DEG = 180.0 / math.pi
        self.R = 1

    #get day number
    def getDayNumber(self, month, day):
        days = [0,
        31, #jan
        28, #feb
        31, #mar
        30, #apr
        31, #may
        30, #jun
        31, #jul
        31, #aug
        30, #sep
        31, #oct
        30, #nov
        31 #dec
        ]
        dn = 0
        for m in range(month):
           dn = dn + days[m]
        dn = dn + day
        return dn

    # get day angle
    def getDayAngle(self):
        return ((self.daynumber / 365.25) * 360.0)

    # get local mean time
    def getLocalMeanTime(self):
        return self.hour + (self.minute / 60.0)

    # get solar declination
    def getSolarDeclination(self):
       return 0.4093*math.sin(((2*math.pi)/365)*(284+self.daynumber))

    # get solar hour angle
    def getSolarHourAngle(self):
        return math.pi+self.getLocalMeanTime()*0.2618

    # get solar altitude
    def getSolarAltitude(self):
        sd = self.getSolarDeclination()
        sha = self.getSolarHourAngle()
        salt = math.asin((math.sin(sd)*math.sin(self.latitude * self.RAD))+(math.cos(sd)*math.cos(self.latitude * self.RAD)*math.cos(sha)))
        if salt > 0:
           return salt
        else:
           return 0.0

    # get solar altitude (degrees)
    def getSolarAltitudeDEG(self):
        return self.getSolarAltitude() * self.DEG

    # get solar azimuth
    def getSolarAzimuth(self):
        lat = self.latitude * self.RAD
        sa = self.getSolarAltitude()
        sd = self.getSolarDeclination()
        sha = self.getSolarHourAngle()
        if sa>0:
           if ((lat>=0 and (math.sin(sa)>=(math.sin(sd)/math.sin(lat)))) or (lat<0 and (math.sin(sa)<(math.sin(sd)/math.sin(lat))))):
              return math.asin((math.cos(sd)*math.sin(sha))/math.cos(sa))
           else:
              if (math.asin((math.cos(sd)*math.sin(sha))/math.cos(sa)))>0:
                 return (math.pi-math.fabs(math.asin((math.cos(sd)*math.sin(sha))/math.cos(sa))))
              else:
                 return -(math.pi-math.fabs(math.asin((math.cos(sd)*math.sin(sha))/math.cos(sa))))
        else:
           return 0.0

    # get solar azimuth (degrees)
    def getSolarAzimuthDEG(self):
        return self.getSolarAzimuth() * self.DEG

    # get solar x coordinate
    def getX(self):
        return self.R * math.cos(self.getSolarAltitude()) * math.cos(self.getSolarAzimuth())

    # get solar y coordinate
    def getY(self):
        return self.R * math.cos(self.getSolarAltitude()) * math.sin(self.getSolarAzimuth())

    # get solar z coordinate
    def getZ(self):
        return self.R * math.sin(self.getSolarAltitude())
# class SunSolarTime: end

def testSun():
 days = [0,
        31, #jan
        28, #feb
        31, #mar
        30, #apr
        31, #may
        30, #jun
        31, #jul
        31, #aug
        30, #sep
        31, #oct
        30, #nov
        31 #dec
        ]
 #scene = Blender.Scene.getCurrent()
 for m in range(6, 6+1, 1):
  #for d in range(1, days[m]+1):
  for d in range(21, 21+1):
    for h in range(0, 23+1, 1):
     for min in range(0, 60, 60):
        #         lat  lon  mer   m   d   h   m
        #sun = SunTime( 37.1,  13.46,  15.0,  m,  d,  h,  min, "solar")
        #Guangzhou lat23d08m=23.12 lon113d19m=113.32 mer120
        #VALENCIA_ESP IWEC Data,     !- Location Name
        #39.50000     ,     !- Latitude {N+ S-}
        #-0.4700000     ,     !- Longitude {W- E+}
        #1.000000     ,     !- Time Zone Relative to GMT {GMT+/-}
        #62.00000     ;     !- Elevation {m}
        #Tromso
        #69.67 18.97 15
        sun = SunTime( 69.67,  18.97,  15.0,  m,  d,  h,  min, "clock")
        sunclock = SunTime( 69.67,  18.97,  15.0,  m,  d,  h,  min, "clock")
        #print sun.month, sun.day, sun.hour, sun.minute,
        #print sun.getSolarAltitudeDEG(), sun.getSolarAzimuthDEG(),
        print sun.getX(), sun.getY(), sun.getZ(),
        #print sun.getDayNumber(m, d),
        #print sun.getSolarAzimuthDEG(), sun.getSolarAltitudeDEG(),
        """
        print sunclock.getMeridianAdjustment(),
        print int(int(sunclock.getMeridianAdjustment())), int(60*(sunclock.getMeridianAdjustment() - int(sunclock.getMeridianAdjustment()))),
        print sunclock.getEquationOfTime(),
        print int(int(sunclock.getEquationOfTime())), int(60*(sunclock.getEquationOfTime() - int(sunclock.getEquationOfTime()))),
        print sunclock.getLocalApparentTime(),
        print int(int(sunclock.getLocalApparentTime())), int(60*(sunclock.getLocalApparentTime() - int(sunclock.getLocalApparentTime()))),
        """
        #print sunclock.getLocalMeanTime()
        #print "solar altitude: ", sun.getSolarAltitudeDEG()
        #print "solar azimuth : ", sun.getSolarAzimuthDEG()
        #print "x :", sun.getX()
        #print "y :", sun.getY()
        #print "z :", sun.getZ()
        print
        """
        if sun.getZ()>0:
         if Blender.Object.Get("Sun"):
           print "#" * 79
           print sun.daynumber
           print sun.hour
           sunobj = Blender.Object.Get("Sun")
           sun.R = 10
           location = (sun.getX(), sun.getY(), sun.getZ())
           sunobj.setLocation(location)
           sun.R = 1
           #sunobj.rot = (1-sun.getSolarAltitude(),0,-sun.getSolarAzimuth())
           sunobj.rot = ((math.pi/2.0) - sun.getSolarAltitude(),0, (math.pi/2.0) - sun.getSolarAzimuth())
           #print "x :", sun.getX()
           #print "y :", sun.getY()
           #print "z :", sun.getZ()
           #print sunobj.rot
           scene.currentFrame(scene.currentFrame()+1)
           Blender.Redraw()
        """

if __name__ == "__main__":
    testSun()
