# #####################################
# Blended Radiance brad_radiance.py #
#####################################

##############################
# (c) Francesco Anselmo 2005 #
##############################

#    This file is part of brad (Blended RADiance).
#
#    brad is free software; you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation; either version 2 of the License, or
#    (at your option) any later version.
#
#    brad is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
#
#    You should have received a copy of the GNU General Public License
#    along with Foobar; if not, write to the Free Software
#    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

import sys

import Blender
import math
import string
import os
import calendar
import shutil

#from matrix import *
from Blender.Draw import *

from brad_i18n import *
from brad_io import *

from Libwin import *

# class Radiance: begin
class Radiance:
    def __init__(self, path,
                      filename,
                      statuswindow,
                      exportwindow,
                      parwindow,
                      radparwindow,
                      timewindow,
                      locationwindow,
                      skyweatherwindow,
                      calculatewindow,
                      importwindow):
        self.statuswindow = statuswindow
        self.exportwindow = exportwindow
        self.parwindow = parwindow
        self.radparwindow = radparwindow
        self.timewindow = timewindow
        self.locationwindow = locationwindow
        self.skyweatherwindow = skyweatherwindow
        self.calculatewindow = calculatewindow
        self.importwindow = importwindow
        self.path = path
        self.filename = filename
        self.scenefilename = ""
        self.skyfilename = ""
        self.file    = None
        self.scene   = None
        self.display = None
        self.lampNames = []
        self.meshNames = []
        self.materialNames = []
        self.geometryHasBeenExported = 0
        self.frameobjs = None
        self.importfilename = self.importwindow.importfilename.val
        registered_data = Blender.Registry.GetKey('brad')
        if registered_data:
          self.brad_path  = registered_data['brad_path']
          self.debug  = registered_data['debug']
          self.convert_path  = registered_data['convert_path']

    def applyTransformation4(self, vertex, matrix):
        vertCopy = Blender.Mathutils.CopyVec(vertex)
        vertCopy.resize4D()
        return Blender.Mathutils.VecMultMat(vertCopy, matrix)

    def applyTransformation3(self, vertex, matrix):
        vertCopy = Blender.Mathutils.CopyVec(vertex)
        vertCopy.resize3D()
        return Blender.Mathutils.VecMultMat(vertCopy, matrix)

    def sameVector3(self, vec1, vec2):
        if vec1[0]==vec2[0] and vec1[1]==vec2[1] and vec1[2]==vec2[2]:
          return True
        else:
          return False

    # not working, therefore not used
    # now retreiving camera position and rotation from worldspace/localspace object matrix
    # see below writeCamera(...)
    """
    def cameraMatrix(self, mat):
        mtx = mat
        scale = Blender.Mathutils.Vector([0,0,0])
        mtx[0].normalize
        mtx[1].normalize
        mtx[2].normalize
        scale[0] = mtx[0].length
        scale[1] = mtx[1].length
        scale[2] = mtx[2].length
        # is scaling negative?
        if mtx.determinant()<0.0:
          for i in range(3):
            scale[i] *= -1.0
            mtx[i][0] *= -1.0
            mtx[i][1] *= -1.0
            mtx[i][2] *= -1.0
        angle_y = -asin(max(min(mtx[0][2], 1.0), -1.0))
        C = cos(angle_y)
        if C!=0.0: C = 1.0/C
        angle_x = atan2(mtx[1][2] * C, mtx[2][2] * C)
        angle_z = atan2(mtx[0][1] * C, mtx[0][0] * C)
        if self.debug:
          print "begin cameraMatrix"
          print "matrix:"
          print mtx
          print "translation:"
          print mtx.translationPart()
          print "rotation:"
          print mtx.rotationPart()
          print "scale:"
          print scale
          print "angles: ", angle_x*180/math.pi, angle_y*180/math.pi, angle_z*180/math.pi
          print "end cameraMatrix"
        return (angle_x, angle_y, angle_z), tuple(scale), tuple(mat[3][:3])
    """

    def importRad(self):
        # recognized Radiance primitives:
        # polygon, plastic, plastic2, metal, metal2, trans, trans2, glass, dielectric
      try:
        registered_data = Blender.Registry.GetKey('brad')
        if registered_data:
          brad_path  = registered_data['brad_path']
          convert_path  = registered_data['convert_path']
          debug  = registered_data['debug']

        if os.path.exists(self.importfilename):
          # tokenize the file
          tokens = []
          radfile = open(self.importfilename, "r")
          radfilelines = radfile.readlines()
          endslash = 0
          for line in radfilelines:
                if line[len(line)-2] == "\\":
                    endslash = 1
                if line[0] != "#" and line[0] != "!" and endslash == 0:
                    for i in range(len(string.split(line))):
                        tokens.append(string.split(line)[i])
                elif endslash == 1:
                    if line[len(line)-2] != "\\":
                      endslash = 0
          radfile.close()
          # creating list of modifiers
          modifiers = []
          for i in range(len(tokens)):
              if modifiers.count(tokens[i])==0:
                try:
                    float(tokens[i])
                except:
                    modifiers.append(tokens[i])
          if self.debug:
             print modifiers

          # creating new mesh for polygons
          newmesh = Blender.NMesh.GetRaw()
          newmesh.name = os.path.basename(self.importfilename)
          i = 0
          while i < len(tokens):
            if tokens[i+1] != "alias":
              snum = int(tokens[i+3])
              inum = int(tokens[i+3+1+snum])
              fnum = int(tokens[i+3+1+1+snum+inum])
              # import Radiance primitives
              if tokens[i+1] == "plastic":
                self.statuswindow.msg = msg_Importing + "plastic" + str(tokens[i+2])
                m = Blender.Material.New()
                m.name=str(tokens[i+2])
                m.setRGBCol(float(tokens[i+3+1+1+1+snum+inum]), float(tokens[i+3+1+1+1+snum+inum+1]), float(tokens[i+3+1+1+1+snum+inum+2]))
                m.setSpec(float(tokens[i+3+1+1+1+snum+inum+3]))
              if tokens[i+1] == "plastic2":
                self.statuswindow.msg = msg_Importing + "plastic2" + str(tokens[i+2])
                m = Blender.Material.New()
                m.name=str(tokens[i+2])
                m.setRGBCol(float(tokens[i+3+1+1+1+snum+inum]), float(tokens[i+3+1+1+1+snum+inum+1]), float(tokens[i+3+1+1+1+snum+inum+2]))
                m.setSpec(float(tokens[i+3+1+1+1+snum+inum+3]))
              if tokens[i+1] == "metal":
                self.statuswindow.msg = msg_Importing + "metal" + str(tokens[i+2])
                m = Blender.Material.New()
                m.name=str(tokens[i+2])
                m.setRGBCol(float(tokens[i+3+1+1+1+snum+inum]), float(tokens[i+3+1+1+1+snum+inum+1]), float(tokens[i+3+1+1+1+snum+inum+2]))
                m.setSpec(float(tokens[i+3+1+1+1+snum+inum+3]))
              if tokens[i+1] == "metal2":
                self.statuswindow.msg = msg_Importing + "metal2" + str(tokens[i+2])
                m = Blender.Material.New()
                m.name=str(tokens[i+2])
                m.setRGBCol(float(tokens[i+3+1+1+1+snum+inum]), float(tokens[i+3+1+1+1+snum+inum+1]), float(tokens[i+3+1+1+1+snum+inum+2]))
                m.setSpec(float(tokens[i+3+1+1+1+snum+inum+3]))
              if tokens[i+1] == "trans":
                self.statuswindow.msg = msg_Importing + "trans" + str(tokens[i+2])
                m = Blender.Material.New()
                m.name=str(tokens[i+2])
                m.setRGBCol(float(tokens[i+3+1+1+1+snum+inum]), float(tokens[i+3+1+1+1+snum+inum+1]), float(tokens[i+3+1+1+1+snum+inum+2]))
                m.setSpec(float(tokens[i+3+1+1+1+snum+inum+3]))
              if tokens[i+1] == "trans2":
                self.statuswindow.msg = msg_Importing + "trans2" + str(tokens[i+2])
                m = Blender.Material.New()
                m.name=str(tokens[i+2])
                m.setRGBCol(float(tokens[i+3+1+1+1+snum+inum]), float(tokens[i+3+1+1+1+snum+inum+1]), float(tokens[i+3+1+1+1+snum+inum+2]))
                m.setSpec(float(tokens[i+3+1+1+1+snum+inum+3]))
              if tokens[i+1] == "glass":
                self.statuswindow.msg = msg_Importing + "glass" + str(tokens[i+2])
                m = Blender.Material.New()
                m.name=str(tokens[i+2])
                m.setRGBCol(float(tokens[i+3+1+1+1+snum+inum]), float(tokens[i+3+1+1+1+snum+inum+1]), float(tokens[i+3+1+1+1+snum+inum+2]))
                m.setAlpha(0.5)
              if tokens[i+1] == "dielectric":
                self.statuswindow.msg = msg_Importing + "dielectric" + str(tokens[i+2])
                m = Blender.Material.New()
                m.name=str(tokens[i+2])
                m.setRGBCol(float(tokens[i+3+1+1+1+snum+inum]), float(tokens[i+3+1+1+1+snum+inum+1]), float(tokens[i+3+1+1+1+snum+inum+2]))
                m.setAlpha(0.5)
              if tokens[i+1] == "polygon":
                self.statuswindow.msg = msg_Importing + "polygon" + str(tokens[i+2])
                base = i+3+1+1+snum+inum
                if fnum == 12 or fnum == 9:
                    newface = Blender.NMesh.Face()
                    for j in range(0, fnum/3):
                      newmesh.verts.append(Blender.NMesh.Vert(float(tokens[base +j*3+1]),float(tokens[base +j*3+2]),float(tokens[base +j*3+3])))
                      newface.v.append(newmesh.verts[len(newmesh.verts)-1])
                    newmesh.faces.append(newface)
                    newmesh.update()
              # increase step
              i += (3 +1+1+1+ snum + inum + fnum)
            else:
              # increase step for alias primitive
              i += 4
          Redraw()
          Blender.NMesh.PutRaw(newmesh, os.path.basename(self.importfilename), 1)
          #ob = Blender.Object.New('Mesh')
          #ob.link(newmesh)
          #sc = Blender.Scene.GetCurrent()
          #sc.link(ob)
          self.statuswindow.msg = msg_Done
        else:
          self.statuswindow.msg = msg_FileNotFound
      except:
        self.statuswindow.msg = msg_ErrorImportRad

    def writeScriptAll(self):
        scriptfilename = os.path.join(self.path, self.filename, self.filename +".py")
        scriptfile = open(scriptfilename, "w")
        scriptfile.write("#!/usr/bin/python\n\n")
        scriptfile.write("import os \n")
        scriptfile.write("import string \n")
        scriptfile.write("try: \n")
        scriptfile.write("   os.mkdir(\"frames\") \n")
        scriptfile.write("except: \n")
        scriptfile.write("   print(\"the frame directory exists\") \n")
        scriptfile.write("for n in range(%s, %s+1): \n" % (self.staframe, self.endframe))
        scriptfile.write("   print \"rendering frame: \", n \n")
        scriptfile.write("   os.chdir(str(n)) \n")
        scriptfile.write("   os.system(\"oconv %s.rad > %s.oct\") \n" % (self.filename, self.filename))
        scriptfile.write("   os.system(\"rpict -vf %s.vf %s.oct > %s.pic\") \n" % (self.filename, self.filename, self.filename))
        scriptfile.write("   os.system(\"pcond -h+ %s.pic > %s_h.pic\") \n" % (self.filename, self.filename))
        scriptfile.write("   scriptcommand = \"ra_tiff %s_h.pic ../frames/\" + zfill(str(n),10) + \".tif\"  \n" % (self.filename))
        scriptfile.write("   os.system(scriptcommand) \n")
        scriptfile.write("   os.chdir(\"../\") \n")
        scriptfile.close()

    def writeScriptCurrentFrame(self):
        scriptfilename = os.path.join(self.path, self.filename, self.filename +".py")
        scriptfile = open(scriptfilename, "w")
        scriptfile.write("#!/usr/bin/python\n\n")
        scriptfile.write("import os \n")
        scriptfile.write("import string \n")
        scriptfile.write("try: \n")
        scriptfile.write("   os.mkdir(\"frames\") \n")
        scriptfile.write("except: \n")
        scriptfile.write("   print(\"the frame directory exists\") \n")
        scriptfile.write("for n in range(%s, %s+1): \n" % (self.curframe, self.curframe))
        scriptfile.write("   print \"rendering frame: \", n \n")
        scriptfile.write("   os.chdir(str(n)) \n")
        scriptfile.write("   os.system(\"oconv %s.rad > %s.oct\") \n" % (self.filename, self.filename))
        scriptfile.write("   os.system(\"rpict -vf %s.vf %s.oct > %s.pic\") \n" % (self.filename, self.filename, self.filename))
        scriptfile.write("   os.system(\"pcond -h+ %s.pic > %s_h.pic\") \n" % (self.filename, self.filename))
        scriptfile.write("   scriptcommand = \"ra_tiff %s_h.pic ../frames/\" + zfill(str(n),10) + \".tif\"  \n" % (self.filename))
        scriptfile.write("   os.system(scriptcommand) \n")
        scriptfile.write("   os.chdir(\"../\") \n")
        scriptfile.close()

    def exportAnimation(self, scene):
        self.statuswindow.msg = msg_Exporting
        Draw()
        #create new scenedir if it doesn't exist
        self.scenedir = os.path.join(self.path, self.filename)
        try:
            os.mkdir(self.scenedir)
            self.statuswindow.msg = msg_SceneDirectoryCreated + self.scenedir
            Draw()
        except:
            self.statuswindow.msg = msg_SceneDirectoryNotCreated + self.scenedir
            Draw()
        self.scene = scene
        self.staframe = Blender.Get("staframe")
        self.endframe = Blender.Get("endframe")
        self.curframe = Blender.Get("curframe")

        if self.staframe<self.endframe:
            for frame in range(self.staframe, self.endframe + 1):
                context = self.scene.getRenderingContext()
                context.currentFrame(frame)
                Blender.Window.Redraw()
                self.writeFrame(frame)
                self.geometryHasBeenExported = 1
        else:
            self.writeFrame(self.curframe)
            Blender.Window.Redraw()
        # static geometry
        if self.exportwindow.ExportStaticGeometry.val==1:
          # write rad scene file if necessary (into first selected frame directory)
          if (self.parwindow.RenderRad.val==1):
              self.writeAnimationRad()
        # dynamic geometry
        if self.exportwindow.ExportStaticGeometry.val==0:
          if (self.parwindow.RenderRad.val==1):
              self.writeAnimationRadScript()
        self.writeEnd()
        self.geometryHasBeenExported = 0
        self.exportwindow.geometryHasBeenExported[0]=1
        self.exportwindow.geometryHasBeenExported[1]=self.staframe
        self.exportwindow.geometryHasBeenExported[2]=self.endframe

    def exportCurrentFrame(self, scene):
        self.statuswindow.msg = msg_Exporting
        Draw()
        #create new scenedir if it doesn't exist
        self.scenedir = os.path.join(self.path, self.filename)
        try:
            os.mkdir(self.scenedir)
            self.statuswindow.msg = msg_SceneDirectoryCreated + self.scenedir
            Draw()
        except:
            self.statuswindow.msg = msg_SceneDirectoryNotCreated + self.scenedir
            Draw()
        self.scene = scene
        self.staframe = Blender.Get("staframe")
        self.endframe = Blender.Get("endframe")
        self.curframe = Blender.Get("curframe")
        # write current frame
        self.writeFrame(self.curframe)
        self.writeFrameRad(self.curframe)
        self.writeEnd()
        self.geometryHasBeenExported = 0
        self.exportwindow.geometryHasBeenExported[0]=1
        self.exportwindow.geometryHasBeenExported[1]=self.curframe
        self.exportwindow.geometryHasBeenExported[2]=self.curframe

    def writeFrame(self, frame):
        self.statuswindow.msg = msg_Exporting + msg_Frame + str(frame)
        #create new framedir if it doesnt exist
        self.framedir = os.path.join(self.path, self.filename, zfill(str(frame),10))
        try:
            os.mkdir(self.framedir)
            self.statuswindow.msg = msg_FrameDirectoryCreated + self.framedir
            Draw()
        except:
            self.statuswindow.msg = msg_FrameDirectoryNotCreated + self.framedir
            Draw()
        self.lampNames = []
        self.meshNames = []
        self.materialNames = []
        self.writeCamera(frame)

        # dynamic geometry
        if self.exportwindow.ExportStaticGeometry.val==0:
          self.writeScene(frame)
          # write rad frame file if necessary
          if (self.parwindow.RenderRad.val==1):
              self.writeFrameRad(frame)
        # static geometry (first frame)
        elif self.exportwindow.ExportStaticGeometry.val==1 and self.geometryHasBeenExported==0:
          self.writeScene(frame)
        Draw()

    def writeEnd(self):
        self.statuswindow.msg = msg_Finished + msg_Exported + self.scenedir
        Draw()

    def writeWorld(self, frame):
        self.skyfilename = os.path.join(self.path, self.filename, zfill(str(frame),10), self.filename + ".sky")
        if self.skyweatherwindow.generateSky.val:
           self.skyweatherwindow.exportSky(self.skyfilename,frame)

    def writeSelectedCamera(self, selectedcamera, frame):
        try:
          cameraname = selectedcamera.name
          #create new camera file if it doesn't exist
          self.camfilename = os.path.join(self.path, self.filename, zfill(str(frame),10), cameraname + ".vf")
          camfile = open(self.camfilename, "w")
          obj = selectedcamera
          # lens attributes
          lens = obj.getData().lens
          factor = math.sqrt(lens * lens + 16 * 16)
          angleX = 2 * (180 * math.acos(lens / factor) / math.pi)
          angleY = (self.parwindow.PictureResolutionY.val * angleX) / self.parwindow.PictureResolutionX.val

          # apply translation and rotation
          matrix = obj.getMatrix('worldspace')
          locmatrix = obj.getMatrix('localspace')
          yvec = Blender.Mathutils.Vector([0,1,0])
          zvec = Blender.Mathutils.Vector([0,0,-1])
          vpvec = Blender.Mathutils.Vector([obj.LocX, obj.LocY, obj.LocZ])
          rotationmatrix=matrix.rotationPart()
          vd = self.applyTransformation3(zvec, rotationmatrix)
          vu = self.applyTransformation3(yvec, rotationmatrix)
          # need to fix camera rotation for tracking camera with nonzero rotation angles !!!
          """
          rotationmatrix=matrix.rotationPart()
          vd = self.applyTransformation3(vd, rotationmatrix)
          vu = self.applyTransformation3(vu, rotationmatrix)
          """
          if not self.sameVector3(matrix.translationPart(), locmatrix.translationPart()):
            vp = self.applyTransformation4(vpvec, matrix)
          else:
            vp = [obj.LocX, obj.LocY, obj.LocZ]

          # reminder: need to add camera settings widget
          vt="v"
          # perspective, cylindric, hemispheric or angular view
          # according to view angles
          if obj.getData().getType()=="persp":
              vt = "v"
              if angleX>=120 and angleY<90:
                vt = "c"
              elif angleX>=120 and angleY<180:
                vt = "h"
              elif angleX>=180 and angleY>=180:
                vt = "a"
          # orthogonal view
          if obj.getData().getType()=="ortho":
              vt = "l"

          vo = obj.getData().clipStart
          va = obj.getData().clipEnd

          #camfile.write("rview -vt%s -vp %s %s %s " % (vt, obj.LocX, obj.LocY, obj.LocZ) +
          #       "-vd %s %s %s " % (vd[0][0], vd[1][0], vd[2][0]) +
          #       "-vu %s %s %s " % (vu[0][0], vu[1][0], vu[2][0]) +
          #       "-vh %s -vv %s " % (angleX, angleY) +
          #       "-vo %s -va %s" % (vo, va))

          camfile.write("rview -vt%s -vp %s %s %s " % (vt, vp[0], vp[1], vp[2]) +
                  "-vd %s %s %s " % (vd[0][0], vd[1][0], vd[2][0]) +
                  "-vu %s %s %s " % (vu[0][0], vu[1][0], vu[2][0]) +
                  "-vh %s -vv %s " % (angleX, angleY))

          camfile.close()
          self.statuswindow.msg = msg_CameraSaved + self.camfilename
          Draw()
        except:
          self.statuswindow.msg = msg_CameraNotSaved + self.camfilename
          Draw()

    def writeCamera(self, frame):
        try:
          #create new camera file
          self.camfilename = os.path.join(self.path, self.filename, zfill(str(frame),10), self.filename + ".vf")
          camfile = open(self.camfilename, "w")

          #self.scene.currentFrame(frame)
          context = self.scene.getRenderingContext()
          context.currentFrame(frame)
          #self.scene.frameSettings(context.startFrame(), context.endFrame(), frame)
          cam = self.scene.getCurrentCamera()
          obj = cam

          # lens attributes
          lens = cam.getData().lens
          factor = math.sqrt(lens * lens + 16 * 16)
          #angle = 2 * (180 * math.acos(lens / factor) / math.pi)
          angleX = 2 * (180 * math.acos(lens / factor) / math.pi)
          angleY = (self.parwindow.PictureResolutionY.val * angleX) / self.parwindow.PictureResolutionX.val

          # apply translation and rotation
          matrix = obj.getMatrix('worldspace')
          locmatrix = obj.getMatrix('localspace')
          yvec = Blender.Mathutils.Vector([0,1,0])
          zvec = Blender.Mathutils.Vector([0,0,-1])
          vpvec = Blender.Mathutils.Vector([obj.LocX, obj.LocY, obj.LocZ])
          rotationmatrix=matrix.rotationPart()
          vd = self.applyTransformation3(zvec, rotationmatrix)
          vu = self.applyTransformation3(yvec, rotationmatrix)
          # need to fix camera rotation for tracking camera with nonzero rotation angles !!!
          """
          rotationmatrix=matrix.rotationPart()
          vd = self.applyTransformation3(vd, rotationmatrix)
          vu = self.applyTransformation3(vu, rotationmatrix)
          """
          if not self.sameVector3(matrix.translationPart(), locmatrix.translationPart()):
            vp = self.applyTransformation4(vpvec, matrix)
          else:
            vp = [obj.LocX, obj.LocY, obj.LocZ]

          # work out projection type according to focal length / view angles
          # need to change this when adding camera settings window !!!
          vt="v"
          # perspective, cylindric, hemispheric or angular view
          # according to view angles
          if obj.getData().getType()=="persp":
              vt = "v"
              if angleX>=120 and angleY<90:
                vt = "c"
              elif angleX>=120 and angleY<180:
                vt = "h"
              elif angleX>=180 and angleY>=180:
                vt = "a"
          # orthogonal view
          if obj.getData().getType()=="ortho":
              vt = "l"

          vo = obj.getData().clipStart
          va = obj.getData().clipEnd

          # need to put again -vo and -va options when adding camera settings window !!!
          """
          camfile.write("rview -vt%s -vp %s %s %s " % (vt, obj.LocX, obj.LocY, obj.LocZ) +
                 "-vd %s %s %s " % (vd[0][0], vd[1][0], vd[2][0]) +
                 "-vu %s %s %s " % (vu[0][0], vu[1][0], vu[2][0]) +
                 "-vh %s -vv %s " % (angleX, angleY) +
                 "-vo %s -va %s" % (vo, va))
          """

          camfile.write("rview -vt%s -vp %s %s %s " % (vt, vp[0], vp[1], vp[2]) +
                  "-vd %s %s %s " % (vd[0], vd[1], vd[2]) +
                  "-vu %s %s %s " % (vu[0], vu[1], vu[2]) +
                  "-vh %s -vv %s " % (angleX, angleY))

          camfile.close()

          #print "camera file for frame n. %s has been created" % str(frame)
          self.statuswindow.msg = msg_CameraSceneSaved + str(frame)
          Draw()
        except:
          print "cannot create camera file for frame n. %s" % str(frame)
          self.statuswindow.msg = msg_CameraSceneNotSaved + str(frame)
          Draw()

    #comment before every primitive
    def writeComment(self, file, name):
        file.write("\n")
        file.write("#" * (len(name) + 4) + "\n")
        file.write("# %s #\n" % name)
        file.write("#" * (len(name) + 4) + "\n")

    #xform transformation after primitive definition
    def writeMatrix(self, file, obj):
        # scaling
        sX, sY, sZ = obj.getSize()
        scale = (sX+sY+sZ)/3.0
        file.write("\t| xform -s %s \\\n" %
              (scale))
        file.write("\t| xform -rx %s -ry %s -rz %s\\\n" %
              (180 * obj.RotX / math.pi,
                180 * obj.RotY / math.pi,
                180 * obj.RotZ / math.pi))
        file.write("\t| xform -t %s %s %s\n" %
              (obj.LocX, obj.LocY, obj.LocZ))

    def writeNoMaterial(self, file):
          file.write("void plastic no_material\n")
          file.write("0\n")
          file.write("0\n")
          file.write("5 %s %s %s %s %s\n" % (1, 1, 1, 0, 0))

    def writeMaterial(self, file, mat, name):
      modlist = loadRadLib(os.path.join(brad_path, "brad", "radiance", "mod"))
      modlist.sort()
      #material from library (if it has the same name of a library material)
      if modlist.count(mat.name)>0:
        matfilename = os.path.join(brad_path, "brad","radiance","mod", mat.name, mat.name+".rad")
        matfile = open(matfilename, "r")
        matfilelines = matfile.readlines()
        for line in matfilelines:
          file.write(line)
        file.write("\n\n")
        matfile.close()
      else:
        # try to define radiance materials according to blender material settings
        if mat.name[:len("Glass")] == "Glass":
              file.write("\nvoid glass %s\n" % name)
              file.write("0\n")
              file.write("0\n\n")
              if mat:
                 file.write("3 %s %s %s \n" % (mat.R, mat.G, mat.B))
              else:
                 file.write("3 %s %s %s \n" % (1, 1, 1))
        elif mat.name[:len("Dielectric")] == "Dielectric":
              file.write("\nvoid dielectric %s\n" % name)
              file.write("0\n")
              file.write("0\n\n")
              if mat:
                 file.write("5 %s %s %s 1.52 0\n" % (mat.R, mat.G, mat.B))
              else:
                 file.write("5 %s %s %s 1.52 0\n" % (1, 1, 1))
        elif mat.name[:len("Metal")] == "Metal":
          # metal
          # mat.hard is used to set roughness
          file.write("void metal %s\n" % name)
          file.write("0\n")
          file.write("0\n")
          if mat:
              file.write("5 %s %s %s %s %s\n" % (mat.R, mat.G, mat.B, mat.spec/2.0, mat.hard/511.0))
          else:
              file.write("5 %s %s %s %s %s\n" % (1, 1, 1, 0.1, 0.15))
        else:
          # plastic material in any other case
          file.write("void plastic %s\n" % name)
          file.write("0\n")
          file.write("0\n")
          if mat:
              file.write("5 %s %s %s %s %s\n" % (mat.R, mat.G, mat.B, mat.spec/2.0, mat.hard/511.0))
          else:
              file.write("5 %s %s %s %s %s\n" % (1, 1, 1, 0, 0))

    def getObjectLayers(self, objlayer):
      l = ""
      objlayer = int(objlayer)
      for i in range(19,-1,-1):
          if (objlayer/int(pow(2,i)))>0:
            l += "1"
            objlayer -= int(pow(2,i))
          else:
            l += "0"
      return l

    def getSelectedLayers(self):
      l = ""
      for i in range(20,0,-1):
          l += str(self.exportwindow.l[i].val)
      return l

    def checkLayers(self, objectLayers, selectedLayers):
      visible=0
      for i in range(len(selectedLayers)):
          if objectLayers[i:i+1]=="1" and selectedLayers[i:i+1]=="1":
            visible=1
      return visible

    def writeMesh(self, frame, mesh, datablock):
      if mesh:
          if len(mesh.faces)>0:
                    index = 0
                    index2 = 0
                    meshfilename = os.path.join(self.path, self.filename, zfill(str(frame),10), mesh.name +".rad")
                    meshfile = open(meshfilename, "w")
                    # material modifiers
                    self.writeComment(meshfile, "no_material")
                    self.writeNoMaterial(meshfile)
                    mats = Blender.Material.get()
                    for mat in mats:
                      name = mat.name
                      self.writeComment(meshfile, name)
                      self.writeMaterial(meshfile, mat, name)
                    #material aliases
                    #for material in mesh.materials:
                    #    meshfile.write("\nvoid alias %s %s\n\n" % (index2, material.name))
                    #    index2 = index2 + 1
                    #status message
                    self.statuswindow.msg = msg_Frame + str(frame) + " | " + msg_Mesh + mesh.name + " | " + msg_NumberMaterials + str(len(mesh.materials))
                    Draw()
                    #export mesh
                    for face in mesh.faces:
                        index = index + 1
                        num = len(face.v)
                        if num == 3 or num == 4:
                            try:
                              meshfile.write("%s polygon %s\n" % (mesh.materials[face.mat].name, index))
                            except:
                              meshfile.write("no_material polygon %s\n" % (index))
                            meshfile.write("0\n")
                            meshfile.write("0\n")
                            meshfile.write("%s\n" % (num * 3,))
                            for vertex in face.v:
                                meshfile.write("\t%s %s %s\n" %
                                                (vertex.co[0] * datablock.SizeX,
                                                 vertex.co[1] * datablock.SizeY,
                                                 vertex.co[2] * datablock.SizeZ))
                            """
                            for vertex in face.v:
                                      # apply translation and rotation
                                      obj = datablock
                                      matrix = obj.getMatrix('worldspace')
                                      locmatrix = obj.getMatrix('localspace')
                                      rotvec = Blender.Mathutils.Vector([vertex.co[0],vertex.co[1],vertex.co[2]])
                                      rotationmatrix=matrix.rotationPart()
                                      rotvec = self.applyTransformation3(rotvec, rotationmatrix)
                                      # need to fix object rotation for tracking object with nonzero rotation angles !!!
                                      if not self.sameVector3(matrix.translationPart(), locmatrix.translationPart()):
                                        vert = self.applyTransformation4(rotvec, matrix)
                                      else:
                                        vert = [obj.LocX, obj.LocY, obj.LocZ]
                                      meshfile.write("\t%s %s %s\n" %
                                               (vert[0] * 1,
                                                vert[1] * 1,
                                                vert[2] * 1))
                            """
                        else:
                            #self.statuswindow.msg = msg_TooManyVertices + str(num)
                            Draw()
                    meshfile.close()

    def oconvMesh(self, frame, mesh):
      if mesh:
        if self.exportwindow.ExportMethod.val == 0:
         if len(mesh.faces)>0:
          meshfilename = os.path.join(self.path, self.filename, zfill(str(frame),10), mesh.name+".rad")
          meshoctreefilename = os.path.join(self.path, self.filename, zfill(str(frame),10), mesh.name +".oct")
          if (self.parwindow.RenderRad.val==1):
            oconvcommand = "oconv -f " +  self.radparwindow.oconvOptions.val + " " +  meshfilename + " > " + meshoctreefilename
          else:
            oconvcommand = "oconv -f " +  meshfilename + " > " + meshoctreefilename
          if self.debug:
            print oconvcommand
          os.system(oconvcommand)

    def convertImages(self,directory):

      registered_data = Blender.Registry.GetKey('brad')
      if registered_data:
        brad_path  = registered_data['brad_path']
        convert_path  = registered_data['convert_path']
        debug  = registered_data['debug']

      for picture in Blender.Image.Get():
        facematerialname = picture.getFilename()
        facematerialbasename = os.path.splitext(os.path.basename(facematerialname))[0]
        facematerialextension = os.path.splitext(os.path.basename(facematerialname))[1]
        #if os.path.exists(facematerialname): # doesn't work !!!
        if True:
         # need to make sure that this check also works for win32
         if os.path.dirname(picture.getFilename()) == "//":
           picturefilename = os.path.join(os.path.dirname(Blender.Get('filename')),facematerialbasename+facematerialextension)
         else:
           picturefilename = picture.getFilename()
         if string.lower(facematerialextension) != ".tga":
           # convert the texture picture first to tga
           convertcommand=self.convert_path+" "+picturefilename+ " " + os.path.join(directory, facematerialbasename+".tga")
           if self.debug:
             print convertcommand
           os.system(convertcommand)
         else:
           try:
             if self.debug:
                print "copying file " + facematerialbasename+".tga" + " ..."
             shutil.copyfile(picturefilename,os.path.join(directory, facematerialbasename+".tga"))
           except:
             pass
             if self.debug:
                print "error copying file " + facematerialbasename+".tga" + " ..."
         # then to Radiance pic format
         rat16command="ra_t16 -r " + os.path.join(directory, facematerialbasename+".tga") + " " + os.path.join(directory, facematerialbasename + ".pic")
         if self.debug:
            print rat16command
         os.system(rat16command)
         # and, finally, normalise it
         normpatcommand="normpat "+ os.path.join(directory, facematerialbasename + ".pic")
         if self.debug:
            print normpatcommand
         os.system(normpatcommand)

    def exportRadianceMeshOBJ(self, object, basefilename, directory):

     #try:

      registered_data = Blender.Registry.GetKey('brad')
      if registered_data:
        brad_path  = registered_data['brad_path']
        convert_path  = registered_data['convert_path']
        debug  = registered_data['debug']

      objname = object.name
      meshname = object.data.name
      mesh = Blender.NMesh.GetRaw(meshname)

      objfilename = basefilename+".obj"
      objfile = open(objfilename, "w")

      mtlfilename = basefilename+".mtl"
      mtlfile = open(mtlfilename, "w")

      radmtlfilename = basefilename+".mat"
      radmtlfile = open(radmtlfilename, "w")

      objfile.write("# brad Radiance Mesh OBJ export module")
      objfile.write("\n# object: " + str(objname))
      objfile.write("\n# total verticis: " + str(len(mesh.verts)))
      objfile.write("\n# total faces: " + str(len(mesh.faces)))

      #objfile.write("maplib " + mtlfilename)
      # useful for syzygy objinput: next line
      objfile.write("\n\nmtllib " + os.path.join(os.getcwd(),mtlfilename) + "\n")
      # useful for syzygy objinput: previous line

      objfile.write("\n\n# Verticis\n")
      for vertex in mesh.verts:
          x, y, z = vertex.co
          #nx, ny, nz = vertex.no
          objfile.write("v %f %f %f\n" % (x, y, z))

      objfile.write("\n# Verticis Normals\n")
      for face in mesh.faces:
         for vertex in face.v:
             objfile.write("vn " + str(vertex.no[0]) + " " + str(vertex.no[1]) + " " + str(vertex.no[2]) + "\n")

      objfile.write("\n# Verticis Textures\n")
      for face in mesh.faces:
         #print face.uv
         for uvco in face.uv:
             w = 1
             h = 1
             if self.debug:
                print face.mode
                print face.image
             if mesh.hasFaceUV() and face.mode == 5 and face.image!=None:
                picturefilename=""
                # need to make sure that this check also works for win32
                if os.path.dirname(face.image.getFilename()) == "//":
                   picturefilename = os.path.join(os.path.dirname(Blender.Get('filename')),os.path.basename(face.image.getFilename()))
                else:
                   picturefilename = face.image.getFilename()
                if self.debug:
                   print picturefilename
                #facematerialname = face.image.getFilename()
                if os.path.exists(picturefilename):
                 try:
                  img = Blender.Image.Get(os.path.basename(face.image.getFilename()))
                  if self.debug:
                     print img
                  w,h = img.getSize()
                 except:
                  pass
                if self.debug:
                   print w, h
                if w>h:
                   objfile.write("vt " + str(uvco[0]*(float(w)/float(h))) + " " + str(uvco[1]) + "\n")
                else:
                   objfile.write("vt " + str(uvco[0]) + " " + str(uvco[1]*(float(h)/float(w))) + "\n")
                w = 1
                h = 1

         #for vertex in face.v:
         #    print "vt", vertex.uvco[0], vertex.uvco[1], vertex.uvco[2]

      if not mesh.materials:             # if there are no materials ...
         newmat = Blender.Material.New()  # create one ...
         mesh.materials.append(newmat)    # and append it to the mesh's list of mats
         mesh.update()

      # useful for syzygy objinput: next 2 lines
      #objfile.write("usemtl " + str(mesh.materials[0].getName()) + "\n\n")
      #mtlfile.write("newmtl " + str(mesh.materials[0].getName()) + "\n")
      # useful for syzygy objinput: previous 2 lines
      r,g,b = mesh.materials[0].getRGBCol()
      mtlfile.write("Kd " + " " +  str(r) + " " +  str(g) + " " +  str(b) + "\n")
      a = mesh.materials[0].getAmb()
      r *= a
      g *= a
      b *= a
      mtlfile.write("Ka " + " " +  str(r) + " " +  str(g) + " " +  str(b) + "\n")
      r,g,b = mesh.materials[0].getSpecCol()
      mtlfile.write("Ks " + " " +  str(r) + " " +  str(g) + " " +  str(b) + "\n\n")

      objfile.write("\n\n# Faces\n")
      objfile.write("usemtl " + str(mesh.materials[0].getName()) + "\n")
      # useful for syzygy objinput: next line
      objfile.write("g " +  str(object.name) + "\n\n")
      # useful for syzygy objinput: previous line

      # material modifiers
      self.writeComment(radmtlfile, "no_material")
      self.writeNoMaterial(radmtlfile)
      mats = Blender.Material.get()
      for mat in mats:
        name = mat.name
        self.writeComment(radmtlfile, name)
        self.writeMaterial(radmtlfile, mat, name)

      faceindex = 0
      for face in mesh.faces:
          #face.v.reverse()    # flips all face normals (is it useful?)
          if mesh.hasFaceUV() and face.mode == 5 and face.image!=None:
              facematerialname = face.image.getFilename()
              facematerialbasename = os.path.splitext(os.path.basename(facematerialname))[0]
              facematerialextension = os.path.splitext(os.path.basename(facematerialname))[1]
              #objfile.write("usemap " os.path.basename(str(face.image.getFilename())))
              # useful for syzygy objinput: next line
              objfile.write("usemtl " + os.path.basename(str(face.image.getFilename())) + "\n")
              # useful for syzygy objinput: previous line
              mtlfile.write("newmtl " + os.path.basename(str(face.image.getFilename())) + "\n")
              r,g,b = mesh.materials[face.mat].getRGBCol()
              specularity = mesh.materials[face.mat].getSpec()
              roughness = "0"
              radmtlfile.write("void colorpict "+facematerialbasename+"\n")
              radmtlfile.write("7 red green blue "+facematerialbasename+".pic"+" . Lu Lv\n")
              radmtlfile.write("0\n")
              radmtlfile.write("0\n\n")
              # only use plastic material (roughness always 0) at the moment
              # must change it to object modifier, as taken from the library
              # or guessed from blender material
              radmtlfile.write(facematerialbasename+" plastic "+ os.path.basename(facematerialname)+ "\n")
              radmtlfile.write("0\n")
              radmtlfile.write("0\n")
              radmtlfile.write("5" + " " +  str(r) + " " +  str(g) + " " +  str(b)+ " " +str(specularity)+ " "  +str(roughness)+"\n\n")
              mtlfile.write("Kd " + " " +  str(r) + " " +  str(g) + " " +  str(b) + "\n")
              a = mesh.materials[face.mat].getAmb()
              r *= a
              g *= a
              b *= a
              mtlfile.write("Ka " + " " +  str(r) + " " +  str(g) + " " +  str(b) + "\n")
              r,g,b = mesh.materials[face.mat].getSpecCol()
              mtlfile.write("Ks " + " " +  str(r) + " " +  str(g) + " " +  str(b) + "\n")
              mtlfile.write("map_Kd " + os.path.basename(facematerialname) + "\n\n")

          faces = ""
          for vertex in face.v:
              #faces += str(vertex.index + 1) + "/" + str(vertex.index + 1)+ "/" + str(vertex.index + 1) + " "
              if face.smooth:
                 vertexnormalindex = str(faceindex + 1)
              else:
                 vertexnormalindex = ""
              if mesh.hasFaceUV() and face.mode == 5 and face.image!=None:
                 faces += str(vertex.index + 1) + "/" + str(faceindex + 1)+ "/" + vertexnormalindex + " "
              else:
                 faces += str(vertex.index + 1) + "//" + vertexnormalindex + " "
              faceindex += 1
	  if len(faces.split())>2:
              objfile.write("f " + faces + "\n")
          objfile.write("usemtl " + str(mesh.materials[0].getName()) + "\n")
      objfile.write("\n")
      radmtlfile.close()
      mtlfile.close()
      objfile.close()
     #except:
     # self.statuswindow.msg = msg_InternalError

    def compileMesh(self, frame, object):
      registered_data = Blender.Registry.GetKey('brad')
      if registered_data:
        brad_path  = registered_data['brad_path']
        debug  = registered_data['debug']

      objname = object.name
      meshname = object.data.name
      mesh = Blender.NMesh.GetRaw(meshname)
      if mesh:
        if self.exportwindow.ExportMethod.val == 2:
         if len(mesh.faces)>0:
          meshfilename = os.path.join(self.path, self.filename, zfill(str(frame),10), mesh.name+".obj")
          meshobjbasename = os.path.join(self.path, self.filename, zfill(str(frame),10), mesh.name)
          self.exportRadianceMeshOBJ(object, meshobjbasename, os.path.join(self.path, self.filename, zfill(str(frame),10)))
          obj2meshcommand = "obj2mesh -w -a " + meshobjbasename + ".mat " +  meshfilename + " " + meshobjbasename + ".rtm"
          if self.debug:
            print obj2meshcommand
          os.system(obj2meshcommand)

    def exportObject(self, scenefile, frame, obj):
            # get rotation, size and position from matrix
            #rot, siz, pos = infoFromMatrix(obj.matrix)
            name = obj.name
            # the object is a grid
            if name[:len("Grid")] == "Grid":
              if obj.getType() == "Mesh":
                meshdata = obj.getData()
                mesh = Blender.NMesh.GetRaw(meshdata.name)
                gridfilename = os.path.join(self.path, self.filename, zfill(str(frame),10), name +".pnt")
                gridfile = open(gridfilename, "w")
                # get transformation matrix
                matrix = obj.getMatrix('worldspace')
                for vertex in mesh.verts:
                  # transform vertex
                  vert = applyTransformation4(vertex.co, matrix)
                  norm = applyTransformation4(vertex.no, matrix)
                  norm.normalize()
                  gridfile.write("%s %s %s %s %s %s \n" %
                                ( vert[0] ,
                                  vert[1],
                                  vert[2],
                                  norm[0],
                                  norm[1],
                                  norm[2],))
                  """
                  gridfile.write("%s %s %s %s %s %s \n" %
                                (vertex.co[0] * obj.SizeX + obj.LocX,
                                  vertex.co[1] * obj.SizeY + obj.LocY,
                                  vertex.co[2] * obj.SizeZ + obj.LocZ,
                                  vertex.no[0],
                                  vertex.no[1],
                                  vertex.no[2],))
                  """
                gridfile.close()
            # the object is a cone
            #if name[:len("Cone")] == "Cone":
            #    self.statuswindow.msg = msg_Frame + str(frame) + " | " + msg_SavingCone + obj.name
            #    Draw()
            #    self.writeComment(scenefile, name)
            #    # create a new file with cone material and geometry
            #    conefilename = os.path.join(self.path, self.filename, zfill(str(frame),10), name +".rad")
            #    conefile = open(conefilename, "w")
            #    try:
            #       mat = Blender.Mesh.get(name).getMaterials()[0].name
            #    except:
            #       mat = "no_material"
            #    conefile.write("%s cone %s\n" % (mat, name))
            #    conefile.write("0\n")
            #    conefile.write("0\n")
            #    conefile.write("8\n")
            #    conefile.write("\t%s %s %s\n" % (0, 0, -1))
            #    conefile.write("\t%s %s %s\n" % (0, 0,  1))
            #    conefile.write("\t%s %s\n" % (math.sqrt(2), 0))
            #    conefile.write("%s ring %s_bottom\n" % (mat, name))
            #    conefile.write("0\n")
            #    conefile.write("0\n")
            #    conefile.write("8\n")
            #    conefile.write("\t%s %s %s\n" % (0, 0, -1))
            #    conefile.write("\t%s %s %s\n" % (0, 0, -1))
            #    conefile.write("\t%s %s\n" % (math.sqrt(2), 0))
            #    conefile.close()
            #    scenefile.write("!xform -s %s -rx %s -ry %s -rz %s -t %s %s %s %s\n" %
            #               (obj.SizeZ,
            #                180 * obj.RotX / math.pi,
            #                180 * obj.RotY / math.pi,
            #                180 * obj.RotZ / math.pi,
            #                obj.LocX, obj.LocY, obj.LocZ,
            #                "%s.rad" % name))
            #elif name[:len("Cube")] == "Cube":
            #    self.statuswindow.msg = msg_Frame + str(frame) + " | " + msg_SavingCube + obj.name
            #    Draw()
            #    self.writeComment(scenefile, name)
            #    try:
            #       mat = Blender.Mesh.get(name).getMaterials()[0].name
            #    except:
            #       mat = "no_material"
            #    scenefile.write("!genbox %s %s %s %s %s \\\n" % (mat, name,
            #                                                obj.SizeX * 2,
            #                                                obj.SizeY * 2,
            #                                                obj.SizeZ * 2))
            #    scenefile.write("\t| xform -t %s %s %s\\\n" % (-obj.SizeX,
            #                                              -obj.SizeY,
            #                                              -obj.SizeZ))
            #    self.writeMatrix(scenefile, obj)
            #elif name[:len("Cylinder")] == "Cylinder":
            #    self.statuswindow.msg = msg_Frame + str(frame) + " | " + msg_SavingCylinder + obj.name
            #    Draw()
            #    self.writeComment(scenefile, name)
            #    cylinderfilename = os.path.join(self.path, self.filename, zfill(str(frame),10), name +".rad")
            #    cylinderfile = open(cylinderfilename, "w")
            #    try:
            #       mat = Blender.Mesh.get(name).getMaterials()[0].name
            #    except:
            #       mat = "no_material"
            #    cylinderfile.write("%s cylinder %s\n" % (mat, name))
            #    cylinderfile.write("0\n")
            #    cylinderfile.write("0\n")
            #    cylinderfile.write("7\n")
            #    cylinderfile.write("\t%s %s %s\n" % (0, 0, -obj.SizeZ))
            #    cylinderfile.write("\t%s %s %s\n" % (0, 0,  obj.SizeZ))
            #    cylinderfile.write("\t%s\n" % math.sqrt(2))
            #    cylinderfile.write("%s ring %sbottom\n" % (mat, name))
            #    cylinderfile.write("0\n")
            #    cylinderfile.write("0\n")
            #    cylinderfile.write("8\n")
            #    cylinderfile.write("\t%s %s %s\n" % (0, 0, -obj.SizeZ))
            #    cylinderfile.write("\t%s %s %s\n" % (0, 0, -1))
            #    cylinderfile.write("\t%s %s\n" % (math.sqrt(2), 0))
            #    cylinderfile.write("%s ring %stop\n" % (mat, name))
            #    cylinderfile.write("0\n")
            #    cylinderfile.write("0\n")
            #    cylinderfile.write("8\n")
            #    cylinderfile.write("\t%s %s %s\n" % (0, 0, obj.SizeZ))
            #    cylinderfile.write("\t%s %s %s\n" % (0, 0, 1))
            #    cylinderfile.write("\t%s %s\n" % (math.sqrt(2), 0))
            #    cylinderfile.close()
            #    scenefile.write("!xform -rx %s -ry %s -rz %s -t %s %s %s %s\n" %
            #               (180 * obj.RotX / math.pi,
            #                180 * obj.RotY / math.pi,
            #                180 * obj.RotZ / math.pi,
            #                obj.LocX, obj.LocY, obj.LocZ,
            #                "%s.rad" % name))
            #elif name[:len("Plane")] == "Plane":
            #    self.statuswindow.msg = msg_Frame + str(frame) + " | " + msg_SavingPlane + obj.name
            #    Draw()
            #    self.writeComment(scenefile, name)
            #    planefilename = os.path.join(self.path, self.filename, zfill(str(frame),10), name +".rad")
            #    planefile = open(planefilename, "w")
            #    mat = Blender.Mesh.get(name).getMaterials()[0].name
            #    planefile.write("%s polygon %s\n" % (mat, name))
            #    planefile.write("0\n")
            #    planefile.write("0\n")
            #    planefile.write("12\n")
            #    planefile.write("\t%s %s %s\n" % (-obj.SizeX, -obj.SizeY, 0))
            #    planefile.write("\t%s %s %s\n" % ( obj.SizeX, -obj.SizeY, 0))
            #    planefile.write("\t%s %s %s\n" % ( obj.SizeX,  obj.SizeY, 0))
            #    planefile.write("\t%s %s %s\n" % (-obj.SizeX,  obj.SizeY, 0))
            #    planefile.close()
            #    scenefile.write("!xform -rx %s -ry %s -rz %s -t %s %s %s %s\n" %
            #               (180 * obj.RotX / math.pi,
            #                180 * obj.RotY / math.pi,
            #                180 * obj.RotZ / math.pi,
            #                obj.LocX, obj.LocY, obj.LocZ,
            #                "%s.rad" % name))
            #elif name[:len("Sphere")] == "Sphere":
            #    self.statuswindow.msg = msg_Frame + str(frame) + " | " + msg_SavingSphere + obj.name
            #    Draw()
            #    self.writeComment(scenefile, name)
            #    spherefilename = os.path.join(self.path, self.filename, zfill(str(frame),10), name +".rad")
            #    try:
            #       mat = Blender.Mesh.get(name).getMaterials()[0].name
            #    except:
            #       mat = "no_material"
            #    spherefile = open(spherefilename, "w")
            #    spherefile.write("%s sphere %s\n" % (mat, name))
            #    spherefile.write("0\n")
            #    spherefile.write("0\n")
            #    spherefile.write("4 %s %s %s %s\n" % (0, 0, 0, math.sqrt(2)))
            #    spherefile.close()
            #    scenefile.write("!xform -s %s -rx %s -ry %s -rz %s -t %s %s %s %s\n" %
            #               (obj.SizeX,
            #                180 * obj.RotX / math.pi,
            #                180 * obj.RotY / math.pi,
            #                180 * obj.RotZ / math.pi,
            #                obj.LocX, obj.LocY, obj.LocZ,
            #                "%s.rad" % name))
            #elif name[:len("SurfDonut")] == "SurfDonut":
            #    self.statuswindow.msg = msg_Frame + str(frame) + " | " + msg_SavingTorus + obj.name
            #    Draw()
            #    try:
            #       mat = Blender.Mesh.get(name).getMaterials()[0].name
            #    except:
            #       mat = "no_material"
            #    scenefile.write("!genrev %s " % mat)
            #    scenefile.write("torus '%s*sin(2*PI*t)' " % (0.25 * obj.SizeX,))
            #    scnenfile.write("'%s+%s*cos(2*PI*t)' " % (0.75 * obj.SizeX,
            #                                         0.25 * obj.SizeX))
            #    scenefile.write("%s \\\n" % 32)
            #    self.writeMatrix(scenefile, obj)
            #elif name[:len("SurfSphere")] == "SurfSphere":
            #    self.statuswindow.msg = msg_Frame + str(frame) + " | " + msg_SavingSphere + obj.name
            #    Draw()
            #    self.writeComment(scenefile, name)
            #    spherefilename = os.path.join(self.path, self.filename, zfill(str(frame),10), name +".rad")
            #    spherefile = open(spherefilename, "w")
            #    try:
            #       mat = Blender.Mesh.get(name).getMaterials()[0].name
            #    except:
            #       mat = "no_material"
            #    spherefile.write("%s sphere %s\n" % (mat, name))
            #    spherefile.write("0\n")
            #    spherefile.write("0\n")
            #    spherefile.write("4 %s %s %s %s\n" % (0, 0, 0, 1))
            #    spherefile.close()
            #    scenefile.write("!xform -s %s -rx %s -ry %s -rz %s -t %s %s %s %s\n" %
            #               (obj.SizeX,
            #                180 * obj.RotX / math.pi,
            #                180 * obj.RotY / math.pi,
            #                180 * obj.RotZ / math.pi,
            #                obj.LocX, obj.LocY, obj.LocZ,
            #                "%s.rad" % name))
            #elif name[:len("SurfTube")] == "SurfTube":
            #    self.statuswindow.msg = msg_Frame + str(frame) + " | " + msg_SavingCylinder + obj.name
            #    Draw()
            #    self.writeComment(scenefile, name)
            #    tubefilename = os.path.join(self.path, self.filename, zfill(str(frame),10), name +".rad")
            #    tubefile = open(tubefilename, "w")
            #    try:
            #       mat = Blender.Mesh.get(name).getMaterials()[0].name
            #    except:
            #       mat = "no_material"
            #    tubefile.write("%s cylinder %s\n" % (mat, name))
            #    tubefile.write("0\n")
            #    tubefile.write("0\n")
            #    tubefile.write("7\n")
            #    tubefile.write("\t%s %s %s\n" % (0, 0, -obj.SizeZ))
            #    tubefile.write("\t%s %s %s\n" % (0, 0,  obj.SizeZ))
            #    tubefile.write("\t%s\n" % obj.SizeX)
            #    tubefile.close()
            #    scenefile.write("!xform -rx %s -ry %s -rz %s -t %s %s %s %s\n" %
            #               (180 * obj.RotX / math.pi,
            #                180 * obj.RotY / math.pi,
            #                180 * obj.RotZ / math.pi,
            #                obj.LocX, obj.LocY, obj.LocZ,
            #                "%s.rad" % name))
            #elif name[:len("Tube")] == "Tube":
            #    self.statuswindow.msg = msg_Frame + str(frame) + " | " + msg_SavingCylinder + obj.name
            #    Draw()
            #    self.writeComment(file, name)
            #    tubefilename = os.path.join(self.path, self.filename, zfill(str(frame),10), name +".rad")
            #    tubefile = open(tubefilename, "w")
            #    try:
            #       mat = Blender.Mesh.get(name).getMaterials()[0].name
            #    except:
            #       mat = "no_material"
            #    tubefile.write("%s cylinder %s\n" % (mat, name))
            #    tubefile.write("0\n")
            #    tubefile.write("0\n")
            #    tubefile.write("7\n")
            #    tubefile.write("\t%s %s %s\n" % (0, 0, -obj.SizeZ))
            #    tubefile.write("\t%s %s %s\n" % (0, 0,  obj.SizeZ))
            #    tubefile.write("\t%s\n" % (math.sqrt(2) * obj.SizeX,))
            #    tubefile.close()
            #    scenefile.write("!xform -rx %s -ry %s -rz %s -t %s %s %s %s\n" %
            #               (180 * obj.RotX / math.pi,
            #                180 * obj.RotY / math.pi,
            #                180 * obj.RotZ / math.pi,
            #                obj.LocX, obj.LocY, obj.LocZ,
            #                "%s.rad" % name))
            else:
             if obj.getType()=="Mesh":
              meshdata = obj.getData()
              mesh = Blender.NMesh.GetRaw(meshdata.name)
              if len(mesh.faces)>0:
                context = self.scene.getRenderingContext()
                context.currentFrame(frame)
                # next line is old syntax
                #self.scene.frameSettings(context.startFrame(), context.endFrame(), frame)
                Blender.Window.Redraw()
                self.writeComment(scenefile, name)
                if self.exportwindow.ExportMethod.val == 0:
                  # scaling
                  sX, sY, sZ = obj.getSize()
                  scale = (sX+sY+sZ)/3.0
                  # declare instance
                  scenefile.write("\nvoid instance %s\n" % (name))
                  scenefile.write("13 %s -s %s -rx %s -ry %s -rz %s -t %s %s %s \n" %
                                ("%s.oct" % meshdata.name,
                                  scale,
                                  180 * obj.RotX / math.pi,
                                  180 * obj.RotY / math.pi,
                                  180 * obj.RotZ / math.pi,
                                  obj.LocX, obj.LocY, obj.LocZ))
                  scenefile.write("0\n")
                  scenefile.write("0\n\n")
                elif self.exportwindow.ExportMethod.val == 1:
                  # scaling
                  sX, sY, sZ = obj.getSize()
                  scale = (sX+sY+sZ)/3.0
                  # declare xforms
                  scenefile.write("!xform -s %s -rx %s -ry %s -rz %s -t %s %s %s %s\n" %
                                ( scale,
                                  180 * obj.RotX / math.pi,
                                  180 * obj.RotY / math.pi,
                                  180 * obj.RotZ / math.pi,
                                  obj.LocX, obj.LocY, obj.LocZ,
                                  "%s.rad" % meshdata.name))
                elif self.exportwindow.ExportMethod.val == 2:
                  # declare mesh primitive as this:
                  #   mod mesh id
                  #   1+ output.rtm [xform args]
                  #   0
                  #   0
                  # scaling
                  sX, sY, sZ = obj.getSize()
                  scale = (sX+sY+sZ)/3.0
                  scenefile.write("\nvoid mesh %s\n" % (name))
                  scenefile.write("13 %s -s %s -rx %s -ry %s -rz %s -t %s %s %s \n" %
                                ("%s.rtm" % meshdata.name,
                                  scale,
                                  180 * obj.RotX / math.pi,
                                  180 * obj.RotY / math.pi,
                                  180 * obj.RotZ / math.pi,
                                  obj.LocX, obj.LocY, obj.LocZ))
                  scenefile.write("0\n")
                  scenefile.write("0\n\n")

    # writeScene: begin
    def writeScene(self, frame):
    # should I export selected objects or selected layers?
        registered_data = Blender.Registry.GetKey('brad')
        if registered_data:
          brad_path  = registered_data['brad_path']
          convert_path  = registered_data['convert_path']
          debug  = registered_data['debug']

        # create new scene file if it doesn't exist
        self.scenefilename = os.path.join(self.path, self.filename, zfill(str(frame),10), self.filename + ".rad")
        scenefile = open(self.scenefilename, "w")

        # go to current frame
        """ old way, not working anymore
        self.scene.currentFrame(frame)
        self.scene.frameSettings(context.startFrame(), context.endFrame(), frame)
        """
        context = self.scene.getRenderingContext()
        context.currentFrame(frame)

        if self.exportwindow.ExportSelected.val==1:
            # get all the selected objects
            objects = Blender.Object.GetSelected()
        else:
            # get all the objects
            objects = Blender.Object.Get()

        # check layer settings and deselect objects that are not in the selected layers
        objs = objects
        for obj in objects:
          if not self.checkLayers(self.getObjectLayers(obj.Layer),self.getSelectedLayers()):
              if self.debug:
                 print "deselecting object ", obj
              objs.remove(obj)
        self.frameobjs = objs

        #check mesh datablocks
        datablocks = []
        meshdatablocks = []
        for obj in objs:
          if obj.getType()=="Mesh":
              if datablocks.count(obj.getData().name)==0:
                datablocks.append(obj.getData().name)
                meshdatablocks.append(obj)
        #convert pictures if using the mesh primive export method
        if self.exportwindow.ExportMethod.val == 2:
          if self.debug:
            print "converting images ..."
          self.convertImages(os.path.join(self.path, self.filename, zfill(str(frame),10)))
        #write and oconv all mesh datablocks
        for datablock in meshdatablocks:
                meshdata = datablock.getData()
                mesh = Blender.NMesh.GetRaw(meshdata.name)
                self.writeMesh(frame, mesh, datablock)
                # oconv polygonal object
                if self.exportwindow.ExportMethod.val == 0:
                  self.oconvMesh(frame, mesh)
                # export as obj and compile mesh primitive
                if self.exportwindow.ExportMethod.val == 2:
                  self.compileMesh(frame, datablock)
        #to be done: export as mesh primitives !!!

        # write header
        scenefile.write("# %s - frame n. %s - exported by brad\n\n" % (self.filename, str(frame)) )

        # load luminaires library
        try:
          lumlist = loadRadLib(os.path.join(brad_path, "brad", "radiance", "lum"))
          lumlist.sort()
        except:
          pass

        # write modifiers
        scenefile.write("\n# =============\n")
        scenefile.write("# = Modifiers =\n")
        scenefile.write("# =============\n")

        # light modifiers and cameras
        # reminder: try to understand why other cameras don't get exported
        # reminder: fix worldmatrix rotation for lamp objects
        for obj in objs:
           name = obj.name
           if obj.getType() == "Lamp":
            if self.debug:
              print "exporting lamp ..."
            objdata = obj.getData()
            lamp = Blender.Lamp.get(objdata.name)
            # luminaire from library (if it has the same name of a library luminaire)
            if lumlist.count(lamp.name)>0:
             try:
              if self.debug:
                print "exporting luminaire", lamp.name, "from library"
              self.writeComment(scenefile, lamp.name)
              lampfilename = os.path.join(self.path, self.filename, zfill(str(frame),10), lamp.name +".rad")
              lumfilename = os.path.join(brad_path, "brad","radiance","lum", lamp.name, lamp.name+".rad")
              shutil.copyfile(lumfilename, lampfilename)
              """
              lampfile = open(lampfilename, "w")
              lumfile = open(lumfilename, "r")
              lumfilelines = lumfile.readlines()
              for line in lumfilelines:
                lampfile.write(line)
                lampfile.write("\n\n")
              lumfile.close()
              """
              datfilename=os.path.join(self.path, self.filename, zfill(str(frame),10), lamp.name +".dat")
              datsrcfilename=os.path.join(brad_path, "brad","radiance","lum", lamp.name, lamp.name+".dat")
              if os.path.exists(datsrcfilename):
                  shutil.copyfile(datsrcfilename, datfilename)
             except:
              pass
            else:
              if lamp.getType() == 0 or lamp.getType() == "Lamp":
                # light
                if self.debug:
                  print "exporting lamp modifier", name
                self.writeComment(scenefile, name)
                scenefile.write("void light %s\n" % name)
                scenefile.write("0\n")
                scenefile.write("0\n")
                scenefile.write("3 %s %s %s\n" % (lamp.energy * lamp.R * 1000, lamp.energy * lamp.G * 1000, lamp.energy * lamp.B * 1000))
              elif lamp.getType() == 2 or lamp.getType() == "Spot":
                # spotlight
                if self.debug:
                  print "exporting spotlight modifier", name
                """
                matx = Matrix(3, 3)
                cosx = math.cos(obj.RotX)
                sinx = math.sin(obj.RotX)
                matx[0] = [1,    0,     0]
                matx[1] = [0, cosx, -sinx]
                matx[2] = [0, sinx,  cosx]
                maty = Matrix(3, 3)
                cosy = math.cos(obj.RotY)
                siny = math.sin(obj.RotY)
                maty[0] = [ cosy, 0, siny]
                maty[1] = [    0, 1,    0]
                maty[2] = [-siny, 0, cosy]
                matz = Matrix(3, 3)
                cosz = math.cos(obj.RotZ)
                sinz = math.sin(obj.RotZ)
                matz[0] = [cosz, -sinz, 0]
                matz[1] = [sinz,  cosz, 0]
                matz[2] = [   0,     0, 1]
                mat = matz * maty * matx
                zvec = Matrix(3, 1)
                zvec[0] = [ 0]
                zvec[1] = [ 0]
                zvec[2] = [-1]
                aim = mat * zvec
                """
                # apply translation and rotation
                matrix = obj.getMatrix('worldspace')
                locmatrix = obj.getMatrix('localspace')
                zvec = Blender.Mathutils.Vector([0,0,-1])
                vpvec = Blender.Mathutils.Vector([obj.LocX, obj.LocY, obj.LocZ])
                rotationmatrix=matrix.rotationPart()
                aim = self.applyTransformation3(zvec, rotationmatrix)
                # need to fix camera rotation for tracking camera with nonzero rotation angles !!!
                """
                rotationmatrix=matrix.rotationPart()
                vd = self.applyTransformation3(vd, rotationmatrix)
                vu = self.applyTransformation3(vu, rotationmatrix)
                """
                if not self.sameVector3(matrix.translationPart(), locmatrix.translationPart()):
                  vp = self.applyTransformation4(vpvec, matrix)
                else:
                  vp = [obj.LocX, obj.LocY, obj.LocZ]

                self.writeComment(scenefile, name)
                scenefile.write("void spotlight %s\n" % name)
                scenefile.write("0\n")
                scenefile.write("0\n")
                scenefile.write("7 %s %s %s %s %s %s %s \n" %
                                (lamp.energy * lamp.R * 1000,
                                  lamp.energy * lamp.G * 1000,
                                  lamp.energy * lamp.B * 1000,
                                  lamp.spotSize,
                                  aim[0],
                                  aim[1],
                                  aim[2]
                                  ))
            if obj.getType() == "Camera":
                if self.debug:
                  print "exporting camera: " + obj.name
                self.writeSelectedCamera(obj, frame)


        # material modifiers
        self.writeComment(scenefile, "no_material")
        self.writeNoMaterial(scenefile)
        mats = Blender.Material.get()
        for mat in mats:
            name = mat.name
            #print mat.rgbCol
            self.writeComment(scenefile, name)
            self.writeMaterial(scenefile, mat, name)


        # write light sources
        scenefile.write("\n# =================\n")
        scenefile.write("# = Light sources =\n")
        scenefile.write("# =================\n")
        for obj in objs:
          #name = obj.name
          if obj.getType() == "Lamp":
            # apply translation and rotation
            matrix = obj.getMatrix('worldspace')
            locmatrix = obj.getMatrix('localspace')
            zvec = Blender.Mathutils.Vector([0,0,-1])
            vpvec = Blender.Mathutils.Vector([obj.LocX, obj.LocY, obj.LocZ])
            rotationmatrix=matrix.rotationPart()
            aim = self.applyTransformation3(zvec, rotationmatrix)
            # need to fix camera rotation for tracking object with nonzero rotation angles !!!
            """
            rotationmatrix=matrix.rotationPart()
            vd = self.applyTransformation3(vd, rotationmatrix)
            vu = self.applyTransformation3(vu, rotationmatrix)
            """
            if not self.sameVector3(matrix.translationPart(), locmatrix.translationPart()):
              #vp = self.applyTransformation4(vpvec, matrix)
              vp = [obj.LocX, obj.LocY, obj.LocZ]
            else:
              vp = [obj.LocX, obj.LocY, obj.LocZ]

            lamp = obj.getData()
            # luminaire from library (if it has the same name of a library luminaire)
            if lumlist.count(lamp.name)>0:
                if self.debug:
                  print "exporting luminaire from library", lamp.name
                scenefile.write("!xform -rx %s -ry %s -rz %s -t %s %s %s %s\n" %
                              (180 * obj.RotX / math.pi,
                                180 * obj.RotY / math.pi,
                                180 * obj.RotZ / math.pi,
                                vp[0], vp[1], vp[2],
                                "%s.rad" % lamp.name))

            else:
                if lamp.mode >= 128:
                    if self.debug:
                      print "exporting area light source", lamp.name
                    self.writeComment(scenefile, lamp.name)
                    scenefile.write("%s polygon fixture\n" % lamp.name)
                    scenefile.write("0\n")
                    scenefile.write("0\n")
                    # todo: check if this still works ...
                    scenefile.write("12 %s %s %s\n" % ((obj.LocX - 1), (obj.LocY * -1 + 1), obj.getLocation()[2]))
                    scenefile.write("\t%s %s %s\n" % ((obj.LocX + 1), (obj.LocY * -1 + 1), obj.getLocation()[2]))
                    scenefile.write("\t%s %s %s\n" % ((obj.LocX + 1), (obj.LocY * -1 - 1), obj.getLocation()[2]))
                    scenefile.write("\t%s %s %s\n" % ((obj.LocX - 1), (obj.LocY * -1 - 1), obj.getLocation()[2]))
                    #print (lamp.name + " - Area lamp")
                    self.statuswindow.msg = msg_Frame + str(frame) + " | " + msg_SavingAreaLamp + lamp.name
                    Draw()
                else:
                  if lamp.getType() == "Lamp" or lamp.getType() == 0:
                    # spherical lamp
                    if self.debug:
                      print "exporting lamp geometry", lamp.name
                    self.writeComment(scenefile, lamp.name)
                    scenefile.write("%s sphere fixture\n" % obj.name)
                    scenefile.write("0\n")
                    scenefile.write("0\n")
                    scenefile.write("4 %s %s %s %s\n" % (vp[0], vp[1], vp[2], 0.05))
                    #print (lamp.name + " - Spherical lamp")
                    self.statuswindow.msg = msg_Frame + str(frame) + " | " + msg_SavingSphericalLamp + lamp.name
                    Draw()
                  elif lamp.getType() == "Spot" or lamp.getType() == 2:
                    # spotlight ring
                    if self.debug:
                      print "exporting spotlight geometry", lamp.name
                    """
                    # rotation matrix
                    matx = Matrix(3, 3)
                    cosx = math.cos(obj.RotX)
                    sinx = math.sin(obj.RotX)
                    matx[0] = [1,    0,     0]
                    matx[1] = [0, cosx, -sinx]
                    matx[2] = [0, sinx,  cosx]
                    maty = Matrix(3, 3)
                    cosy = math.cos(obj.RotY)
                    siny = math.sin(obj.RotY)
                    maty[0] = [ cosy, 0, siny]
                    maty[1] = [    0, 1,    0]
                    maty[2] = [-siny, 0, cosy]
                    matz = Matrix(3, 3)
                    cosz = math.cos(obj.RotZ)
                    sinz = math.sin(obj.RotZ)
                    matz[0] = [cosz, -sinz, 0]
                    matz[1] = [sinz,  cosz, 0]
                    matz[2] = [   0,     0, 1]
                    mat = matz * maty * matx
                    zvec = Matrix(3, 1)
                    zvec[0] = [ 0]
                    zvec[1] = [ 0]
                    zvec[2] = [-1]
                    aim = mat * zvec
                    """
                    self.writeComment(scenefile, lamp.name)
                    scenefile.write("%s ring spotlamp\n" % obj.name)
                    scenefile.write("0\n")
                    scenefile.write("0\n")
                    scenefile.write("8 %s %s %s %s %s %s %s %s\n" %
                                      (vp[0],vp[1],vp[2],
                                      aim[0],
                                      aim[1],
                                      aim[2],
                                      0, 0.125))
                    #obj.LocX, obj.LocY, obj.LocZ,
                    self.statuswindow.msg = msg_Frame + str(frame) + " | " + msg_SavingSpotLamp + lamp.name
                    Draw()


        #write geometry
        scenefile.write("\n# ============\n")
        scenefile.write("# = Geometry =\n")
        scenefile.write("# ============\n")
        for obj in objs:
            self.exportObject(scenefile, frame, obj)

        scenefile.close()

        #write world
        self.writeWorld(frame)

        #oconv scene
        prevpath = os.getcwd()
        framepath = os.path.join(self.path, self.filename, zfill(str(frame),10))
        os.chdir(framepath)
        self.skyfilename = os.path.join(self.path, self.filename, zfill(str(frame),10), self.filename + ".sky")
        self.scenefilename = os.path.join(self.path, self.filename, zfill(str(frame),10), self.filename + ".rad")
        sceneoctreefilename = os.path.join(self.path, self.filename, zfill(str(frame),10), self.filename + ".oct")
        if (self.parwindow.RenderRad.val==1):
          if self.skyweatherwindow.generateSky.val:
             oconvcommand = "oconv " +  self.radparwindow.oconvOptions.val + " "+ self.skyfilename + " " + self.scenefilename + " > " + sceneoctreefilename
          else:
             oconvcommand = "oconv " +  self.radparwindow.oconvOptions.val + " " + self.scenefilename + " > " + sceneoctreefilename
        else:
          if self.skyweatherwindow.generateSky.val:
             oconvcommand = "oconv " + self.skyfilename+ " " + self.scenefilename + " > " + sceneoctreefilename
          else:
             oconvcommand = "oconv " + " " + self.scenefilename + " > " + sceneoctreefilename
        if self.debug:
          print oconvcommand
        os.system(oconvcommand)
        os.chdir(prevpath)
    # writeScene: end

    def writeAnimationRad(self):
        self.writeFrameRad(0)

    def writeAnimationRadScript(self):
        scriptfilename = os.path.join(self.path, self.filename, self.filename +".py")
        scriptfile = open(scriptfilename, "w")
        scriptfile.write("#!/usr/bin/python\n\n")
        scriptfile.write("import os \n")
        scriptfile.write("import os.path \n")
        scriptfile.write("import string \n")
        scriptfile.write("try: \n")
        scriptfile.write("   os.mkdir(\"pic\") \n")
        scriptfile.write("except: \n")
        scriptfile.write("   print(\"the radiance pictures directory exists\") \n")
        scriptfile.write("try: \n")
        scriptfile.write("   os.mkdir(\"tif\") \n")
        scriptfile.write("except: \n")
        scriptfile.write("   print(\"the tiff pictures directory exists\") \n")
        scriptfile.write("for dirname in os.listdir(os.getcwd()):\n")
        scriptfile.write(" if os.path.isdir(dirname):\n")
        scriptfile.write("   print \"rendering frame: \", dirname \n")
        scriptfile.write("   os.chdir(dirname) \n")
        scriptfile.write("   radcommand = \"rad " + self.filename + ".rif\"\n")
        scriptfile.write("   tiffcommand = \"ra_tiff " + self.filename + "_\" + dirname + \".pic " + self.filename + "_\" + dirname + \".tif\"\n")
        scriptfile.write("   os.system(radcommand)\n")
        scriptfile.write("   os.system(tiffcommand)\n")
        scriptfile.write("   os.system(\"cp *.pic ../pic\")\n")
        scriptfile.write("   os.system(\"cp *.tif ../tif\")\n")
        scriptfile.write("   os.chdir(\"../\")\n")
        scriptfile.close()

    def writeFrameRad(self, frame):
        if frame == 0:
                riffilename = os.path.join(self.path, self.filename, zfill(str(self.staframe),10), self.filename + ".rif")
        else:
                riffilename = os.path.join(self.path, self.filename, zfill(str(frame),10), self.filename + ".rif")
        riffile = open(riffilename, "w")
        IntExt = ["Exterior", "Interior"]
        QVD = ["Low", "Medium", "High"]
        TrueFalse = ["False", "True"]
        # ZONE
        riffile.write("ZONE= "
                      + IntExt[self.parwindow.ZoneInterior.val] + " "
                      + str(self.parwindow.ZoneMinX.val) + " "
                      + str(self.parwindow.ZoneMaxX.val) + " "
                      + str(self.parwindow.ZoneMinY.val) + " "
                      + str(self.parwindow.ZoneMaxY.val) + " "
                      + str(self.parwindow.ZoneMinZ.val) + " "
                      + str(self.parwindow.ZoneMaxZ.val) + "\n"
                      )
        # scene
        if self.skyweatherwindow.generateSky.val:
           riffile.write("scene= " + self.filename + ".sky " + self.filename + ".rad \n")
        else:
           riffile.write("scene= " + self.filename + ".rad \n")
        # QUALITY
        riffile.write("QUALITY= " + QVD[self.parwindow.Quality.val] + "\n")
        # VARIABILITY
        riffile.write("VARIABILITY= " + QVD[self.parwindow.Variability.val] + "\n")
        # DETAIL
        riffile.write("DETAIL= " + QVD[self.parwindow.Detail.val] + "\n")
        # PENUMBRAS
        riffile.write("PENUMBRAS= " + TrueFalse[self.parwindow.Penumbras.val] + "\n")
        # INDIRECT
        riffile.write("INDIRECT= " + str(self.parwindow.Indirect.val) + "\n")
        # RESOLUTION
        riffile.write("RESOLUTION= " + str(self.parwindow.PictureResolutionX.val) + " " + str(self.parwindow.PictureResolutionY.val) + "\n")
        if self.radparwindow.PictureName.val!="":
          # PICTURE
          riffile.write("PICTURE= " + self.radparwindow.PictureName.val + "\n")
        if self.radparwindow.RawFile.val!="":
          # RAWFILE
          riffile.write("RAWFILE= " + self.radparwindow.RawFile.val + "\n")
        if self.radparwindow.OctreeName.val!="":
          # OCTREE
          riffile.write("OCTREE= " + self.radparwindow.OctreeName.val + "\n")
        if self.radparwindow.OptionsFile.val!="":
          # OPTFILE
          riffile.write("OPTFILE= " + self.radparwindow.OptionsFile.val + "\n")
        if self.radparwindow.AmbientFile.val!="":
          # AMBFILE
          riffile.write("AMBFILE= " + self.radparwindow.AmbientFile.val + "\n")
        if self.radparwindow.ZFile.val!="":
          # ZFILE
          riffile.write("ZFILE= " + self.radparwindow.ZFile.val + "\n")
        if self.radparwindow.oconvOptions.val!="":
          # oconv
          riffile.write("oconv= " + self.radparwindow.oconvOptions.val + "\n")
        if self.radparwindow.mkillumOptions.val!="":
          # mkillum
          riffile.write("mkillum= " + self.radparwindow.mkillumOptions.val + "\n")
        if self.radparwindow.renderOptions.val!="":
          # render
          riffile.write("render= " + self.radparwindow.renderOptions.val + " -w \n")
        if self.radparwindow.pfiltOptions.val!="":
          # pfilt
          riffile.write("pfilt= " + self.radparwindow.pfiltOptions.val + "\n")
        if self.radparwindow.Exposure.val!=0.0:
          # EXPOSURE
          riffile.write("EXPOSURE= " + str(self.radparwindow.Exposure.val) + "\n")
        if self.radparwindow.Eyesep.val!=0.0:
          # EYESEP
          riffile.write("EYESEP= " + str(self.radparwindow.Eyesep.val) + "\n")
        # REPORT
        riffile.write("REPORT= " + str(self.radparwindow.ReportTime.val) + " " + self.radparwindow.ReportFile.val + "\n")
        # views
        if frame == 0:
          if self.staframe<=self.endframe:
            for frame in range(self.staframe, self.endframe + 1):
                riffile.write("view = " + zfill(str(frame),10) + " -vf " + os.path.join(self.path, self.filename, zfill(str(frame),10), self.filename) + ".vf \n")
        else:
          riffile.write("view = " + zfill(str(frame),10) + " -vf " + self.filename + ".vf \n")
        riffile.close()

    def renderInteractive(self, scene, export):
      self.curframe = Blender.Get("curframe")
      self.scene = scene
      # export geometry
      if export:
          self.exportCurrentFrame(scene)
      # export geometry if necessary
      else:
        if self.exportwindow.geometryHasBeenExported[0]==0:
          self.exportCurrentFrame(scene)
        elif self.curframe<self.exportwindow.geometryHasBeenExported[1] and self.curframe>self.exportwindow.geometryHasBeenExported[2]:
          self.exportCurrentFrame(scene)
      # re-create rif and view file
      self.writeFrameRad(self.curframe)
      self.writeCamera(self.curframe)
      # use rad to render if rad is the selected rendering method
      if self.parwindow.RenderRad.val == 1:
          prevpath = os.getcwd()
          framepath = os.path.join(self.path, self.filename, zfill(str(self.curframe),10))
          os.chdir(framepath)
          # launch rad -o x11
          if sys.platform=='win32':
              radcommand = "rad -o x11 " + self.filename + ".rif"
          else:
              radcommand = "rad -o x11 " + self.filename + ".rif &"
          try:
            os.system(radcommand)
          except:
            self.statuswindow.msg = msg_InternalError
          os.chdir(prevpath)

    # reminder: fix worldspace rotation for grids and vtkgrids
    def calculateGrids(self, scene):
      self.curframe = Blender.Get("curframe")
      self.scene = scene
      # export geometry
      self.exportCurrentFrame(scene)
      # re-create rif and view file
      self.writeFrameRad(self.curframe)
      self.writeCamera(self.curframe)
      # use rad to render if rad is the selected rendering method
      prevpath = os.getcwd()
      framepath = os.path.join(self.path, self.filename, zfill(str(self.curframe),10))
      os.chdir(framepath)
      for grid in self.frameobjs:
          if grid.name[:len("Grid")] == "Grid":
            gridfilename = os.path.join(self.path, self.filename, zfill(str(self.curframe),10), grid.name +".pnt")
            # launch rtrace
            if sys.platform=='win32':
                rtracecommand = "rtrace -h -w " + self.calculatewindow.rtraceOptions.val + " " + self.filename + ".oct < " + grid.name + ".pnt | rcalc -e '" + self.calculatewindow.postProcessOptions.val + "' > " + grid.name + ".out"
            else:
                rtracecommand = "rtrace -h -w " + self.calculatewindow.rtraceOptions.val + " " + self.filename + ".oct < " + grid.name + ".pnt | rcalc -e '" + self.calculatewindow.postProcessOptions.val + "' > " + grid.name + ".out"
            os.system(rtracecommand)

            meshdata = grid.getData()
            mesh = Blender.NMesh.GetRaw(meshdata.name)
            #count number of vertex in the current grid
            vnum=0
            for vert in mesh.verts:
                vnum+=1
            #export VTKPolyData + scalars
            self.writeVtkPolydata(mesh, grid)
            vtksrcfilename = os.path.join(self.path, self.filename, zfill(str(self.curframe),10), mesh.name +".vtk")
            vtkdstfilename = os.path.join(self.path, self.filename, zfill(str(self.curframe),10), mesh.name +".out.vtk")
            shutil.copyfile(vtksrcfilename, vtkdstfilename)
            vtkdstfile = open(vtkdstfilename, 'a')
            vtkdstfile.write("POINT_DATA " + str(vnum) + "\n")
            vtkdstfile.write("SCALARS values float\nLOOKUP_TABLE default\n")
            outfilename = os.path.join(self.path, self.filename, zfill(str(self.curframe),10), mesh.name +".out")
            outfile = open(outfilename, "r")
            outfilelines = outfile.readlines()
            for line in outfilelines:
                vtkdstfile.write(line)
                #vtkdstfile.write("\n")
            outfile.close()
            vtkdstfile.close()
      os.chdir(prevpath)

    def updateGrids(self, scene):
      registered_data = Blender.Registry.GetKey('brad')
      if registered_data:
        brad_path  = registered_data['brad_path']
        convert_path  = registered_data['convert_path']
        debug  = registered_data['debug']
      self.curframe = Blender.Get("curframe")
      self.scene = scene
      for grid in self.frameobjs:
          gridoutfilename = os.path.join(self.path, self.filename, zfill(str(self.curframe),10), grid.name +".out")
          if grid.name[:len("Grid")] == "Grid" and os.path.exists(gridoutfilename):
            if self.debug:
              print gridoutfilename
            if grid.getType() == "Mesh":
                meshdata = grid.getData()
                mesh = Blender.NMesh.GetRaw(meshdata.name)
                gridoutfile = open(gridoutfilename, "r")
                # get max value
                max=0.0
                values = []
                for vertex in mesh.verts:
                  value = float(string.strip(gridoutfile.readline()))
                  values.append(value)
                  if value>max:
                      max=value
                if max == 0.0:
                  max = 0.000000001
                # get min value
                min=max
                i=0
                for vertex in mesh.verts:
                  if values[i]<min:
                      min=values[i]
                  i=+1
                # get average value
                sum = 0
                i=0
                for vertex in mesh.verts:
                  sum += values[i]
                  i+=1
                sum /= i
                if sum == 0.0:
                  sum = 0.000000001
                for face in mesh.faces:
                  vi = []
                  vc = []
                  i = 0
                  for vert in face.v:
                      vi.append(vert.index)
                      if values[vert.index]>=(max/2):
                        vc.append(Blender.NMesh.Col(int(values[vert.index]/max*255),255-int(values[vert.index]/max*255),0))
                      else:
                        vc.append(Blender.NMesh.Col(0,int(values[vert.index]/max*255),255-int(values[vert.index]/max*255)))
                      if self.debug:
                        print values[vert.index]
                      i+=1
                  face.col = vc
                mesh.update()
                gridoutfile.close()
                self.statuswindow.msg = "min: " + str(round(min,3)) + " | " + "max: " + str(round(max,3)) + " | av: " + str(round(sum,3)) + " | unif: " + str(round(min/sum,3))

    def writeVtkPolydata(self, mesh, obj):
      self.curframe = Blender.Get("curframe")
      self.framedir = os.path.join(self.path, self.filename, zfill(str(self.curframe),10))
      try:
          os.mkdir(self.framedir)
          self.statuswindow.msg = msg_FrameDirectoryCreated + self.framedir
          Draw()
      except:
          self.statuswindow.msg = msg_FrameDirectoryNotCreated + self.framedir
          Draw()
      if mesh:
          if len(mesh.faces)>0:
                    meshfilename = os.path.join(self.path, self.filename, zfill(str(self.curframe),10), obj.name +".vtk")
                    meshfile = open(meshfilename, "w")
                    #VTK legacy file header
                    meshfile.write("# vtk DataFile Version 2.0\n")
                    meshfile.write(mesh.name+"\n")
                    meshfile.write("ASCII\n")
                    meshfile.write("DATASET POLYDATA\n")
                    meshfile.write("POINTS "+ str(len(mesh.verts)) + " float\n")
                    # get transformation matrix
                    matrix = obj.getMatrix('worldspace')
                    #export points
                    for vertex in mesh.verts:
                        # transform vertex
                        vert = applyTransformation4(vertex.co, matrix)
                        norm = applyTransformation4(vertex.no, matrix)
                        norm.normalize()
                        meshfile.write(str(vert[0]) + " " + str(vert[1]) + " "  + str(vert[2]) + "\n")
                        #meshfile.write(str(vert.co[0] * obj.SizeX + obj.LocX) + " " + str(vert.co[1] * obj.SizeY + obj.LocY) + " "  + str(vert.co[2] * obj.SizeZ + obj.LocZ) + "\n")
                    #status message
                    self.statuswindow.msg = msg_SelectedVTKExported
                    Draw()
                    #export polygons
                    total = 0
                    for face in mesh.faces:
                      #if len(face.v) > 2:
                      total += (len(face.v)+1)
                    meshfile.write("POLYGONS "+ str(len(mesh.faces)) + " " + str(total) + "\n")
                    for face in mesh.faces:
                        num = len(face.v)
                        line = str(num) + " "
                        #if len(face.v) > 2:
                        for vertex in face.v:
                          line +=  str(vertex.index) + " "
                        meshfile.write(line+ "\n")
                    #export normals
                    #meshfile.write("NORMALS "+ "object_normals" + " " + "float" + "\n")
                    #for vert in mesh.verts:
                    #     meshfile.write(str(vert.no[0]) + " " + str(vert.no[1]) + " "  + str(vert.no[2]) + "\n")
                    #export cells
                    """
                    meshfile.write("CELL_DATA "+ str(len(mesh.faces)) + "\n")
                    meshfile.write("SCALARS cell_scalars int\n")
                    meshfile.write("LOOKUP_TABLE default\n")
                    i=0
                    for face in mesh.faces:
                        meshfile.write(str(i)+ "\n")
                        i+=1
                    """
                    meshfile.close()

    def writeGrid(self, obj):
      self.curframe = Blender.Get("curframe")
      self.framedir = os.path.join(self.path, self.filename, zfill(str(self.curframe),10))
      try:
                os.mkdir(self.framedir)
                self.statuswindow.msg = msg_FrameDirectoryCreated + self.framedir
                Draw()
      except:
                self.statuswindow.msg = msg_FrameDirectoryNotCreated + self.framedir
                Draw()
      if obj.getType() == "Mesh":
                meshdata = obj.getData()
                mesh = Blender.NMesh.GetRaw(meshdata.name)
                gridfilename = os.path.join(self.path, self.filename, zfill(str(self.curframe),10), mesh.name +".pnt")
                gridfile = open(gridfilename, "w")
                # get transformation matrix
                matrix = obj.getMatrix('worldspace')
                for vertex in mesh.verts:
                  # transform vertex
                  vert = applyTransformation4(vertex.co, matrix)
                  norm = applyTransformation4(vertex.no, matrix)
                  norm.normalize()
                  gridfile.write("%s %s %s %s %s %s \n" %
                                ( vert[0] ,
                                  vert[1],
                                  vert[2],
                                  norm[0],
                                  norm[1],
                                  norm[2],))
                gridfile.close()
                #status message
                self.statuswindow.msg = msg_SelectedGridExported


    def exportLuminairesPositionsOBJ(self, objects):
      registered_data = Blender.Registry.GetKey('brad')
      if registered_data:
        brad_path  = registered_data['brad_path']
        convert_path  = registered_data['convert_path']
        debug  = registered_data['debug']
      self.curframe = Blender.Get("curframe")
      self.framedir = os.path.join(self.path, self.filename, zfill(str(self.curframe),10))
      try:
                os.mkdir(self.framedir)
                self.statuswindow.msg = msg_FrameDirectoryCreated + self.framedir
                Draw()
      except:
                self.statuswindow.msg = msg_FrameDirectoryNotCreated + self.framedir
                Draw()
      lumfilename = os.path.join(self.path, self.filename, zfill(str(self.curframe),10), self.filename +"_lum.obj")
      lumfile = open(lumfilename, "w")
      objsize = .1
      objcount = 0
      for obj in objects:
          #name = obj.name
          if obj.getType() == "Lamp":
            lamp = obj.getData()
            # luminaire from library (if it has the same name of a library luminaire)
            if self.debug:
              print "exporting luminaire from library", lamp.name
            x = obj.LocX
            y = obj.LocY
            z = obj.LocZ
            lumfile.write("v %f %f %f\n" % (x-objsize, y-objsize, z))
            lumfile.write("v %f %f %f\n" % (x+objsize, y-objsize, z))
            lumfile.write("v %f %f %f\n" % (x+objsize, y+objsize, z))
            lumfile.write("v %f %f %f\n" % (x-objsize, y+objsize, z))
            lumfile.write("f " + str(4*objcount+1) + "// " + str(4*objcount+2) + "// " + str(4*objcount+3) + "// " + str(4*objcount+4) + "//\n")
            """
            lumfile.write("!xform -rx %s -ry %s -rz %s -t %s %s %s %s\n" %
                              (180 * obj.RotX / math.pi,
                                180 * obj.RotY / math.pi,
                                180 * obj.RotZ / math.pi,
                                obj.LocX, obj.LocY, obj.LocZ,
                                "%s.rad" % lamp.name))
            """
            objcount += 1
      lumfile.close()
      #status message
      self.statuswindow.msg = msg_LuminairePositionExported

    def exportLuminairesPositionsDXF(self, objects):
      self.curframe = Blender.Get("curframe")
      self.framedir = os.path.join(self.path, self.filename, zfill(str(self.curframe),10))
      try:
                os.mkdir(self.framedir)
                self.statuswindow.msg = msg_FrameDirectoryCreated + self.framedir
                Draw()
      except:
                self.statuswindow.msg = msg_FrameDirectoryNotCreated + self.framedir
                Draw()
      lumfilename = os.path.join(self.path, self.filename, zfill(str(self.curframe),10), self.filename +"_lum.dxf")
      lumfile = open(lumfilename, "w")

      deltaX     = 0
      deltaY     = 0
      deltaZ     = 0
      deltaPhi   = 0
      deltaTheta = 0

      csvfilename = os.path.join(self.path, self.filename, zfill(str(self.curframe),10), self.filename +"_lum.csv")
      csvfile = open(csvfilename, "w")

      # write CSV header
      csvfile.write("number,type,id,abs_x,abs_y,abs_z,i,j,k,abs_phi,abs_theta,rel_x,rel_y,rel_z,rel_phi,rel_theta,lamp_type,power,luminous_flux,cost\n")

      # write DXF comment
      lumfile.write("999\nDXF created from [brad]\n")
      # write DXF header
      lumfile.write("0\nSECTION\n")
      lumfile.write("2\nHEADER\n")
      lumfile.write("9\n$ACADVER\n1\nAC1006\n")
      lumfile.write("9\n$LIMMIN\n 10\n-5.000000\n 20\n-5.000000\n")
      lumfile.write("9\n$LIMMAX\n 10\n5.000000\n 20\n5.000000\n")
      lumfile.write("9\n$EXTMIN\n 10\n-5.000000\n 20\n-5.000000\n")
      lumfile.write("9\n$EXTMAX\n 10\n5.000000\n 20\n5.000000\n")
      lumfile.write("9\n$ANGBASE\n 50\n0.000000\n")
      lumfile.write("9\n$ANGDIR\n 70\n0\n")
      lumfile.write("9\n$INSUNITS\n 70\n5\n")
      lumfile.write("9\n$DIMALT\n 70\n 5\n")
      lumfile.write("9\n$DIMLUNIT\n 70\n2\n")
      lumfile.write("9\n$DIMASZ\n 40\n3.000000\n")
      lumfile.write("9\n$DIMTXT\n 40\n5.000000\n")
      lumfile.write("9\n$DIMRND\n 40\n0.010000\n")
      lumfile.write("9\n$DIMEXE\n 40\n2.000000\n")
      lumfile.write("9\n$DIMEXO\n 40\n2.000000\n")
      lumfile.write("9\n$DIMAUNIT\n 70\n 0\n")
      lumfile.write("9\n$DIMADEC\n 70\n 2\n")
      lumfile.write("9\n$GRIDUNIT\n 10\n0.001000\n 20\n0.001000\n")
      lumfile.write("9\n$PLIMMIN\n 10\n0.000000\n 20\n0.000000\n")
      lumfile.write("9\n$PLIMMAX\n 10\n21.000000\n 20\n29.700000\n")
      lumfile.write("9\n$PSVPSCALE\n 40\n1.000000\n")
      lumfile.write("0\nENDSEC\n")
      # write DXF tables
      lumfile.write("0\nSECTION\n")
      lumfile.write("2\nTABLES\n")
      lumfile.write("0\nTABLE\n  2\nLTYPE\n 70\n     1\n  0\nLTYPE\n  2\nCONTINUOUS\n 70\n    64\n  3\nAusgez\n 72\n    65\n 73\n  0\n 40\n0.000000\n  0\nENDTAB\n")
      lumfile.write("0\nTABLE\n  2\nLAYER\n 70\n     1\n  0\nLAYER\n  2\nLuminaires\n 70\n    64\n 62\n7\n  6\nCONTINUOUS\n  0\nENDTAB\n")
      lumfile.write("0\nENDSEC\n")
      # write DXF blocks
      lumfile.write("0\nSECTION\n")
      lumfile.write("2\nBLOCKS\n")
      lumfile.write("0\nENDSEC\n")
      # write DXF entities
      lumfile.write("0\nSECTION\n")
      lumfile.write("2\nENTITIES\n")

      objsize = 1
      objcount = 0
      for object in objects:
          objsize = .45
          #name = obj.name
          if object.getType() == "Lamp":
            lamp = object.getData()
            name_id = object.name
            # luminaire from library (if it has the same name of a library luminaire)
            #print "exporting luminaire position from library", lamp.name
            x = object.LocX
            y = object.LocY
            z = object.LocZ

            # hour line
            x1 = object.LocX
            y1 = object.LocY
            z1 = object.LocZ - 5*objsize
            """
            xa1 = object.LocX
            ya1 = object.LocY + objsize
            za1 = object.LocZ - 4*objsize
            xb1 = object.LocX
            yb1 = object.LocY - objsize
            zb1 = object.LocZ - 4*objsize
            """

            # rotation matrix: rotation around x
            matx = Matrix(3, 3)
            cosx = math.cos(object.RotX)
            sinx = math.sin(object.RotX)
            matx[0] = [1,    0,     0]
            matx[1] = [0, cosx, -sinx]
            matx[2] = [0, sinx,  cosx]
            # rotation matrix: rotation around y
            maty = Matrix(3, 3)
            cosy = math.cos(object.RotY)
            siny = math.sin(object.RotY)
            maty[0] = [ cosy, 0, siny]
            maty[1] = [    0, 1,    0]
            maty[2] = [-siny, 0, cosy]
            # rotation matrix: rotation around z
            matz = Matrix(3, 3)
            cosz = math.cos(object.RotZ)
            sinz = math.sin(object.RotZ)
            matz[0] = [cosz, -sinz, 0]
            matz[1] = [sinz,  cosz, 0]
            matz[2] = [   0,     0, 1]
            # rotation matrix
            mat = matz * maty * matx
            # rotate vertex

            v = Matrix(3, 1)
            v[0] = [0]
            v[1] = [0]
            v[2] = [-1]
            vrot = mat * v

            """
            # calculate rotated line

            zvec = Matrix(3, 1)
            zvec[0] = [ 0]
            zvec[1] = [ 0]
            zvec[2] = [-1]
            vd = mat * zvec

            zavec = Matrix(3, 1)
            zavec[0] = [ 0]
            zavec[1] = [ math.sin(math.atan(1/4))]
            zavec[2] = [-math.cos(math.atan(1/4))]
            vda = mat * zavec

            zbvec = Matrix(3, 1)
            zbvec[0] = [ 0]
            zbvec[1] = [-math.sin(math.atan(1/4))]
            zbvec[2] = [-math.cos(math.atan(1/4))]
            vdb = mat * zbvec
            """

            """
            x2 = x1 + 5*vd[0][0]
            y2 = y1 + 5*vd[1][0]
            z2 = z1 + 5*vd[2][0]

            xa2 = xa1 + 5*vda[0][0]
            ya2 = ya1 + 5*vda[1][0]
            za2 = za1 + 5*vda[2][0]
            xb2 = xb1 + 5*vdb[0][0]
            yb2 = yb1 + 5*vdb[1][0]
            zb2 = zb1 + 5*vdb[2][0]
            """

            """
            xa = x + vrot[0][0]*1.5*objsize
            ya = y + vrot[1][0]*1.5*objsize

            xb = x + vrot[0][0]*3*objsize
            yb = y + vrot[1][0]*3*objsize
            """

            theta = math.asin(vrot[2][0])
            phi   = math.asin(vrot[1][0]/math.cos(theta))

            theta_degrees = 180.0*theta/math.pi
            phi_degrees   = 180.0*phi/math.pi

            # first quadrant
            xa = x + math.cos(phi)*objsize
            ya = y + math.sin(phi)*objsize

            xb = x + math.cos(phi)*2*objsize
            yb = y + math.sin(phi)*2*objsize

            # second quadrant
            if vrot[0][0] < 0 and vrot[1][0] > 0:
                xa = x - math.cos(phi)*objsize
                xb = x - math.cos(phi)*2*objsize
                phi_degrees = 180.0 - math.fabs(phi_degrees)

            # third quadrant
            if vrot[0][0] < 0 and vrot[1][0] < 0:
                xa = x - math.cos(phi)*objsize
                xb = x - math.cos(phi)*2*objsize
                phi_degrees = 180.0 + math.fabs(phi_degrees)

            # fourth quadrant
            if vrot[0][0] > 0 and vrot[1][0] < 0:
                phi_degrees = 360.0 - math.fabs(phi_degrees)


            # dxf lines

            lumfile.write("0\nLINE\n")
            lumfile.write("8\nLuminaires\n")
            lumfile.write("6\nCONTINUOUS\n")
            lumfile.write("10\n"+str(xa)+"\n 20\n"+str(ya)+"\n 30\n"+str(z)+"\n 39\n0\n 62\n7\n")
            lumfile.write("11\n"+str(xb)+"\n 21\n"+str(yb)+"\n 31\n"+str(z)+"\n")

            """
            lumfile.write("0\nLINE\n")
            lumfile.write("8\nLuminaires\n")
            lumfile.write("6\nCONTINUOUS\n")
            lumfile.write("10\n"+str(x)+"\n 20\n"+str(y)+"\n 30\n"+str(z)+"\n 39\n0\n 62\n7\n")
            lumfile.write("11\n"+str(x2)+"\n 21\n"+str(y2)+"\n 31\n"+str(z2)+"\n")
            lumfile.write("0\nLINE\n")
            lumfile.write("8\nLuminaires\n")
            lumfile.write("6\nCONTINUOUS\n")
            lumfile.write("10\n"+str(xa2)+"\n 20\n"+str(ya2)+"\n 30\n"+str(za2)+"\n 39\n0\n 62\n7\n")
            lumfile.write("11\n"+str(x2)+"\n 21\n"+str(y2)+"\n 31\n"+str(z2)+"\n")
            lumfile.write("0\nLINE\n")
            lumfile.write("8\nLuminaires\n")
            lumfile.write("6\nCONTINUOUS\n")
            lumfile.write("10\n"+str(xb2)+"\n 20\n"+str(yb2)+"\n 30\n"+str(zb2)+"\n 39\n0\n 62\n7\n")
            lumfile.write("11\n"+str(x2)+"\n 21\n"+str(y2)+"\n 31\n"+str(z2)+"\n")
            """
            # dxf circle
            lumfile.write("0\nCIRCLE\n")
            lumfile.write("8\nLuminaires\n")
            lumfile.write("6\nCONTINUOUS\n")
            lumfile.write("10\n"+str(x)+"\n 20\n"+str(y)+"\n 30\n"+str(z)+"\n 40\n"+str(objsize)+"\n 39\n0\n 62\n7\n")
            # dxf text label
            lumfile.write("0\nTEXT\n")
            lumfile.write("8\nLuminaires\n")
            lumfile.write("6\nCONTINUOUS\n")
            lumfile.write("10\n"+str(x-objsize/1.5)+"\n 20\n"+str(y-objsize/1.5)+"\n 30\n"+str(z)+"\n 40\n"+str(objsize/1.5)+"\n 39\n0\n 62\n7\n")
            lumfile.write("11\n"+str(x+objsize/1.5)+"\n 21\n"+str(y+objsize/1.5)+"\n 31\n"+str(z)+"\n")
            lumfile.write("1\n"+str(lamp.name)+str(objcount+1)+"\n 72\n5\n 30\n"+str(z)+"\n 73\n0\n")

            csvfile.write(str(objcount+1)+","+str(lamp.name)+","+str(name_id)+","+str(x)+","+str(y)+","+str(z)+","+str(vrot[0][0])+","+str(vrot[1][0])+","+str(vrot[2][0])+","+str(phi_degrees)+","+str(theta_degrees)+",rel_x,rel_y,"+str(z+deltaZ)+","+str(phi_degrees+deltaPhi)+","+str(theta_degrees+deltaTheta)+",lamp_type,power,luminous_flux,cost\n")

            """
            lumfile.write("v %f %f %f\n" % (x-objsize, y-objsize, z))
            lumfile.write("v %f %f %f\n" % (x+objsize, y-objsize, z))
            lumfile.write("v %f %f %f\n" % (x+objsize, y+objsize, z))
            lumfile.write("v %f %f %f\n" % (x-objsize, y+objsize, z))
            lumfile.write("f " + str(4*objcount+1) + "// " + str(4*objcount+2) + "// " + str(4*objcount+3) + "// " + str(4*objcount+4) + "//\n")
            lumfile.write("!xform -rx %s -ry %s -rz %s -t %s %s %s %s\n" %
                              (180 * obj.RotX / math.pi,
                                180 * obj.RotY / math.pi,
                                180 * obj.RotZ / math.pi,
                                obj.LocX, obj.LocY, obj.LocZ,
                                "%s.rad" % lamp.name))
            """
            objcount += 1
      lumfile.write("0\nENDSEC\n")
      lumfile.write("0\nEOF\n")
      lumfile.close()

      csvfile.close()
      #status message
      self.statuswindow.msg = msg_LuminairePositionExported

# class Radiance: end

# supplementary functions: begin

def applyTransformation4(vertex, matrix):
        vertCopy = Blender.Mathutils.CopyVec(vertex)
        vertCopy.resize4D()
        return Blender.Mathutils.VecMultMat(vertCopy, matrix)

def applyTransformation3(vertex, matrix):
        vertCopy = Blender.Mathutils.CopyVec(vertex)
        vertCopy.resize3D()
        return Blender.Mathutils.VecMultMat(vertCopy, matrix)

def sameVector3(vec1, vec2):
        if vec1[0]==vec2[0] and vec1[1]==vec2[1] and vec1[2]==vec2[2]:
          return True
        else:
          return False


def getBBox(object, parwindow, statuswindow):
  if object:
      if object.getType() == "Mesh":
        meshdata = object.getData()
        mesh = Blender.NMesh.GetRaw(meshdata.name)
        parwindow.ZoneMinX.val=0.0
        parwindow.ZoneMaxX.val=0.0
        parwindow.ZoneMinY.val=0.0
        parwindow.ZoneMaxY.val=0.0
        parwindow.ZoneMinZ.val=0.0
        parwindow.ZoneMaxZ.val=0.0
        if mesh:
            matrix = object.getMatrix('worldspace')
            # check vertices
            for vertex in mesh.verts:
              # transform vertex
              vert = applyTransformation4(vertex.co, matrix)

              # check vertex
              if vert[0] < parwindow.ZoneMinX.val:
                  parwindow.ZoneMinX.val = vert[0]
              if vert[0] > parwindow.ZoneMaxX.val:
                  parwindow.ZoneMaxX.val = vert[0]
              if vert[1] < parwindow.ZoneMinY.val:
                  parwindow.ZoneMinY.val = vert[1]
              if vert[1] > parwindow.ZoneMaxY.val:
                  parwindow.ZoneMaxY.val = vert[1]
              if vert[2] < parwindow.ZoneMinZ.val:
                  parwindow.ZoneMinZ.val = vert[2]
              if vert[2] > parwindow.ZoneMaxZ.val:
                  parwindow.ZoneMaxZ.val = vert[2]

            statuswindow.msg=msg_SetBBox
      else:
        statuswindow.msg=msg_NoSelectedMesh

"""
def getBBoxOld(object, parwindow, statuswindow):
  if object:
      if object.getType() == "Mesh":
        meshdata = object.getData()
        mesh = Blender.NMesh.GetRaw(meshdata.name)
        parwindow.ZoneMinX.val=0.0
        parwindow.ZoneMaxX.val=0.0
        parwindow.ZoneMinY.val=0.0
        parwindow.ZoneMaxY.val=0.0
        parwindow.ZoneMinZ.val=0.0
        parwindow.ZoneMaxZ.val=0.0
        if mesh:
            # rotation matrix: rotation around x
            matx = Matrix(3, 3)
            cosx = math.cos(object.RotX)
            sinx = math.sin(object.RotX)
            matx[0] = [1,    0,     0]
            matx[1] = [0, cosx, -sinx]
            matx[2] = [0, sinx,  cosx]
            # rotation matrix: rotation around y
            maty = Matrix(3, 3)
            cosy = math.cos(object.RotY)
            siny = math.sin(object.RotY)
            maty[0] = [ cosy, 0, siny]
            maty[1] = [    0, 1,    0]
            maty[2] = [-siny, 0, cosy]
            # rotation matrix: rotation around z
            matz = Matrix(3, 3)
            cosz = math.cos(object.RotZ)
            sinz = math.sin(object.RotZ)
            matz[0] = [cosz, -sinz, 0]
            matz[1] = [sinz,  cosz, 0]
            matz[2] = [   0,     0, 1]
            # rotation matrix
            mat = matz * maty * matx

            # check vertices
            for vertex in mesh.verts:
              # rotate vertex

              v = Matrix(3, 1)
              v[0] = [vertex.co[0]*object.SizeX]
              v[1] = [vertex.co[1]*object.SizeY]
              v[2] = [vertex.co[2]*object.SizeZ]
              vrot = mat * v

              # check vertex
              if vrot[0][0] < parwindow.ZoneMinX.val:
                  parwindow.ZoneMinX.val = vrot[0][0]
              if vrot[0][0] > parwindow.ZoneMaxX.val:
                  parwindow.ZoneMaxX.val = vrot[0][0]
              if vrot[1][0] < parwindow.ZoneMinY.val:
                  parwindow.ZoneMinY.val = vrot[1][0]
              if vrot[1][0] > parwindow.ZoneMaxY.val:
                  parwindow.ZoneMaxY.val = vrot[1][0]
              if vrot[2][0] < parwindow.ZoneMinZ.val:
                  parwindow.ZoneMinZ.val = vrot[2][0]
              if vrot[2][0] > parwindow.ZoneMaxZ.val:
                  parwindow.ZoneMaxZ.val = vrot[2][0]

            # add translation
            parwindow.ZoneMinX.val += object.LocX
            parwindow.ZoneMinY.val += object.LocY
            parwindow.ZoneMinZ.val += object.LocZ
            parwindow.ZoneMaxX.val += object.LocX
            parwindow.ZoneMaxY.val += object.LocY
            parwindow.ZoneMaxZ.val += object.LocZ
            statuswindow.msg=msg_SetBBox
      else:
        statuswindow.msg=msg_NoSelectedMesh
"""

# supplementary functions: end

if __name__ == "__main__":
    print "brad_radiance module"


