/*
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */
/***************************************************************************
 *            widgets.c
 *
 *  Sat Aug 21 23:56:24 2004
 *  Copyright  2004  Gergely Polonkai
 *  polesz@techinfo.hu
 ***************************************************************************/
/***************************************************************************
 * Widget creating, and some callbacks
 ***************************************************************************/
#include "botcommander.h"

#include <gtk/gtk.h>
#include <vte/vte.h>
#include <string.h>

#include "variables.h"
#include "functions.h"

static unsigned int menu_mode_m = 0;
static unsigned int menu_mode_e = 1;
static unsigned int menu_mode_b = 2;

/*
 * menu_*()
 * Callback functions for the menu items.
 */
static void
menu_exit(GtkWidget *widget, gpointer data)
{
	exit_cleanly(TRUE);
}

static void
menu_change_mode(GtkWidget *item, gpointer data)
{
	tab_data_t *tab;
	static char modes[4] = "MEB";
  unsigned int which = *((unsigned int *)data);
	
	if ((tab = get_active_tab()) == NULL)
		return;
	
	if ((GTK_CHECK_MENU_ITEM(item)->active) && (tab->current_mode != modes[which]))
		change_mode(tab, TRUE, modes[which]);
}

static void
menu_settings(GtkWidget *item, gpointer data)
{
	open_prefs();
}

static void
menu_new_tab(GtkWidget *item, gpointer data)
{
	add_tab();
}

static void
menu_close_tab(GtkWidget *item, gpointer data)
{
	tab_data_t *tab;
	
	if ((tab = get_active_tab()) != NULL)
		remove_tab(tab);
}

static void
menu_botlist(GtkWidget *item, gpointer data)
{
	open_botlist();
}

/*
 * create_menu()
 * Creates the main menubar.
 */
GtkWidget *
create_menu(void)
{
	GtkWidget *menubar,
	          *submenu,
	          *menuitem;
	GSList *group = NULL;
	
	menubar = gtk_menu_bar_new();
	
	submenu = gtk_menu_new();
	
	menuitem = gtk_menu_item_new_with_mnemonic("E_xit");
	gtk_menu_shell_append(GTK_MENU_SHELL(submenu), menuitem);
	gtk_signal_connect(GTK_OBJECT(menuitem), "activate", GTK_SIGNAL_FUNC(menu_exit), NULL);
	gtk_widget_show(menuitem);
	
	menuitem = gtk_menu_item_new_with_mnemonic("_Bot");
	gtk_menu_item_set_submenu(GTK_MENU_ITEM(menuitem), submenu);
	gtk_menu_shell_append(GTK_MENU_SHELL(menubar), menuitem);
	gtk_widget_show(menuitem);
	
	submenu = gtk_menu_new();
	
	menuitem = gtk_radio_menu_item_new_with_mnemonic(group, "_Message mode");
	group = gtk_radio_menu_item_get_group(GTK_RADIO_MENU_ITEM(menuitem));
	gtk_menu_shell_append(GTK_MENU_SHELL(submenu), menuitem);
	gtk_signal_connect(GTK_OBJECT(menuitem), "toggled", GTK_SIGNAL_FUNC(menu_change_mode), &menu_mode_m);
	gtk_widget_show(menuitem);
	static_data.menu_mode_m = menuitem;
	
	menuitem = gtk_radio_menu_item_new_with_mnemonic(group, "_Eggdrop command mode");
	group = gtk_radio_menu_item_get_group(GTK_RADIO_MENU_ITEM(menuitem));
	gtk_menu_shell_append(GTK_MENU_SHELL(submenu), menuitem);
	gtk_signal_connect(GTK_OBJECT(menuitem), "toggled", GTK_SIGNAL_FUNC(menu_change_mode), &menu_mode_e);
	gtk_widget_show(menuitem);
	static_data.menu_mode_e = menuitem;
	
	menuitem = gtk_radio_menu_item_new_with_mnemonic(group, "_BotCommander command mode");
	group = gtk_radio_menu_item_get_group(GTK_RADIO_MENU_ITEM(menuitem));
	gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(menuitem), TRUE);
	gtk_menu_shell_append(GTK_MENU_SHELL(submenu), menuitem);
	gtk_signal_connect(GTK_OBJECT(menuitem), "toggled", GTK_SIGNAL_FUNC(menu_change_mode), &menu_mode_b);
	gtk_widget_show(menuitem);
	static_data.menu_mode_b = menuitem;
	
	menuitem = gtk_menu_item_new_with_mnemonic("_Mode");
	gtk_menu_item_set_submenu(GTK_MENU_ITEM(menuitem), submenu);
	gtk_menu_shell_append(GTK_MENU_SHELL(menubar), menuitem);
	gtk_widget_show(menuitem);
	
	submenu = gtk_menu_new();
	
	menuitem = gtk_menu_item_new_with_mnemonic("_Bot list");
	gtk_menu_shell_append(GTK_MENU_SHELL(submenu), menuitem);
	gtk_signal_connect(GTK_OBJECT(menuitem), "activate", GTK_SIGNAL_FUNC(menu_botlist), NULL);
	gtk_widget_show(menuitem);
	
	menuitem = gtk_menu_item_new_with_mnemonic("_Preferences");
	gtk_menu_shell_append(GTK_MENU_SHELL(submenu), menuitem);
	gtk_signal_connect(GTK_OBJECT(menuitem), "activate", GTK_SIGNAL_FUNC(menu_settings), NULL);
	gtk_widget_show(menuitem);
	
	menuitem = gtk_menu_item_new_with_mnemonic("_Setup");
	gtk_menu_item_set_submenu(GTK_MENU_ITEM(menuitem), submenu);
	gtk_menu_shell_append(GTK_MENU_SHELL(menubar), menuitem);
	gtk_widget_show(menuitem);
	
	submenu = gtk_menu_new();
	
	menuitem = gtk_menu_item_new_with_mnemonic("_New tab");
	gtk_menu_shell_append(GTK_MENU_SHELL(submenu), menuitem);
	gtk_signal_connect(GTK_OBJECT(menuitem), "activate", GTK_SIGNAL_FUNC(menu_new_tab), NULL);
	gtk_widget_show(menuitem);
	
	menuitem = gtk_menu_item_new_with_mnemonic("_Close tab");
	gtk_menu_shell_append(GTK_MENU_SHELL(submenu), menuitem);
	gtk_signal_connect(GTK_OBJECT(menuitem), "activate", GTK_SIGNAL_FUNC(menu_close_tab), NULL);
	gtk_widget_show(menuitem);
	
	menuitem = gtk_menu_item_new_with_mnemonic("_Tab");
	gtk_menu_item_set_submenu(GTK_MENU_ITEM(menuitem), submenu);
	gtk_menu_shell_append(GTK_MENU_SHELL(menubar), menuitem);
	gtk_widget_show(menuitem);
	
	submenu = gtk_menu_new();
	
	menuitem = gtk_menu_item_new_with_mnemonic("_About");
	gtk_menu_shell_append(GTK_MENU_SHELL(submenu), menuitem);
	gtk_widget_show(menuitem);

	menuitem = gtk_menu_item_new_with_mnemonic("_Help");
	gtk_menu_item_set_submenu(GTK_MENU_ITEM(menuitem), submenu);
	gtk_menu_shell_append(GTK_MENU_SHELL(menubar), menuitem);
	gtk_menu_item_right_justify(GTK_MENU_ITEM(menuitem));
	gtk_widget_show(menuitem);

	return menubar;
}

/*
 * toolbar_*
 * Toolbar callbacks
 */
static void
toolbar_quit(GtkButton *button, gpointer data)
{
	exit_cleanly(FALSE);
}

static void
book_change_tab(GtkNotebook *book, GtkNotebookPage *newpage, guint pagenum, gpointer data)
{
	tab_data_t *tab = get_active_tab();
	
	/* TODO */
	if (tab)
	{
		char buf[8];

		/* TODO: Those menu items are stored statically, and it is not good */
		gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM((tab->current_mode == 'B') ? static_data.menu_mode_b : ((tab->current_mode == 'E') ? static_data.menu_mode_e : static_data.menu_mode_m)), TRUE);
		sprintf((char *)&buf, "Mode: %c", tab->current_mode);
		gtk_label_set_text(GTK_LABEL(static_data.main_statuslabel_mode), (const char *)&buf);
		set_title(NULL, "%s", gtk_label_get_text(GTK_LABEL(tab->label)));
	}
}

/*
 * create_main_window()
 * Creates the main window with all of its components.
 */
gboolean
create_main_window(void)
{
	GtkWidget *frame;
	static_data.main_window = gtk_window_new(GTK_WINDOW_TOPLEVEL);
	g_signal_connect(G_OBJECT(static_data.main_window), "delete_event", G_CALLBACK(delete_event), NULL);
	g_signal_connect(G_OBJECT(static_data.main_window), "destroy", G_CALLBACK(destroy_event), NULL);
	gtk_window_set_icon_from_file(GTK_WINDOW(static_data.main_window), "../icon/bc48.png", NULL);
	set_title(NULL, "%s", package_version);
	
	static_data.main_vbox = gtk_vbox_new(FALSE, 0);
	
	static_data.main_menu = create_menu();
	gtk_box_pack_start(GTK_BOX(static_data.main_vbox), static_data.main_menu, FALSE, TRUE, 0);
	gtk_widget_show(static_data.main_menu);
	
	static_data.main_toolbar = gtk_toolbar_new();
	gtk_toolbar_insert_stock(GTK_TOOLBAR(static_data.main_toolbar), GTK_STOCK_QUIT, "Exit BotCommander", NULL, G_CALLBACK(toolbar_quit), NULL, -1);
	gtk_toolbar_set_style(GTK_TOOLBAR(static_data.main_toolbar), GTK_TOOLBAR_ICONS);
	gtk_box_pack_start(GTK_BOX(static_data.main_vbox), static_data.main_toolbar, FALSE, TRUE, 0);
	gtk_widget_show(static_data.main_toolbar);
	
	static_data.main_book = gtk_notebook_new();
	gtk_notebook_set_tab_pos(GTK_NOTEBOOK(static_data.main_book), GTK_POS_TOP);
	gtk_notebook_set_show_tabs(GTK_NOTEBOOK(static_data.main_book), TRUE);
	gtk_notebook_set_show_border(GTK_NOTEBOOK(static_data.main_book), TRUE);
	gtk_notebook_set_scrollable(GTK_NOTEBOOK(static_data.main_book), TRUE);
	gtk_box_pack_start(GTK_BOX(static_data.main_vbox), static_data.main_book, TRUE, TRUE, 0);
	gtk_signal_connect(GTK_OBJECT(static_data.main_book), "switch-page", GTK_SIGNAL_FUNC(book_change_tab), NULL);
	gtk_widget_show(static_data.main_book);
	
	/* As I want special things which a normal gtk_statusbar cannot do, I create a hbox */
	static_data.main_statusbox = gtk_hbox_new(FALSE, 0);
	
	frame = gtk_frame_new(NULL);
	gtk_frame_set_shadow_type(GTK_FRAME(frame), GTK_SHADOW_ETCHED_IN);
	static_data.main_statuslabel_message = gtk_label_new(package_version);
	gtk_label_set_justify(GTK_LABEL(static_data.main_statuslabel_message), GTK_JUSTIFY_LEFT);
	gtk_container_add(GTK_CONTAINER(frame), static_data.main_statuslabel_message);
	gtk_widget_show(frame);
	gtk_box_pack_start(GTK_BOX(static_data.main_statusbox), frame, TRUE, TRUE, 0);
	gtk_widget_show(static_data.main_statuslabel_message);
	
	frame = gtk_frame_new(NULL);
	static_data.main_statuslabel_mode = gtk_label_new("Mode: B");
	gtk_label_set_justify(GTK_LABEL(static_data.main_statuslabel_mode), GTK_JUSTIFY_LEFT);
	gtk_container_add(GTK_CONTAINER(frame), static_data.main_statuslabel_mode);
	gtk_widget_show(frame);
	gtk_box_pack_end(GTK_BOX(static_data.main_statusbox), frame, FALSE, FALSE, 0);
	gtk_widget_show(static_data.main_statuslabel_mode);
	
	gtk_box_pack_start(GTK_BOX(static_data.main_vbox), static_data.main_statusbox, FALSE, FALSE, 0);
	gtk_widget_show(static_data.main_statusbox);
	
	/* Put the vertical box into the main window */
	gtk_container_add(GTK_CONTAINER(static_data.main_window), static_data.main_vbox);
	gtk_widget_show(static_data.main_vbox);
	
	gtk_widget_show(static_data.main_window);
	
	return TRUE;
}

/*
 * feed_line_to_terminal()
 * Feeds some text to the terminal widget.
 */
void
feed_line_to_terminal(tab_data_t *tab, gboolean need_crlf, gchar *line, va_list list)
{
	gchar *string;
	gint len;
	
	len = valist_to_string(&string, line, list);

	vte_terminal_feed((VteTerminal *)tab->vte, string, strlen(string));
	if (need_crlf)
	{
		vte_terminal_feed((VteTerminal *)tab->vte, "\r\n", 2);
	}
}

/*
 * feed_crlf_to_terminal()
 * Feeds a CD LF pair to the terminal, if needed.
 */
void
feed_crlf_to_terminal(tab_data_t *tab, gboolean need_it)
{
	if (need_it)
	{
		vte_terminal_feed((VteTerminal *)tab->vte, "\r\n", 2);
	}
}

/*
 * feed_message_to_terminal()
 * Feeds some informational (un-coloured) text to the terminal, always appending
 * a line-feed.
 */
void
feed_message_to_terminal(tab_data_t *tab, gboolean need_crlf, gchar *line, ...)
{
	va_list pl;
	
	va_start(pl, line);
	feed_line_to_terminal(tab, FALSE, line, pl);
	va_end(pl);
	feed_crlf_to_terminal(tab, need_crlf);
}

/*
 * feed_error_to_terminal()
 * Feeds some error (red) text to the terminal, always appending a line-feed
 */
void
feed_error_to_terminal(tab_data_t *tab, gboolean need_crlf, gchar *line, ...)
{
	va_list pl;

	va_start(pl, line);
	feed_message_to_terminal(tab, FALSE, "\033[1;31m");
	feed_line_to_terminal(tab, FALSE, line, pl);
	va_end(pl);
	feed_message_to_terminal(tab, need_crlf, "\033[0m");
}

/*
 * feed_info_to_terminal()
 * Feeds some informational (green) text to the terminal, always appending a
 * line-feed.
 */
void
feed_info_to_terminal(tab_data_t *tab, gboolean need_crlf, gchar *line, ...)
{
	va_list pl;

	va_start(pl, line);
	feed_message_to_terminal(tab, FALSE, "\033[1;32m");
	feed_line_to_terminal(tab, FALSE, line, pl);
	va_end(pl);
	feed_message_to_terminal(tab, need_crlf, "\033[0m");
}

/*
 * set_title()
 * Sets the main window's titlebar
 */
void
set_title(tab_data_t *tab, gchar *fmt, ...)
{
	gchar tbuf[1024];
	gchar *ptbuf = (gchar *)&tbuf;
	va_list args;

	memset(ptbuf, 0, 1024);
	va_start(args, fmt);
	valist_to_string(&ptbuf, fmt, args);
	va_end(args);

	if (tab == NULL)
		gtk_window_set_title(GTK_WINDOW(static_data.main_window), (const gchar *)ptbuf);
	else
	{
		gtk_label_set_text(GTK_LABEL(tab->label), (const char *)ptbuf);
		if (get_active_tab() == tab)
			gtk_window_set_title(GTK_WINDOW(static_data.main_window), (const gchar *)ptbuf);
	}
}
