/*
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */
/***************************************************************************
 *            string.c
 *
 *  Thu Aug 26 22:29:25 2004
 *  Copyright  2004  Gergely POLONKAI
 *  polesz@techinfo.hu
 ***************************************************************************/
/***************************************************************************
 * String handling routines
 ***************************************************************************/

#include <gtk/gtk.h>
#include <string.h>
#include <ctype.h>
#include <stdio.h>
#include <sys/stat.h>
#include <stdlib.h>
#include <iconv.h>
#include <errno.h>
#include <math.h>

#include "botcommander.h"

/*
 * wrap_string()
 * Wraps a string into words. Words can be separated by the chars which seps
 * contains. If retlist is not NULL, a pointer is returned in it to the word
 * list. The return value is the number of words.
 */
signed int
wrap_string(gchar *string, gchar *seps, gchar ***retlist)
{
	gchar *a, **ret;
	int *start = NULL, *end = NULL, current_sep = 0, previous_sep = 1, wnum = 0, i, j;
	signed int cword = -1;
	
	if (string == NULL)
		return -1;
	if (*string == 0)
		return 0;

	for (a = string; *a; a++)
	{
		if (strchr(seps, *a))
			current_sep = 1;
		else
			current_sep = 0;

		if (!current_sep && previous_sep)
			wnum++;

		previous_sep = current_sep;
	}

  if (retlist == NULL)
    return wnum;
	
	if (wnum == 0)
		return 0;
	
	if ((start = g_malloc(wnum * sizeof(int))) == NULL)
    return -1;
  
	if ((end = g_malloc(wnum * sizeof(int))) == NULL)
  {
    g_free(start);
    return -1;
  }
  
  if ((ret = g_malloc((wnum + 1) * sizeof(char *))) == NULL)
  {
    g_free(end);
    g_free(start);
    return -1;
  }
	
	current_sep = 0;
	previous_sep = 1;
	
	for (a = string; *a; a++)
	{
		if (strchr(seps, *a))
			current_sep = 1;
		else
			current_sep = 0;
		
		if (previous_sep && !current_sep) {
			start[++cword] = (a - string);
      end[cword] = strlen(string);
    }
		if (!previous_sep && current_sep)
			end[cword] = (a - string);
		
		previous_sep = current_sep;
	}
	
	for (i = 0; i < wnum; i++)
	{
    if ((ret[i] = g_malloc(end[i] - start[i] + 1)) == NULL)
    {
      for (j = 0; j < i; j++)
        g_free(ret[j]);
      g_free(ret);
      g_free(end);
      g_free(start);
      return -1;
    }
    memset(ret[i], 0, end[i] - start[i] + 1);
    strncpy(ret[i], string + start[i], end[i] - start[i]);
	}

  *retlist = ret;

  return wnum;
}

/*
 * wl_free()
 * free()'s a word list created with wrap_string()
 */
void
wl_free(gchar **wl, gint num)
{
	gint i;
	
	if (!wl)
		return;

	for (i = 0; i < num; i++)
		if (wl[i])
			g_free(wl[i]);
	
	g_free(wl);
}

/*
 * str_eq()
 * Finds if two strings are the same. It can make case sensitive and
 * case-insensitive comparisons, depending on the value of cs.
 */
gboolean
str_eq(gchar *s1, gchar *s2, gboolean cs)
{
	gchar *a;

	if (!s1 && !s2)
		return TRUE;

	if (!s1 || !s2)
		return FALSE;

	if (strlen(s1) != strlen(s2))
		return FALSE;

	if (cs)
		return (strcmp(s1, s2) == 0) ? TRUE : FALSE;

	for (a = s1; *a; a++)
	{
		if (toupper(*a) != toupper(*(s2 + (a - s1))))
			return FALSE;
	}
	
	return TRUE;
}

/*
 * valist_to_string()
 * Creates a string from a va_list.
 */
signed int
valist_to_string(gchar **ret, gchar *fmt, va_list list)
{
	gchar *retval;
	int needed_len;
	
	if ((retval = g_malloc(strlen(fmt) + 1)) == NULL)
	{
		return -1;
	}
	memset(retval, 0, strlen(fmt) + 1);
	
	needed_len = vsnprintf(retval, strlen(fmt), fmt, list);
	
	if (needed_len < 0)
	{
		return -1;
	}
	
	if ((retval = g_realloc(retval, needed_len + 2)) == NULL)
	{
		return -1;
	}
	memset(retval, 0, needed_len + 2);
	
	if (vsnprintf(retval, needed_len + 1, fmt, list) < 0)
	{
		return -1;
	}
	
	if (ret)
	{
		*ret = retval;
	}
	
	return needed_len;
}

/*
 * is_numeric()
 * Returns TRUE if the string contains only numbers, FALSE otherwise.
 */
gboolean
is_numeric(gchar *str)
{
	gchar *a;
	
	if (!str)
		return FALSE;
	
	for (a = str; *a; a++)
	{
		if (!isdigit(*a))
			return FALSE;
	}
	
	return TRUE;
}

/*
 * trim()
 * Trims whitespace from the beginning and the end of the given string.
 * If white is not NULL and not empty, it contains the characters to trim.
 * Otherwise it defaults to " \t\r\n".
*/
gboolean
trim(gchar *original, gchar *ret, gchar *white)
{
	char rrwhite[] = " \t\r\n";
	char *rwhite, *a, *r;
	gboolean begun = FALSE;

	if ((original == NULL) || (ret == NULL))
		return FALSE;
	
	if (white == NULL)
		rwhite = (char *)&rrwhite;
	else
		rwhite = white;
	
	for (a = original, r = ret; *a; a++)
	{
		if (strchr(rwhite, *a) == NULL)
			begun = TRUE;
		if (begun)
			*(r++) = *a;
	}
	
	r--;
	a--;
	for (a = r; (strchr(rwhite, *a) && (a != ret)); a--);
	
	*(++a) = 0;
	
	return TRUE;
}

/*
 * read_file()
 * Reads the whole content of a file
 */
gboolean
read_file(gchar *filename, gchar **content)
{
	char *myfilename;
	FILE *fh;
	struct stat statbuf;
	
	if ((filename[0] == '~') && (filename[1] == '/'))
	{
		gchar *home;
		
		home = getenv("HOME");
		myfilename = g_malloc(strlen(home) + strlen(DEFAULT_CONFIG));
		memset(myfilename, 0, strlen(home) + strlen(DEFAULT_CONFIG));
		strcpy(myfilename, home);
		if (myfilename[strlen(myfilename) - 1] != '/')
			strcat(myfilename, "/");
		strcat(myfilename, (DEFAULT_CONFIG) + 2);
	}
	else
		myfilename = filename;
	
	if (content == NULL)
	{
		return FALSE;
	}
	
	if ((fh = fopen(myfilename, "r")) == NULL)
	{
		return FALSE;
	}
	
	if (stat(myfilename, &statbuf) != 0)
	{
		fclose(fh);
		return FALSE;
	}
	
	if ((*content = g_malloc(statbuf.st_size)) == NULL)
	{
		fclose(fh);
		return FALSE;
	}
	
	fread(*content, sizeof(gchar), statbuf.st_size, fh);
	fclose(fh);
	
	return TRUE;
}

/*
 * write_file()
 * Writes data to a file
 */
gboolean
write_file(gchar *filename, const gchar *content)
{
	FILE *fh;
	
	/* TODO: interpret ~/ in filename */
	
	if ((fh = fopen(filename, "w")) == NULL)
		return FALSE;
	
	fwrite(content, sizeof(gchar), strlen(content), fh);
	return TRUE;
}

/*
 * iconvert()
 * Converts an UTF-8 text to ISO-8859-1 and vice-verse
 */
gchar *
iconvert(gchar *source, gboolean fromutf8)
{
	/* The following code is from php source code. Maybe I will have to clean it, but it works :) */
	char *src = g_strdup(source);
	char *outbuf = NULL;
	int src_len = strlen(src);
	int nl;
	int *new_len = &nl;

	if (src && src_len)
	{
		int outlen = src_len;
    int outlenleft = src_len;
		int inlenleft = src_len;
		iconv_t ic;
		char* src_ptr = (char*)src;
		char* out_ptr = 0;

		ic = (fromutf8) ? iconv_open("UTF-8", "ISO-8859-2") : iconv_open("ISO-8859-2", "UTF-8");

		if(ic != (iconv_t)-1)
		{
			size_t st;
			outbuf = (char*)g_malloc(outlen + 1);

			if(outbuf)
			{
				out_ptr = (char*)outbuf;
				while(inlenleft)
				{
					st = iconv(ic, &src_ptr, &inlenleft, &out_ptr, &outlenleft);
					if(st == -1)
					{
            if(errno == E2BIG)
						{
              int diff = out_ptr - outbuf;
              outlen += inlenleft;
              outlenleft += inlenleft;
              outbuf = (char*)realloc(outbuf, outlen + 1);
              if(!outbuf)
							{
                break;
              }
              out_ptr = outbuf + diff;
            }
            else
						{
              free(outbuf);
              outbuf = 0;
              break;
            }
					}
				}
			}
			iconv_close(ic);
		}
		outlen -= outlenleft;

		if(new_len)
		{
			*new_len = outbuf ? outlen : 0;
		}
		if(outbuf)
		{
			outbuf[outlen] = 0;
		}
	}
	
	return outbuf;
}

gchar *
int_to_str(gint number)
{
	int n;
	gchar *retval;

	for (n = 0; number - pow(10, n) >= 0; n++);
		
	if ((retval = g_malloc(n + 2)) == NULL)
		return NULL;
	memset(retval, 0, n + 2);
	
	snprintf(retval, n + 1, "%d", number);
	
	return retval;
}
